#!/bin/bash
# Kernel Source Prep , for RedHat 7.x distros
# Offered freely, use at your own risk
# - Bob Arendt 4/02/2002 Original
# - Bob Arendt 5/18/2002 Updated for RecHat 7.3
# 
#
/bin/cat <<EOF
This will check to see if you have the corresponding RedHat kernel source
package, and setup and build the infamous modversion files that give a
kernel it's "flavor"

This script will:
  1) Verify that a kernel-source RPM corresponding to the running
         kernel has been installed
  2) link /usr/src/linux to the source dir
  3) uninstall  kernel-headers (if installed)
  4) make links /usr/include/asm and /usr/include/linux to the source dir
  5) Minor edits to the kernel Makefile (tweak EXTRAVERSION)
  6) copy the proper kernel config file to the source dir, based on "uname -m"
  7) make the kernel dependency files

*Caution* - if the kernel source is uninstalled, install the
proper kernel-headers files, or the lack of /usr/include/linux and
/usr/include/asm directories may inhibit future builds.

This script is offered freely, use at your own risk.
EOF

if [ `whoami` != root ] ; then
  echo "*------- Error -------*"
  echo "This script must be run as root"
  exit
fi

read -p "To continue: Enter (cntrl-C to abort)"

#-- Pedantic checking of installed versions
echo " "
echo "Checking version of current OS and rpm installed versions"
kful=`uname -r`
kver=`echo $kful | cut -d- -f 1`
kext=`echo $kful | cut -d- -f 2`
arch=`uname -m`

set `rpm -q --qf '%{NAME} %{VERSION} %{RELEASE}\n' kernel-source`
if [ "$1" != "kernel-source" ] ; then
  echo "*------- Error -------*"
  echo "A kernel-source RPM is not installed"
  echo "Ideally you need kernel-source-${kful}.i386.rpm, though"
  echo "a package matching the first three digits should suffice"
  echo " "
  echo "Download and install using:"
  echo "  rpm -ivh kernel-source-${kful}.i386.rpm"
  exit
fi
sver=$2
sext=$3
sful=${sver}-${sext}

echo "Running   kernel version: ${kful}"
echo "Installed source version: ${sful}  (kernel-source-${sful}.i386.rpm)"
if [ $kful = $sful ] ; then
  echo "Great! - kernel and source are an exact match"
elif [ $kver = $sver ] ; then
  echo "Okay - kernel and source are a close match - probably will work"
  echo "  Going to fake out the kernel EXTRAVERSION in"
  echo "      /usr/src/linux/Makefile"
else
  echo "*------- Error -------*"
  echo "Wrong kernel-source version"
  echo "Installed = kernel-source-${kful}.i386.rpm"
  echo "   Need   = kernel-source-${sful}.i386.rpm"
  echo " "
  echo "Download and re-install using:"
  echo "  rpm -e --justdb --nodeps kernel-source"
  echo "  rpm -ivh kernel-source-${kful}.i386.rpm"
  exit
fi

cd /usr/src
if [ ! -d linux-${sful} ] ;  then
  echo "*------- Error -------*"
  echo "The source directory that should exist from"
  echo "     kernel-source-${sful}.i386.rpm"
  echo "     = /usr/src/linux-${sful}"
  echo "doesn't exist.  Reinstall it using:"
  echo "  rpm -e --justdb --nodeps kernel-source"
  echo "  rpm -ivh kernel-source-${sful}.i386.rpm"
  exit
fi

#-- Now do the work
echo " "
echo "Creating source symlink"
/bin/rm -f linux
/bin/ln -s linux-${sful} linux
/bin/ls -l /usr/src/linux

echo " "
echo "Seeing if  kernel-headers package installed"
rp=$(rpm -qf /usr/include/linux/a.out.h)
if [[ $rp ]] ; then
  echo "Uninstalling rpm $rp"
  rpm -e --nodeps $rp
  rm -rf /usr/include/linux
fi
rp=$(rpm -qf /usr/include/asm/a.out.h)
if [[ $rp ]] ; then
  echo "Uninstalling rpm $rp"
  rpm -e --nodeps $rp
  rm -rf /usr/include/asm
fi

echo " "
echo "Creating /usr/include symlinks"
cd /usr/include
/bin/ln -snf /usr/src/linux/include/linux    linux
/bin/ln -snf /usr/src/linux/include/asm-i386 asm
ls -l /usr/include/{asm,linux}

echo "Fixing kernel-source Makefile so it has the proper EXTRAVERSION"
cd /usr/src/linux
if [ ! -e Makefile.orig ] ; then
  /bin/mv Makefile{,.orig}
fi

sed <Makefile.orig  \
  -e "/^EXTRAVERSION/s/.*/EXTRAVERSION = -$kext/"  \
  -e "/INSTALL_PATH=/s/^#//"  \
  > Makefile

echo " "
echo "------- Cleaning kernel source tree ----------"
/usr/bin/make mrproper
echo " "
echo "----------------------------------------------"
echo " Using kernel config:"
echo "   configs/kernel-${sful}-${arch}.config"
f=configs/kernel-${sful}-${arch}.config
if [[ ! -e $f ]] ; then 
   echo "no kernel config file $f"
   f=configs/kernel-${sver}-${arch}.config
   if [[ ! -e $f ]] ; then 
      echo "no kernel config file $f"
      exit
   fi
fi
echo "Using kernel config /usr/src/linux/$f"
/bin/cp $f .config 
echo " "
echo "----- Generating kernel dependency files -----"
/usr/bin/make oldconfig
/usr/bin/make dep
echo " "
echo "----------------------------------------------"
echo "Here are mods to the /usr/src/linux/Makefile :"
echo "(Original Makfile is /usr/src/linux/Makefile.orig)"
/usr/bin/diff -u Makefile.orig  Makefile
echo "----------------------------------------------"
echo "You should now be able to run the build_module script to"
echo "create the lt_modem & lt_serial driver binarys."


