#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/wait.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <stdio.h>
#include <errno.h>
#include <stdarg.h>

#include "common.h"

char *strip_cr(char *s)
{
	int len;

	assert(s);

	len = strlen(s);
	s[len - 1] = s[len - 1] == '\n' ? 0x00 : s[len - 1];

	return s;
}

int read_key(char *buf, int size)
{
	int fnval = 1;

	assert(buf);
	assert(size > 0);

	if (fgets(buf, size, stdin) == NULL) {
		fnval = 0;
		goto _return;
	}

	strip_cr(buf);

      _return:
	return fnval;
}

void msg(int verbose, const char *format, ...)
{
	assert(format != NULL);

	if (verbose) {
		va_list args;

		va_start(args, format);
		vfprintf(stdout, format, args);
		va_end(args);
	}
}

int run_cryptsetup_luksFormat(const char *block_key_cipher,
			      const char *device,
			      const char *passphrase, const int key_len)
{
	pid_t child;
	int fnval = 1, pipefd[2], child_exit;
	char *key_len_str;

	assert(block_key_cipher != NULL);
	assert(device != NULL);
	assert(passphrase != NULL);
	assert(key_len > 0);

	if (asprintf(&key_len_str, "%d", key_len * 8) == -1) {
		fprintf(stderr, "Failed to allocate memory, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return_no_free;
	}

	if (pipe(pipefd) == -1) {
		fprintf(stderr, "Failed to create pipe, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	}

	child = fork();

	if (child < 0) {
		fprintf(stderr, "Failed to fork, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	} else if (child == 0) {
		close(0);
		if (dup(pipefd[0]) == -1) {
			fprintf(stderr, "Failed to perform dup\n");
			fnval = 0;
			goto _return;
		}
		close(pipefd[0]);
		close(pipefd[1]);
		execl(CRYPTSETUP, "cryptsetup", "-s", key_len_str, "-c",
		      block_key_cipher, "luksFormat", device, NULL);
		fprintf(stderr, "Failed to execute %s, err=%s\n",
			CRYPTSETUP, strerror(errno));
		exit(EXIT_FAILURE);
	} else {
		close(pipefd[0]);
		if (write(pipefd[1], passphrase, strlen(passphrase)) == -1) {
			fprintf(stderr, "Failed to write passphrase\n");
			fnval = 0;
			goto _return;
		}
		close(pipefd[1]);
		waitpid(child, &child_exit, 0);
		fnval = !WEXITSTATUS(child_exit);
		goto _return;
	}

      _return:
	free(key_len_str);
      _return_no_free:
	return fnval;
}

int run_cryptsetup_luksUUID(const char *device, char *uuid)
{
	pid_t child;
	int pipefd[2], fnval = 1, child_exit;

	assert(device != NULL); /* size must be UUIDLEN + 1 */

	if (pipe(pipefd) == -1) {
		fprintf(stderr, "Failed to create pipe, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	}

	child = fork();

	if (child < 0) {
		fprintf(stderr, "Failed to fork, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	} else if (child == 0) {
		close(1);
		if (dup(pipefd[1]) == -1) {
			fprintf(stderr, "Failed to perform dup\n");
			fnval = 0;
			goto _return;
		}
		close(pipefd[1]);
		close(pipefd[0]);
		execl(CRYPTSETUP, "cryptsetup", "luksUUID", device, NULL);
		fprintf(stderr, "Failed to execute %s, err=%s\n",
			CRYPTSETUP, strerror(errno));
		exit(EXIT_FAILURE);
	} else {
		close(pipefd[1]);
		if (read(pipefd[0], uuid, UUIDLEN) == -1) {
			fprintf(stderr, "Failed read UUID\n");
			fnval = 0;
			goto _return;
		}
		uuid[UUIDLEN] = 0x00;
		close(pipefd[0]);
		waitpid(child, &child_exit, 0);
		fnval = !WEXITSTATUS(child_exit);
		goto _return;
	}

      _return:
	return fnval;
}

int run_cryptsetup_isLuks(const char *device)
{
	pid_t child;
	int fnval = 1, child_exit;

	assert(device != NULL);

	child = fork();

	if (child < 0) {
		fprintf(stderr, "Failed to fork, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	} else if (child == 0) {
		execl(CRYPTSETUP, "cryptsetup", "isLuks", device, NULL);
		fprintf(stderr, "Failed to execute %s, err=%s\n",
			CRYPTSETUP, strerror(errno));
		exit(EXIT_FAILURE);
	} else {
		waitpid(child, &child_exit, 0);
		fnval = !WEXITSTATUS(child_exit);
		goto _return;
	}

      _return:
	return fnval;
}

int run_cryptsetup_luksOpen(const char *prefix, const char *uuid, 
			    const char *device, const char *passphrase)
{
	pid_t child;
	int fnval = 1, pipefd[2], child_exit;
	char dmname[PATH_MAX + 1];

	assert(prefix != NULL);
	assert(uuid != NULL);
	assert(device != NULL);
	assert(passphrase != NULL);

	strncpy(dmname, prefix, sizeof dmname - strlen(dmname));
	if (strlen(device) + strlen(uuid) > PATH_MAX) {
		fprintf(stderr, "Uuid %s is too long\n", uuid);
		fnval = 0;
		goto _return;
	}
	strncat(dmname, uuid, sizeof dmname - strlen(dmname));

	if (pipe(pipefd) == -1) {
		fprintf(stderr, "Failed to create pipe, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	}

	child = fork();

	if (child < 0) {
		fprintf(stderr, "Failed to fork, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	} else if (child == 0) {
		close(0);
		if (dup(pipefd[0]) == -1) {
			fprintf(stderr, "Failed to perform dup\n");
			fnval = 0;
			goto _return;
		}
		close(pipefd[0]);
		close(pipefd[1]);
		execl(CRYPTSETUP, "cryptsetup", "luksOpen", device, dmname, 
		      NULL);
		fprintf(stderr, "Failed to execute %s, err=%s\n",
			CRYPTSETUP, strerror(errno));
		exit(EXIT_FAILURE);
	} else {
		close(pipefd[0]);
		if (write(pipefd[1], passphrase, strlen(passphrase)) == -1) {
			fprintf(stderr, "Failed to write passphrase\n");
			fnval = 0;
			goto _return;
		}
		close(pipefd[1]);
		waitpid(child, &child_exit, 0);
		fnval = !WEXITSTATUS(child_exit);
		goto _return;
	}

      _return:
	return fnval;
}

int run_cryptunsetup(const char *prefix, const char *uuid, const char *device)
{
	pid_t child;
	int fnval = 1, child_exit;
	char dmname[PATH_MAX + 1];

	assert(prefix != NULL);
	assert(device != NULL);
	assert(uuid != NULL);

	strcpy(dmname, prefix);
	if (strlen(dmname) + strlen(uuid) > PATH_MAX) {
		fprintf(stderr, "Uuid %s is too long\n", uuid);
		fnval = 0;
		goto _return;
	}
	strncat(dmname, uuid, sizeof dmname - strlen(dmname));

	child = fork();

	if (child < 0) {
		fprintf(stderr, "Failed to fork, err=%s\n",
			strerror(errno));
		fnval = 0;
		goto _return;
	} else if (child == 0) {
		execl(CRYPTSETUP, "cryptsetup", "remove", dmname, device,
		      NULL);
		fprintf(stderr, "Failed to execute %s, err=%s\n",
			CRYPTSETUP, strerror(errno));
		exit(EXIT_FAILURE);
	} else {
		waitpid(child, &child_exit, 0);
		fnval = !WEXITSTATUS(child_exit);
		goto _return;
	}

      _return:
	return fnval;
}
