
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <assert.h>
#include <limits.h>
#include <getopt.h>
#include <glib.h>
#include <mntent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <uuid/uuid.h>

#include "common.h"

static int verbose = 0;
static int dry_run = 0;
static char *fs_type = "ext2";
static char *fs_cipher = "aes";
static size_t fs_keylen = 16;
static int do_check = 0;
static char *device = NULL;
static char *volume_name = "Untitled";

static void print_usage(const int exitcode, const char *error,
			const char *more)
{
	if (error)
		assert(more);

	fprintf(stderr, "luks-setup [options] device\n\n"
		"-h, --help\n"
		"	print a list of options\n\n"
		"-v, --verbose\n"
		"	verbose display of messages\n\n"
		"-d, --dry-run\n"
		"	do not actually perform any operations on device\n\n"
		"-t, --fs-type type\n"
		"	set the filesystem type to use		[%s]\n\n"
		"-c, --fs-cipher cipher\n"
		"	cipher used to encrypt the filesystem	[%s]\n\n"
		"-l, --fs-keylen length\n"
		"	length in bits of encryption key	[%zu]\n\n"
		"-n, --volume-name name\n"
		"	set the name of the filesystem		[%s]\n\n"
		"-k, --check\n"
		"	check device for bad blocks		[%s]\n\n",
		fs_type,
		fs_cipher,
		fs_keylen * 8, volume_name, do_check ? "on" : "off");

	if (error)
		fprintf(stderr, "%s: %s\n", error, more);

	exit(exitcode);
}

/* FIXME: do all Unices have an /etc/mtab? */
static int mounted(const char *match)
{
	int mounted = 0;
	FILE *mtab;
	struct mntent *mtab_record;

	assert(match != NULL);

	if (!(mtab = fopen("/etc/mtab", "r"))) {
		fprintf(stderr,
			"Could not open /etc/mtab, assuming mounted\n");
		mounted = 1;
		goto _return;
	}
	while ((mtab_record = getmntent(mtab)) != NULL) {
		char const *mnt_fsname = mtab_record->mnt_fsname;
		if (!strcasecmp(mnt_fsname, match)) {
			mounted = 1;
			fprintf(stderr,
				"Volume %s currently mounted at %s\n",
				device, mtab_record->mnt_dir);
		}
	}
      _return:
	fclose(mtab);
	return mounted;
}

static void print_output(FILE * out, int in)
{
	char c[1];

	assert(out);
	assert(in >= 0);

	/* unbuffered I/O so we can catch progress indicators */
	while (read(in, c, 1) != 0)
		fprintf(out, "%c", *c);
}

/* This is necessary because mkfs.vfat uses -n and mkfs.ext[23] uses -L */
static const char **build_argv_vfat(const char *fs_type, const char *dmname,
				    int check)
{
	int argc = 5;
	const char **argv = (const char **) g_new0(char *, 8);

	assert(fs_type != NULL);
	assert(dmname != NULL);

	argv[0] = MKFS;
	argv[1] = "-t";
	argv[2] = fs_type;
	argv[3] = "-n";
	argv[4] = volume_name;

	if (check)
		argv[argc++] = "-c";
	
	argv[argc++] = dmname;

	return argv;
}

static const char **build_argv_ext(const char *fs_type, const char *dmname,
				   int check)
{
	int argc = 5;
	const char **argv = (const char **) g_new0(char *, 8);

	assert(fs_type != NULL);
	assert(dmname != NULL);

	argv[0] = MKFS;
	argv[1] = "-t";
	argv[2] = fs_type;
	argv[3] = "-L";
	argv[4] = volume_name;

	if (check)
		argv[argc++] = "-c";
	
	argv[argc++] = dmname;

	return argv;
}

static int run_mkfs(const char *uuid, const char *fs_type, int check)
{
	pid_t pid;
	const char **argv;
	GError *err = NULL;
	char dmname[PATH_MAX + 1];
	int fnval = 1, child_exit, pstderr = -1;

	assert(uuid != NULL);
	assert(fs_type != NULL);

	strcpy(dmname, DMDIR);
	strncat(dmname, DMCRYPT_TMP_PREFIX,
		sizeof dmname - strlen(dmname));
	if (strlen(device) + strlen(uuid) > PATH_MAX) {
		fprintf(stderr, "Uuid %s is too long\n", uuid);
		fnval = 0;
		goto _return;
	}
	strncat(dmname, uuid, sizeof dmname - strlen(dmname));

	if (!strcmp(fs_type, "ext2"))
		argv = build_argv_ext(fs_type, dmname, check);
	else if (!strcmp(fs_type, "ext3"))
		argv = build_argv_ext(fs_type, dmname, check);
	else if (!strcmp(fs_type, "vfat"))
		argv = build_argv_vfat(fs_type, dmname, check);
	else {
		fprintf(stderr, "Filesystem type %s not supported\n",
			fs_type);
		fnval = 0;
		goto _return;
	}

	if (argv == NULL) {
		fnval = 0;
		goto _return;
	}

	if (verbose) {
		int i = 0;
		printf("executing:");
		while (argv[i])
			printf(" %s", argv[i++]);
		printf("\n");
	}

	if (g_spawn_async_with_pipes
	    (NULL, (char **) argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD, NULL,
	     NULL, &pid, NULL, NULL, &pstderr, &err) == FALSE) {
		fprintf(stderr, "%s\n", err->message);
		g_error_free(err);
		fnval = 0;
		goto _return;
	}

	g_free(argv);

	print_output(stderr, pstderr);

	if (waitpid(pid, &child_exit, 0) == -1) {
		fprintf(stderr, "Error waiting for child\n");
		fnval = 0;
		goto _return;
	}

	fnval = !WEXITSTATUS(child_exit);

      _return:
	return fnval;
}

static int run_randomize(const char *device)
{
	pid_t pid;
	GError *err = NULL;
	char of[BUFSIZ + 1];
	int fnval = 1, child_exit, pstderr = -1;
	const char *argv[] = { DD, "if=/dev/urandom", "", NULL };

	assert(device);

	strcpy(of, "of=");
	if (strlen(device) + strlen(of) > BUFSIZ) {
		fprintf(stderr, "Device name %s is too long\n", device);
		fnval = 0;
		goto _return;
	}
	strcat(of, device);
	argv[2] = of;

	if (g_spawn_async_with_pipes
	    (NULL, (char **) argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD, NULL,
	     NULL, &pid, NULL, NULL, &pstderr, &err) == FALSE) {
		fprintf(stderr, "%s\n", err->message);
		g_error_free(err);
		fnval = 0;
		goto _return;
	}

	print_output(stderr, pstderr);

	if (waitpid(pid, &child_exit, 0) == -1) {
		fprintf(stderr, "Error waiting for child\n");
		fnval = 0;
		goto _return;
	}

	fnval = !WEXITSTATUS(child_exit);

      _return:
	return fnval;
}

int main(int argc, char *argv[])
{
	char passphrase[BUFSIZ + 1], uuid[UUIDLEN + 1];
	int c, opt_index = 0, status = EXIT_SUCCESS;
	struct option opts[] = {
		{"help", 0, 0, 'h'},
		{"dry-run", 0, 0, 'd'},
		{"verbose", 0, 0, 'v'},
		{"fs-type", 1, 0, 't'},
		{"fs-cipher", 1, 0, 'c'},
		{"fs-keylen", 1, 0, 'l'},
		{"volume-name", 1, 0, 'n'},
		{"check", 1, 0, 'k'},
		{0, 0, 0, 0}
	};

	while ((c =
		getopt_long(argc, argv, "hvdt:c:l:n:k:", opts, &opt_index))
	       >= 0) {
		switch (c) {
		case 'h':
			print_usage(EXIT_SUCCESS, NULL, NULL);
		case 'v':
			verbose = 1;
			break;
		case 'd':
			dry_run = 1;
			break;
		case 't':
			fs_type = optarg;
			break;
		case 'c':
			fs_cipher = optarg;
			break;
		case 'l':
			fs_keylen = atoi(optarg) / 8;
			break;
		case 'n':
			volume_name = optarg;
			break;
		case 'k':
			do_check = 1;
			break;
		default:
			print_usage(EXIT_FAILURE, NULL, NULL);
		}
	}

	if (argv[optind] == NULL)
		print_usage(EXIT_FAILURE, NULL, NULL);
	device = argv[optind];

	if (mounted(device)) {
		status = EXIT_FAILURE;
		goto _exit;
	}

	if (read_key(passphrase, BUFSIZ) == 0) {
		fprintf(stderr, "Could not read key\n");
		status = EXIT_FAILURE;
		goto _exit;
	}

	/* FIXME: slow and causes "No space left on device" */
	msg(verbose, "randomizing %s\n", device);
	if (0 && /* FIXME */run_randomize(device) == 0) {
		status = EXIT_FAILURE;
		goto _exit;
	}

	msg(verbose, "initializing LUKS on %s using %s (%d bit key)\n",
	    device, fs_cipher, fs_keylen * 8);
	if (!dry_run)
		if (run_cryptsetup_luksFormat
		    (fs_cipher, device, passphrase, fs_keylen) == 0) {
			status = EXIT_FAILURE;
			goto _exit;
		}

	msg(verbose, "checking UUID assigned to device\n");
	if (!dry_run)
		if (run_cryptsetup_luksUUID(device, uuid) == 0) {
			status = EXIT_FAILURE;
			goto _exit;
		}
	msg(verbose, "UUID is %s\n", uuid);

	msg(verbose, "setting up dmcrypt device\n");
	if (!dry_run)
		if (run_cryptsetup_luksOpen
		    (DMCRYPT_TMP_PREFIX, uuid, device, passphrase) == 0) {
			status = EXIT_FAILURE;
			goto _exit;
		}

	msg(verbose, "creating %s filesystem\n", fs_type);
	if (!dry_run)
		if (run_mkfs(uuid, fs_type, do_check) == 0) {
			status = EXIT_FAILURE;
			goto _exit;
		}

	msg(verbose, "removing temporary dm-crypt device\n");
	if (!dry_run)
		if (run_cryptunsetup(DMCRYPT_TMP_PREFIX, uuid, device) ==
		    0) {
			status = EXIT_FAILURE;
			goto _exit;
		}

	msg(verbose, "re-initializing dm-crypt device for hald\n");
	if (!dry_run) {
		if (run_cryptsetup_luksOpen
		    (DMCRYPT_PREFIX, uuid, device, passphrase) == 0) {
			status = EXIT_FAILURE;
			goto _exit;
		}
	}

      _exit:
	exit(status);
}
