/******************************************************************************
 * $Id: lwlint.c,v 1.17 2005/04/25 22:01:06 devel Exp $
 * Author:		Jim Brooks <tools@jimbrooks.org>
 * Start Date:	2003/09/28
 * Description:	C/C++ lint program.
 * License:		GNU GENERAL PUBLIC LICENSE (GPL)
 * Notes:
 ******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>

#include "lwlint.h"
#include "parser.h"

extern void Parse( char* pBuf, int bufLen, char* pFilename_ );

int debug		= 0;
int lintErrors	= 0;	/* returned as UNIX exit code */
struct disable_t disable;

/*****************************************************************************
 * Clean _some_ lint out of a C/C++ source file.
 *****************************************************************************/
void
Lint( char* pFnameInput )
{
	int		fdInput		= -1;
	char*	pBuf		= NULL;
	int		bufLen;
	off_t	offset;

	/*
	 * Open input file.
	 */
	fdInput = open( pFnameInput, O_RDONLY );
	if ( fdInput < 0 )
	{
		fprintf( stderr, PROGNAME ": couldn't open '%s'  ", pFnameInput );
		goto ERROR;
	}

	/*
	 * Allocate a buffer equal to size of input file.
	 */
	offset = lseek( fdInput, 0, SEEK_END ); /* move offset to EOF */
	if ( offset < 0 )
		goto ERROR;
	if ( offset < 20 )						/* skip very tiny files */
		goto RETURN;
	bufLen = offset;
	offset = lseek( fdInput, 0, SEEK_SET ); /* rewind file */
	if ( offset < 0 )
		goto ERROR;
	pBuf = malloc( bufLen + 10 );			/* extra for null byte */
	if ( !pBuf )
		goto ERROR;

	/*
	 * Read the entire input file.
	 */
	if ( read( fdInput, pBuf, bufLen ) != bufLen )
	{
		fprintf( stderr, PROGNAME ": couldn't read '%s'  ", pFnameInput );
		goto ERROR;
	}
	pBuf[bufLen] = 0;						/* terminate buffer with null byte */

	/*
	 * Send input buffer to parser.
	 */
	Parse( pBuf, bufLen, pFnameInput );

RETURN:
	free( pBuf );
	if ( fdInput > 0 ) close( fdInput );
	return;

ERROR:
	perror("");
	goto RETURN;
}

/*****************************************************************************
 * main.
 *****************************************************************************/
int
main( int argc, char** argv )
{
	int		i;
	int		inputFiles	= 0;	/* will there be any input files? */

	/*
	 * Initialize structs.
	 */
	bzero( &disable, sizeof(disable) );
	disable.multidecl = TRUE; /* see parser.l for why this is disabled by default */

	/*
	 * Parse args:
	 */
	for ( i = 1; i < argc; ++i )
	{
		if ( STREQ( argv[i], "-all"  ) ) /* enable all warnings */
		{
			bzero( &disable, sizeof(disable) ); /* zero as no warnings are disabled */
			continue;
		}
		if ( STREQ( argv[i], "-none"  ) ) /* disable all warnings */
		{
			memset( &disable, TRUE, sizeof(disable) ); /* yes, size is third arg */
			continue;
		}
		else if (    STREQ( argv[i], "-octal" )
                  || STREQ( argv[i], "+octal" ) )
		{
			disable.octal = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-multidecl" )
                  || STREQ( argv[i], "+multidecl" ) )
		{
			disable.multidecl = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-dangling-else" )
                  || STREQ( argv[i], "+dangling-else" )
                  || STREQ( argv[i], "-dangle-else"   )
                  || STREQ( argv[i], "+dangle-else"   ) )
		{
			disable.danglingElse = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-dangling-else-weak" )
                  || STREQ( argv[i], "+dangling-else-weak" )
                  || STREQ( argv[i], "-dangle-else-weak"   )
                  || STREQ( argv[i], "+dangle-else-weak"   ) )
		{
			disable.danglingElseWeak = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-else-if" )
                  || STREQ( argv[i], "+else-if" )
                  || STREQ( argv[i], "-elseif"  )
                  || STREQ( argv[i], "+elseif"  ) )
		{
			disable.elseIf = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-if-assign" )
                  || STREQ( argv[i], "+if-assign" ) )
		{
			disable.ifAssign = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-bit-test" )
                  || STREQ( argv[i], "+bit-test" )
                  || STREQ( argv[i], "-bit-tests")
                  || STREQ( argv[i], "+bit-tests") )
		{
			disable.bitTests = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-test-true" )
                  || STREQ( argv[i], "+test-true" ) )
		{
			disable.testTrue = (*argv[i] == '-');
			continue;
		}
		else if (    STREQ( argv[i], "-test-false" )
                  || STREQ( argv[i], "+test-false" ) )
		{
			disable.testFalse = (*argv[i] == '-');
			continue;
		}
		else if ( STREQ( argv[i], "-debug" ) )
		{
			debug = TRUE;
			continue;
		}
		else if (    STREQ( argv[i], "-h" )
                  || STREQ( argv[i], "-help" )
                  || STREQ( argv[i], "--help" ) )
		{
			goto Help;
		}
		/* (must be final case) In case the user passed an unknown/mistyped option. */
		else if ( *argv[i] == '-' )
		{
			fprintf( stderr, PROGNAME ": unrecognized arg '%s' \n", argv[i] );
			goto Help;
		}

		/*
		 * This arg is assumed to be an input file.
		 */
 		Lint( argv[i] );
		++inputFiles; /* presumably */
	}

	/*
	 * At least one input file is required.
	 */
	if ( !inputFiles ) goto Help;

	/*
	 * Done.
	 */
	exit( lintErrors ? 1 : 0 );

Help:
	fprintf( stderr, "lwlint   by Jim Brooks  http://www.jimbrooks.org/tools/ \n" );
	fprintf( stderr, "Lightweight C/C++ lint program   License: GNU GPL (no warranty) \n" );
	fprintf( stderr, "Usage: lwlint [-opts] file(s) \n" );
	fprintf( stderr, "-all               enables all warnings \n" );
	fprintf( stderr, "-none              disables all warnings \n" );
	fprintf( stderr, "For these args with '-+', '-' means to disable, '+' means to enable \n" );
	fprintf( stderr, "-+octal            warning about using octal constants \n" );
	fprintf( stderr, "-+multidecl        warning if one statement declares pointers & non-pointers\n" );
	fprintf( stderr, "-+dangle-else      warning about any kind of dangling 'else' \n" );
	fprintf( stderr, "-+dangle-else-weak warning if indentation of dangling 'else' seems OK \n" );
	fprintf( stderr, "-+else-if          warning about 'if' line inside series of else/if lines \n" );
	fprintf( stderr, "-+if-assign        warning about assignment within 'if' \n" );
	fprintf( stderr, "-+bit-test         warning about bad bit tests: if (reg&bit==mask) \n" );
	fprintf( stderr, "-+test-true/false  warning about testing true/false directly: if (f==TRUE) \n" );
	fprintf( stderr, "Args which follow override args which precede.  lwlint effectively starts with\n" );
	fprintf( stderr, "these initial args to disable less useful (nuisance) warnings: \n" );
	fprintf( stderr, "-all -multidecl \n" );
	exit(1);
}
