/*****************************************************************************
 * $Id: parser.y,v 1.23 2005/04/25 22:01:06 devel Exp $
 * Author:		Jim Brooks <tools@jimbrooks.org>
 * Start Date:	2003/09/28
 * Description:	C/C++ lint program.
 * License:		GNU GENERAL PUBLIC LICENSE (GPL)
 * Notes:
 *****************************************************************************/

/*
 * token struct ($<str>$).
 */
%union
{
   char* str;
}

/*
 * tokens
 */
%nonassoc TOKEN_LEFT_BRACE
%nonassoc TOKEN_RIGHT_BRACE
%nonassoc TOKEN_DANGLING_ELSE
%nonassoc TOKEN_BAD_ELSE_IF
%nonassoc TOKEN_IF_ASSIGN
%nonassoc TOKEN_BAD_BIT_TEST
%nonassoc TOKEN_OCTAL_RVALUE
%nonassoc TOKEN_OCTAL_ARG
%nonassoc TOKEN_PTR_NONPTR_DECL
%nonassoc TOKEN_TEST_TRUE
%nonassoc TOKEN_TEST_FALSE

/*===========================================================================*/

%{

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include "parser.h"
#include "lwlint.h"
#include "str_heap.h"

static char filename[500];
int eof = 0;
STR_HEAP strHeap = { NULL, 0, NULL };
extern char* yytext;
extern int debug;
extern int lineLex;
extern int lineLexPre;
extern int lintErrors;
extern struct disable_t disable;

extern void*
LexBufferUse( char* pBuf );
extern void
LexBufferDiscard( void* hnd );
extern int
yylex( void );
int
yyerror( char* s );

static void
Error( char* msg, int line )
{
	++lintErrors;
	fprintf( stderr, PROGNAME ": %s \n    at line %d in %s \n", msg, line, filename );
}

/*****************************************************************************
 * Analyze the identation to give either a weaker or strong warning.
 * Strong warning is given if indentation of if/else is misleading.
 * May fail to give the strong warning in rare case where programmer
 * uses different whitespace on one line but not the others.
 *
 * Eg, a weak warning shall be given as the indentation matches OK (isn't misleading):
 *
 *    if (ssize < len)
 *        if (doasyncfree)
 *            bdwrite(ebp);
 *        else
 *            bwrite(ebp);
 *****************************************************************************/
static void
DanglingElse( const char* pLexeme, int line )
{
	const char* p;
	const char*	pIf;
	const char*	pElse;
	int			charsPrecedingIf;
	int			charsPrecedingDanglingElse;		/* the second "else" */
	int			lexemeLen;

	if ( disable.danglingElse ) return;

	lexemeLen = strlen(pLexeme);
	if ( ! pLexeme || lexemeLen < 10 || lexemeLen > MAX_LEXEME_LEN )
		goto WEAK;

	/*
	 * Determine the amount of chars (whitespace) preceding the first "if".
	 * The first line should contain the first "if".
	 */
	pIf = strstr( pLexeme, "if" );
	if ( ! pIf ) goto WEAK;
	charsPrecedingIf = pIf - pLexeme;
	if ( charsPrecedingIf <= 0 ) goto WEAK;

	/*
	 * Now determine the amount of chars preceding the "else" on its line.
	 * Scan backwards until the newline char of the previous line is encountered.
	 */
	pElse = strstr( pLexeme, "else" );
	for ( p = pElse-1, charsPrecedingDanglingElse = 1;
          p > pLexeme;
          --p, ++charsPrecedingDanglingElse )
	{
		if ( *p == '\n' ) break;
		if ( charsPrecedingDanglingElse > MAX_LEXEME_LEN ) goto WEAK;
	}
	--charsPrecedingDanglingElse;

	/*
	 * Issue stronger warning if same amount of whitespace was counted.
	 */
	if ( charsPrecedingIf == charsPrecedingDanglingElse )
	{
		if ( ! disable.danglingElse )
			Error( "Dangling 'else' found (braces STRONGLY suggested).", line );
		return;
	}
	/* fall thru to WEAK */
WEAK:
	if ( ! disable.danglingElse  &&  ! disable.danglingElseWeak )
		Error( "Suggest braces around possible dangling 'else'.", line );
	return;
}

%}

%%

/*===========================================================================*/

program_s:
		program
	|	program_s  program
	;

program:
		TOKEN_LEFT_BRACE
	|	TOKEN_RIGHT_BRACE
	|	dangling_else
	|	bad_elseif
	|	if_assign
	|	bad_bit_test
	|   octal_rvalue
	|	octal_arg
	|	ptr_nonptr_decl
	|	test_true
	|	test_false
	;

dangling_else:
		TOKEN_DANGLING_ELSE {
		char* yyvalstr = $<str>$;
		DanglingElse( yyvalstr, lineLex ); }
	;

bad_elseif:
		TOKEN_BAD_ELSE_IF {
		if ( ! disable.elseIf ) Error( "Bad else/if series found.", lineLexPre ); }
	;

if_assign:
		TOKEN_IF_ASSIGN {
		if ( ! disable.ifAssign ) Error( "Assignment within 'if'.", lineLex ); }
	;

bad_bit_test:
		TOKEN_BAD_BIT_TEST {
		if ( ! disable.bitTests ) Error( "Bad bit test.", lineLex ); }
	;

octal_rvalue:
		TOKEN_OCTAL_RVALUE {
		if ( ! disable.octal ) Error( "Octal constant assigned.", lineLex ); }
	;

octal_arg:
		TOKEN_OCTAL_ARG {
		if ( ! disable.octal ) Error( "Octal constant passed as an arg.", lineLex ); }
	;

ptr_nonptr_decl:
		TOKEN_PTR_NONPTR_DECL {
		if ( ! disable.multidecl ) Error( "Pointer and non-pointer declared in single statement.", lineLex ); }
	;

test_true:
		TOKEN_TEST_TRUE {
		if ( ! disable.testTrue ) Error( "if (flag == true) could be wrong, suggest if (flag) instead .", lineLex ); }
	;

test_false:
		TOKEN_TEST_FALSE {
		if ( ! disable.testFalse ) Error( "if (flag == false) could be wrong, suggest if (! flag) instead.", lineLex ); }
	;

%%

/*===========================================================================*/

/*----------------------------------------------------------------------------
 * public functions
 *---------------------------------------------------------------------------*/

/*****************************************************************************
 * Main parse routine.
 *
 * Tell lex to use a memory buffer (rather than stdin).
 * Create a memory heap for storing patterns matched by lex.
 * A yacc action can retrieve the string using $<str>$.
 *****************************************************************************/
void
Parse( char* pBuf, int bufLen, char* pFilename_ )
{
	void*	hnd;
	int		strHeapSize = bufLen * 2; /* plenty extra to be safe */

	strcpy( filename, pFilename_ );

	/*
	 * Clear vars.
	 */
	lineLex = 1;
	eof = 0;

	StrHeapCreate( &strHeap, strHeapSize );

	hnd = LexBufferUse( pBuf );
	yyparse();
	LexBufferDiscard( hnd );

	StrHeapDelete( &strHeap );

	return;
}

/*----------------------------------------------------------------------------
 * internal functions
 *---------------------------------------------------------------------------*/

/*****************************************************************************
 * What to do if a parse error is detected in the input.
 *****************************************************************************/
int
yyerror( char* s )
{
	// This point is reached if no tokens were parsed
	// which isn't considered an error in this program.
	if ( ! eof  &&  debug )
		fprintf( stderr, PROGNAME ": (parse error) \n    at line %d in %s \n", lineLex, filename );

	return 0;
}
