/**
 * \file ControlExternal.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Asger Alstrup
 * \author John Levon
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS
 */

#include <config.h>

#include "ControlExternal.h"
#include "buffer.h"
#include "BufferView.h"
#include "gettext.h"
#include "helper_funcs.h"
#include "lyxrc.h"
#include "support/filetools.h"

#include <vector>

using std::vector;


ControlExternal::ControlExternal(LyXView & lv, Dialogs & d)
	: ControlInset<InsetExternal, InsetExternal::Params>(lv, d)
{}


InsetExternal::Params const ControlExternal::getParams(string const &)
{
	return InsetExternal::Params();
}


InsetExternal::Params const
ControlExternal::getParams(InsetExternal const & inset)
{
	return inset.params();
}


void ControlExternal::applyParamsToInset()
{
	inset()->setFromParams(params());
	bufferview()->updateInset(inset(), true);
}


void ControlExternal::editExternal()
{
	// fill the local, controller's copy of the Params struct with
	// the contents of the dialog's fields.
	view().apply();

	// Create a local copy of the inset and initialise it with this
	// params struct.
	boost::scoped_ptr<InsetExternal> ie;
	ie.reset(static_cast<InsetExternal *>(inset()->clone(*buffer())));
	ie->setFromParams(params());

	ie->editExternal();
}


void ControlExternal::viewExternal()
{
	view().apply();

	boost::scoped_ptr<InsetExternal> ie;
	ie.reset(static_cast<InsetExternal *>(inset()->clone(*buffer())));
	ie->setFromParams(params());

	ie->viewExternal();
}


void ControlExternal::updateExternal()
{
	view().apply();

	boost::scoped_ptr<InsetExternal> ie;
	ie.reset(static_cast<InsetExternal *>(inset()->clone(*buffer())));
	ie->setFromParams(params());

	ie->updateExternal();
}


vector<string> const ControlExternal::getTemplates() const
{
	vector<string> result;

	ExternalTemplateManager::Templates::const_iterator i1, i2;
	i1 = ExternalTemplateManager::get().getTemplates().begin();
	i2 = ExternalTemplateManager::get().getTemplates().end();

	for (; i1 != i2; ++i1) {
		result.push_back(i1->second.lyxName);
	}
	return result;
}


int ControlExternal::getTemplateNumber(string const & name) const
{
	ExternalTemplateManager::Templates::const_iterator i1, i2;
	i1 = ExternalTemplateManager::get().getTemplates().begin();
	i2 = ExternalTemplateManager::get().getTemplates().end();
	for (int i = 0; i1 != i2; ++i1, ++i) {
		if (i1->second.lyxName == name)
			return i;
	}

	// we can get here if a LyX document has a template not installed
	// on this machine.
	return -1;
}


ExternalTemplate ControlExternal::getTemplate(int i) const
{
	ExternalTemplateManager::Templates::const_iterator i1
		= ExternalTemplateManager::get().getTemplates().begin();

	std::advance(i1,  i);

	return i1->second;
}


string const ControlExternal::Browse(string const & input) const
{
	string const title =  _("Select external file");

	string const bufpath = buffer()->filePath();

	/// Determine the template file extension
	ExternalTemplate const & et = params().templ;
	string pattern = et.fileRegExp;
	if (pattern.empty())
		pattern = "*";

	std::pair<string, string> dir1(N_("Documents|#o#O"),
				  string(lyxrc.document_path));

	return browseRelFile(&lv_, true, input, bufpath, title, pattern, false, dir1);
}
