/*
    libMakeMKV - MKV multiplexer library

    Copyright (C) 2009-2010 GuinpinSoft inc <libmkv@makemkv.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/
#ifndef LGPL_MEMORY_H_INCLUDED
#define LGPL_MEMORY_H_INCLUDED

#include "smartptr.h"
#include "world.h"

class memory_buffer
{
private:
    memory_buffer(const memory_buffer&);
    void operator=(const memory_buffer&);
private:
    unsigned int    m_size;
    void*           m_data;
public:
    memory_buffer() 
        : m_data(NULL) , m_size(0)
    {
    };
    ~memory_buffer()
    {
        free_data();
    }
    void alloc(int size)
    {
        free_data();
        m_size=size;
#ifdef BLD_DEBUG
        m_data=lgpl_safe_alloc_dbg(size,__FILE__,__LINE__);
#else
        m_data=lgpl_safe_alloc(size);
#endif
    }
    void alloc_dbg(int size,const char *file,unsigned int line)
    {
        free_data();
        m_size=size;
#ifdef BLD_DEBUG
        m_data=lgpl_safe_alloc_dbg(size,file,line);
#else
        m_data=lgpl_safe_alloc(size);
#endif
    }
    void alloc_fake(int size)
    {
        free_data();
        m_size=size;
    }
    void reduce_size(int size)
    {
        if ( ((unsigned int)size) < m_size )
        {
            m_size=((unsigned int)size);
        }
    }
    unsigned int get_size() const
    {
        return m_size;
    }
    unsigned char* get() const
    {
        return (unsigned char*) m_data;
    }
    void move_from(memory_buffer& buf)
    {
        free_data();
        m_data = buf.m_data;
        m_size = buf.m_size;
        buf.m_data = NULL;
        buf.m_size = 0;
    }
    void free_data()
    {
        if (NULL!=m_data)
        {
#ifdef BLD_DEBUG
            lgpl_safe_free_dbg(m_data);
#else
            lgpl_safe_free(m_data);
#endif
            m_data=NULL;
        }
        m_size=0;
    }
};

class memory_c : counted_ptr_base<memory_buffer>
{
public:
    void alloc(unsigned int size)
    {
        detach();
        memory_buffer* p = base_get();
        p->alloc(size);
    }
    void alloc_fake(unsigned int size)
    {
        detach();
        memory_buffer* p = base_get();
        p->alloc_fake(size);
    }
    void reduce_size(int size)
    {
        memory_buffer* p = base_get();
        if (NULL==p) return;
        p->reduce_size(size);
    }
    void alloc_dbg(unsigned int size,const char *file,unsigned int line)
    {
        detach();
        memory_buffer* p = base_get();
        p->alloc_dbg(size,file,line);
    }
    unsigned int get_size() const
    {
        const memory_buffer* buf = base_get_const();
        if (NULL==buf) return 0;
        return buf->get_size();
    }
    unsigned char* get()
    {
        return base_get()->get();
    }
    const unsigned char* get_const() const
    {
        const memory_buffer* buf = base_get_const();
        if (NULL==buf) return NULL;
        return buf->get();
    }
    void clear()
    {
        detach();
    }
    bool can_modify_inplace() const
    {
        return (base_get_count()<=1);
    }
};

#ifdef BLD_DEBUG
#define alloc(s) alloc_dbg(s,__FILE__,__LINE__)
#endif

#endif // LGPL_MEMORY_H_INCLUDED
