/*
    libMakeMKV - MKV multiplexer library

    Copyright (C) 2009-2010 GuinpinSoft inc <libmkv@makemkv.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/
#ifndef LGPL_SMARTPTR_H_INCLUDED
#define LGPL_SMARTPTR_H_INCLUDED

#include <memory>
#include "world.h"
//
// Modified auto_ptr. Modifications include:
// 1. remove unsafe assign
//
template<class T>
class m_auto_ptr : public std::auto_ptr<T>
{
private:
    typedef std::auto_ptr<T> _parent;
    typedef T* TPTR;
public:
    m_auto_ptr() : _parent() {};
    m_auto_ptr(T* p) : _parent(p) {};
    void operator=(T* p)
    {
        reset(p);
    }
    //T& operator*() const { return std::auto_ptr<T>::operator*(); };
    //T* operator->() const { return std::auto_ptr<T>::operator->(); };
    operator TPTR() const { return _parent::get(); };
#ifdef __GNUC__
    m_auto_ptr(const _parent& s) : _parent()
    {
        const _parent* p = &s;
        reset( ((_parent*)p)->release() );
    };
#endif
};

// generic counted ptr
template <class T>
class counted_ptr_base
{
private:
    class control_area
    {
    public:
        void* operator new(size_t size)
        {
            return lgpl_safe_alloc((unsigned int)size);
        }
        void operator delete(void*p)
        {
            lgpl_safe_free(p);
        }
    public:
        int     count;
        T       value;
    };
private:
    control_area    *c;
public:
    counted_ptr_base() : c(NULL)
    {
    };
    counted_ptr_base(const counted_ptr_base &src) : c(NULL)
    {
        assign(src);
    }
    ~counted_ptr_base()
    {
        detach();
    }
    T* base_get()
    {
        denull();
        return & c->value;
    }
    T* base_get_const() const
    {
        if (c==NULL) return NULL;
        return & c->value;
    }
    int base_get_count() const
    {
        if (NULL==c) return 0;
        return c->count;
    }
    void operator=(const counted_ptr_base &src)
    {
        assign(src);
    }
protected:
    void assign(const counted_ptr_base &src)
    {
        detach();
        if (src.c!=NULL)
        {
            c = src.c;
            c->count++;
        }
    }
    void detach()
    {
        if (c!=NULL)
        {
            c->count--;
            if (c->count==0)
            {
                delete c;
            }
            c=NULL;
        }
    }
    void denull()
    {
        if (c==NULL)
        {
            c = new control_area();
            c->count=1;
        }
    }
};

// counted pointer
template<class T>
class counted_ptr : public counted_ptr_base< m_auto_ptr<T> >
{
public:
    typedef counted_ptr_base< m_auto_ptr<T> > _parent;
    typedef T*  TPTR;
public:
    counted_ptr()
    {
    }
    counted_ptr(TPTR p)
    {
        if (p!=NULL)
        {
            *(_parent::base_get()) = p;
        }
    }
    T* get() const
    {
        if (_parent::base_get_const()==NULL) return NULL;
        return _parent::base_get_const()->get();
    }
    T* release()
    {
        if (_parent::base_get_count()!=1) return NULL;
        return _parent::base_get()->release();
    }
};

#endif // LGPL_SMARTPTR_H_INCLUDED

