/*
    libMakeMKV - MKV multiplexer library

    Copyright (C) 2009-2010 GuinpinSoft inc <libmkv@makemkv.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/
#ifndef LIBMKV_H_INCLUDED
#define LIBMKV_H_INCLUDED

#include <lgpl/memory.h>
#include <string.h>

typedef char mkv_utf8_t;

#ifdef HAVE_FSOURCE_PRIV
#include <fsource/libmkvpriv.h>
#else

typedef struct _mkv_chunk_priv_pub
{
    void*       p0;
    int64_t     p1;
    int64_t     p2;
    int         p3;
    int         p4;
    int         p5;
    int         p6;
    bool        p7;
} mkv_chunk_priv_t;

static inline void mkv_chunk_priv_init(mkv_chunk_priv_t& priv)
{
    memset(&priv,0,sizeof(mkv_chunk_priv_t));
}

typedef struct _mkv_track_video_priv_pub
{
    int     t0;
    int     t1;
    bool    t2;
} mkv_track_video_priv_t;

#endif

class CMkvChunk
{
public:
    memory_c    data;
    int64_t     timecode;
    int64_t     duration;
    bool        keyframe;
    bool        cluster_start;
    bool        chapter_mark;

    // private, do not use
    mkv_chunk_priv_t    priv;
public:
    CMkvChunk()
      : timecode(-1) ,
        duration(-1) ,
        keyframe(false) ,
        cluster_start(false) ,
        chapter_mark(false) 
    {
        mkv_chunk_priv_init(priv);
    }
};

typedef enum _MkvTrackType
{
    mttUnknown,
    mttVideo,
    mttAudio,
    mttSubtitle
} MkvTrackType;

typedef struct _MkvTrackInfo
{
    MkvTrackType        type;
    const char*         codec_id;
    const char*         lang;
    const mkv_utf8_t*   name;
    const mkv_utf8_t*   type_extension;

    const unsigned char *codec_private;
    int                 codec_private_size;

    const unsigned char *header_comp_data;
    int                 header_comp_size;

    bool                use_lacing;
    int64_t             default_duration;
    int64_t             dts_adjust;

    unsigned int        bitrate;

    uint32_t            stream_flags;
    struct _temp
    {
        uint8_t     p1;
        uint8_t     p2;
    } temp;

    union _u
    {
        struct _video
        {
            int     pixel_h;
            int     pixel_v;

            int     display_h;
            int     display_v;

            int     fps_n;
            int     fps_d;

            // private, do not use
            mkv_track_video_priv_t  priv;
        } video;

        struct _audio
        {
            int     sample_rate;
            int     channels_count;
            int     bits_per_sample;

        } audio;

        struct _subtitles
        {
        } subtitles;
    } u;
} MkvTrackInfo;

class IMkvFrameSource
{
public:
    virtual unsigned int    GetAvailableFramesCount()=0;
    virtual void            PopFrame()=0;
    virtual void            PopAllFrames()=0;
    virtual bool            SourceFinished()=0;
    virtual bool            FetchFrames(unsigned int Count,bool Force=false)=0;
    virtual CMkvChunk*      PeekFrame(unsigned int Index)=0;
    virtual bool            UpdateTrackInfo(MkvTrackInfo* Info)=0;
    virtual void            GetProgress(uint64_t* Value,uint64_t* MaxValue)=0;
    virtual void            ReleaseBuffers()=0;
};

class IMkvWriteTarget
{
public:
    virtual bool            Write(const void *Data,unsigned int Size)=0;
    virtual bool            Overwrite(uint64_t Offset,const void *Data,unsigned int Size)=0;
};

class IMkvTrack
{
public:
    virtual unsigned int MkvGetStreamCount()=0;
    virtual IMkvFrameSource* MkvGetStream(unsigned int Index)=0;
    virtual bool MkvPreallocateBuffers()=0;
public:
    inline void ReleaseBuffers()
    {
        for (unsigned int i=0;i<MkvGetStreamCount();i++)
        {
            MkvGetStream(i)->ReleaseBuffers();
        }
    }
};

typedef enum _MkvChapterType
{
    mccUnknown,
    mccChapterMark,
    mccTimecode
} MkvChapterType;

static const unsigned int MKV_MAX_CHAPTER_NAMES=25;

typedef struct _MkvChapterInfo
{
    MkvChapterType      type;
    uint64_t            time_c;
    unsigned int        name_count;
    const char*         name_lang[MKV_MAX_CHAPTER_NAMES];
    const mkv_utf8_t*   name_text[MKV_MAX_CHAPTER_NAMES];
} MkvChapterInfo;

class IMkvTitleInfo
{
public:
    virtual unsigned int GetChapterCount()=0;
    virtual void GetChapterInfo(MkvChapterInfo *info,unsigned int chapter_id)=0;
};

extern "C"
bool __cdecl MkvCreateFile(IMkvWriteTarget* Output,IMkvTrack *Input,const mkv_utf8_t *WritingApp,IMkvTitleInfo* TitleInfo) throw();

#endif // LIBMKV_H_INCLUDED
