/*
    libMakeMKV - MKV multiplexer library

    Copyright (C) 2009-2010 GuinpinSoft inc <libmkv@makemkv.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/
#include <libmkv/libmkv.h>
#include <libmkv/internal.h>
#include <libmkv/ebmlwrite.h>
#include <lgpl/cassert>
#include <exception>
#include <list>
#include <lgpl/sstring.h>
#include <time.h>

#define TIMECODE_SCALE              1000000
#define MAX_TIMECODE_SIZE_BYTES     6
#define MAX_TIMECODE                ((((uint64_t)1)<<(8*MAX_TIMECODE_SIZE_BYTES))-1)


template <class Tv,class Te>
static inline Tv& GetChild(EbmlMaster *node)
{
    return * (static_cast<Tv *>(&GetChild<Te>( *node )));
}

template <class Tv,class Te>
static inline Tv& GetChild(EbmlMaster &node)
{
    return GetChild<Tv,Te>(&node);
}

static inline int64_t ScaleTimecode(int64_t UnscaledTimecode)
{
    return UnscaledTimecode / TIMECODE_SCALE;
}

static inline int64_t TimecodeFromClock(int64_t MkvClockTimecode)
{
    return ((MkvClockTimecode*25)/27);
}

template <class Te>
static inline KaxSeek* AddSeekEntry(KaxSeekHead& Seek)
{
    KaxSegment tmp_seg;
    Te& tmp_val = GetChild<Te>(tmp_seg);

    return Seek.IndexThis(tmp_val,tmp_seg);
}

static inline void UpdateSeekEntry(KaxSeek* Seek,IOCallback &File,const EbmlElement & aElt, const KaxSegment & ParentSegment)
{
    GetChild<EbmlUInteger,KaxSeekPosition>(Seek) = ParentSegment.GetRelativePosition(aElt);
    GetChild<KaxSeekPosition>(*Seek).OverwriteData(File,true);
}

static inline void VoidElement(EbmlMaster* Element,IOCallback &File)
{
    EbmlVoid v;
    v.Overwrite(*Element,File);
}

static void FinishCluster(KaxCluster *cluster,IOCallback *File,KaxSegment *Segment);

static bool AtKeyFrame(IMkvTrack *Input)
{
    bool rtn = false;
    for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
    {
        IMkvFrameSource *p;
        p = Input->MkvGetStream(i);

        if (false==p->FetchFrames(1)) return false;
        if (0==p->GetAvailableFramesCount()) continue;

        if (false==p->PeekFrame(0)->keyframe) return false;
    }

    return true;
}

static bool AtClusterStart(IMkvTrack *Input)
{
    bool rtn = false;
    for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
    {
        IMkvFrameSource *p;
        p = Input->MkvGetStream(i);

        if (false==p->FetchFrames(1)) return false;
        if (0==p->GetAvailableFramesCount()) continue;

        if (false==p->PeekFrame(0)->cluster_start) return false;
    }

    return true;
}

static int64_t GetClusterTimecode(IMkvTrack *Input)
{
    int64_t rtn = -1;
    for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
    {
        IMkvFrameSource *p;
        p = Input->MkvGetStream(i);

        if (false==p->FetchFrames(1)) return false;
        for (unsigned int j=0;j<p->GetAvailableFramesCount();j++)
        {
            int64_t ft;

            ft = p->PeekFrame(j)->timecode;

            MKV_ASSERT(ft!=-1);

            if (rtn==-1) rtn = ft;

            if (ft<rtn) rtn=ft;
        }
    }

    MKV_ASSERT(rtn!=-1);

    return rtn;
}

static UTFstring UTF8string(const char *str)
{
    UTFstring rtn;

    rtn.SetUTF8(str);

    return rtn;
}

class CChapters
{
private:
    KaxChapters         m_Chapters;
    std::vector<KaxChapterAtom*>    m_ChapterMarks;
    unsigned int        m_RealCount;
    bool                m_DoRender;
    IMkvTitleInfo*      m_TitleInfo;
public:
    CChapters(IMkvTitleInfo* TitleInfo) : m_RealCount(0) , m_TitleInfo(TitleInfo)
    {
        Init();
    }
    void Init()
    {
        MkvChapterType  ctype;
        MkvChapterInfo  ti;
        unsigned int chap_count = m_TitleInfo->GetChapterCount();

        m_DoRender=false;

        if (chap_count==0) return;

        memset(&ti,0,sizeof(ti));
        m_TitleInfo->GetChapterInfo(&ti,0);
        ctype = ti.type;

        // verify type
        for (unsigned int i=0;i<chap_count;i++)
        {
            memset(&ti,0,sizeof(ti));
            m_TitleInfo->GetChapterInfo(&ti,i);
            if (ti.type!=ctype)
            {
                throw mkv_error_exception("Mixed chapter types are not supported (yet?).");
            }
        }

        switch(ctype)
        {
        case mccChapterMark:
            ReserveChapterMarks(1+chap_count);
            SetEmptyChapterName(m_ChapterMarks[0],0);
            for (unsigned int i=0;i<chap_count;i++)
            {
                memset(&ti,0,sizeof(ti));
                m_TitleInfo->GetChapterInfo(&ti,i);
                if (ti.name_count)
                {
                    for (unsigned int j=0;j<ti.name_count;j++)
                    {
                        SetChapterName(m_ChapterMarks[i+1],ti.name_lang[j],ti.name_text[j]);
                    }
                } else {
                    SetEmptyChapterName(m_ChapterMarks[i+1],i+1);
                }
            }
            m_DoRender = (chap_count>0);
            break;
        case mccTimecode:
            {
                KaxEditionEntry &edit = CreateEdition();
                for (unsigned int i=0;i<chap_count;i++)
                {
                    memset(&ti,0,sizeof(ti));
                    m_TitleInfo->GetChapterInfo(&ti,i);

                    if (i==0)
                    {
                        if (IsVerySmallTimecode(ti.time_c))
                        {
                            ti.time_c=0;
                        } else {
                            KaxChapterAtom& first_atom = AddNewChild<KaxChapterAtom>(edit);
                            GetChild<EbmlUInteger,KaxChapterUID>( first_atom ) = 1;
                            GetChild<EbmlUInteger,KaxChapterTimeStart>( first_atom ) = 0;
                            SetEmptyChapterName(&first_atom,0);
                            m_RealCount++;
                        }
                    }

                    KaxChapterAtom& atom = AddNewChild<KaxChapterAtom>(edit);
                    GetChild<EbmlUInteger,KaxChapterUID>( atom ) = (m_RealCount+1);
                    GetChild<EbmlUInteger,KaxChapterTimeStart>( atom ) = TimecodeFromClock(ti.time_c);

                    if (ti.name_count)
                    {
                        for (unsigned int j=0;j<ti.name_count;j++)
                        {
                            SetChapterName(&atom,ti.name_lang[j],ti.name_text[j]);
                        }
                    } else {
                        SetEmptyChapterName(&atom,m_RealCount);
                    }
                    m_RealCount++;
                }
            }
            m_DoRender = (m_RealCount>1);
            break;
        default:
            throw mkv_error_exception("Unsupported chapter type");
            break;
        }
    }
    void Render(IOCallback &File,KaxSeek* Seek,const KaxSegment & FileSegment)
    {
        if (m_DoRender)
        {
            m_Chapters.Render(File,true);
            UpdateSeekEntry(Seek,File,m_Chapters,FileSegment);
        } else {
            VoidElement(Seek,File);
        }
    }
    void AddChapterMark(uint64_t Timecode,IOCallback &File)
    {
        if (!m_DoRender) return;

        MKV_ASSERT(m_RealCount < m_ChapterMarks.size() );
        MKV_ASSERT(Timecode < MAX_TIMECODE);

        if (m_RealCount==0)
        {
            MKV_ASSERT( m_ChapterMarks.size() >= 2 );

            if (IsVerySmallTimecode(Timecode))
            {
                m_ChapterMarks[0]->VoidMe(File);
                Timecode = 0;
            } else {
                GetChild<EbmlUInteger,KaxChapterTimeStart>( *(m_ChapterMarks[0]) ) = 0;
                m_ChapterMarks[0]->OverwriteData(File,true);
            }
            m_RealCount=1;
        }

        GetChild<EbmlUInteger,KaxChapterTimeStart>( *(m_ChapterMarks[m_RealCount]) ) = TimecodeFromClock(Timecode);
        m_ChapterMarks[m_RealCount]->OverwriteData(File,true);
        m_RealCount++;
    }
    void Finalize(IOCallback &File,KaxSeek* Seek)
    {
        if (!m_DoRender) return;

        if (!m_ChapterMarks.empty())
        {
            for (unsigned int i=m_RealCount;i<m_ChapterMarks.size();i++)
            {
                m_ChapterMarks[i]->VoidMe(File);
            }
        }
        if (m_RealCount<2)
        {
            m_Chapters.VoidMe(File);
            VoidElement(Seek,File);
        }
    }
    static bool IsVerySmallTimecode(uint64_t Timecode)
    {
        static const uint64_t SmallTimecode = 108000000ll;  // 0.1 sec
        return (Timecode<=SmallTimecode);
    }
private:
    void SetChapterName(KaxChapterAtom* atom,const char *lang,const mkv_utf8_t* name)
    {
        KaxChapterDisplay& disp = GetChild<KaxChapterDisplay>(*atom);
        GetChild<EbmlString,KaxChapterLanguage>(disp) = lang;
        GetChild<EbmlUnicodeString,KaxChapterString>(disp) = UTF8string(name);
    }
    void SetEmptyChapterName(KaxChapterAtom* atom,unsigned int Id)
    {
        char ChapterName[128];
        sprintf_s(ChapterName,128,"Chapter %02u",Id+0);
        SetChapterName(atom,"eng",ChapterName);
    }
    void ReserveChapterMarks(unsigned int Count)
    {
        if (0==Count) return;

        m_ChapterMarks.resize(Count);

        KaxEditionEntry &edit = CreateEdition();

        for (unsigned int i=0;i<Count;i++)
        {
            m_ChapterMarks[i] = & AddNewChild<KaxChapterAtom>(edit);
            GetChild<EbmlUInteger,KaxChapterUID>( *(m_ChapterMarks[i]) ) = (i+1);

            GetChild<EbmlUInteger,KaxChapterTimeStart>( *(m_ChapterMarks[i]) ) = 0;
            GetChild<EbmlUInteger,KaxChapterTimeStart>( *(m_ChapterMarks[i]) ).SetDefaultSize(MAX_TIMECODE_SIZE_BYTES);
        }
    }
    KaxEditionEntry& CreateEdition()
    {
        KaxEditionEntry &edit = GetChild<KaxEditionEntry>(m_Chapters);
        GetChild<EbmlUInteger,KaxEditionFlagDefault>(edit) = 1;

        uint32_t EditionUid;
        for (unsigned int i=0;i<sizeof(EditionUid);i++)
        {
            ((uint8_t*)&EditionUid)[i]=lgpl_get_random_byte();
        }
        GetChild<EbmlUInteger,KaxEditionUID>(edit) = EditionUid;

        return edit;
    }
};

static bool CompareTimecode(int64_t Timecode1,int64_t Timecode2)
{
    int64_t Diff;
    if (Timecode1>Timecode2)
    {
        Diff = Timecode1 - Timecode2;
    } else {
        Diff = Timecode2 - Timecode1;
    }
    return (Diff<3);
}

static bool IsFirstTrackOfType(IMkvTrack *Input,MkvTrackType Type,unsigned int Id)
{
    MkvTrackInfo ti;

    for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
    {
        memset(&ti,0,sizeof(ti));
        if (false==Input->MkvGetStream(i)->UpdateTrackInfo(&ti))
        {
            throw mkv_error_exception("UpdateTrackInfo failed");
        }
        if (ti.type!=Type) continue;
        return (Id==i);
    }
    return false;
}

class MyMkvTrackInfo
{
public:
    MkvTrackInfo    info;
    unsigned int    frame_count;
};

static void MkvCreateFileInternal(IOCallback &File,IMkvTrack *Input,IMkvTitleInfo* TitleInfo,const char *WritingApp)
{
    EbmlHead FileHead;

    EDocType & MyDocType = GetChild<EDocType>(FileHead);
    *static_cast<EbmlString *>(&MyDocType) = "matroska";

    EDocTypeVersion & MyDocTypeVer = GetChild<EDocTypeVersion>(FileHead);
    *(static_cast<EbmlUInteger *>(&MyDocTypeVer)) = MATROSKA_VERSION;

    EDocTypeReadVersion & MyDocTypeReadVer = GetChild<EDocTypeReadVersion>(FileHead);
    *(static_cast<EbmlUInteger *>(&MyDocTypeReadVer)) = 2;

    FileHead.Render(File);

    // objects
    KaxSegment FileSegment;

    KaxCues AllCues;
    AllCues.SetGlobalTimecodeScale(TIMECODE_SCALE);

    CChapters Chapters(TitleInfo);

    // start render
    uint64_t SegmentSize = FileSegment.WriteHead(File, 8);

    KaxSeekHead & MetaSeek = AddNewChild<KaxSeekHead>(FileSegment);
    KaxSeekHead & ClusterSeek = AddNewChild<KaxSeekHead>(FileSegment);

    KaxSeek* seek_infos = AddSeekEntry<KaxInfo>(MetaSeek);
    KaxSeek* seek_tracks = AddSeekEntry<KaxTracks>(MetaSeek);
    KaxSeek* seek_cues = AddSeekEntry<KaxCues>(MetaSeek);
    KaxSeek* seek_clust1 = AddSeekEntry<KaxCluster>(MetaSeek);
    KaxSeek* seek_chap = AddSeekEntry<KaxChapters>(MetaSeek);
    KaxSeek* seek_seek2 = AddSeekEntry<KaxSeekHead>(MetaSeek);

    MetaSeek.Render(File);

    // fill the mandatory Info section
    KaxInfo & MyInfos = GetChild<KaxInfo>(FileSegment);

    GetChild<EbmlUInteger,KaxTimecodeScale>(MyInfos) = TIMECODE_SCALE;
    GetChild<EbmlFloat,KaxDuration>(MyInfos) = (double)0;

    GetChild<EbmlUnicodeString,KaxMuxingApp>(MyInfos) = GetLibraryVersionString();
    GetChild<EbmlUnicodeString,KaxWritingApp>(MyInfos) = UTF8string(WritingApp);

    uint8_t SegmentUid[16];
    for (unsigned int i=0;i<16;i++)
    {
        SegmentUid[i] = lgpl_get_random_byte();
    }
    GetChild<EbmlBinary,KaxSegmentUID>(MyInfos).CopyBuffer(SegmentUid, 16);
    GetChild<EbmlDate,KaxDateUTC>(MyInfos).SetEpochDate((uint32_t)time(NULL));


    MyInfos.Render(File,true);


    // tracks
    KaxTracks & MyTracks = GetChild<KaxTracks>(FileSegment);

    std::vector<MyMkvTrackInfo> track_info;
    std::vector<KaxTrackEntry*> tracks;

    track_info.resize(Input->MkvGetStreamCount());
    tracks.resize(Input->MkvGetStreamCount());

    for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
    {
        MkvTrackInfo *ti;
        KaxTrackEntry *cur_track;

        ti = & track_info[i].info;
        track_info[i].frame_count = 0;

        memset(ti,0,sizeof(*ti));
        if (false==Input->MkvGetStream(i)->UpdateTrackInfo(ti))
        {
            throw mkv_error_exception("UpdateTrackInfo failed");
        }

        cur_track = & AddNewChild<KaxTrackEntry>(MyTracks);
        cur_track->SetGlobalTimecodeScale(TIMECODE_SCALE);

        tracks[i] = cur_track;

        GetChild<EbmlUInteger,KaxTrackNumber>(cur_track) = i+1;
        GetChild<EbmlUInteger,KaxTrackUID>(cur_track) = i+1;

        track_type ttype;
        switch(ti->type)
        {
        case mttVideo : ttype = track_video; break;
        case mttAudio : ttype = track_audio; break;
        case mttSubtitle : ttype = track_subtitle; break;
        default: throw mkv_error_exception("bad track type");
        }

        GetChild<EbmlUInteger,KaxTrackType>(cur_track) = ttype;
        GetChild<EbmlUInteger,KaxTrackFlagDefault>(cur_track) = IsFirstTrackOfType(Input,ti->type,i) ? 1 : 0;

        if (NULL!=ti->lang)
        {
            GetChild<EbmlString,KaxTrackLanguage>(cur_track) = ti->lang;
        }

        GetChild<EbmlString,KaxCodecID>(cur_track) = ti->codec_id;

        if (0!=ti->codec_private_size)
        {
            GetChild<EbmlBinary,KaxCodecPrivate>(cur_track).CopyBuffer(ti->codec_private,ti->codec_private_size);
        }

        if (0!=ti->default_duration)
        {
            GetChild<EbmlUInteger,KaxTrackDefaultDuration>(cur_track) = TimecodeFromClock(ti->default_duration);
        }

        if (NULL!=ti->name)
        {
            UTFstring tmp;
            tmp = UTF8string(ti->name);
            GetChild<EbmlUnicodeString,KaxTrackName>(cur_track) = tmp;
        }

        GetChild<EbmlUInteger,KaxTrackFlagLacing>(cur_track) = (ti->use_lacing)?1:0;

        if (mttVideo==ti->type)
        {
            KaxTrackVideo*  vid_track = & GetChild<KaxTrackVideo>(*cur_track);

            GetChild<EbmlUInteger,KaxVideoPixelWidth>(vid_track) = ti->u.video.pixel_h;
            GetChild<EbmlUInteger,KaxVideoPixelHeight>(vid_track) = ti->u.video.pixel_v;

            GetChild<EbmlUInteger,KaxVideoDisplayHeight>(vid_track) = ti->u.video.display_v;
            GetChild<EbmlUInteger,KaxVideoDisplayWidth>(vid_track) = ti->u.video.display_h;

            GetChild<EbmlUInteger,KaxVideoDisplayUnit>(vid_track) = 0;


            if (ti->u.video.fps_n!=0)
            {
                char fps_string[128];

                fps_string[sizeof(fps_string)-1]=0;

                if (ti->u.video.fps_d==1)
                {
                    sprintf_s(fps_string,sizeof(fps_string),"%d",ti->u.video.fps_n);
                }
                else 
                {
                    int fps_1=ti->u.video.fps_n/ti->u.video.fps_d;
                    int64_t fps_2=( ( ((int64_t)ti->u.video.fps_n) * 10000) / ti->u.video.fps_d) - (((int64_t)fps_1)*10000);


                    while( ((fps_2%10)==0) && (fps_2!=0) )
                    {
                        fps_2/=10;
                    }

                    sprintf_s(fps_string,sizeof(fps_string),"%d.%d (%d/%d)",fps_1,((int)fps_2),ti->u.video.fps_n,ti->u.video.fps_d);
                }
            }
        }

        if (mttAudio==ti->type)
        {
            KaxTrackAudio*  aud_track = & GetChild<KaxTrackAudio>(*cur_track);

            GetChild<EbmlFloat,KaxAudioSamplingFreq>(aud_track) = ti->u.audio.sample_rate;
            GetChild<EbmlUInteger,KaxAudioChannels>(aud_track) = ti->u.audio.channels_count;
            if (0!=ti->u.audio.bits_per_sample)
            {
                GetChild<EbmlUInteger,KaxAudioBitDepth>(aud_track) = ti->u.audio.bits_per_sample;
            }
        }

    }

    MyTracks.Render(File,true);

    Chapters.Render(File,seek_chap,FileSegment);

    // finish all meta info
    UpdateSeekEntry(seek_infos,File,MyInfos,FileSegment);
    UpdateSeekEntry(seek_tracks,File,MyTracks,FileSegment);

    KaxCluster *curr_cluster=NULL,*prev_cluster=NULL;
    int64_t cluster_timecode=0;
    int64_t max_duration=0;
    unsigned int prg_val,prg_last=0;

    while(true)
    {
        int64_t min_timecode;
        unsigned int stream_id;

        const int64_t max_timecode = (1ll<<62);
        min_timecode=max_timecode;

        for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
        {
            bool force_fetch;

            force_fetch = ( (track_info[i].info.type==mttVideo) || (track_info[i].info.type==mttAudio) );

            if (false==Input->MkvGetStream(i)->FetchFrames(1,force_fetch))
            {
                throw mkv_error_exception("Error while reading input");
            }
            if (Input->MkvGetStream(i)->GetAvailableFramesCount()>0)
            {
                int64_t dts;

                CMkvChunk* tf = Input->MkvGetStream(i)->PeekFrame(0);

                dts = tf->timecode - track_info[i].info.dts_adjust;

                if ( (dts<min_timecode) )
                {
                    if (tf->timecode==-1) throw mkv_error_exception("Frame not timestamped");

                    min_timecode = dts;
                    stream_id = i;
                }
            }
        }
        if (max_timecode==min_timecode)
        {
            // no more frames
            break;
        }

        // update UI
        if (0==stream_id)
        {
            uint64_t prgv,prgm;
            Input->MkvGetStream(0)->GetProgress(&prgv,&prgm);
            
            prg_val = (unsigned int) ((prgv*2000)/prgm);
            if (prg_val!=prg_last)
            {
                prg_last=prg_val;
                lgpl_update_current_progress(prgv);
            }
        }

        CMkvChunk*  frame;

        frame = Input->MkvGetStream(stream_id)->PeekFrame(0);

        if (frame->chapter_mark)
        {
            // dvd-style position based chapters
            Chapters.AddChapterMark(frame->timecode,File);
        }

        bool new_cluster;

        new_cluster=false;
        if (NULL==curr_cluster)
        {
            new_cluster=true;
        } else {
            int64_t cluster_duration;

            cluster_duration = frame->timecode - cluster_timecode;

            //MKV_ASSERT(frame->timecode>=cluster_timecode);

            // split if we have a full keyframe after 0.8 sec
            if (cluster_duration > 864000000ll)
            {
                if (AtKeyFrame(Input)) 
                {
                    new_cluster=true;
                }
            }

            // cluster is going to overflow soon, 
            // split at any I-frame
            if (cluster_duration > 1620000000ll)
            {
                if (AtClusterStart(Input)) 
                {
                    new_cluster=true;
                }
            }
        }

        if (new_cluster)
        {
            // hack: cluster must start with video frame
            /*if (0!=Input->MkvGetStream(0)->GetAvailableFramesCount())
            {
                stream_id=0;
                frame = Input->MkvGetStream(stream_id)->PeekFrame(0);
            }*/

            if (NULL!=curr_cluster)
            {
                FinishCluster(curr_cluster,&File,&FileSegment);
            }
            prev_cluster = curr_cluster;
            curr_cluster = & AddNewChild<KaxCluster>(FileSegment);
            curr_cluster->SetSizeInfinite();
            curr_cluster->SetParent(FileSegment);
            
            curr_cluster->WriteHead(File, 5);

            if (prev_cluster==NULL)
            {
                // 1st cluster
                UpdateSeekEntry(seek_clust1,File,*curr_cluster,FileSegment);
            } else {
                ClusterSeek.IndexThis(*curr_cluster,FileSegment);
            }

            cluster_timecode = GetClusterTimecode(Input);

            curr_cluster->InitTimecode(ScaleTimecode(TimecodeFromClock(cluster_timecode)),TIMECODE_SCALE);

            GetChild<EbmlUInteger,KaxClusterTimecode>(curr_cluster) = ScaleTimecode(TimecodeFromClock(cluster_timecode));
            GetChild<KaxClusterTimecode>(*curr_cluster).Render(File);

            // add position info
            GetChild<EbmlUInteger,KaxClusterPosition>(curr_cluster) =  
                curr_cluster->GetElementPosition() - FileSegment.GetElementPosition();
            GetChild<KaxClusterPosition>(*curr_cluster).Render(File);

            if (NULL!=prev_cluster)
            {
                GetChild<EbmlUInteger,KaxClusterPrevSize>(curr_cluster) = prev_cluster->GetSize();
                GetChild<KaxClusterPrevSize>(*curr_cluster).Render(File);
            }
        }

        // have a frame and a cluster

        // update entire file duration
        int64_t frame_end;
        frame_end=frame->timecode+frame->duration;
        if (frame_end>max_duration) max_duration = frame_end;

        // create block
        std::list<memory_c> frame_data_keeper;
        KaxBlockGroup *blk_grp;
        KaxBlockBlob  *blk_blob;

        blk_grp  = NULL;
        blk_blob = NULL;
            
        if (track_info[stream_id].info.use_lacing && 
            (0!=track_info[stream_id].info.default_duration) && 
            (frame->keyframe) &&
            (NULL!=prev_cluster) // no lacing in first cluster, let the buffers fill
            )
        {
            blk_blob = new KaxBlockBlob(BLOCK_BLOB_ALWAYS_SIMPLE);
            blk_blob->SetParent(*curr_cluster);

            unsigned int lacing_frames = (unsigned int)(216000000 / track_info[stream_id].info.default_duration);
            if (lacing_frames>255)
            {
                lacing_frames = 255;
            }
            if (false==Input->MkvGetStream(stream_id)->FetchFrames(lacing_frames+1)) 
            {
                throw mkv_error_exception("Error while reading input");
            }

            unsigned int frames_count = Input->MkvGetStream(stream_id)->GetAvailableFramesCount();

            if (frames_count>lacing_frames) frames_count=lacing_frames;
            
            MKV_ASSERT(frames_count>0);

            // GetChild<EbmlUInteger,KaxBlockDuration>(blk) = ScaleTimecode(track_info[stream_id].default_duration)*frames_count;

            unsigned int last_keyframe = 0;
            unsigned int frames_to_scan = Input->MkvGetStream(stream_id)->GetAvailableFramesCount();
            if (frames_to_scan>frames_count) frames_to_scan=frames_count+1;
            for (unsigned int i=(frames_to_scan-1);i!=0;i--)
            {
                frame=Input->MkvGetStream(stream_id)->PeekFrame(i);
                if (frame->keyframe)
                {
                    last_keyframe = i;
                    break;
                }
            }
            if (last_keyframe>3)
            {
                frames_count = last_keyframe;
            }

            bool all_same = true;
            unsigned int same_frame_size = Input->MkvGetStream(stream_id)->PeekFrame(0)->data.get_size();
            for (unsigned int i=0;i<frames_count;i++)
            {
                frame=Input->MkvGetStream(stream_id)->PeekFrame(i);
                if (frame->data.get_size()!=same_frame_size)
                {
                    all_same = false;
                    break;
                }
            }

            for (unsigned int i=0;i<frames_count;i++)
            {
                memory_c frame_data;

                frame=Input->MkvGetStream(stream_id)->PeekFrame(0);
                frame_data = frame->data;
                frame_data_keeper.push_front(frame_data);

                DataBuffer* mkv_buffer = new DataBuffer(frame_data.get(),frame_data.get_size());
                blk_blob->AddFrameAuto( * (tracks[stream_id]) , TimecodeFromClock(frame->timecode) , *mkv_buffer , all_same?LACING_FIXED:LACING_EBML );

                if (i!=(frames_count-1))
                {
                    if ( !CompareTimecode( frame->timecode+frame->duration , Input->MkvGetStream(stream_id)->PeekFrame(1)->timecode ))
                    {
                        track_info[stream_id].frame_count++;
                        Input->MkvGetStream(stream_id)->PopFrame();
                        break;
                    }
                }

                track_info[stream_id].frame_count++;
                Input->MkvGetStream(stream_id)->PopFrame();
            }
            {
                KaxSimpleBlock & blk = (*blk_blob);
                blk.Render(File);
            }

        } else {
            memory_c frame_data;

            blk_grp = new KaxBlockGroup();
            blk_grp->SetParent(*curr_cluster);

            frame_data = frame->data;
            frame_data_keeper.push_front(frame_data);

            GetChild<EbmlUInteger,KaxBlockDuration>(blk_grp) = ScaleTimecode(TimecodeFromClock(frame->duration+frame->timecode)) - ScaleTimecode(TimecodeFromClock(frame->timecode));
            /* killed for good
            if (frame->bref!=-1)
            {
                AddNewChild<KaxReferenceBlock>(*blk).SetReferencedTimecode( ScaleTimecode(frame->bref) - ScaleTimecode(frame->timecode) );
            }
            if (frame->fref!=-1)
            {
                AddNewChild<KaxReferenceBlock>(*blk).SetReferencedTimecode( ScaleTimecode(frame->fref) - ScaleTimecode(frame->timecode) );
            }
            */

            KaxBlock& iblk = GetChild<KaxBlock>(*blk_grp);

            DataBuffer* mkv_buffer = new DataBuffer(frame_data.get(),frame_data.get_size());

            iblk.AddFrame( * (tracks[stream_id]) , TimecodeFromClock(frame->timecode) , *mkv_buffer , LACING_NONE);

            track_info[stream_id].frame_count++;
            Input->MkvGetStream(stream_id)->PopFrame();

            blk_grp->Render(File);
        }

        frame_data_keeper.clear();

        // add cluster cue
        if (new_cluster)
        {
            KaxCuePoint *cp = & AddNewChild<KaxCuePoint>(AllCues);
            if (blk_blob!=NULL)
            {
                MKV_ASSERT(blk_grp==NULL);
                cp->PositionSet(*blk_blob,TIMECODE_SCALE);
            } else {
                MKV_ASSERT(blk_blob==NULL);
                cp->PositionSet(*blk_grp,TIMECODE_SCALE);
            }
        }

        delete blk_blob;
        delete blk_grp;
    }
    FinishCluster(curr_cluster,&File,&FileSegment);

    // update total duration
    GetChild<EbmlFloat,KaxDuration>(MyInfos) = (double) ScaleTimecode(TimecodeFromClock(max_duration));
    GetChild<KaxDuration>(MyInfos).OverwriteData(File,true);

    AllCues.Render(File);
    UpdateSeekEntry(seek_cues,File,AllCues,FileSegment);

    ClusterSeek.Render(File);
    UpdateSeekEntry(seek_seek2,File,ClusterSeek,FileSegment);

    Chapters.Finalize(File,seek_chap);

    // delete empty tracks
    for (unsigned int i=0;i<Input->MkvGetStreamCount();i++)
    {
        if (track_info[i].frame_count>0) continue;
        VoidElement(tracks[i],File);
        my_world()->uc_emptytrack(Input,i,&track_info[i].info);
    }

    // end

    // Set the correct size for the segment.
    FileSegment.ForceSize(File.getFilePointer() - ( FileSegment.GetElementPosition() + FileSegment.HeadSize() ) );
    FileSegment.OverwriteHead(File);
}

static void FinishCluster(KaxCluster *cluster,IOCallback *File,KaxSegment *Segment)
{
    // add position info 
    GetChild<EbmlUInteger,KaxClusterPosition>(cluster) =  
        cluster->GetElementPosition() - Segment->GetElementPosition();
    GetChild<KaxClusterPosition>(*cluster).Render(*File);

    // correct size
    cluster->ForceSize(File->getFilePointer() - ( cluster->GetElementPosition() + cluster->HeadSize() ) );
    cluster->OverwriteHead(*File);
}

extern "C"
bool __cdecl MkvCreateFile(IMkvWriteTarget* Output,IMkvTrack *Input,const char *WritingApp,IMkvTitleInfo* TitleInfo) throw()
{
    CEbmlWrite  wrt(Output);
    try 
    {
        MkvCreateFileInternal(wrt,Input,TitleInfo,WritingApp);
        Input->ReleaseBuffers();
        return true;
    } catch(std::exception &Ex)
    {
        // no memory allocations here
        char tstr[512];
        strcpy(tstr,"Exception: ");
        strncat(tstr,Ex.what(),sizeof(tstr)-1);
        tstr[sizeof(tstr)-1]=0;
        lgpl_trace(tstr);
    } catch(...)
    {
        lgpl_trace("Exception: unknown");
    }
    Input->ReleaseBuffers();
    return false;
}

