/*
    MakeMKV GUI - Graphics user interface application for MakeMKV

    Written by GuinpinSoft inc <makemkvgui@makemkv.com>

    This file is hereby placed into public domain,
    no copyright is claimed.

*/
#ifndef LGPL_APROXY_H_INCLUDED
#define LGPL_APROXY_H_INCLUDED

#include <stddef.h>
#include <stdint.h>
#include <libmkv/libmkv.h>
#include <vector>

typedef uint16_t utf16_t;

static inline size_t utf16len(const utf16_t *SrcString)
{
    const uint16_t *p=SrcString;
    while(*p!=0)
    {
        p++;
    }
    return (p-SrcString);
}

static inline const utf16_t* utf16chr(const utf16_t *SrcString,utf16_t c)
{
    const uint16_t *p=SrcString;
    while(*p!=0)
    {
        if (*p==c) return p;
        p++;
    }
    return NULL;
}

static bool utf16compare(const utf16_t *String1,const utf16_t *String2)
{
    size_t len = utf16len(String1);
    if (utf16len(String2)!=len) return false;
    return (memcmp(String1,String2,len*sizeof(utf16_t))==0);
}

//
// sh mem
//
#ifdef _MSC_VER
#define ALIGN_PACKED
#pragma pack(1)
#endif

#ifdef __GNUC__
#define ALIGN_PACKED __attribute((packed)) 
#endif

typedef struct _AP_SHMEM
{
    uint32_t    cmd;
    uint8_t     start;
    uint8_t     abort_nomem;
    uint8_t     pad0[2];
    uint64_t    args[32];
    utf16_t     strbuf[32768];
} ALIGN_PACKED AP_SHMEM;

#ifdef _MSC_VER
#pragma pack()
#endif

//
// sem
//
typedef struct _AP_SGRP
{
    uintptr_t   gid;
    uintptr_t   a_id;
    uintptr_t   b_id;
    uintptr_t   win_pph;
} AP_SGRP;

//
// commands
//
typedef enum _AP_CMD
{
    apNop=0,
    apReturn,
    apClientDone,
    apCallSignalExit,
    apCallOnIdle,
    apCallCancelAllJobs,
    apCallSetOutputFolder,
    apCallUpdateAvailableDrives,
    apCallOpenFile,
    apCallOpenCdDisk,
    apCallOpenTitleCollection,
    apCallCloseDisk,
    apCallEjectDisk,
    apCallSaveAllSelectedTitlesToMkv,
    apCallGetUiItemState,
    apCallSetUiItemState,
    apCallGetUiItemInfo,
    apCallGetSettingInt,
    apCallGetSettingString,
    apCallSetSettingInt,
    apCallSetSettingString,
    apCallSaveSettings,
    apCallAppGetString,
    apCallStartStreaming,
    apCallBackupDisc,
    apCallGetInterfaceLanguage,
    apCallGetInterfaceLanguageData,

    apBackEnterJobMode=192,
    apBackLeaveJobMode,
    apBackUpdateDrive,
    apBackUpdateCurrentBar,
    apBackUpdateTotalBar,
    apBackUpdateLayout,
    apBackSetTotalName,
    apBackUpdateCurrentInfo,
    apBackReportUiMesaage,
    apBackExit,

    apBackSetTitleCollInfo,
    apBackSetTitleInfo,
    apBackSetTrackInfo,

    apBackFatalCommError=250,
    apBackOutOfMem,

    apUnknown
} AP_CMD;

#define AP_CRE_VER "C0008"
#define AP_GUI_VER "G0008"

typedef uint32_t AP_DiskFsFlags;

#include <lgpl/apdefs.h>

//
// Wrappers
//
class CApClient;

class AP_UiItem
{
private:
    CApClient*  m_client;
    uint64_t    m_handle;
    bool        m_stateknown;
    bool        m_state;
    utf16_t*    m_infos[ap_iaMaxValue];
    char        m_infos_known[ap_iaMaxValue];
    char        m_infos_alloc[ap_iaMaxValue];
public:
    AP_UiItem();
    ~AP_UiItem();
    void Clear();
    bool    get_Enabled();
    void    set_Enabled(bool State);
    const utf16_t* GetInfo(AP_ItemAttributeId Id);
    uint64_t GetInfoNumeric(AP_ItemAttributeId Id);
    void SetInfo(uint64_t handle,CApClient* client);
    void MoveFrom(const AP_UiItem *src);
public:
    void operator=(const AP_UiItem &src)
    {
        MoveFrom(&src);
    }
    AP_UiItem(const AP_UiItem &src);
};

class AP_UiTrack : public AP_UiItem
{
    friend class CApClient;
public:
    MkvTrackType        m_Type;
};

class AP_UiTitle : public AP_UiItem
{
    friend class CApClient;
private:
    std::vector<AP_UiTrack> m_Tracks;
public:
    unsigned int GetTrackCount()
    {
        return (unsigned int) m_Tracks.size();
    }
    AP_UiTrack* GetTrack(unsigned int Index)
    {
        return &m_Tracks[Index];
    }
};

class AP_UiTitleCollection : public AP_UiItem
{
    friend class CApClient;
private:
    std::vector<AP_UiTitle> m_Titles;
public:
    bool        m_Updated;
public:
    unsigned int GetCount()
    {
        return (unsigned int) m_Titles.size();
    }
    AP_UiTitle* GetTitle(unsigned int Index)
    {
        return &m_Titles[Index];
    }
};

//
// ap class
//
class CApClient
{
public:
    class INotifier
    {
    public:
        virtual void SetTotalName(unsigned long Name)=0;
        virtual void UpdateCurrentBar(unsigned int Value)=0;
        virtual void UpdateTotalBar(unsigned int Value)=0;
        virtual void UpdateLayout(unsigned long CurrentName,unsigned int NameSubindex,unsigned int Flags,unsigned int Size,const unsigned long* Names)=0;
        virtual void UpdateCurrentInfo(unsigned int Index,const utf16_t* Value)=0;
        virtual void EnterJobMode()=0;
        virtual void LeaveJobMode()=0;
        virtual void ExitApp()=0;
        virtual void UpdateDrive(unsigned int Index,const utf16_t *DriveName,bool Visible,bool Enabled,const utf16_t *DiskName,AP_DiskFsFlags DiskFlags,const void* DiskData,unsigned int DiskDataSize)=0;
        virtual int  ReportUiMessage(unsigned long Code,unsigned long Flags,const utf16_t* Text)=0;
    };
private:
    AP_SGRP     m_sem;
    INotifier*  m_Ui;
    bool        m_shutdown;
    uintptr_t   m_debug;
public:
    volatile AP_SHMEM *m_mem;
    void ExecCmd(AP_CMD cmd);
private:
    AP_CMD Transact(AP_CMD cmd);
    AP_CMD FatalCommError();
public:
    bool EnableDebug(const char* LogName);
    bool Init(char Type,const char* AppName);
    void SetUiNotifier(CApClient::INotifier* Notifier)
    {
        m_Ui = Notifier;
    }
    CApClient()
    {
        m_Ui = NULL;
        m_debug = 0;
    }
public:
    AP_UiTitleCollection    m_TitleCollection;
public:
    void SignalExit();
    void CancelAllJobs();
    void OnIdle();

    void SetOutputFolder(const utf16_t *Name);

    bool UpdateAvailableDrives();
    bool OpenFile(const utf16_t* FileName);
    bool OpenCdDisk(unsigned int Id);
    bool OpenTitleCollection(const utf16_t* Source);
    bool CloseDisk();
    bool EjectDisk(unsigned int Id);
    bool SaveAllSelectedTitlesToMkv();
    bool StartStreaming();
    bool BackupDisc(unsigned int Id,const utf16_t* Folder,uint32_t Flags);
    int GetSettingInt(ApSettingId id);
    const utf16_t* GetSettingString(ApSettingId id);
    void SetSettingInt(ApSettingId id,int Value);
    void SetSettingString(ApSettingId id,const utf16_t* Value);
    bool SaveSettings();
    const utf16_t* GetAppString(unsigned int Id);
    bool GetInterfaceLanguage(unsigned int Id,utf16_t** Name,uint64_t** Param);
    const void* GetInterfaceLanguageData(unsigned int Id,unsigned int* Size1,unsigned int* Size2);
private:
    void SetTitleCollInfo(uint64_t handle,unsigned int Count);
    void SetTitleInfo(unsigned int id,uint64_t handle,unsigned int Count);
    void SetTrackInfo(unsigned int id,unsigned int trkid,uint64_t handle,MkvTrackType type);
    void DebugFailRoutine(uintptr_t arg0,const char* arg1,int line);
    void DebugOut(const char* string);
};
#define DebugFail(arg0,arg1) DebugFailRoutine(arg0,arg1,__LINE__)


#define AP_SEM_TIMEOUT 9

//
// API
//
int  ApSpawnApp(const char* verstr,const char* AppName,char* response,size_t responselength);
void*   ApOpenShmem(const char *name);
bool ApOpenSgrp(AP_SGRP *sgrp,const uint64_t *data);
bool ApSemInc(AP_SGRP *sgrp,uintptr_t sid);
bool ApSemDec(AP_SGRP *sgrp,uintptr_t sid);
uintptr_t ApDebugOpen(const char* name);
void ApDebugOut(uintptr_t file,const char* string);

bool SemWatchdogStart(AP_SGRP *sgrp,uintptr_t sid);
void SemWatchdogStop();
void SemWatchdogArm(bool Enable);
bool SemWatchdogFired();

//
// UI 
//
extern const utf16_t* AppGetString(unsigned long code);
extern bool AppGetInterfaceLanguageData(CApClient* app);

#define AP_UI_STRING(msg) AppGetString(msg)

#endif
