/*
    MakeMKV GUI - Graphics user interface application for MakeMKV

    Copyright (C) 2009-2010 GuinpinSoft inc <makemkvgui@makemkv.com>

    The contents of this file are subject to the Mozilla Public License
    Version 1.1 (the "License"); you may not use this file except in
    compliance with the License. You may obtain a copy of the License at
    http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS IS"
    basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
    License for the specific language governing rights and limitations
    under the License.

*/
#include <lgpl/aproxy.h>
#include <lgpl/smem.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <semaphore.h>
#include <sys/time.h>
#include <fcntl.h>

#include <stdio.h>

int posix_getmyname(char* buffer,int length);
int posix_launch(char** argv,int fdstdout);

int ApSpawnApp(const char* verstr,const char* AppName,char* response,size_t responselength)
{
    int     pipe_fd[2],err;
    char    app_path[1024+32];
    char*   argv[4];
    char*   p;
    char    str_guiserver[sizeof("guiserver")+2];
    char    str_apver[8];
    int     app_len;

    if (pipe(pipe_fd))
    {
        return errno|0x80000000;
    }

    app_len = posix_getmyname(app_path,(int)(sizeof(app_path)-1));
    if (app_len<=0)
    {
        return -2;
    }
    app_path[app_len]=0;
    p=app_path+app_len;
    while(p!=app_path)
    {
        if(*p=='/')
        {
            p++;
            break;
        }
        p--;
    }
    strcpy(p,AppName);

    strcpy(str_guiserver,"guiserver");
    strcpy(str_apver,verstr);

    argv[0]=app_path;
    argv[1]=str_guiserver;
    argv[2]=str_apver;
    argv[3]=NULL;

    err = posix_launch(argv,pipe_fd[1]);

    close(pipe_fd[1]);

    if (err)
    {
        return err;
    }

    for (unsigned int i=0;i<(responselength-1);i++)
    {
        if (1!=read(pipe_fd[0],response+i,1))
        {
            return errno|0x80000000;
        }

        if (response[i]=='$')
        {
            response[i]=0;

            // break pipe
            close(pipe_fd[0]);
            return 0;
        }
    }

    return -3;
}

void* ApOpenShmem(const char *Name)
{
    int fd;
    void* pmap;
    struct stat st;

    fd=shm_open(Name,O_RDWR,0);
    if (fd<0)
    {
        return NULL;
    }
    shm_unlink(Name);

    if (fstat(fd,&st))
    {
        return NULL;
    }

    pmap=mmap(NULL,st.st_size,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);
    if (MAP_FAILED==pmap)
    {
        return NULL;
    }

    return pmap;
}

bool ApOpenSgrp(AP_SGRP *sgrp,const uint64_t* data)
{
    sem_t* psem[2];
    for (unsigned int i=0;i<=1;i++)
    {
        const char* name;

        name = ((const char*)data) + data[i];

        psem[i] = sem_open(name,O_RDWR);

        sem_unlink(name);

        if (psem[i]==((sem_t*)SEM_FAILED))
        {
            return false;
        }
    }

    sgrp->a_id = (uintptr_t)(void*)psem[0];
    sgrp->b_id = (uintptr_t)(void*)psem[1];

    return true;
}

bool ApSemInc(AP_SGRP *sgrp,uintptr_t sid)
{
    if (0!=sem_post( (sem_t*)(void*)sid ))
    {
        return false;
    }
    return true;
}

#if !defined(_darwin_)
bool ApSemDec(AP_SGRP *sgrp,uintptr_t sid)
{
    struct timespec tm;
    struct timeval  tv;

    gettimeofday(&tv,NULL);

    tm.tv_sec  = tv.tv_sec+AP_SEM_TIMEOUT;
    tm.tv_nsec = tv.tv_usec*1000;

    if (0!=sem_timedwait( (sem_t*)(void*)sid , & tm ))
    {
        return false;
    }

    return true;
}
#else
bool ApSemDec(AP_SGRP *sgrp,uintptr_t sid)
{
    SemWatchdogArm(true);
    if (0!=sem_wait( (sem_t*)(void*)sid ))
    {
        SemWatchdogArm(false);
        return false;
    }
    SemWatchdogArm(false);
    if (SemWatchdogFired())
    {
        return false;
    }
    return true;
}
#endif

uintptr_t ApDebugOpen(const char* name)
{
    int fd = open(name,O_WRONLY|O_CREAT|O_TRUNC|O_SYNC,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
    if (fd<0) return 0;
    return (uintptr_t)fd;
}

void ApDebugOut(uintptr_t file,const char* string)
{
    int written;

    if (file==0) return;

    size_t len = strlen(string);
    while(len>0)
    {
        written = write((int)file,string,len);
        if (written<0) return;
        len -= written;
        string += written;
        fsync((int)file);
    }
}
