/*
    MakeMKV GUI - Graphics user interface application for MakeMKV

    Copyright (C) 2009-2010 GuinpinSoft inc <makemkvgui@makemkv.com>

    The contents of this file are subject to the Mozilla Public License
    Version 1.1 (the "License"); you may not use this file except in
    compliance with the License. You may obtain a copy of the License at
    http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS IS"
    basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
    License for the specific language governing rights and limitations
    under the License.

*/
#include "mainwnd.h"
#include <lgpl/appshared.h>
#include <lgpl/smem.h>

QString     get_auto_path();
uint64_t    get_free_space(const utf16_t* Folder);

typedef enum _DestMethod
{
    destNone=0,
    destAuto,
    destSemiAuto,
    destCustom
} DestMethod;

void MainWnd::ReReadSettings()
{
    setting_DestinationType = m_app->GetSettingInt(apset_app_DestinationType);
    setting_DestinationDir = QStringFromUtf16(m_app->GetSettingString(apset_app_DestinationDir));
    setting_ShowDebug = ( m_app->GetSettingInt(apset_app_ShowDebug) != 0 );
}

QString MainWnd::GetOutputBaseName()
{
    DestMethod method = (DestMethod) setting_DestinationType;

    QString base_path;
    if ( (method==destSemiAuto) || (method==destCustom) )
    {
        base_path = setting_DestinationDir;
    } else {
        base_path = get_auto_path();
    }
    return base_path;
}


QString MainWnd::GetOutputFolderName()
{
    QString label;
    const utf16_t* label_str = m_app->m_TitleCollection.GetInfo(ap_iaName);
    if (label_str==NULL)
    {
        label = UI_QSTRING(APP_IFACE_DEFAULT_FOLDER_NAME);
    } else {
        label = QStringFromUtf16(label_str);
    }
    AppMakeCorrectFileName(QStringAccessBufferRW(label));
    QStringFixLen(label);
    return label;
}

void MainWnd::CalculateDefaultOutputDir()
{
    DestMethod method = (DestMethod) setting_DestinationType;
    QString base_path = setting_DestinationDir;
    QString str;

    switch(method)
    {
    case destNone:
        break;
    case destAuto:
        base_path = get_auto_path();
    case destSemiAuto:
        str = base_path;
        if (GetOutputFolderName().isEmpty()) return;
        str += QChar::fromLatin1('/');
        str += GetOutputFolderName();
        saveFolderBox->setText(str);
        break;

    case destCustom:
        saveFolderBox->setText(base_path);
        break;

    default:
        break;
    }
}

QString FormatDiskFreeSpace(const utf16_t* FolderName)
{
    uint64_t free_sp = get_free_space(FolderName);

    char suf;

    free_sp >>=(10-8);
    if (free_sp < 2*1024*256)
    {
        suf = 'K';
    } else {
        free_sp >>= 10;
        suf = 'M';
        if (free_sp > (20*1024*256))
        {
            free_sp >>= 10;
            suf = 'G';
        }
    }

    unsigned int sz = (unsigned int) free_sp;
    unsigned int dt = ((sz&0xff)*10)>>8;
    sz >>= 8;

    QString str;
    return str.sprintf("%u.%u %c",sz,dt,suf); 
}

static inline bool wcmp(const utf16_t *s1,const char* s2)
{
    size_t l1,l2;

    if (s1==NULL) return false;
    if (s2==NULL) return false;
    l1=utf16len(s1);
    l2=strlen(s2);
    if (l1!=l2) return false;
    for (size_t i=0;i<l1;i++)
    {
        if (s1[i]!=s2[i]) return false;
    }
    return true;
}

static inline bool wcmp(const utf16_t *s1,const utf16_t* s2)
{
    size_t l1,l2;

    if (s1==NULL) return false;
    if (s2==NULL) return false;
    l1=utf16len(s1);
    l2=utf16len(s2);
    if (l1!=l2) return false;
    return (0==smemcmp(s1,s2,l1));
}


static bool HaveHiQTrack(AP_UiTitle *Title,const utf16_t* Language)
{
    if (NULL==Language) return false;
    for (unsigned int i=0;i<Title->GetTrackCount();i++)
    {
        AP_UiTrack* trk;
        trk=Title->GetTrack(i);
        if (trk->m_Type!=mttAudio) continue;
        if (false==wcmp(trk->GetInfo(ap_iaLangCode),Language)) continue;
        if ( wcmp(trk->GetInfo(ap_iaCodecId),"A_AC3") || wcmp(trk->GetInfo(ap_iaCodecId),"A_DTS") )
        {
            if (NULL!=trk->GetInfo(ap_iaAudioChannelsCount))
            {
                unsigned long cc;
                cc=QStringFromConstUtf16(trk->GetInfo(ap_iaAudioChannelsCount)).toInt();
                if (cc>=5)
                {
                    return true;
                }
            }
        }
    }
    return false;
}

static void ApplyDefaultSelection(AP_UiTitle *Title,const utf16_t* PreferredLanguage)
{
    bool haveAudioPreferred=false;
    bool haveSubtitlePreferred=false;
    for (unsigned int i=0;i<Title->GetTrackCount();i++)
    {
        AP_UiTrack* trk;
        const utf16_t* language;
        unsigned int flags;

        trk=Title->GetTrack(i);
        language=trk->GetInfo(ap_iaLangCode);
        if (trk->m_Type==mttAudio)
        {
            flags = (unsigned int) trk->GetInfoNumeric(ap_iaStreamFlags);

            // deselect 2.0 audio tracks if higher quality is available
            if ( (true==wcmp(trk->GetInfo(ap_iaCodecId),"A_AC3")) &&
                (true==wcmp(trk->GetInfo(ap_iaAudioChannelsCount),"2")) &&
                (true==HaveHiQTrack(Title,language)) )
            {
                trk->set_Enabled(false);
                continue;
            }

            // deselect secondary audio
            if ((flags&AP_AVStreamFlag_SecondaryAudio)!=0)
            {
                trk->set_Enabled(false);
                continue;
            }

            // deselect HD if core present
            if ((flags&AP_AVStreamFlag_HasCoreAudio)!=0)
            {
                trk->set_Enabled(false);
                continue;
            }

            if ((PreferredLanguage) && (language))
            {
                if (utf16compare(PreferredLanguage,language))
                {
                    haveAudioPreferred = true;
                }
            }
        }

        if (trk->m_Type==mttSubtitle)
        {
            if ((PreferredLanguage) && (language))
            {
                if (utf16compare(PreferredLanguage,language))
                {
                    haveSubtitlePreferred = true;
                }
            }
        }
    }

    for (unsigned int i=0;i<Title->GetTrackCount();i++)
    {
        AP_UiTrack* trk;
        const utf16_t* language;

        trk=Title->GetTrack(i);
        language=trk->GetInfo(ap_iaLangCode);

        if ( (trk->m_Type==mttAudio) && (haveAudioPreferred) && (language) )
        {
            if (!utf16compare(PreferredLanguage,language))
            {
                trk->set_Enabled(false);
            }
        }

        if ( (trk->m_Type==mttSubtitle) && (haveSubtitlePreferred) && (language) )
        {
            if (!utf16compare(PreferredLanguage,language))
            {
                trk->set_Enabled(false);
            }
        }
    }


    // deselect title if not primary angle
    const utf16_t *angle;
    angle=Title->GetInfo(ap_iaAngleInfo);
    if (angle!=NULL)
    {
        if (false==wcmp(angle,"1"))
        {
            Title->set_Enabled(false);
        }
    }

}

void ApplyDefaultSelection(AP_UiTitleCollection *Coll,const utf16_t* PreferredLanguage)
{
    for (unsigned int i=0;i<Coll->GetCount();i++)
    {
        ApplyDefaultSelection(Coll->GetTitle(i),PreferredLanguage);
    }
}

