/*
    MakeMKV GUI - Graphics user interface application for MakeMKV

    Copyright (C) 2009-2010 GuinpinSoft inc <makemkvgui@makemkv.com>

    The contents of this file are subject to the Mozilla Public License
    Version 1.1 (the "License"); you may not use this file except in
    compliance with the License. You may obtain a copy of the License at
    http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS IS"
    basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
    License for the specific language governing rights and limitations
    under the License.

*/
#include "qtapp.h"
#include <driveio/driveio.h>
#include <lgpl/smem.h>

static inline void append_item(QString &Qstr,const char *Str)
{
    if (Qstr.length()>0)
    {
        append_const(Qstr,",");
    }
    append_const(Qstr,Str);
}

static QString FormatProtectionString(AP_DiskFsFlags FsFlags,const uint8_t* CopyrightInfo,const uint8_t* MkbSmall)
{
    bool have_css=false,have_aacs=false,have_bdsvm=false,have_cprm=false;
    unsigned int aacs_ver = 0;

    if (0!=(FsFlags&AP_DskFsFlagDiskIsLoading)) return QString();

    if (NULL!=CopyrightInfo)
    {
        switch(CopyrightInfo[4])
        {
        case 0x01:
            have_css=true;
            break;
        case 0x02:
            have_cprm=true;
            break;
        case 0x03:
        case 0x10:
            have_aacs=true;
            break;
        }
    }

    if (0!=(FsFlags&AP_DskFsFlagAacsFilesPresent)) have_aacs=true;
    if (0!=(FsFlags&AP_DskFsFlagBdsvmFilesPresent)) have_bdsvm=true;

    if (MkbSmall!=NULL)
    {
        if (MkbSmall[0]==0x10)
        {
            aacs_ver =
                (((uint32_t)MkbSmall[8])<<(3*8)) |
                (((uint32_t)MkbSmall[9])<<(2*8)) |
                (((uint32_t)MkbSmall[10])<<(1*8)) |
                (((uint32_t)MkbSmall[11])<<(0*8)) ;
        }
    }

    QString str;
    str.reserve(20);

    if (have_css)   append_item(str,"CSS/CPPM");
    if (have_cprm)  append_item(str,"CPRM/CPPM");
    if (have_aacs)  append_item(str,"AACS");
    if (aacs_ver)
    {
        char    ver[8];
        sprintf(ver," v%u",aacs_ver);
        append_const(str,ver);
    }
    if (have_bdsvm) append_item(str,"BD+");

    return str;
}

static inline void append_const(QString &Qstr,const uint8_t *Str,size_t Size)
{
    append_const(Qstr,(const char*)Str,Size);
}

static inline void append_trimmed(QString &Qstr,const char *Str,size_t Size)
{
    while ( (Size!=0) && (Str[Size-1]==' ') )
    {
        Size--;
    }
    append_const(Qstr,Str,Size);
}

static inline void append_trimmed(QString &Qstr,const uint8_t *Str,size_t Size)
{
    append_trimmed(Qstr,(const char*)Str,Size);
}

static const char* GetMMCProfileString(unsigned int id)
{
    switch(id)
    {
    case 0x0008: return "CD-ROM";
    case 0x0009: return "CD-R";
    case 0x000A: return "CD-RW";
    case 0x0010: return "DVD-ROM";
    case 0x0011: return "DVD-R";
    case 0x0012: return "DVD-RAM";
    case 0x0013: return "DVD-RW";
    case 0x0014: return "DVD-RW";
    case 0x0015: return "DVD-R DL SR";
    case 0x0016: return "DVD-R DL JR";
    case 0x0017: return "DVD-RW DL";
    case 0x001A: return "DVD+RW";
    case 0x001B: return "DVD+R";
    case 0x002A: return "DVD+RW DL";
    case 0x002B: return "DVD+R DL";
    case 0x0040: return "BD-ROM";
    case 0x0041: return "BD-R SRM";
    case 0x0042: return "BD-R RRM";
    case 0x0043: return "BD-RE";
    case 0x0050: return "HD DVD-ROM";
    case 0x0051: return "HD DVD-R";
    case 0x0052: return "HD DVD-RAM";
    case 0x0053: return "HD DVD-RW";
    case 0x0058: return "HD DVD-R DL";
    case 0x005A: return "HD DVD-RW DL";
    }
    return NULL;
}


bool FormatDriveDiskInfo(QString& ProtectionString,QString& FullInfoString,const void* DiskData,unsigned int DiskDataSize,AP_DiskFsFlags FsFlags)
{
    struct _items{
        DriveInfoItem   inquiry,drive_serial,firmware_date,firmware_string,current_profile;
        DriveInfoItem   copyright_info,dvd_physical_info,capacity,bd_disc_info,mkb_small;
    } items;

    // extract all interesting items
    memset(&items,0,sizeof(items));
    for (unsigned int offset=0;offset!=DiskDataSize;offset+=DriveInfoList_GetSerializedChunkSize( ((const char*)DiskData) + offset) )
    {
        DriveInfoItem   item;
        DriveInfoList_GetSerializedChunkInfo( ((const char*)DiskData) + offset , &item );
        switch(item.Id)
        {
        case diid_InquiryData:
            items.inquiry = item;
            break;
        case diid_FeatureDescriptor_DriveSerialNumber:
            items.drive_serial = item;
            break;
        case diid_FeatureDescriptor_FirmwareInformation:
            items.firmware_date = item;
            break;
        case diid_FirmwareDetailsString:
            items.firmware_string=item;
            break;
        case diid_CurrentProfile:
            items.current_profile=item;
            break;
        case diid_DiscCapacity:
            items.capacity=item;
            break;
        case diid_DiscStructure_DVD_CopyrightInformation:
            items.copyright_info = item;
            break;
        case diid_DiscStructure_DVD_PhysicalFormat:
            items.dvd_physical_info=item;
            break;
        case diid_DiscStructure_BD_DiscInformation:
            items.bd_disc_info=item;
            break;
        case 0x05102201:
            items.mkb_small=item;
            break;
        }
    }

    //
    // Firstly, protection info
    //
    ProtectionString=FormatProtectionString(FsFlags,
        (items.copyright_info.Size>=8)?items.copyright_info.Data:NULL,
        (items.mkb_small.Size>=12)?items.mkb_small.Data:NULL
        );

    QString str;
    static const unsigned int TypicalInfoSize = 512;
    str.reserve(TypicalInfoSize);

    //
    // Drive info
    //
    append_const(str,"<b>Drive Information</b><br>");

    if (items.current_profile.Size>=2)
    {
        unsigned int id = 
            (((unsigned int)items.current_profile.Data[0])<<8) |
            (((unsigned int)items.current_profile.Data[1])<<0);
        const char *name = GetMMCProfileString(id);
        if (NULL!=name)
        {
            append_const(str,"Current profile: ");
            append_const(str,name);
            append_const(str,"<br>");
        }
    }

    if (items.inquiry.Size>=36)
    {
        append_const(str,"Manufacturer: ");
        append_trimmed(str,items.inquiry.Data+8,8);
        append_const(str,"<br>");
        append_const(str,"Product: ");
        append_trimmed(str,items.inquiry.Data+16,16);
        append_const(str,"<br>");
        append_const(str,"Revision: ");
        append_trimmed(str,items.inquiry.Data+32,4);
        append_const(str,"<br>");
    }
    if (items.drive_serial.Size>=8)
    {
        append_const(str,"Serial number: ");
        append_trimmed(str,items.drive_serial.Data+4,items.drive_serial.Data[3]);
        append_const(str,"<br>");
    }
    if (items.firmware_string.Data!=NULL)
    {
        append_const(str,"Firmware details: ");
        append_const(str,items.firmware_string.Data,items.firmware_string.Size);
        append_const(str,"<br>");
    }
    if (items.firmware_date.Size==20)
    {
        char fstr[20];
        append_const(str,"Firmware date: ");
        cmemcpy(fstr+0,items.firmware_date.Data+4,4);
        fstr[4]='-';
        cmemcpy(fstr+5,items.firmware_date.Data+8,2);
        fstr[7]='-';
        cmemcpy(fstr+8,items.firmware_date.Data+10,2);
        fstr[10]=' ';
        cmemcpy(fstr+11,items.firmware_date.Data+12,2);
        fstr[13]=':';
        cmemcpy(fstr+14,items.firmware_date.Data+14,2);
        fstr[16]=':';
        cmemcpy(fstr+17,items.firmware_date.Data+16,2);
        if (0==cmemcmp(fstr+11,"00:00:00",8))
        {
            append_const(str,fstr,10);
        } else {
            append_const(str,fstr,19);
        }
        append_const(str,"<br>");
    }

    //
    // Disk info
    //
    if (0!=(FsFlags&AP_DskFsFlagDiskIsAbsent))
    {
        append_const(str,"<br><b>No disc inserted</b><br>");
    } else {
        append_const(str,"<br><b>Disc Information</b><br>");

        if (0!=(FsFlags&AP_DskFsFlagDiskIsLoading))
        {
            // disk is loading
            append_const(str,"Disc is being loaded<br>");
        } else {
            // capacity
            if (items.capacity.Size>=8)
            {
                uint32_t size_sec =
                    (((uint32_t)items.capacity.Data[0])<<(3*8)) |
                    (((uint32_t)items.capacity.Data[1])<<(2*8)) |
                    (((uint32_t)items.capacity.Data[2])<<(1*8)) |
                    (((uint32_t)items.capacity.Data[3])<<(0*8)) ;

                unsigned int size_mb=size_sec/(1024/2);

                char szstr[64];
                sprintf(szstr,"%u.%u",(size_mb/1024),((size_mb%1024)*100)/1024);
                append_const(str,"Data capacity: ");
                append_const(str,szstr);
                append_const(str," Gb<br>");
            }

            if (items.dvd_physical_info.Size>=16)
            {
                const char* p;
                switch(items.dvd_physical_info.Data[4+0]>>4)
                {
                case 0x0: p="DVD-ROM"; break;
                case 0x1: p="DVD-RAM"; break;
                case 0x2: p="DVD-R"; break;
                case 0x3: p="DVD-RW"; break;
                case 0x4: p="HD DVD-ROM"; break;
                case 0x5: p="HD DVD-RAM"; break;
                case 0x6: p="HD DVD-R"; break;
                case 0x9: p="DVD+RW"; break;
                case 0xA: p="DVD+R"; break;
                case 0xD: p="DVD+RW DL"; break;
                case 0xE: p="DVD+R DL"; break;
                default: p=NULL;
                }
                if (NULL!=p)
                {
                    append_const(str,"Disc type: ");
                    append_const(str,p);
                    append_const(str,"<br>");
                }

                switch(items.dvd_physical_info.Data[4+1]>>4)
                {
                case 0x0: p="120mm"; break;
                case 0x1: p="80mm"; break;
                default: p=NULL;
                }
                if (NULL!=p)
                {
                    append_const(str,"Disc size: ");
                    append_const(str,p);
                    append_const(str,"<br>");
                }

                switch(items.dvd_physical_info.Data[4+1]&0x0f)
                {
                case 0x0: p="2.52 Mbps [0.25x]"; break;
                case 0x1: p="5.04 Mbps [0.5x]"; break;
                case 0x2: p="10.08 Mbps [1x]"; break;
                case 0x3: p="20.16 Mbps [2x]"; break;
                case 0x4: p="30.24 Mbps [3x]"; break;
                default: p=NULL;
                }
                if (NULL!=p)
                {
                    append_const(str,"Maximum read rate: ");
                    append_const(str,p);
                    append_const(str,"<br>");
                }

                switch((items.dvd_physical_info.Data[4+2]>>5)&3)
                {
                case 0x0: p="1"; break;
                case 0x1: p="2"; break;
                default: p=NULL;
                }
                if (NULL!=p)
                {
                    append_const(str,"Number of layers: ");
                    append_const(str,p);
                    if (0==(items.dvd_physical_info.Data[4+2]&0x10))
                    {
                        append_const(str," (PTP)");
                    } else {
                        append_const(str," (OTP)");
                    }
                    append_const(str,"<br>");
                }
            }

            if (items.bd_disc_info.Size>=(4+16))
            {
                // print only L0 DI descriptor
                if ( (0==cmemcmp(items.bd_disc_info.Data+4,"DI\01",3)) &&
                    ( 
                    (0==cmemcmp(items.bd_disc_info.Data+4+8,"BDO",3)) ||
                    (0==cmemcmp(items.bd_disc_info.Data+4+8,"BDW",3)) ||
                    (0==cmemcmp(items.bd_disc_info.Data+4+8,"BDR",3)) 
                    ) &&
                    true)
                {
                    const char* p;
                    switch(items.bd_disc_info.Data[4+12]&0x0f)
                    {
                    case 1:     p="BD-ROM"; break;
                    case 2:     p="BD-R"; break;
                    case 4:     p="BD-RE"; break;
                    default:    p=NULL; break;
                    }
                    if (NULL!=p)
                    {
                        append_const(str,"Disc type: ");
                        append_const(str,p);
                        append_const(str,"<br>");
                    }

                    char laystr[8];
                    sprintf(laystr,"%u",(items.bd_disc_info.Data[4+12]>>4));
                    append_const(str,"Number of layers: ");
                    append_const(str,laystr);
                    append_const(str,"<br>");

                    switch(items.bd_disc_info.Data[4+13]&0x0f)
                    {
                    case 1:     p="74,5 nm (25.0 GB max. per layer)"; break;
                    case 2:     p="69,0 nm (27.0 GB max. per layer)"; break;
                    default:    p=NULL; break;
                    }
                    if (NULL!=p)
                    {
                        append_const(str,"Channel bit length: ");
                        append_const(str,p);
                        append_const(str,"<br>");
                    }
                }
            }
        }
    }

    // finish
    if (str.size()>=TypicalInfoSize)
    {
        qDebug("info string reallocated, size=%u",((unsigned int)str.size()));
    }
    FullInfoString = str;
    return true;
}

