/*
    MakeMKV GUI - Graphics user interface application for MakeMKV

    Copyright (C) 2009-2010 GuinpinSoft inc <makemkvgui@makemkv.com>

    The contents of this file are subject to the Mozilla Public License
    Version 1.1 (the "License"); you may not use this file except in
    compliance with the License. You may obtain a copy of the License at
    http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS IS"
    basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
    License for the specific language governing rights and limitations
    under the License.

*/
#include <QtGui/QtGui>
#include "settingdlg.h"
#include "mainwnd.h"
#include <lgpl/sstring.h>

CSettingDialog::CSettingDialog(CApClient* ap_client,IIconPath* iconPath,QIcon* icon,QWidget *parent) : QDialog(parent)
{
    client = ap_client;

    setWindowIcon(*icon);
    setWindowTitle(UI_QSTRING(APP_SETTINGDLG_TITLE));

    show_restart_box = true;

    tabWidget = new QTabWidget();
    buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel | QDialogButtonBox::Apply , Qt::Horizontal);

    // general
    generalTab = new CGeneralTab(iconPath);
    tabWidget->addTab(generalTab , UI_QSTRING(APP_IFACE_SETTINGS_TAB_GENERAL));

    // Language
    languageTab = new CLanguageTab(client);
    tabWidget->addTab(languageTab , UI_QSTRING(APP_IFACE_SETTINGS_TAB_LANGUAGE));

    // IO
    ioTab = new CIOTab();
    tabWidget->addTab(ioTab, QLatin1String("IO"));

    // dvd
    dvdTab = new CDVDTab();
    tabWidget->addTab(dvdTab, UI_QSTRING(APP_TTREE_VIDEO));

    // stream
    streamTab = new CStreamTab();
    tabWidget->addTab(streamTab, QLatin1String("Streaming"));

    // aacs
    aacsTab = new CAACSTab(iconPath);
    tabWidget->addTab(aacsTab, QLatin1String("AACS"));

    // bd+
    bdpTab = new CBDPTab(iconPath);
    tabWidget->addTab(bdpTab, QLatin1String("BD+"));


    QBoxLayout *lay = new QVBoxLayout();
    lay->addWidget(tabWidget);
    lay->addStretch(10);
    lay->addWidget(buttonBox);
    this->setLayout(lay);

    connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
    connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    connect(buttonBox->button(QDialogButtonBox::Apply), SIGNAL(clicked()), this, SLOT(SlotApply()));

    connect(this, SIGNAL(accepted()) , this , SLOT(SlotApply()));

    ReadSettings();
};

void CSettingDialog::SlotApply()
{
    if (false==WriteSettings())
    {
        QMessageBox::critical(this,UI_QSTRING(APP_CAPTION_MSG),UI_QSTRING(APP_IFACE_SETTINGS_MSG_FAILED));
    }
    ReadSettings();
    if (show_restart_box)
    {
        show_restart_box=false;
        QMessageBox::information(this,UI_QSTRING(APP_CAPTION_MSG),
            UI_QSTRING(APP_IFACE_SETTINGS_MSG_RESTART));
    }
}

void CSettingDialog::ReadSettings()
{
    static const utf16_t zero[1]={0};

    // general
    const utf16_t *dest_path = client->GetSettingString(apset_app_DestinationDir);
    if (NULL==dest_path) dest_path=zero;
    generalTab->destinationDir->setText(QStringFromUtf16(dest_path));

    int dest_type = client->GetSettingInt(apset_app_DestinationType);
    generalTab->destinationDir->setIndexValue(dest_type);

    int show_debug = client->GetSettingInt(apset_app_ShowDebug);
    generalTab->check_DebugLog->setCheckState( (show_debug==0) ? Qt::Unchecked : Qt::Checked );

    int site_access = client->GetSettingInt(apset_app_UpdateEnable);
    generalTab->check_SiteAccess->setCheckState( (site_access==0) ? Qt::Unchecked : Qt::Checked );

    // language
    languageTab->setValue(languageTab->comboBoxInterfaceLanguage,client->GetSettingString(apset_app_InterfaceLanguage));
    languageTab->setValue(languageTab->comboBoxPreferredLanguage,client->GetSettingString(apset_app_PreferredLanguage));

    // io
    int retry_count = client->GetSettingInt(apset_io_ErrorRetryCount);
    ioTab->spinBoxRetryCount->setValue(retry_count);
    int buf_size = client->GetSettingInt(apset_io_RBufSizeMB);
    int darwin_workaround = client->GetSettingInt(apset_io_DarwinK2Workaround);

    if (0==buf_size)
    {
        ioTab->comboBoxRBufSize->setCurrentIndex(0);
    } else {
        char sbuf[32];
        sprintf_s(sbuf,32,"%u",buf_size);
        QString sstr=QLatin1String(sbuf);
        if (ioTab->comboBoxRBufSize->findText(sstr)<0)
        {
            ioTab->comboBoxRBufSize->addItem(sstr);
        }
        ioTab->comboBoxRBufSize->setCurrentIndex(ioTab->comboBoxRBufSize->findText(sstr));
    }
    ioTab->comboBoxDarwinK2Workaround->setCurrentIndex(darwin_workaround);

    int disable_direct = client->GetSettingInt(apset_io_NoDirectAccess);
    ioTab->checkNoDirectIo->setCheckState( (disable_direct==0) ? Qt::Unchecked : Qt::Checked );

    // dvd
    int MinimumTitleLength = client->GetSettingInt(apset_dvd_MinimumTitleLength);
    dvdTab->spinBoxMinimumTitleLength->setValue(MinimumTitleLength);

    int SpRemoveMethod = client->GetSettingInt(apset_dvd_SPRemoveMethod);
    dvdTab->comboBoxSpRemoveMethod->setCurrentIndex(SpRemoveMethod);

    // aacs
    const utf16_t *keypath = client->GetSettingString(apset_aacs_KeyFilesDir);
    if (NULL==keypath) keypath=zero;
    aacsTab->keyDir->setText(QStringFromUtf16(keypath));

    // bd+
    const utf16_t *svqpath = client->GetSettingString(apset_bdplus_SvqDir);
    if (NULL==svqpath) svqpath=zero;
    bdpTab->svqDir->setText(QStringFromUtf16(svqpath));
    int dump_always = client->GetSettingInt(apset_bdplus_DumpAlways);
    bdpTab->check_DumpAlways->setCheckState( (dump_always==0) ? Qt::Unchecked : Qt::Checked );

    const utf16_t *dumppath = client->GetSettingString(apset_bdplus_DumpDir);
    if (NULL==dumppath) dumppath=zero;
    bdpTab->dumpDir->setText(QStringFromUtf16(dumppath));

    // stream
    int Enableupnp = client->GetSettingInt(apset_stream_EnableUPNP);
    streamTab->checkEnableUpnp->setCheckState( (Enableupnp==0) ? Qt::Unchecked : Qt::Checked );

    const utf16_t *ipaddress = client->GetSettingString(apset_stream_BindIp);
    if (ipaddress)
    {
        if (ipaddress[0]==0) ipaddress=NULL;
    }
    if (!ipaddress)
    {
        streamTab->comboAddress->setCurrentIndex(0);
    } else {
        if (streamTab->comboAddress->findText(QStringFromUtf16(ipaddress))<0)
        {
            streamTab->comboAddress->addItem(QStringFromUtf16(ipaddress));
        }
        streamTab->comboAddress->setCurrentIndex(streamTab->comboAddress->findText(QStringFromUtf16(ipaddress)));
    }

    int BindPort = client->GetSettingInt(apset_stream_BindPort);
    streamTab->spinPort->setValue(BindPort);
}

bool CSettingDialog::WriteSettings()
{
    QString string;

    // general
    client->SetSettingString(apset_app_DestinationDir,Utf16FromQString(generalTab->destinationDir->text()));
    client->SetSettingInt(apset_app_DestinationType,generalTab->destinationDir->getIndexValue());

    client->SetSettingInt( apset_app_ShowDebug , (generalTab->check_DebugLog->checkState() == Qt::Checked) ? 1 : 0 );
    client->SetSettingInt( apset_app_UpdateEnable , (generalTab->check_SiteAccess->checkState() == Qt::Checked) ? 1 : 0 );

    // language
    client->SetSettingString(apset_app_InterfaceLanguage,languageTab->getValue(languageTab->comboBoxInterfaceLanguage,string));
    client->SetSettingString(apset_app_PreferredLanguage,languageTab->getValue(languageTab->comboBoxPreferredLanguage,string));

    // io
    client->SetSettingInt(apset_io_ErrorRetryCount,ioTab->spinBoxRetryCount->value());
    int rbuf_size;
    if (ioTab->comboBoxRBufSize->currentIndex()==0)
    {
        rbuf_size=0;
    } else {
        rbuf_size=ioTab->comboBoxRBufSize->currentText().toInt();
    }
    client->SetSettingInt(apset_io_RBufSizeMB,rbuf_size);
    client->SetSettingInt(apset_io_NoDirectAccess,(ioTab->checkNoDirectIo->checkState() == Qt::Checked) ? 1 : 0 );
    client->SetSettingInt(apset_io_DarwinK2Workaround,ioTab->comboBoxDarwinK2Workaround->currentIndex());

    // dvd
    client->SetSettingInt(apset_dvd_MinimumTitleLength,dvdTab->spinBoxMinimumTitleLength->value());
    client->SetSettingInt(apset_dvd_SPRemoveMethod,dvdTab->comboBoxSpRemoveMethod->currentIndex());

    // aacs
    client->SetSettingString(apset_aacs_KeyFilesDir,Utf16FromQString(aacsTab->keyDir->text()));

    // bd+
    client->SetSettingString(apset_bdplus_SvqDir,Utf16FromQString(bdpTab->svqDir->text()));
    client->SetSettingString(apset_bdplus_DumpDir,Utf16FromQString(bdpTab->dumpDir->text()));
    client->SetSettingInt( apset_bdplus_DumpAlways , (bdpTab->check_DumpAlways->checkState() == Qt::Checked) ? 1 : 0 );

    // stream
    client->SetSettingInt(apset_stream_EnableUPNP , (streamTab->checkEnableUpnp->checkState() == Qt::Checked) ? 1 : 0 );

    if ( (streamTab->comboAddress->currentIndex()==0) && (streamTab->comboAddress->currentText()==streamTab->comboAddress->itemText(0)) )
    {
        client->SetSettingString(apset_stream_BindIp,NULL);
    } else {
        client->SetSettingString(apset_stream_BindIp,Utf16FromQString(streamTab->comboAddress->currentText()));
    }
    client->SetSettingInt(apset_stream_BindPort,streamTab->spinPort->value());

    // flush
    return client->SaveSettings();
}

CIOTab::CIOTab(QWidget *parent) : QWidget(parent)
{
    QGroupBox* box = new QGroupBox(UI_QSTRING(APP_IFACE_SETTINGS_IO_OPTIONS));

    spinBoxRetryCount = new QSpinBox();
    comboBoxRBufSize = new QComboBox();
    comboBoxRBufSize->addItem(QString(UI_QSTRING(APP_IFACE_SETTINGS_IO_AUTO)));
    comboBoxRBufSize->addItem(QLatin1String("64"));
    comboBoxRBufSize->addItem(QLatin1String("256"));
    comboBoxRBufSize->addItem(QLatin1String("512"));
    comboBoxRBufSize->addItem(QLatin1String("1024"));
    comboBoxRBufSize->addItem(QLatin1String("2048"));
    checkNoDirectIo = new QCheckBox();
    comboBoxDarwinK2Workaround = new QComboBox();
    comboBoxDarwinK2Workaround->addItem(QLatin1String("0"));
    comboBoxDarwinK2Workaround->addItem(QLatin1String("1"));
    comboBoxDarwinK2Workaround->addItem(QLatin1String("2"));

    QGridLayout *b_lay = new QGridLayout();
    b_lay->setColumnStretch(0,2);

    bool osx_k2bug_affected = false;

#ifdef Q_OS_DARWIN
    osx_k2bug_affected = true;
#endif

    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_IO_READ_RETRY)),0,0,Qt::AlignRight);
    b_lay->addWidget(spinBoxRetryCount,0,1);
    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_IO_READ_BUFFER)),1,0,Qt::AlignRight);
    b_lay->addWidget(comboBoxRBufSize,1,1);
    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_IO_NO_DIRECT_ACCESS)),2,0,Qt::AlignRight);
    b_lay->addWidget(checkNoDirectIo,2,1);
    if (osx_k2bug_affected)
    {
        b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_IO_DARWIN_K2_WORKAROUND)),3,0,Qt::AlignRight);
        b_lay->addWidget(comboBoxDarwinK2Workaround,3,1);
    }

    box->setLayout(b_lay);

    QBoxLayout *lay = new QVBoxLayout();
    lay->addWidget(box);

    lay->addStretch(2);
    this->setLayout(lay);
}

CDVDTab::CDVDTab(QWidget *parent) : QWidget(parent)
{
    QGroupBox* box = new QGroupBox(UI_QSTRING(APP_IFACE_SETTINGS_DVD_OPTIONS));

    spinBoxMinimumTitleLength = new QSpinBox();
    spinBoxMinimumTitleLength->setMaximum(9999);
    comboBoxSpRemoveMethod = new QComboBox();
    comboBoxSpRemoveMethod->addItem(UI_QSTRING(APP_IFACE_SETTINGS_DVD_AUTO));
    comboBoxSpRemoveMethod->addItem(QLatin1String("CellWalk"));
    comboBoxSpRemoveMethod->addItem(QLatin1String("CellTrim"));

    QGridLayout *b_lay = new QGridLayout();
    b_lay->setColumnStretch(0,2);

    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_DVD_MIN_LENGTH)),0,0,Qt::AlignRight);
    b_lay->addWidget(spinBoxMinimumTitleLength,0,1);

    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_DVD_SP_REMOVE)),1,0,Qt::AlignRight);
    b_lay->addWidget(comboBoxSpRemoveMethod,1,1);

    box->setLayout(b_lay);
    QBoxLayout *lay = new QVBoxLayout();
    lay->addWidget(box);
    lay->addStretch(2);
    this->setLayout(lay);
}

CAACSTab::CAACSTab(IIconPath* iconPath,QWidget *parent) : QWidget(parent)
{
    keyDir = new CDirSelectBox(UI_QSTRING(APP_IFACE_SETTINGS_AACS_KEY_DIR),iconPath);

    QBoxLayout *lay = new QVBoxLayout();

    lay->addWidget(keyDir);

    lay->addStretch(2);
    this->setLayout(lay);
}

CBDPTab::CBDPTab(IIconPath* iconPath,QWidget *parent) : QWidget(parent)
{
    svqDir = new CDirSelectBox(UI_QSTRING(APP_IFACE_SETTINGS_BDP_SVQ_DIR),iconPath);
    dumpDir = new CDirSelectBox(UI_QSTRING(APP_IFACE_SETTINGS_BDP_DUMP_DIR),iconPath);

    QGroupBox* miscBox = new QGroupBox(UI_QSTRING(APP_IFACE_SETTINGS_BDP_MISC));
    QGridLayout *m_lay = new QGridLayout();
    m_lay->setColumnStretch(0,2);

    check_DumpAlways = new QCheckBox();
    m_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_BDP_DUMP_ALWAYS)),0,0,Qt::AlignRight);
    m_lay->addWidget(check_DumpAlways,0,1);
    miscBox->setLayout(m_lay);

    QBoxLayout *lay = new QVBoxLayout();

    lay->addWidget(svqDir);
    lay->addWidget(dumpDir);
    lay->addWidget(miscBox);

    lay->addStretch(2);
    this->setLayout(lay);
}

CStreamTab::CStreamTab(QWidget *parent) : QWidget(parent)
{
    QGroupBox* box = new QGroupBox(UI_QSTRING(APP_IFACE_SETTINGS_STREAM_OPTIONS));

    checkEnableUpnp = new QCheckBox();

    comboAddress = new QComboBox();
    comboAddress->addItem(QString(UI_QSTRING(APP_IFACE_SETTINGS_IO_AUTO)));
    comboAddress->setEditable(true);

    spinPort = new QSpinBox();
    spinPort->setMinimum(1025);
    spinPort->setMaximum(65535);

    QGridLayout *b_lay = new QGridLayout();

    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_STREAM_ENABLE_UPNP)),0,0,Qt::AlignRight);
    b_lay->addWidget(checkEnableUpnp,0,1);
    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_STREAM_BIND_IP)),1,0,Qt::AlignRight);
    b_lay->addWidget(comboAddress,1,1);
    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_STREAM_BIND_PORT)),2,0,Qt::AlignRight);
    b_lay->addWidget(spinPort,2,1);
    box->setLayout(b_lay);

    QBoxLayout *lay = new QVBoxLayout();
    lay->addWidget(box);

    lay->addStretch(2);
    this->setLayout(lay);
}

CGeneralTab::CGeneralTab(IIconPath* iconPath,QWidget *parent) : QWidget(parent)
{
    radio_None = new QRadioButton(UI_QSTRING(APP_IFACE_SETTINGS_DEST_TYPE_NONE));
    radio_Auto = new QRadioButton(UI_QSTRING(APP_IFACE_SETTINGS_DEST_TYPE_AUTO));
    radio_SemiAuto = new QRadioButton(UI_QSTRING(APP_IFACE_SETTINGS_DEST_TYPE_SEMIAUTO));
    radio_Custom = new QRadioButton(UI_QSTRING(APP_IFACE_SETTINGS_DEST_TYPE_CUSTOM));

    QWidgetList lst;
    lst.append(radio_None);
    lst.append(radio_Auto);
    lst.append(radio_SemiAuto);
    lst.append(radio_Custom);

    destinationDir = new CDirSelectBox(UI_QSTRING(APP_IFACE_SETTINGS_DESTDIR),iconPath,lst);

    connect(destinationDir, SIGNAL(SignalIndexChanged()) , this , SLOT(SlotIndexChanged()) );

    QGroupBox* miscBox = new QGroupBox(UI_QSTRING(APP_IFACE_SETTINGS_GENERAL_MISC));
    QGridLayout *m_lay = new QGridLayout();
    m_lay->setColumnStretch(0,2);

    check_DebugLog = new QCheckBox();
    m_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_LOG_DEBUG_MSG)),0,0,Qt::AlignRight);
    m_lay->addWidget(check_DebugLog,0,1);
    check_SiteAccess = new QCheckBox();
    m_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_ENABLE_SITE_ACCESS)),1,0,Qt::AlignRight);
    m_lay->addWidget(check_SiteAccess,1,1);
    miscBox->setLayout(m_lay);

    QBoxLayout *lay = new QVBoxLayout();
    lay->addWidget(destinationDir);
    lay->addWidget(miscBox);

    lay->addStretch(2);

    this->setLayout(lay);
}

void CGeneralTab::SlotIndexChanged()
{
    int index = destinationDir->getIndexValue();
    if ( (index==0) || (index==1) )
    {
        destinationDir->setDirEnabled(false);
    }
    if ( (index==2) || (index==3) )
    {
        destinationDir->setDirEnabled(true);
    }
}

typedef struct _lang_info_t
{
    const char* code;
    const char* a1;
    const char* a2;
    const char* name;
} lang_info_t;

#include <lgpl/iso639tbl.h>


CLanguageTab::CLanguageTab(CApClient* ap_client,QWidget *parent) : QWidget(parent)
{
    QGroupBox* box = new QGroupBox(UI_QSTRING(APP_IFACE_SETTINGS_LANGUAGE_OPTIONS));

    comboBoxInterfaceLanguage = new QComboBox();
    comboBoxInterfaceLanguage->addItem(QString(UI_QSTRING(APP_IFACE_SETTINGS_LANGUAGE_AUTO)));

    for (unsigned int i=0;i<AP_APP_LOC_MAX;i++)
    {
        uint16_t* name;
        uint64_t* param;
        QString qstrName;

        if (!ap_client->GetInterfaceLanguage(i,&name,&param)) break;

        qstrName.clear();
        qstrName.reserve(10+utf16len(name));
        qstrName.append(QStringFromUtf16(name).mid(0,3));
        qstrName.append(QString::fromLatin1(" : "));
        qstrName.append(QStringFromUtf16(name).mid(4));

        comboBoxInterfaceLanguage->addItem(qstrName);
    }


    comboBoxPreferredLanguage = new QComboBox();
    comboBoxPreferredLanguage->addItem(QString(UI_QSTRING(APP_IFACE_SETTINGS_LANGUAGE_NONE)));

    for (size_t i=0;i<(sizeof(lang_table)/sizeof(lang_table[0]));i++)
    {
        QString langString;

        if (!lang_table[i].a1) continue;

        langString.clear();
        langString.reserve(8+strlen(lang_table[i].name));

        langString.append(QString::fromLatin1(lang_table[i].code));
        langString.append(QString::fromLatin1(" : "));
        langString.append(QString::fromUtf8(lang_table[i].name));

        comboBoxPreferredLanguage->addItem(QString(langString));
    }

    QGridLayout *b_lay = new QGridLayout();
    b_lay->setColumnStretch(0,2);

    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_LANG_INTERFACE)),0,0,Qt::AlignRight);
    b_lay->addWidget(comboBoxInterfaceLanguage,0,1);
    b_lay->addWidget(createLabel(UI_QSTRING(APP_IFACE_SETTINGS_LANG_PREFERRED)),1,0,Qt::AlignRight);
    b_lay->addWidget(comboBoxPreferredLanguage,1,1);

    box->setLayout(b_lay);

    QBoxLayout *lay = new QVBoxLayout();
    lay->addWidget(box);

    lay->addStretch(2);
    this->setLayout(lay);
}

void CLanguageTab::setValue(QComboBox*  comboBox,const utf16_t *value)
{
    comboBox->setCurrentIndex(0);
    if (!value)
    {
        return;
    }
    if (!value[0])
    {
        return;
    }

    QString valueStr = QStringFromUtf16(value);

    for (int i=1;i<comboBox->count();i++)
    {
        if (comboBox->itemText(i).startsWith(valueStr))
        {
            comboBox->setCurrentIndex(i);
            break;
        }
    }
}

const utf16_t* CLanguageTab::getValue(QComboBox*  comboBox,QString &buffer)
{
    if (comboBox->currentIndex()==0) return NULL;

    buffer.clear();
    buffer.append(comboBox->currentText().mid(0,3));

    return Utf16FromQString(buffer);
}
