/*
    MakeMKV GUI - Graphics user interface application for MakeMKV

    Copyright (C) 2009-2010 GuinpinSoft inc <makemkvgui@makemkv.com>

    The contents of this file are subject to the Mozilla Public License
    Version 1.1 (the "License"); you may not use this file except in
    compliance with the License. You may obtain a copy of the License at
    http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS IS"
    basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
    License for the specific language governing rights and limitations
    under the License.

*/
#include <QtGui/QtGui>
#include "mainwnd.h"

class CCheckTreeItem : public QTreeWidgetItem
{
    static const int    MyType = 1002;
    Q_DISABLE_COPY(CCheckTreeItem)
    AP_UiItem*        m_UiItem;
public:
    bool            m_Checkable;
    bool            m_AlwaysExpanded;
    QString         m_Info;
public:
    CCheckTreeItem(AP_UiItem* pItem,bool Checkable) 
        : m_UiItem(pItem) , QTreeWidgetItem(MyType)
    {
        m_Checkable = Checkable;
        m_AlwaysExpanded = false;
        if (false==Checkable)
        {
            setFlags( Qt::ItemIsSelectable | Qt::ItemIsEnabled);
        } else {
            setFlags( Qt::ItemIsSelectable | Qt::ItemIsUserCheckable | Qt::ItemIsEnabled);
            if (true==m_UiItem->get_Enabled())
            {
                setCheckState(0,Qt::Checked);
            } else {
                setCheckState(0,Qt::Unchecked);
            }
        }
    }
public:
    void setData(int column, int role, const QVariant &value);
    static CCheckTreeItem* cast(QTreeWidgetItem* TreeItem)
    {
        if (NULL==TreeItem) return NULL;
        if (TreeItem->type() == MyType)
        {
            return (CCheckTreeItem*) TreeItem;
        } else {
            return NULL;
        }
    }
    bool Enabled()
    {
        return m_UiItem->get_Enabled();
    }
};

void CCheckTreeItem::setData(int column, int role, const QVariant &value)
{
    QTreeWidgetItem::setData(column,role,value);

    if ( (column==0) && (role==Qt::CheckStateRole) )
    {
        int state = value.toInt();
        bool val = (state==Qt::Checked);

        m_UiItem->set_Enabled(val);
    }
}

void MainWnd::ResetInfoPane()
{
    itemInfoEdit->setHtmlBody(QString());
}

static CCheckTreeItem* GetSelectedItem(QTreeWidget* tree)
{
    QList<QTreeWidgetItem *> sel_list;

    sel_list = tree->selectedItems();

    if (sel_list.empty())
    {
        return NULL;
    } else {
        CCheckTreeItem* item = CCheckTreeItem::cast( *sel_list.begin() );
        return item;
    }
}

void MainWnd::SlotTreeSelectionChanged()
{
    CCheckTreeItem* item = GetSelectedItem(titleTreeView);

    if (NULL==item)
    {
        ResetInfoPane();
        m_tree_toggle->setEnabled(false);
        m_tree_select->setEnabled(false);
        m_tree_unselect->setEnabled(false);
    } else {
        // info
        itemInfoEdit->setHtmlBody(item->m_Info);
        m_tree_toggle->setEnabled(item->m_Checkable);
        m_tree_select->setEnabled(true);
        m_tree_unselect->setEnabled(true);
    }
}

void MainWnd::SlotTreeItemCollapsed(QTreeWidgetItem *item)
{
    CCheckTreeItem* titem = CCheckTreeItem::cast(item);
    if (titem==NULL) return;
    if (titem->m_AlwaysExpanded)
    {
        titem->setExpanded(true);
    }
}

void MainWnd::SlotTreeItemExpanded(QTreeWidgetItem *item)
{
    CCheckTreeItem* titem = CCheckTreeItem::cast(item);
    if (titem==NULL) return;

    for (int i=0;i<item->childCount();i++)
    {
        CCheckTreeItem* citem;

        citem = CCheckTreeItem::cast(titem->child(i));
        if (citem==NULL) continue;

        if (citem->m_AlwaysExpanded)
        {
            citem->setExpanded(true);
        }
    }
}

static void SetSiblingsState(CCheckTreeItem* item,bool check)
{
    CCheckTreeItem* parent,*child;
    
    if (NULL==item) return;
    parent = CCheckTreeItem::cast(item->parent());

    if (NULL==parent)
    {
        parent=item;
    }
    for (int i=0;i<parent->childCount();i++)
    {
        child = CCheckTreeItem::cast(parent->child(i));
        if (NULL==child) continue;
        if (false==child->m_Checkable) continue;
        if (check!=child->Enabled())
        {
            if (check)
            {
                child->setCheckState(0,Qt::Checked);
            } else {
                child->setCheckState(0,Qt::Unchecked);
            }
        }
    }
}

void MainWnd::SlotSelectTreeItem()
{
    SetSiblingsState(GetSelectedItem(titleTreeView),true);
}

void MainWnd::SlotUnselectTreeItem()
{
    SetSiblingsState(GetSelectedItem(titleTreeView),false);
}

void MainWnd::SlotToggleTreeItem()
{
    CCheckTreeItem* item = GetSelectedItem(titleTreeView);
    if (NULL!=item)
    {
        if (item->m_Checkable)
        {
            if (true==item->Enabled())
            {
                item->setCheckState(0,Qt::Unchecked);
            } else {
                item->setCheckState(0,Qt::Checked);
            }
        }
    }
}

void MainWnd::Update_TitleInfo_from_app()
{
    if (0==m_app->m_TitleCollection.GetCount())
    {
        saveFolderBox->selectDialogAction()->setEnabled(false);
        closeDiskAct->setEnabled(false);
        saveAllMkvAct->setEnabled(false);
        startStreamingAct->setEnabled(false);
        RefreshEmptyFrame();
    } else {
        saveFolderBox->selectDialogAction()->setEnabled(true);
        closeDiskAct->setEnabled(true);
        startStreamingAct->setEnabled(true);
        backupAct->setEnabled(false);
        
        if (false==saveFolderBox->IsDirValid())
        {
            saveAllMkvAct->setEnabled(false);
        } else {
            saveAllMkvAct->setEnabled(true);
        }
        if (m_disc_opened)
        {
            // just opened a disc
            m_do_disc_opened=true;
        }
    }
    m_disc_opened=false;
}

void MainWnd::EmptyFrameAddDrive(unsigned int Index,const utf16_t* DriveName,const utf16_t* DiskName,AP_DiskFsFlags FsFlags,const void* DiskData,unsigned int DiskDataSize)
{
    DriveInfo[Index].Update(DriveName,DiskName,FsFlags,DiskData,DiskDataSize);
    RefreshEmptyFrame();
}

void MainWnd::EmptyFrameRemoveDrive(unsigned int Index)
{
    DriveInfo[Index].valid=false;
    RefreshEmptyFrame();
}

int MainWnd::GetEmptyBoxDriveId()
{
    int cur_ndx=-1;

    int cmb_ndx = emptyDriveBox->currentIndex();
    if (cmb_ndx>=0)
    {
        cur_ndx = emptyDriveBox->itemData(cmb_ndx).toInt();
        if (DriveInfo[cur_ndx].valid==false)
        {
            cur_ndx = -1;
        }
    }
    return cur_ndx;
}


void MainWnd::RefreshEmptyFrame()
{
    int cur_ndx = GetEmptyBoxDriveId();

    if ( (cur_ndx>=0) && (DriveInfo[cur_ndx].show_open==false) )
    {
        // check if we have a video disk in addition to this
        // cd-rom and select it
        int good_ndx=-1;
        for (unsigned int i=0;i<AP_MaxCdromDevices;i++)
        {
            if (DriveInfo[i].valid==false) continue;
            if (DriveInfo[i].show_open==false) continue;
            good_ndx=i;
            break;
        }
        // see if we have a "loading" drive
        if (good_ndx<0)
        {
            for (unsigned int i=0;i<AP_MaxCdromDevices;i++)
            {
                if (DriveInfo[i].valid==false) continue;
                if (0==(DriveInfo[i].fs_flags&AP_DskFsFlagDiskIsLoading)) continue;
                good_ndx=i;
                break;
            }
        }
        if (good_ndx>=0)
        {
            cur_ndx=good_ndx;
        }
    }

    // re-create
    int cmb_ndx=-1;
    emptyDriveBox->clear();
    for (unsigned int i=0;i<AP_MaxCdromDevices;i++)
    {
        if (DriveInfo[i].valid==false) continue;

        emptyDriveBox->addItem(DriveInfo[i].name,QVariant(i));
        if (i==cur_ndx)
        {
            cmb_ndx = emptyDriveBox->count()-1;
        }
    }

    if ( (cmb_ndx==-1) && (emptyDriveBox->count()>0) )
    {
        cmb_ndx=0;
    }

    emptyDriveBox->setCurrentIndex(cmb_ndx);
    SlotEmptyBoxChanged();
}

void MainWnd::SlotEmptyBoxChanged()
{
    int cur_ndx = GetEmptyBoxDriveId();

    if (cur_ndx==-1)
    {
        empty_type->setText(QString());
        empty_label->setText(QString());
        empty_prot->setText(QString());
        empty_right_info->setHtmlBody(QString());
        empty_right_info->setEnabled(false);
        empty_big_btn->setEnabled(false);
        emptyDriveBox->setEnabled(false);
        backupAct->setEnabled(false);
    } else {
        CDriveInfo* info = &DriveInfo[cur_ndx];
        empty_type->setText(info->type);
        empty_label->setText(info->label);
        empty_prot->setText(info->prot);
        empty_right_info->setHtmlBody(info->right_info);
        empty_right_info->setEnabled(true);

        QAction* btn_act;

        switch(info->disk_type)
        {
            case dtBluray: 
                btn_act=blurayToHdAct; 
                backupAct->setEnabled(true);
                break;
            case dtHdvd: 
                btn_act=hddvdToHdAct; 
                backupAct->setEnabled(false);
                break;
            case dtLoading: 
                btn_act=loadingDiskAct; 
                backupAct->setEnabled(false);
                break;
            default:
            case dtDvd: 
                btn_act=dvdToHdAct; 
                backupAct->setEnabled(false);
                break;
        }

        empty_big_btn->setDefaultAction(btn_act);
        emptyDriveBox->setEnabled(true);
        empty_big_btn->setEnabled(info->show_open);
    }
}

QString BuildCollectionItemInfo(AP_UiTitleCollection *Title);
QString BuildTitleItemInfo(AP_UiTitle *Title);
QString BuildTrackItemInfo(AP_UiTrack *Title);

void MainWnd::Update_TitleTree_from_app()
{
    QTreeWidget*        tree = titleTreeView;
    AP_UiTitleCollection* coll = &m_app->m_TitleCollection;
    
    tree->clear();
    tree->setColumnCount(2);

    QStringList hdr_labels;
    hdr_labels += UI_QSTRING(APP_TTREE_HDR_TYPE);
    hdr_labels += UI_QSTRING(APP_TTREE_HDR_DESC);
    tree->setHeaderLabels(hdr_labels);
    

    if (coll->GetCount()==0)
    {
        // "no title opened" splash
//        topStackedWidget->setCurrentIndex(2);

    } else {
  //      topStackedWidget->setCurrentIndex(0);

        CCheckTreeItem *c_item;
        c_item = new CCheckTreeItem(coll,false);
        const utf16_t* coll_name = coll->GetInfo(ap_iaType);
        if (NULL!=coll_name)
        {
            c_item->setText(0,QStringFromConstUtf16(coll_name));
        } else {
            c_item->setText(0,QLatin1String("Titles"));
        }
        const utf16_t* coll_desc = coll->GetInfo(ap_iaName);
        if (NULL!=coll_desc)
        {
            c_item->setText(1,QStringFromConstUtf16(coll_desc));
        }
        tree->addTopLevelItem(c_item);
        c_item->m_Info = BuildCollectionItemInfo(coll);

        for (unsigned int i=0;i<coll->GetCount();i++)
        {
            AP_UiTitle *title = coll->GetTitle(i);

            CCheckTreeItem*    t_item,*item,*main_item;
            t_item = new CCheckTreeItem(title,true);
            t_item->setText(0,UI_QSTRING(APP_TTREE_TITLE));

            {
                QString t_text;
                bool first=true;

                const utf16_t* name = title->GetInfo(ap_iaName);
                if (NULL!=name)
                {
                    t_text += QStringFromConstUtf16(name);
                    first = false;
                } else {
                    const utf16_t* ttime = title->GetInfo(ap_iaDateTime);
                    if (NULL!=ttime)
                    {
                        t_text += QStringFromConstUtf16(ttime);
                        first = false;
                    }
                }

                const utf16_t* c_cnt = title->GetInfo(ap_iaChapterCount);
                if (NULL!=c_cnt)
                {
                    if (false==first)
                    {
                        t_text.append(QLatin1String(" - "));
                    }
                    QString chap_str;
                    chap_str = UI_QSTRING(APP_TTREE_CHAP_DESC).arg(QStringFromConstUtf16(c_cnt));
                    t_text += chap_str;
                    first = false;
                }

                const utf16_t * t_sz = title->GetInfo(ap_iaDiskSize);
                if (NULL!=t_sz)
                {
                    if (false==first)
                    {
                        t_text.append(QLatin1String(" , "));
                    }
                    t_text += QStringFromConstUtf16(t_sz);
                    first = false;
                }

                const utf16_t* a_cnt = title->GetInfo(ap_iaAngleInfo);
                if (NULL!=a_cnt)
                {
                    if (false==first)
                    {
                        t_text.append(QLatin1String(" "));
                    }
                    QString angle_str;
                    angle_str = UI_QSTRING(APP_TTREE_ANGLE_DESC).arg(QStringFromConstUtf16(a_cnt));
                    t_text += angle_str;
                    first = false;
                }

                t_item->setText(1,t_text);
            }

            t_item->m_Info = BuildTitleItemInfo(title);

            // all tracks
            static const MkvTrackType types[3]={mttVideo,mttAudio,mttSubtitle};
            for (unsigned int k=0;k<3;k++)
            {
                main_item = NULL;
                for (unsigned int j=0;j<title->GetTrackCount();j++)
                {
                    AP_UiTrack* track;
                    unsigned int flags;

                    track = title->GetTrack(j);

                    if (track->m_Type != types[k]) continue;

                    item = new CCheckTreeItem(track,(mttVideo==types[k])?false:true);
                    item->setText(0,QStringFromConstUtf16(track->GetInfo(ap_iaType)));

                    QString text;

                    const utf16_t *wname,*wlang;
                    text.append( QStringFromConstUtf16(track->GetInfo(ap_iaCodecShort)) );

                    wname = track->GetInfo(ap_iaName);
                    if (NULL!=wname)
                    {
                        text.append( QLatin1String(" ") );
                        text.append( QStringFromConstUtf16(wname) );
                    }
                    wlang = track->GetInfo(ap_iaLangName);
                    if (NULL!=wlang)
                    {
                        text.append( QLatin1String(" ") );
                        text.append( QStringFromConstUtf16(wlang) );
                    }

                    flags = (unsigned int) track->GetInfoNumeric(ap_iaStreamFlags);

                    if ((flags&AP_AVStreamFlag_ForcedSubtitles)!=0)
                    {
                        text.append( QLatin1String(" ") );
                        text.append( UI_QSTRING(APP_TTREE_FORCED_SUBTITLES) );
                    }

                    item->setText(1,text);

                    item->m_Info = BuildTrackItemInfo(track);

                    if (flags&AP_AVStreamFlag_DerivedStream)
                    {
                        main_item->m_AlwaysExpanded = true;
                        main_item->addChild(item);
                    } else {
                        t_item->addChild(item);
                        main_item = item;
                    }
                    item=NULL;
                }
            }

            //tree->addTopLevelItem(t_item);
            c_item->addChild(t_item); t_item=NULL;
        }

        ResetInfoPane();

        tree->expandAll();
        tree->resizeColumnToContents(0);
        tree->resizeColumnToContents(1);
        tree->collapseAll();

        tree->expandItem(c_item);
        tree->setCurrentItem(c_item);
    }
}

void CDriveInfo::Update(const utf16_t* DriveName,const utf16_t* DiskName,AP_DiskFsFlags FsFlags,const void* DiskData,unsigned int DiskDataSize)
{
    valid = true;
    name = QStringFromUtf16(DriveName);
    label = QStringFromUtf16(DiskName);

    fs_flags = FsFlags;

    bool need_prot;

    if (0==(FsFlags&(AP_DskFsFlagDvdFilesPresent|AP_DskFsFlagHdvdFilesPresent|AP_DskFsFlagBlurayFilesPresent)))
    {
        show_open = false;

        if (0!=(FsFlags&AP_DskFsFlagDiskIsLoading))
        {
            type=QString(UI_QSTRING(APP_IFACE_DRIVEINFO_LOADING));
            label=QString(UI_QSTRING(APP_IFACE_DRIVEINFO_WAIT));
            disk_type=dtLoading;
        } else {
            if (0!=(FsFlags&AP_DskFsFlagDiskIsAbsent))
            {
                type=QString(UI_QSTRING(APP_IFACE_DRIVEINFO_NODISC));
                label=QString();
                disk_type=dtUnknown;
            } else {
                type=QString(UI_QSTRING(APP_IFACE_DRIVEINFO_DATADISC));
                disk_type=dtUnknown;
            }
        }
        prot.clear();
        need_prot=false;
    } else {
        show_open = true;

        // type
        type.clear();
        disk_type=dtUnknown;
        if (0!=(FsFlags&AP_DskFsFlagBlurayFilesPresent))
        {
            type=QLatin1String("Blu-ray");
            disk_type=dtBluray;
        } else {
            if (0!=(FsFlags&AP_DskFsFlagHdvdFilesPresent))
            {
                type = QLatin1String("HD-DVD");
                disk_type=dtHdvd;
            } else {
                if (0!=(FsFlags&AP_DskFsFlagDvdFilesPresent))
                {
                    type = QLatin1String("DVD");
                    disk_type=dtDvd;
                }
            }
        }

        // prot
        need_prot=true;
    }
    QString tprot;

    // for all disks
    if ( (0==DiskDataSize) || (false==FormatDriveDiskInfo(tprot,right_info,DiskData,DiskDataSize,FsFlags)) )
    {
        tprot.clear();
        right_info=UI_QSTRING(APP_IFACE_DRIVEINFO_NONE);
    }
    if (need_prot) prot = tprot;
}

void QGrayTextViewer::setHtmlBody(const QString& str)
{
    QString txt;
    char tclr[10];

    // #AARRGGBB
    unsigned int clr = palette().window().color().rgb();
    sprintf(tclr,"#%02X%02X%02X",(clr>>16)&0xff,(clr>>8)&0xff,(clr>>0)&0xff);

    txt.reserve(str.size()+30);

    append_const(txt,"<body bgcolor='");
    append_const(txt,tclr);
    append_const(txt,"'>");
    txt+=str;
    append_const(txt,"</body>");

    setHtml(txt);
}


