// license:BSD-3-Clause
// copyright-holders:Olivier Galibert, R. Belmont
//============================================================
//
//  window.h - SDL window handling
//
//  SDLMAME by Olivier Galibert and R. Belmont
//
//============================================================

#ifndef __SDLWINDOW__
#define __SDLWINDOW__

#include "osdsdl.h"
#include "video.h"

#include "modules/osdwindow.h"

#include <cstdint>
#include <memory>
#include <list>


//============================================================
//  TYPE DEFINITIONS
//============================================================

// forward of SDL_DisplayMode not possible (typedef struct) - define wrapper

class SDL_DM_Wrapper;

typedef uintptr_t HashT;

#define OSDWORK_CALLBACK(name)  void *name(void *param, ATTR_UNUSED int threadid)

class sdl_window_info : public osd_window
{
public:
	sdl_window_info(running_machine &a_machine, int index, osd_monitor_info *a_monitor,
			const osd_window_config *config);

	~sdl_window_info();

	int window_init();

	void update();
	void toggle_full_screen();
	void modify_prescale(int dir);
	void resize(INT32 width, INT32 height);
	void destroy();

	void capture_pointer() override;
	void release_pointer() override;
	void show_pointer() override;
	void hide_pointer() override;

	void notify_changed();

	osd_dim get_size() override;

	int xy_to_render_target(int x, int y, int *xt, int *yt);

	running_machine &machine() const override { return m_machine; }
	osd_monitor_info *monitor() const override { return m_monitor; }
	int fullscreen() const override { return m_fullscreen; }

	render_target *target() override { return m_target; }

	int prescale() const { return m_prescale; }

	// Pointer to next window
	sdl_window_info *   m_next;

private:
	// window handle and info
	char                m_title[256];
	int                 m_startmaximized;

	// dimensions
	osd_dim             m_minimum_dim;
	osd_dim             m_windowed_dim;

	// rendering info
	osd_event           m_rendered_event;
	render_target *     m_target;

	// Original display_mode
	SDL_DM_Wrapper      *m_original_mode;

	int                 m_extra_flags;

	static OSDWORK_CALLBACK( complete_create_wt );

private:
	int wnd_extra_width();
	int wnd_extra_height();
	void set_starting_view(int index, const char *defview, const char *view);
	osd_rect constrain_to_aspect_ratio(const osd_rect &rect, int adjustment);
	osd_dim get_min_bounds(int constrain);
	osd_dim get_max_bounds(int constrain);
	void update_cursor_state();
	osd_dim pick_best_mode();
	void set_fullscreen(int afullscreen) { m_fullscreen = afullscreen; }

	// Pointer to machine
	running_machine &   m_machine;
	// monitor info
	osd_monitor_info *  m_monitor;
	int                 m_fullscreen;

	// static callbacks ...

	static OSDWORK_CALLBACK( sdlwindow_resize_wt );
	static OSDWORK_CALLBACK( draw_video_contents_wt );
	static OSDWORK_CALLBACK( sdlwindow_video_window_destroy_wt );
	static OSDWORK_CALLBACK( sdlwindow_toggle_full_screen_wt );
	static OSDWORK_CALLBACK( notify_changed_wt );
	static OSDWORK_CALLBACK( update_cursor_state_wt );

	void measure_fps(int update);

};

struct osd_draw_callbacks
{
	osd_renderer *(*create)(osd_window *window);
};

//============================================================
//  GLOBAL VARIABLES
//============================================================

// window - list
extern std::list<std::shared_ptr<sdl_window_info>> sdl_window_list;

//============================================================
//  PROTOTYPES
//============================================================

//============================================================
// PROTOTYPES - drawsdl.c
//============================================================

int drawsdl_scale_mode(const char *s);

//============================================================
// PROTOTYPES - drawogl.c
//============================================================

int drawogl_init(running_machine &machine, osd_draw_callbacks *callbacks);

//============================================================
// PROTOTYPES - draw13.c
//============================================================

int drawsdl2_init(running_machine &machine, osd_draw_callbacks *callbacks);

//============================================================
// PROTOTYPES - drawbgfx.c
//============================================================

int drawbgfx_init(running_machine &machine, osd_draw_callbacks *callbacks);

#endif /* __SDLWINDOW__ */
