// license:BSD-3-Clause
// copyright-holders:Sergey Svishchev
/***************************************************************************

    15IE-00-013 Terminal

    A serial (RS232 or current loop) green-screen terminal, mostly VT52
    compatible (no Hold Screen mode and no graphics character set).

    Alternate character set (selected by SO/SI chars) is Cyrillic.

****************************************************************************/


#include "emu.h"

#include "bus/rs232/rs232.h"
#include "cpu/ie15/ie15.h"
#include "machine/ie15_kbd.h"
#include "sound/beep.h"
#include "ie15.lh"

#define SCREEN_PAGE (80*48)

#define IE_TRUE         0x80
#define IE_FALSE    0

#define IE15_TOTAL_HORZ 1000
#define IE15_DISP_HORZ  800
#define IE15_HORZ_START 200

#define IE15_TOTAL_VERT (28*11)
#define IE15_DISP_VERT  (25*11)
#define IE15_VERT_START (2*11)
#define IE15_STATUSLINE 11


#define VERBOSE_DBG 1       /* general debug messages */

#define DBG_LOG(N,M,A) \
	do { \
		if(VERBOSE_DBG>=N) \
		{ \
			if( M ) \
				logerror("%11.6f at %s: %-24s",machine().time().as_double(),machine().describe_context(),(char*)M ); \
			logerror A; \
		} \
	} while (0)


class ie15_state : public driver_device,
	public device_serial_interface
{
public:
	ie15_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag)
		, device_serial_interface(mconfig, *this)
		, m_maincpu(*this, "maincpu")
		, m_p_videoram(*this, "video")
		, m_p_chargen(*this, "chargen")
		, m_beeper(*this, "beeper")
		, m_rs232(*this, "rs232")
		, m_screen(*this, "screen")
		, m_io_keyboard(*this, "keyboard")
	{ }

	DECLARE_WRITE16_MEMBER( kbd_put );
	DECLARE_WRITE_LINE_MEMBER( serial_rx_callback );
	DECLARE_WRITE8_MEMBER( mem_w );
	DECLARE_READ8_MEMBER( mem_r );
	DECLARE_WRITE8_MEMBER( mem_addr_lo_w );
	DECLARE_WRITE8_MEMBER( mem_addr_hi_w );
	DECLARE_WRITE8_MEMBER( mem_addr_inc_w );
	DECLARE_WRITE8_MEMBER( mem_addr_dec_w );
	DECLARE_READ8_MEMBER( flag_r );
	DECLARE_WRITE8_MEMBER( flag_w );
	DECLARE_WRITE8_MEMBER( beep_w );
	DECLARE_READ8_MEMBER( kb_r );
	DECLARE_READ8_MEMBER( kb_ready_r );
	DECLARE_READ8_MEMBER( kb_s_red_r );
	DECLARE_READ8_MEMBER( kb_s_sdv_r );
	DECLARE_READ8_MEMBER( kb_s_dk_r );
	DECLARE_READ8_MEMBER( kb_s_dupl_r );
	DECLARE_READ8_MEMBER( kb_s_lin_r );
	DECLARE_WRITE8_MEMBER( kb_ready_w );
	DECLARE_READ8_MEMBER( serial_tx_ready_r );
	DECLARE_WRITE8_MEMBER( serial_w );
	DECLARE_READ8_MEMBER( serial_rx_ready_r );
	DECLARE_READ8_MEMBER( serial_r );
	DECLARE_WRITE8_MEMBER( serial_speed_w );
	DECLARE_PALETTE_INIT( ie15 );
	TIMER_CALLBACK_MEMBER(ie15_beepoff);
	uint32_t screen_update(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect);

private:
	static const device_timer_id TIMER_HBLANK = 0;
	void scanline_callback();
	void update_leds();
	void draw_scanline(uint32_t *p, uint16_t offset, uint8_t scanline);
	std::unique_ptr<uint32_t[]> m_tmpbmp;

	emu_timer *m_hblank_timer;

	uint8_t m_long_beep;
	uint8_t m_kb_control;
	uint8_t m_kb_data;
	uint8_t m_kb_flag0;
	uint8_t m_kb_flag;
	uint8_t m_kb_ruslat;
	uint8_t m_latch;

	struct {
		uint8_t cursor;
		uint8_t enable;
		uint8_t line25;
		uint32_t ptr1;
		uint32_t ptr2;
	} m_video;

	uint8_t m_serial_rx_ready;
	uint8_t m_serial_tx_ready;
	int m_hblank;
	int m_vpos;
	int m_marker_scanline;

	virtual void machine_start() override;
	virtual void machine_reset() override;
	virtual void video_start() override;
	virtual void rcv_complete() override;
	virtual void tra_callback() override;
	virtual void tra_complete() override;
	virtual void device_timer(emu_timer &timer, device_timer_id id, int param, void *ptr) override;

	required_device<cpu_device> m_maincpu;
	required_region_ptr<u8> m_p_videoram;
	required_region_ptr<u8> m_p_chargen;
	required_device<beep_device> m_beeper;
	required_device<rs232_port_device> m_rs232;
	required_device<screen_device> m_screen;
	required_ioport m_io_keyboard;
};

READ8_MEMBER( ie15_state::mem_r ) {
	uint8_t ret;

	ret = m_p_videoram[m_video.ptr1];
	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0 && m_video.ptr1 >= SCREEN_PAGE)
	{
		DBG_LOG(2,"memory",("R @ %03x == %02x\n", m_video.ptr1, ret));
	}
	m_video.ptr1++;
	m_video.ptr1 &= 0xfff;
	m_latch = 0;
	return ret;
}

WRITE8_MEMBER( ie15_state::mem_w ) {
	if ((m_latch ^= 1) == 0) {
		if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0 && m_video.ptr1 >= SCREEN_PAGE)
		{
			DBG_LOG(2,"memory",("W @ %03x <- %02x\n", m_video.ptr1, data));
		}
		m_p_videoram[m_video.ptr1++] = data;
		m_video.ptr1 &= 0xfff;
	}
}

WRITE8_MEMBER( ie15_state::mem_addr_inc_w ) {
	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0)
	{
		DBG_LOG(2,"memory",("++ %03x\n", m_video.ptr1));
	}
	m_video.ptr1++;
	m_video.ptr1 &= 0xfff;
	if (m_video.enable)
		m_video.ptr2 = m_video.ptr1;
}

WRITE8_MEMBER( ie15_state::mem_addr_dec_w ) {
	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0)
	{
		DBG_LOG(2,"memory",("-- %03x\n", m_video.ptr1));
	}
	m_video.ptr1--;
	m_video.ptr1 &= 0xfff;
	if (m_video.enable)
		m_video.ptr2 = m_video.ptr1;
}

WRITE8_MEMBER( ie15_state::mem_addr_lo_w ) {
	uint16_t tmp = m_video.ptr1;

	tmp &= 0xff0;
	tmp |= ((data >> 4) & 0xf);
	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0)
	{
		DBG_LOG(2,"memory",("lo %03x <- %02x = %03x\n", m_video.ptr1, data, tmp));
	}
	m_video.ptr1 = tmp;
	if (m_video.enable)
		m_video.ptr2 = tmp;
}

WRITE8_MEMBER( ie15_state::mem_addr_hi_w ) {
	uint16_t tmp = m_video.ptr1;

	tmp &= 0xf;
	tmp |= (data << 4);
	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0)
	{
		DBG_LOG(2,"memory",("hi %03x <- %02x = %03x\n", m_video.ptr1, data, tmp));
	}
	m_video.ptr1 = tmp;
	if (m_video.enable)
		m_video.ptr2 = tmp;
}

TIMER_CALLBACK_MEMBER(ie15_state::ie15_beepoff)
{
	m_beeper->set_state(0);
}

WRITE8_MEMBER( ie15_state::beep_w ) {
	uint16_t length = (m_long_beep & IE_TRUE) ? 150 : 400;

	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0)
	{
		DBG_LOG(1,"beep",("(%s)\n", m_long_beep ? "short" : "long"));
	}
	machine().scheduler().timer_set(attotime::from_msec(length), timer_expired_delegate(FUNC(ie15_state::ie15_beepoff),this));
	m_beeper->set_state(1);
}

/* keyboard */

// active high
READ8_MEMBER( ie15_state::kb_r ) {
	DBG_LOG(2,"keyboard",("R %02X '%c'\n", m_kb_data, m_kb_data < 0x20 ? ' ' : m_kb_data));
	return m_kb_data;
}

// active low
READ8_MEMBER( ie15_state::kb_ready_r ) {
	m_kb_flag &= IE_TRUE;
	if (m_kb_flag != m_kb_flag0) {
		DBG_LOG(2,"keyboard",("? %c\n", m_kb_flag ? 'n' : 'y'));
		m_kb_flag0 = m_kb_flag;
	}
	return m_kb_flag;
}

// active low
WRITE8_MEMBER( ie15_state::kb_ready_w ) {
	DBG_LOG(2,"keyboard",("clear ready\n"));
	m_kb_flag = IE_TRUE | IE_KB_ACK;
}


// active high; active = interpret controls, inactive = display controls
READ8_MEMBER( ie15_state::kb_s_red_r ) {
	return m_io_keyboard->read() & IE_KB_RED ? IE_TRUE : 0;
}

// active high; active = setup mode
READ8_MEMBER( ie15_state::kb_s_sdv_r ) {
	return m_kb_control & IE_KB_SDV ? IE_TRUE : 0;
}

// active high; active = keypress detected on aux keypad
READ8_MEMBER( ie15_state::kb_s_dk_r ) {
	return m_kb_control & IE_KB_DK ? IE_TRUE : 0;
}

// active low; active = full duplex, inactive = half duplex
READ8_MEMBER( ie15_state::kb_s_dupl_r ) {
	return m_io_keyboard->read() & IE_KB_DUP ? IE_TRUE : 0;
}

// active high; active = on-line, inactive = local editing
READ8_MEMBER( ie15_state::kb_s_lin_r ) {
	return m_io_keyboard->read() & IE_KB_LIN ? IE_TRUE : 0;
}

/* serial port */

void ie15_state::device_timer(emu_timer &timer, device_timer_id id, int param, void *ptr)
{
	device_serial_interface::device_timer(timer, id, param, ptr);

	switch(id)
	{
		case TIMER_HBLANK:
			if (m_hblank) // Transitioning from in blanking to out of blanking
			{
				m_hblank = 0;
				m_hblank_timer->adjust(m_screen->time_until_pos((m_vpos+1) % IE15_TOTAL_VERT, 0));
				scanline_callback();
			}
			else // Transitioning from out of blanking to in blanking
			{
				m_hblank = 1;
				m_hblank_timer->adjust(m_screen->time_until_pos(m_vpos, IE15_HORZ_START));
			}
			break;
	}
}

WRITE_LINE_MEMBER( ie15_state::serial_rx_callback )
{
	device_serial_interface::rx_w(state);
}

void ie15_state::rcv_complete()
{
	receive_register_extract();
	m_serial_rx_ready = IE_FALSE;
}

void ie15_state::tra_callback()
{
	uint8_t bit = transmit_register_get_data_bit();
	m_rs232->write_txd(bit);
}

void ie15_state::tra_complete()
{
	m_serial_tx_ready = IE_TRUE;
}

// active low
READ8_MEMBER( ie15_state::serial_rx_ready_r ) {
	return m_serial_rx_ready;
}

// active high
READ8_MEMBER( ie15_state::serial_tx_ready_r ) {
	return m_serial_tx_ready;
}

// not called unless data are ready
READ8_MEMBER( ie15_state::serial_r ) {
	uint8_t data;

	data = get_received_char();
	m_serial_rx_ready = IE_TRUE;
	DBG_LOG(1,"serial",("R %02X '%c'\n", data, data < 0x20?' ':data));
	return data;
}

WRITE8_MEMBER( ie15_state::serial_w ) {
	DBG_LOG(1,"serial",("W %02X '%c'\n", data, data < 0x20?' ':data));

	m_serial_tx_ready = IE_FALSE;
	transmit_register_setup(data);
}

WRITE8_MEMBER( ie15_state::serial_speed_w ) {
	return;
}

READ8_MEMBER( ie15_state::flag_r ) {
	switch (offset)
	{
		case 0: // hsync pulse (not hblank)
			return m_hblank;
		case 1: // marker scanline
			return m_marker_scanline;
		case 2: // vblank
			return !m_screen->vblank();
		case 4:
			return m_kb_ruslat;
		default:
			break;
	}
	if (machine().debug_flags & DEBUG_FLAG_ENABLED)
	{
		DBG_LOG(2,"flag",("read %d: ?\n", offset));
	}
	return 0;
}

WRITE8_MEMBER( ie15_state::flag_w ) {
	switch (offset)
	{
		case 0:
			m_video.enable = data;
			break;
		case 1:
			m_video.cursor = data;
			break;
		case 2:
			m_long_beep = data;
			break;
		case 3:
			m_video.line25 = data;
			break;
		case 4:
			m_kb_ruslat = data;
			break;
		default:
			break;
	}
	if ((machine().debug_flags & DEBUG_FLAG_ENABLED) != 0 && !offset)
	{
		DBG_LOG(2,"flag",("%sset %d\n", data?"":"re", offset));
	}
}

static ADDRESS_MAP_START( ie15_mem, AS_PROGRAM, 8, ie15_state )
	ADDRESS_MAP_UNMAP_HIGH
	AM_RANGE(0x0000, 0x0fff) AM_ROM
ADDRESS_MAP_END

static ADDRESS_MAP_START( ie15_io, AS_IO, 8, ie15_state )
	ADDRESS_MAP_UNMAP_HIGH
	AM_RANGE(000, 000) AM_READWRITE(mem_r, mem_w)   // 00h W: memory request, R: memory data [6.1.2.2]
	AM_RANGE(001, 001) AM_READ(serial_rx_ready_r) AM_WRITENOP   // 01h W: memory latch [6.1.2.2]
	AM_RANGE(002, 002) AM_WRITE(mem_addr_hi_w)      // 02h W: memory address high [6.1.2.2]
	AM_RANGE(003, 003) AM_WRITE(mem_addr_lo_w)      // 03h W: memory address low [6.1.2.2]
	AM_RANGE(004, 004) AM_WRITE(mem_addr_inc_w)     // 04h W: memory address counter + [6.1.2.2]
	AM_RANGE(005, 005) AM_WRITE(mem_addr_dec_w)     // 05h W: memory address counter - [6.1.2.2]
	AM_RANGE(006, 006) AM_READWRITE(serial_r, serial_w)     // 06h W: serial port data [6.1.5.4]
// port 7 is handled in cpu core
	AM_RANGE(010, 010) AM_READWRITE(serial_tx_ready_r, beep_w)  // 08h W: speaker control [6.1.5.4]
	AM_RANGE(011, 011) AM_READ(kb_r)            // 09h R: keyboard data [6.1.5.2]
	AM_RANGE(012, 012) AM_READ(kb_s_red_r)          // 0Ah I: keyboard mode "RED" [6.1.5.2]
	AM_RANGE(013, 013) AM_READ(kb_ready_r)          // 0Bh R: keyboard data ready [6.1.5.2]
	AM_RANGE(014, 014) AM_READWRITE(kb_s_sdv_r, serial_speed_w) // 0Ch W: serial port speed [6.1.3.1], R: keyboard mode "SDV" [6.1.5.2]
	AM_RANGE(015, 015) AM_READWRITE(kb_s_dk_r, kb_ready_w)  // 0Dh I: keyboard mode "DK" [6.1.5.2]
	AM_RANGE(016, 016) AM_READ(kb_s_dupl_r)         // 0Eh I: keyboard mode "DUPL" [6.1.5.2]
	AM_RANGE(017, 017) AM_READ(kb_s_lin_r)          // 0Fh I: keyboard mode "LIN" [6.1.5.2]
// simulation of flag registers
	AM_RANGE(020, 027) AM_READWRITE(flag_r, flag_w)
ADDRESS_MAP_END

/* Input ports */
static INPUT_PORTS_START( ie15 )
	PORT_START("keyboard")
	PORT_DIPNAME(IE_KB_RED, IE_KB_RED, "RED (Interpret controls)")
	PORT_DIPSETTING(0x00, "Off")
	PORT_DIPSETTING(IE_KB_RED, "On")
	PORT_DIPNAME(IE_KB_DUP, IE_KB_DUP, "DUP (Full duplex)")
	PORT_DIPSETTING(0x00, "Off")
	PORT_DIPSETTING(IE_KB_DUP, "On")
	PORT_DIPNAME(IE_KB_LIN, IE_KB_LIN, "LIN (Online)")
	PORT_DIPSETTING(0x00, "Off")
	PORT_DIPSETTING(IE_KB_LIN, "On")
INPUT_PORTS_END

WRITE16_MEMBER( ie15_state::kbd_put )
{
	DBG_LOG(2,"keyboard",("W %02X<-%02X '%c' %02X (%c)\n", m_kb_data, data, 'x' /* data < 0x20 ? ' ' : (data & 255) */,
		m_kb_flag, m_kb_flag ? 'n' : 'y'));
	m_kb_control = (data >> 8) & 255;
	// send new key only when firmware has processed previous one
	if (m_kb_flag == IE_TRUE) {
		m_kb_data = data & 255;
		m_kb_flag = 0;
	}
}

void ie15_state::machine_start()
{
	m_hblank_timer = timer_alloc(TIMER_HBLANK);
	m_hblank_timer->adjust(attotime::never);
}

void ie15_state::machine_reset()
{
	memset(&m_video, 0, sizeof(m_video));
	m_kb_ruslat = m_long_beep = m_kb_control = m_kb_data = m_kb_flag0 = 0;
	m_kb_flag = IE_TRUE;

	m_hblank = 1;
	m_hblank_timer->adjust(m_screen->time_until_pos(0, IE15_HORZ_START));
	m_vpos = m_screen->vpos();
	m_marker_scanline = (m_vpos % 11) > 7;

	m_beeper->set_state(0);

	m_serial_tx_ready = m_serial_rx_ready = IE_TRUE;
	set_data_frame(1 /* start bits */, 8 /* data bits */, PARITY_NONE, STOP_BITS_1);
	// device supports rates from 150 to 9600 baud but null_modem has hardcoded 9600
	set_rate(9600);
}

void ie15_state::video_start()
{
	m_video.ptr1 = m_video.ptr2 = m_latch = 0;

	m_tmpbmp = std::make_unique<uint32_t[]>(IE15_TOTAL_HORZ * IE15_TOTAL_VERT);
}

/*
    Usable raster is 800 x 275 pixels (80 x 25 characters).  24 lines are
    available to the user and 25th (topmost) line is the status line.
    Status line, if enabled, displays current serial port speed, 16 setup
    bits, and clock.  There is no NVRAM, so setup bits are always 0 after
    reset and clock starts counting at 0 XXX.

    No character attributes are available, but in 'display controls' mode
    control characters stored in memory are shown as blinking chars.

    Character cell is 10 x 11; character generator provides 7 x 8 of that.
    3 extra horizontal pixels are always blank.  Blinking cursor may be
    displayed on 3 extra scan lines.

    On each scan line, video board draws 80 characters from any location
    in video memory; this is used by firmware to provide instant scroll
    and cursor, which is a character with code 0x7F stored in XXX.

    Video board output is controlled by
    - control flag 0 "disable video": 0 == disable
    - control flag 1 "cursor": 0 == if this scan line is one of extra 3,
      enable video every 5 frames.
    - control flag 3 "status line": 0 == current scan line is part of status line
    - keyboard mode 'RED' ('display controls'): if character code is
      less than 0x20 and RED is set, enable video every 5 frames; if RED is
      unset, disable video.

    XXX 'dotted look' is caused by strobe at 2x pixel clock -- use HLSL for this.
*/

void ie15_state::draw_scanline(uint32_t *p, uint16_t offset, uint8_t scanline)
{
	static const uint32_t palette[2] = { 0xff000000, 0xff00c000 };

	uint8_t ra = scanline % 8;
	uint32_t ra_high = 0x200 | ra;
	bool blink((m_screen->frame_number() % 10) > 4);
	bool red(m_io_keyboard->read() & IE_KB_RED);
	bool blink_red_line25 = blink && red && m_video.line25;
	bool cursor_blank = scanline > 7 && (!m_video.cursor || blink);

	if (cursor_blank)
	{
		for (uint16_t x = 0; x < 80*10; x++)
		{
			*p++ = palette[0];
		}
	}
	else
	{
		for (uint16_t x = offset; x < offset + 80; x++)
		{
			uint16_t chr = m_p_videoram[x] << 3;
			uint8_t gfx = m_p_chargen[chr | ra];

			if (chr < (0x20<<3))
			{
				if (blink_red_line25)
					gfx = m_p_chargen[chr | ra_high];
				else
					gfx = 0;
			}

			*p++ = palette[BIT(gfx, 7)];
			*p++ = palette[BIT(gfx, 6)];
			*p++ = palette[BIT(gfx, 5)];
			*p++ = palette[BIT(gfx, 4)];
			*p++ = palette[BIT(gfx, 3)];
			*p++ = palette[BIT(gfx, 2)];
			*p++ = palette[BIT(gfx, 1)];
			*p++ = palette[0];
			*p++ = palette[0];
			*p++ = palette[0];
		}
	}
}

void ie15_state::update_leds()
{
	uint8_t data = m_io_keyboard->read();
	output().set_value("lat_led", m_kb_ruslat ^ 1);
	output().set_value("nr_led", BIT(m_kb_control, IE_KB_NR_BIT) ^ 1);
	output().set_value("pch_led", BIT(data, IE_KB_PCH_BIT) ^ 1);
	output().set_value("dup_led", BIT(data, IE_KB_DUP_BIT) ^ 1);
	output().set_value("lin_led", BIT(data, IE_KB_LIN_BIT) ^ 1);
	output().set_value("red_led", BIT(data, IE_KB_RED_BIT) ^ 1);
	output().set_value("sdv_led", BIT(m_kb_control, IE_KB_SDV_BIT) ^ 1);
	output().set_value("prd_led", 1); // XXX
}

/*
    VBlank is active for 3 topmost on-screen rows and 1 at the bottom; however, control flag 3 overrides VBlank,
    allowing status line to be switched on and off.
*/
void ie15_state::scanline_callback()
{
	int y = m_vpos;

	m_vpos++;
	m_vpos %= IE15_TOTAL_VERT;
	m_marker_scanline = (m_vpos % 11) > 7;

	DBG_LOG(3,"scanline_cb",
		("addr %03x frame %d x %.4d y %.3d row %.2d e:c:s %d:%d:%d\n",
		m_video.ptr2, (int)m_screen->frame_number(), m_screen->hpos(), y,
		y%11, m_video.enable, m_video.cursor, m_video.line25));

	if (y < IE15_VERT_START) return;
	y -= IE15_VERT_START;
	if (y >= IE15_DISP_VERT) return;

	if (!m_video.enable || (y < IE15_STATUSLINE && m_video.line25)) {
		memset(&m_tmpbmp[(y + IE15_VERT_START) * IE15_TOTAL_HORZ], 0, sizeof(uint32_t) * IE15_TOTAL_HORZ);
	} else {
		draw_scanline(&m_tmpbmp[(y + IE15_VERT_START) * IE15_TOTAL_HORZ + IE15_HORZ_START], m_video.ptr2, y % 11);
	}
}

uint32_t ie15_state::screen_update(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect)
{
	update_leds();
	memcpy(&bitmap.pix32(0), &m_tmpbmp[0], sizeof(uint32_t) * IE15_TOTAL_HORZ * IE15_TOTAL_VERT);
	return 0;
}


/* F4 Character Displayer */
static const gfx_layout ie15_charlayout =
{
	7, 8,                   /* 7x8 pixels in 10x11 cell */
	256,                    /* 256 characters */
	1,                  /* 1 bits per pixel */
	{ 0 },                  /* no bitplanes */
	/* x offsets */
	{ 0, 1, 2, 3, 4, 5, 6 },
	/* y offsets */
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8 },
	8*8                 /* every char takes 8 bytes */
};

static GFXDECODE_START( ie15 )
	GFXDECODE_ENTRY("chargen", 0x0000, ie15_charlayout, 0, 1)
GFXDECODE_END

PALETTE_INIT_MEMBER( ie15_state, ie15 )
{
	palette.set_pen_color(0, rgb_t::black()); // black
	palette.set_pen_color(1, 0x00, 0xc0, 0x00); // green
}

static MACHINE_CONFIG_START( ie15, ie15_state )
	/* Basic machine hardware */
	MCFG_CPU_ADD("maincpu", IE15, XTAL_30_8MHz/10)
	MCFG_CPU_PROGRAM_MAP(ie15_mem)
	MCFG_CPU_IO_MAP(ie15_io)

	/* Video hardware */
	MCFG_SCREEN_ADD_MONOCHROME("screen", RASTER, rgb_t::green())
	MCFG_SCREEN_UPDATE_DRIVER(ie15_state, screen_update)
	MCFG_SCREEN_RAW_PARAMS(XTAL_30_8MHz/2, IE15_TOTAL_HORZ, IE15_HORZ_START,
		IE15_HORZ_START+IE15_DISP_HORZ, IE15_TOTAL_VERT, IE15_VERT_START,
		IE15_VERT_START+IE15_DISP_VERT);

	MCFG_DEFAULT_LAYOUT(layout_ie15)

	MCFG_GFXDECODE_ADD("gfxdecode", "palette", ie15)
	MCFG_PALETTE_ADD_MONOCHROME("palette")

	/* Devices */
	MCFG_DEVICE_ADD("keyboard", IE15_KEYBOARD, 0)
	MCFG_IE15_KEYBOARD_CB(WRITE16(ie15_state, kbd_put))

	MCFG_RS232_PORT_ADD("rs232", default_rs232_devices, "null_modem")
	MCFG_RS232_RXD_HANDLER(WRITELINE(ie15_state, serial_rx_callback))

	MCFG_SPEAKER_STANDARD_MONO("mono")
	MCFG_SOUND_ADD("beeper", BEEP, 2400)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.15)
MACHINE_CONFIG_END


/* ROM definition */
ROM_START( ie15 )
	ROM_REGION(0x1000, "maincpu", ROMREGION_ERASE00)
	ROM_DEFAULT_BIOS("5chip")
	ROM_SYSTEM_BIOS(0, "5chip", "5-chip firmware (newer)")
	ROMX_LOAD("dump1.bin", 0x0000, 0x1000, CRC(14b82284) SHA1(5ac4159fbb1c3b81445605e26cd97a713ae12b5f), ROM_BIOS(1))
	ROM_SYSTEM_BIOS(1, "6chip", "6-chip firmware (older)")
	ROMX_LOAD("dump5.bin", 0x0000, 0x1000, CRC(01f2e065) SHA1(2b72dc0594e38a528400cd25aed0c47e0c432895), ROM_BIOS(2))

	ROM_REGION(0x1000, "video", ROMREGION_ERASE00)

	ROM_REGION(0x0800, "chargen", ROMREGION_ERASE00)
	ROM_LOAD("chargen-15ie.bin", 0x0000, 0x0800, CRC(ed16bf6b) SHA1(6af9fb75f5375943d5c0ce9ed408e0fb4621b17e))
ROM_END

/* Driver */

/*    YEAR  NAME      PARENT  COMPAT   MACHINE    INPUT    INIT                      COMPANY     FULLNAME       FLAGS */
COMP( 1980, ie15,     0,      0,       ie15,      ie15,    driver_device,     0,     "USSR",     "15IE-00-013", 0)
