// license:BSD-3-Clause
// copyright-holders:Manuel Abadia, Ernesto Corvi, Nicola Salmoria
/***************************************************************************

Galaga 3 / Gaplus (c) 1984 Namco

driver by Manuel Abadia, Ernesto Corvi, Nicola Salmoria


Custom ICs:
----------
11XX     gfx data shifter and mixer (16-bit in, 4-bit out) [1]
15XX     sound control
16XX     I/O control
CUS20    tilemap and sprite address generator
CUS21    sprite generator
CUS26    starfield generator
CUS29    sprite line buffer and sprite/tilemap mixer
CUS33    timing generator
CUS34    address decoder
56XX     I/O
58XX     I/O
CUS62    I/O and explosion generator
98XX     lamp/coin output
99XX     sound volume


memory map
----------
Most of the address decoding for main and sound CPU is done by a custom IC (34XX),
so the memory map is largely deducted by program behaviour. The 34XX also handles
internally the main and sub irq, and a watchdog.
Most of the address decoding for sub CPU is done by a PAL which was read and
decoded, but there are some doubts about its validity.
There is also some additional decoding for tile/sprite RAM done by the 20XX
tilemap and sprite address generator.

Note: chip positions are based on the Midway version schematics. The Namco
version has a different layout (see later for the known correspondencies)

MAIN CPU:

Address          Dir Data     Name      Description
---------------- --- -------- --------- -----------------------
00000xxxxxxxxxxx R/W xxxxxxxx RAM 9J    tilemap RAM (shared with sub CPU)
00001xxxxxxxxxxx R/W xxxxxxxx RAM 3M    work RAM (shared with sub CPU)
000011111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (sprite number & color)
00010xxxxxxxxxxx R/W xxxxxxxx RAM 3K    work RAM (shared with sub CPU)
000101111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (x, y)
00011xxxxxxxxxxx R/W xxxxxxxx RAM 3L    work RAM (shared with sub CPU)
000111111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (x msb, flip, size)
01100-xxxxxxxxxx R/W xxxxxxxx SOUND     RAM (shared with sound CPU)
01101-----xxxxxx R/W ----xxxx FBIT      I/O chips
0111x-----------   W --------           main CPU irq enable (data is in A11) (MIRQ generated by 34XX)
01111----------- R   --------           watchdog reset (MRESET generated by 34XX)
1000x-----------   W -------- SRESET    reset sub and sound CPU, sound enable (data is in A11) (latch in 34XX)
1001x-----------   W -------- FRESET    reset I/O chips (data is in A11) (latch in 34XX)
10100---------xx   W xxxxxxxx STWR      to custom 26XX (starfield control)
10-xxxxxxxxxxxxx R   xxxxxxxx ROM 9E    program ROM (can optionally be a 27128)
110xxxxxxxxxxxxx R   xxxxxxxx ROM 9D    program ROM
111xxxxxxxxxxxxx R   xxxxxxxx ROM 9C    program ROM

[1] Program uses addresses with A10 = 1, e.g. 7400, 7c00, but A10 is not used.
On startup, it also writes to 7820-782f. This might be a bug, the intended range
being 6820-682f to address the 3rd I/O chip.


SOUND CPU:

Address          Dir Data     Name      Description
---------------- --- -------- --------- -----------------------
000---xxxxxxxxxx R/W xxxxxxxx SOUND2    RAM (shared with main CPU)
001------------- R/W --------           watchdog reset? (34XX) [1]
01x-------------   W --------           sound CPU irq enable (data is in A13) (SIRQ generated by 34XX)
11-xxxxxxxxxxxxx R   xxxxxxxx ROM 7B    program ROM (can optionally be a 27128)

[1] Program writes to 3000 and on startup reads from 3000.
On startup it also writes to 2007, but there doesn't seem to be anything else there.


SUB CPU:

Address          Dir Data     Name      Description
---------------- --- -------- --------- -----------------------
00000xxxxxxxxxxx R/W xxxxxxxx RAM 9J    tilemap RAM (shared with main CPU)
00001xxxxxxxxxxx R/W xxxxxxxx RAM 3M    work RAM (shared with main CPU)
000011111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (sprite number & color)
00010xxxxxxxxxxx R/W xxxxxxxx RAM 3K    work RAM (shared with main CPU)
000101111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (x, y)
00011xxxxxxxxxxx R/W xxxxxxxx RAM 3L    work RAM (shared with main CPU)
000111111xxxxxxx R/W xxxxxxxx           portion holding sprite registers (x msb, flip, size)
0110-----------x     -------- VINTON    sub CPU irq enable (data is in A0) [1]
10-xxxxxxxxxxxxx R   xxxxxxxx ROM 6L    program ROM (can optionally be a 27128)
110xxxxxxxxxxxxx R   xxxxxxxx ROM 6M    program ROM
111xxxxxxxxxxxxx R   xxxxxxxx ROM 6N    program ROM

[1] Program normally uses 6080/6081, but 6001 is written on startup.
500F is also written on startup, whose meaning is unknown.


ROM chip placements
-------------------
Midway  Namco
------  -----
9C      8B
9D      8C
9E      8D
6N      11B
6M      11C
6L      11D
7B      4B
9L      8S
5K      11R
5L      11N
5M      11P
5N      11M

----------------------------------------------------------------------------


Notes:
------
- Easter egg:
  - enter service mode
  - keep P1 start and P1 button pressed
  - move joystick left until sound reaches 19
  (c) 1984 NAMCO will appear on the screen

- most sets always say "I/O OK", even if the custom I/O checks fail. Only
  gaplus and gaplusd stop working; these two also don't do the usual
  Namco-trademark RAM test on startup, and use the first I/O chip in "coin" mode,
  while the others use it in "switch/lamp" mode.

- gaplusd has the 58XX and 56XX inverted. Why would they do that?

- To use Round Advance: turn the dip switch on before the start of a level. Push
  joystick up to pick a later level, then set the dip switch back to off.

- The only difference between galaga3b and galaga3m is the bonus life settings.

TODO:
- The starfield is wrong.

- schematics show 4 lines going from the 58XX I/O chip to the 26XX (starfield generator).
  Function and operation unknown.

- Complete 62XX custom emulation (machine/namco62.c) (though it's quite different from 56XX and 58XX).

- Is the sprite generator the same as Phozon? This isn't clear yet. They are
  very similar, especially in the way the size flags are layed out.

***************************************************************************/

#include "emu.h"
#include "includes/gaplus.h"

#include "cpu/m6809/m6809.h"
#include "machine/namco62.h"
#include "machine/watchdog.h"
#include "sound/samples.h"
#include "speaker.h"


WRITE8_MEMBER(gaplus_state::irq_1_ctrl_w)
{
	int bit = !BIT(offset, 11);
	m_main_irq_mask = bit & 1;
	if (!bit)
		m_maincpu->set_input_line(0, CLEAR_LINE);
}

WRITE8_MEMBER(gaplus_state::irq_2_ctrl_w)
{
	int bit = offset & 1;
	m_sub_irq_mask = bit & 1;
	if (!bit)
		m_subcpu->set_input_line(0, CLEAR_LINE);
}

WRITE8_MEMBER(gaplus_state::irq_3_ctrl_w)
{
	int bit = !BIT(offset, 13);
	m_sub2_irq_mask = bit & 1;
	if (!bit)
		m_subcpu2->set_input_line(0, CLEAR_LINE);
}

WRITE8_MEMBER(gaplus_state::sreset_w)
{
	int bit = !BIT(offset, 11);
	m_subcpu->set_input_line(INPUT_LINE_RESET, bit ? CLEAR_LINE : ASSERT_LINE);
	m_subcpu2->set_input_line(INPUT_LINE_RESET, bit ? CLEAR_LINE : ASSERT_LINE);
	m_namco_15xx->mappy_sound_enable(bit);
}

WRITE8_MEMBER(gaplus_state::freset_w)
{
	int bit = !BIT(offset, 11);

	logerror("%04x: freset %d\n",space.device().safe_pc(), bit);

	m_namco58xx->set_reset_line(bit ? CLEAR_LINE : ASSERT_LINE);
	m_namco56xx->set_reset_line(bit ? CLEAR_LINE : ASSERT_LINE);
}

void gaplus_state::machine_reset()
{
	/* on reset, VINTON is reset, while the other flags don't seem to be affected */
	m_sub_irq_mask = 0;
	m_subcpu->set_input_line(0, CLEAR_LINE);
}

void gaplus_state::device_timer(emu_timer &timer, device_timer_id id, int param, void *ptr)
{
	switch (id)
	{
	case TIMER_NAMCOIO0_RUN:
		namcoio0_run(ptr, param);
		break;
	case TIMER_NAMCOIO1_RUN:
		namcoio1_run(ptr, param);
		break;
	default:
		assert_always(false, "Unknown id in gaplus_state::device_timer");
	}
}

TIMER_CALLBACK_MEMBER(gaplus_state::namcoio0_run)
{
	m_namco58xx->customio_run();
}

TIMER_CALLBACK_MEMBER(gaplus_state::namcoio1_run)
{
	m_namco56xx->customio_run();
}

INTERRUPT_GEN_MEMBER(gaplus_state::vblank_main_irq)
{
	if(m_main_irq_mask)
		m_maincpu->set_input_line(0, ASSERT_LINE);

	if (!m_namco58xx->read_reset_line())       /* give the cpu a tiny bit of time to write the command before processing it */
		m_namcoio0_run_timer->adjust(attotime::from_usec(50));

	if (!m_namco56xx->read_reset_line())       /* give the cpu a tiny bit of time to write the command before processing it */
		m_namcoio1_run_timer->adjust(attotime::from_usec(50));
}

INTERRUPT_GEN_MEMBER(gaplus_state::gapluso_vblank_main_irq)
{
	if(m_main_irq_mask)
		m_maincpu->set_input_line(0, ASSERT_LINE);

	if (!m_namco58xx->read_reset_line())       /* give the cpu a tiny bit of time to write the command before processing it */
		m_namcoio1_run_timer->adjust(attotime::from_usec(50));

	if (!m_namco56xx->read_reset_line())       /* give the cpu a tiny bit of time to write the command before processing it */
		m_namcoio0_run_timer->adjust(attotime::from_usec(50));
}

INTERRUPT_GEN_MEMBER(gaplus_state::vblank_sub_irq)
{
	if(m_sub_irq_mask)
		m_subcpu->set_input_line(0, ASSERT_LINE);
}

INTERRUPT_GEN_MEMBER(gaplus_state::vblank_sub2_irq)
{
	if(m_sub2_irq_mask)
		m_subcpu2->set_input_line(0, ASSERT_LINE);
}


static ADDRESS_MAP_START( cpu1_map, AS_PROGRAM, 8, gaplus_state )
	AM_RANGE(0x0000, 0x07ff) AM_RAM_WRITE(videoram_w) AM_SHARE("videoram")        /* tilemap RAM (shared with CPU #2) */
	AM_RANGE(0x0800, 0x1fff) AM_RAM AM_SHARE("spriteram") /* shared RAM with CPU #2 (includes sprite RAM) */
	AM_RANGE(0x6000, 0x63ff) AM_DEVREADWRITE("namco", namco_15xx_device, sharedram_r, sharedram_w)                                      /* shared RAM with CPU #3 */
	AM_RANGE(0x6800, 0x680f) AM_DEVREADWRITE("namcoio_1", namcoio_device, read, write)                                                   /* custom I/O chips interface */
	AM_RANGE(0x6810, 0x681f) AM_DEVREADWRITE("namcoio_2", namcoio_device, read, write)                                                   /* custom I/O chips interface */
	AM_RANGE(0x6820, 0x682f) AM_READWRITE(customio_3_r, customio_3_w) AM_SHARE("customio_3")  /* custom I/O chip #3 interface */
	AM_RANGE(0x7000, 0x7fff) AM_WRITE(irq_1_ctrl_w)                                                      /* main CPU irq control */
	AM_RANGE(0x7800, 0x7fff) AM_DEVREAD("watchdog", watchdog_timer_device, reset_r)
	AM_RANGE(0x8000, 0x8fff) AM_WRITE(sreset_w)                                                          /* reset CPU #2 & #3, enable sound */
	AM_RANGE(0x9000, 0x9fff) AM_WRITE(freset_w)                                                          /* reset I/O chips */
	AM_RANGE(0xa000, 0xa7ff) AM_WRITE(starfield_control_w)               /* starfield control */
	AM_RANGE(0xa000, 0xffff) AM_ROM                                                                             /* ROM */
ADDRESS_MAP_END

static ADDRESS_MAP_START( cpu2_map, AS_PROGRAM, 8, gaplus_state )
	AM_RANGE(0x0000, 0x07ff) AM_RAM_WRITE(videoram_w) AM_SHARE("videoram")   /* tilemap RAM (shared with CPU #1) */
	AM_RANGE(0x0800, 0x1fff) AM_RAM AM_SHARE("spriteram")                           /* shared RAM with CPU #1 */
//  AM_RANGE(0x500f, 0x500f) AM_WRITENOP                                            /* ??? written 256 times on startup */
	AM_RANGE(0x6000, 0x6fff) AM_WRITE(irq_2_ctrl_w)                          /* IRQ 2 control */
	AM_RANGE(0xa000, 0xffff) AM_ROM                                                 /* ROM */
ADDRESS_MAP_END

static ADDRESS_MAP_START( cpu3_map, AS_PROGRAM, 8, gaplus_state )
	AM_RANGE(0x0000, 0x03ff) AM_DEVREADWRITE("namco", namco_15xx_device, sharedram_r, sharedram_w)  /* shared RAM with the main CPU + sound registers */
	AM_RANGE(0x2000, 0x3fff) AM_DEVREADWRITE("watchdog", watchdog_timer_device, reset_r, reset_w)  /* watchdog? */
	AM_RANGE(0x4000, 0x7fff) AM_WRITE(irq_3_ctrl_w)                                          /* interrupt enable/disable */
	AM_RANGE(0xe000, 0xffff) AM_ROM                                                                 /* ROM */
ADDRESS_MAP_END

static INPUT_PORTS_START( gaplus )
	/* The inputs are not memory mapped, they are handled by three I/O chips. */
	PORT_START("P1")    /* 56XX #0 pins 22-29 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY

	PORT_START("P2")    /* 56XX #0 pins 22-29 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_COCKTAIL
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_COCKTAIL

	PORT_START("BUTTONS")   /* 56XX #0 pins 30-33 and 38-41 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_COCKTAIL
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START("COINS") /* 56XX #0 pins 30-33 and 38-41 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START("DSWA_LOW")  /* 58XX #1 pins 30-33 and 38-41 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_B ) ) PORT_DIPLOCATION("SW1:7,8")
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPUNUSED_DIPLOC( 0x04, 0x04, "SW1:6" ) /* Listed as "Unused" */
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Demo_Sounds ) ) PORT_DIPLOCATION("SW1:5")
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )

	PORT_START("DSWA_HIGH") /* 58XX #1 pins 30-33 and 38-41 */
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) ) PORT_DIPLOCATION("SW1:3,4")
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Lives ) ) PORT_DIPLOCATION("SW1:1,2")
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x0c, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )

	PORT_START("DSWB_LOW")  /* 58XX #1 pins 22-29 */
	PORT_DIPNAME( 0x08, 0x08, "Round Advance" ) PORT_DIPLOCATION("SW2:5")
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x07, 0x00, DEF_STR( Bonus_Life ) ) PORT_DIPLOCATION("SW2:6,7,8")
	PORT_DIPSETTING(    0x00, "30k 70k and every 70k" )
	PORT_DIPSETTING(    0x01, "30k 100k and every 100k" )
	PORT_DIPSETTING(    0x02, "30k 100k and every 200k" )
	PORT_DIPSETTING(    0x03, "50k 100k and every 100k" )
	PORT_DIPSETTING(    0x04, "50k 100k and every 200k" )
	PORT_DIPSETTING(    0x07, "50k 150k and every 150k" )
	PORT_DIPSETTING(    0x05, "50k 150k and every 300k" )
	PORT_DIPSETTING(    0x06, "50k 150k" )

	PORT_START("DSWB_HIGH") /* 58XX #1 pins 22-29 */
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) ) PORT_DIPLOCATION("SW2:1")
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x07, 0x07, DEF_STR( Difficulty ) ) PORT_DIPLOCATION("SW2:2,3,4")
	PORT_DIPSETTING(    0x07, "0 - Standard" )
	PORT_DIPSETTING(    0x06, "1 - Easiest" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x01, "6" )
	PORT_DIPSETTING(    0x00, "7 - Hardest" )

	PORT_START("IN2")   /* 62XX #2 pins 24-27 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_SERVICE( 0x08, IP_ACTIVE_LOW )
INPUT_PORTS_END

/* identical to gaplus, but service mode is a dip switch instead of coming from edge connector */
static INPUT_PORTS_START( gapluso )
	PORT_INCLUDE( gaplus )

	PORT_MODIFY("DSWB_HIGH")
	PORT_SERVICE_DIPLOC(  0x08, IP_ACTIVE_LOW, "SW2:1" )

	PORT_MODIFY("IN2")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )    // doesn't seem to be used
INPUT_PORTS_END

/* identical to gaplus, but different bonus life settings */
static INPUT_PORTS_START( galaga3a )
	PORT_INCLUDE( gaplus )

	PORT_MODIFY("DSWB_LOW")
	PORT_DIPNAME( 0x07, 0x02, DEF_STR( Bonus_Life ) ) PORT_DIPLOCATION("SW2:6,7,8")
	PORT_DIPSETTING(    0x02, "30k 80k and every 100k" )
	PORT_DIPSETTING(    0x03, "30k 100k and every 100k" )
	PORT_DIPSETTING(    0x04, "30k 100k and every 150k" )
	PORT_DIPSETTING(    0x07, "30k 100k and every 200k" )
	PORT_DIPSETTING(    0x05, "30k 100k and every 300k" )
	PORT_DIPSETTING(    0x06, "30k 150k" )
	PORT_DIPSETTING(    0x00, "50k 150k and every 150k" )
	PORT_DIPSETTING(    0x01, "50k 150k and every 200k" )
INPUT_PORTS_END

/* identical to gaplus, but different bonus life settings */
static INPUT_PORTS_START( galaga3m )
	PORT_INCLUDE( gaplus )

	PORT_MODIFY("DSWB_LOW")
	PORT_DIPNAME( 0x07, 0x00, DEF_STR( Bonus_Life ) ) PORT_DIPLOCATION("SW2:6,7,8")
	PORT_DIPSETTING(    0x00, "30k 150k and every 600k" )
	PORT_DIPSETTING(    0x01, "50k 150k and every 300k" )
	PORT_DIPSETTING(    0x02, "50k 150k and every 600k" )
	PORT_DIPSETTING(    0x03, "50k 200k and every 300k" )
	PORT_DIPSETTING(    0x04, "100k 300k and every 300k" )
	PORT_DIPSETTING(    0x07, "100k 300k and every 600k" )
	PORT_DIPSETTING(    0x05, "150k 400k and every 900k" )
	PORT_DIPSETTING(    0x06, "150k 400k" )
INPUT_PORTS_END



static const gfx_layout charlayout =
{
	8,8,
	RGN_FRAC(1,1),
	2,
	{ 4, 6 },
	{ 16*8, 16*8+1, 24*8, 24*8+1, 0, 1, 8*8, 8*8+1 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8 },
	32*8
};

static const gfx_layout spritelayout =
{
	16,16,
	RGN_FRAC(1,2),
	3,
	{ RGN_FRAC(1,2), 0, 4 },
	{ 0, 1, 2, 3, 8*8, 8*8+1, 8*8+2, 8*8+3,
		16*8+0, 16*8+1, 16*8+2, 16*8+3, 24*8+0, 24*8+1, 24*8+2, 24*8+3 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8,
		32*8, 33*8, 34*8, 35*8, 36*8, 37*8, 38*8, 39*8 },
	64*8
};

static GFXDECODE_START( gaplus )
	GFXDECODE_ENTRY( "gfx1", 0x0000, charlayout,      0, 64 )
	GFXDECODE_ENTRY( "gfx2", 0x0000, spritelayout, 64*4, 64 )
GFXDECODE_END

static const char *const gaplus_sample_names[] =
{
	"*gaplus",
	"bang",
	nullptr
};

/***************************************************************************

  Custom I/O initialization

***************************************************************************/

WRITE8_MEMBER(gaplus_state::out_lamps0)
{
	output().set_led_value(0, data & 1);
	output().set_led_value(1, data & 2);
	machine().bookkeeping().coin_lockout_global_w(data & 4);
	machine().bookkeeping().coin_counter_w(0, ~data & 8);
}

WRITE8_MEMBER(gaplus_state::out_lamps1)
{
	machine().bookkeeping().coin_counter_w(1, ~data & 1);
}

void gaplus_state::machine_start()
{
	m_namcoio0_run_timer = timer_alloc(TIMER_NAMCOIO0_RUN);
	m_namcoio1_run_timer = timer_alloc(TIMER_NAMCOIO1_RUN);

	switch (m_type)
	{
		case GAME_GALAGA3:
		case GAME_GAPLUS:
			m_namco56xx = machine().device<namco56xx_device>("namcoio_1");
			m_namco58xx = machine().device<namco58xx_device>("namcoio_2");
			break;
		case GAME_GAPLUSD:
			m_namco58xx = machine().device<namco58xx_device>("namcoio_1");
			m_namco56xx = machine().device<namco56xx_device>("namcoio_2");
			break;
	}

	save_item(NAME(m_main_irq_mask));
	save_item(NAME(m_sub_irq_mask));
	save_item(NAME(m_sub2_irq_mask));
}


static MACHINE_CONFIG_START( gaplus )

	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M6809,  24576000/16)    /* 1.536 MHz */
	MCFG_CPU_PROGRAM_MAP(cpu1_map)
	MCFG_CPU_VBLANK_INT_DRIVER("screen", gaplus_state,  vblank_main_irq)

	MCFG_CPU_ADD("sub", M6809,  24576000/16)    /* 1.536 MHz */
	MCFG_CPU_PROGRAM_MAP(cpu2_map)
	MCFG_CPU_VBLANK_INT_DRIVER("screen", gaplus_state,  vblank_sub_irq)

	MCFG_CPU_ADD("sub2", M6809, 24576000/16)    /* 1.536 MHz */
	MCFG_CPU_PROGRAM_MAP(cpu3_map)
	MCFG_CPU_VBLANK_INT_DRIVER("screen", gaplus_state,  vblank_sub2_irq)

	MCFG_QUANTUM_TIME(attotime::from_hz(6000))  /* a high value to ensure proper synchronization of the CPUs */

	MCFG_WATCHDOG_ADD("watchdog")

	MCFG_DEVICE_ADD("namcoio_1", NAMCO_56XX, 0)
	MCFG_NAMCO56XX_IN_0_CB(IOPORT("COINS"))
	MCFG_NAMCO56XX_IN_1_CB(IOPORT("P1"))
	MCFG_NAMCO56XX_IN_2_CB(IOPORT("P2"))
	MCFG_NAMCO56XX_IN_3_CB(IOPORT("BUTTONS"))
	MCFG_NAMCO56XX_OUT_0_CB(WRITE8(gaplus_state, out_lamps0))
	MCFG_NAMCO56XX_OUT_1_CB(WRITE8(gaplus_state, out_lamps1))

	MCFG_DEVICE_ADD("namcoio_2", NAMCO_58XX, 0)
	MCFG_NAMCO58XX_IN_0_CB(IOPORT("DSWA_HIGH"))
	MCFG_NAMCO58XX_IN_1_CB(IOPORT("DSWB_LOW"))
	MCFG_NAMCO58XX_IN_2_CB(IOPORT("DSWB_HIGH"))
	MCFG_NAMCO58XX_IN_3_CB(IOPORT("DSWA_LOW"))

	MCFG_NAMCO_62XX_ADD("62xx", 24576000/6/2)  /* totally made up - TODO: fix */
	//MCFG_NAMCO_62XX_INPUT_0_CB(IOPORT("IN0L"))
	//MCFG_NAMCO_62XX_INPUT_1_CB(IOPORT("IN0H"))
	//MCFG_NAMCO_62XX_INPUT_2_CB(IOPORT("IN1L"))
	//MCFG_NAMCO_62XX_INPUT_3_CB(IOPORT("IN1H"))
	//MCFG_NAMCO_62XX_OUTPUT_0_CB(WRITE8(gaplus_state,out_0))
	//MCFG_NAMCO_62XX_OUTPUT_1_CB(WRITE8(gaplus_state,out_1))

	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_REFRESH_RATE(60.606060)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC(0))
	MCFG_SCREEN_SIZE(36*8, 28*8)
	MCFG_SCREEN_VISIBLE_AREA(0*8, 36*8-1, 0*8, 28*8-1)
	MCFG_SCREEN_UPDATE_DRIVER(gaplus_state, screen_update)
	MCFG_SCREEN_VBLANK_CALLBACK(WRITELINE(gaplus_state, screen_vblank))
	MCFG_SCREEN_PALETTE("palette")

	MCFG_GFXDECODE_ADD("gfxdecode", "palette", gaplus)
	MCFG_PALETTE_ADD("palette", 64*4+64*8)
	MCFG_PALETTE_INDIRECT_ENTRIES(256)
	MCFG_PALETTE_INIT_OWNER(gaplus_state, gaplus)

	/* sound hardware */
	MCFG_SPEAKER_STANDARD_MONO("mono")

	MCFG_SOUND_ADD("namco", NAMCO_15XX, 24576000/1024)
	MCFG_NAMCO_AUDIO_VOICES(8)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.0)

	MCFG_SOUND_ADD("samples", SAMPLES, 0)
	MCFG_SAMPLES_CHANNELS(1)
	MCFG_SAMPLES_NAMES(gaplus_sample_names)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 0.80)
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( gaplusd, gaplus )

	MCFG_DEVICE_REPLACE("namcoio_1", NAMCO_58XX, 0)
	MCFG_NAMCO58XX_IN_0_CB(IOPORT("COINS"))
	MCFG_NAMCO58XX_IN_1_CB(IOPORT("P1"))
	MCFG_NAMCO58XX_IN_2_CB(IOPORT("P2"))
	MCFG_NAMCO58XX_IN_3_CB(IOPORT("BUTTONS"))

	MCFG_DEVICE_REPLACE("namcoio_2", NAMCO_56XX, 0)
	MCFG_NAMCO56XX_IN_0_CB(IOPORT("DSWA_HIGH"))
	MCFG_NAMCO56XX_IN_1_CB(IOPORT("DSWB_LOW"))
	MCFG_NAMCO56XX_IN_2_CB(IOPORT("DSWB_HIGH"))
	MCFG_NAMCO56XX_IN_3_CB(IOPORT("DSWA_LOW"))
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( gapluso, gaplusd )

	/* basic machine hardware */
	MCFG_CPU_MODIFY("maincpu")
	MCFG_CPU_VBLANK_INT_DRIVER("screen", gaplus_state,  gapluso_vblank_main_irq)

	MCFG_DEVICE_REPLACE("namcoio_1", NAMCO_56XX, 0)
	MCFG_NAMCO56XX_IN_0_CB(IOPORT("COINS"))
	MCFG_NAMCO56XX_IN_1_CB(IOPORT("P1"))
	MCFG_NAMCO56XX_IN_2_CB(IOPORT("P2"))
	MCFG_NAMCO56XX_IN_3_CB(IOPORT("BUTTONS"))

	MCFG_DEVICE_REPLACE("namcoio_2", NAMCO_58XX, 0)
	MCFG_NAMCO58XX_IN_0_CB(IOPORT("DSWA_HIGH"))
	MCFG_NAMCO58XX_IN_1_CB(IOPORT("DSWB_LOW"))
	MCFG_NAMCO58XX_IN_2_CB(IOPORT("DSWB_HIGH"))
	MCFG_NAMCO58XX_IN_3_CB(IOPORT("DSWA_LOW"))
MACHINE_CONFIG_END



ROM_START( gaplus ) /* Version 2 or 3 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp2-4.8d",  0xa000, 0x2000, CRC(e525d75d) SHA1(93fcd8b940491abf6344181811d0b35765d7e45c) )
	ROM_LOAD( "gp2-3b.8c", 0xc000, 0x2000, CRC(d77840a4) SHA1(81402b28a2d5ac2d1301252534afa0cb65d7e162) )
	ROM_LOAD( "gp2-2b.8b", 0xe000, 0x2000, CRC(b3cb90db) SHA1(025c2f3978772e1ecbbf36842dc7c2203ee91a1f) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp2-8.11d", 0xa000, 0x2000, CRC(42b9fd7c) SHA1(f230eb0ad757f0714c0ac81c812e950778452947) )
	ROM_LOAD( "gp2-7.11c", 0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp2-6.11b", 0xe000, 0x2000, CRC(75b18652) SHA1(398059da967c80321a9ec94d982a6c0b3c970c5f) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp2-5.8s", 0x0000, 0x2000, CRC(f3d19987) SHA1(a0107fa4659597ac42c875ab1c0deb845534268b) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp2-6.6p", 0x0400, 0x0200, CRC(6f99c2da) SHA1(955dcef363870ee8e91edc73b9ea3ce489738aad) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp2-5.6n", 0x0600, 0x0200, CRC(c7d31657) SHA1(a93a5bc448dc127e1389d10a9cb06acadfe940cf) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END

ROM_START( gaplusa ) /* Version 2 or 3 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp2-4.8d",  0xa000, 0x2000, CRC(e525d75d) SHA1(93fcd8b940491abf6344181811d0b35765d7e45c) )
	ROM_LOAD( "gp2-3b.8c", 0xc000, 0x2000, CRC(d77840a4) SHA1(81402b28a2d5ac2d1301252534afa0cb65d7e162) )
	ROM_LOAD( "gp2-2.8b",  0xe000, 0x2000, CRC(61f6cc65) SHA1(0f9b30722ba03a63c70ff9f1bd9712aa1a4a6a3d) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp2-8.11d", 0xa000, 0x2000, CRC(42b9fd7c) SHA1(f230eb0ad757f0714c0ac81c812e950778452947) )
	ROM_LOAD( "gp2-7.11c", 0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp2-6.11b", 0xe000, 0x2000, CRC(75b18652) SHA1(398059da967c80321a9ec94d982a6c0b3c970c5f) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp2-5.8s", 0x0000, 0x2000, CRC(f3d19987) SHA1(a0107fa4659597ac42c875ab1c0deb845534268b) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp2-6.6p", 0x0400, 0x0200, CRC(6f99c2da) SHA1(955dcef363870ee8e91edc73b9ea3ce489738aad) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp2-5.6n", 0x0600, 0x0200, CRC(c7d31657) SHA1(a93a5bc448dc127e1389d10a9cb06acadfe940cf) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END

ROM_START( gaplusd ) /* Alternate hardware */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp2-4b.8d", 0xa000, 0x2000, CRC(484f11e0) SHA1(659756ae183dac3817440c8975f203c7dbe08c6b) )
	ROM_LOAD( "gp2-3c.8c", 0xc000, 0x2000, CRC(a74b0266) SHA1(a534c6b4af569ed545bf52769c7d5ceb5f2c4935) )
	ROM_LOAD( "gp2-2d.8b", 0xe000, 0x2000, CRC(69fdfdb7) SHA1(aec611336b8767897ad493d581d70b1f0e75aeba) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp2-8b.11d", 0xa000, 0x2000, CRC(bff601a6) SHA1(e1a04354d8d0bc0d51d7341a46bd23cbd2158ee9) ) /* Revised for alternate hardware */
	ROM_LOAD( "gp2-7.11c",  0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp2-6b.11b", 0xe000, 0x2000, CRC(14cd61ea) SHA1(05605abebcf2791e60b2d810dafcdd8582a87d9b) ) /* Revised for alternate hardware */

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp2-5.8s", 0x0000, 0x2000, CRC(f3d19987) SHA1(a0107fa4659597ac42c875ab1c0deb845534268b) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp2-6.6p", 0x0400, 0x0200, CRC(6f99c2da) SHA1(955dcef363870ee8e91edc73b9ea3ce489738aad) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp2-5.6n", 0x0600, 0x0200, CRC(c7d31657) SHA1(a93a5bc448dc127e1389d10a9cb06acadfe940cf) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END

ROM_START( gaplust ) /* Tecfri PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp2_4.4", 0xa000, 0x2000, CRC(d891a70d) SHA1(ec906d623f936335e194d3bf9484ca4e82691272) ) // galaga3m - m1.9e                   99.865723%
	ROM_LOAD( "gp2_3.3", 0xc000, 0x2000, CRC(1df6e319) SHA1(beb7bd22ff8bcb1c39f676e8bbb607e06e4f20d6) ) // galaga3m - m2.9d                   6.762695%
	ROM_LOAD( "gp2_2.2", 0xe000, 0x2000, CRC(fc764728) SHA1(5a8bd3e83fbea2bb6cc06748c85b56e24a706f37) ) // galaga3m - m3.9c                   81.530762%

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp2_8.8", 0xa000, 0x2000, CRC(9ec3dce5) SHA1(196a975aff59be19f55041a44b201aafef083ba7) ) // matches roms from the Midway PCB sets
	ROM_LOAD( "gp2_7.7", 0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp2_6.6", 0xe000, 0x2000, CRC(6a2942c5) SHA1(6fb2c4dcb2ad393220917b81f1a42e571d209d76) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp2-5.8s", 0x0000, 0x2000, CRC(f3d19987) SHA1(a0107fa4659597ac42c875ab1c0deb845534268b) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp2-6.6p", 0x0400, 0x0200, CRC(6f99c2da) SHA1(955dcef363870ee8e91edc73b9ea3ce489738aad) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp2-5.6n", 0x0600, 0x0200, CRC(c7d31657) SHA1(a93a5bc448dc127e1389d10a9cb06acadfe940cf) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END

ROM_START( galaga3 ) /* Version 2 or 3 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp3-4c.8d", 0xa000, 0x2000, CRC(10d7f64c) SHA1(e39f77af16016d28170e4ac1c2a784b0a7ec5454) )
	ROM_LOAD( "gp3-3c.8c", 0xc000, 0x2000, CRC(962411e8) SHA1(2b6bb2a5d77a837810180391ef6c0ce745bfed64) )
	ROM_LOAD( "gp3-2d.8b", 0xe000, 0x2000, CRC(ecc01bdb) SHA1(b176b46bd6f2501d3a74ed11186be8411fd1105b) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp3-8b.11d", 0xa000, 0x2000, CRC(f5e056d1) SHA1(bbed2056dc28dc2828e29987c16d89fb16e7059e) )
	ROM_LOAD( "gp2-7.11c",  0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp3-6b.11b", 0xe000, 0x2000, CRC(026491b6) SHA1(a19f2942dafc899d686a42240fc2f7a7a7d3b1f5) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp3-5.8s", 0x0000, 0x2000, CRC(8d4dcebf) SHA1(0a556b45976bc36eb99048b1512c446b472da1d2) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp3-6.6p", 0x0400, 0x0200, CRC(d48c0eef) SHA1(6d0512958bc522d22e69336677369507847f8f6f) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp3-5.6n", 0x0600, 0x0200, CRC(417ba0dc) SHA1(2ba51ccdd0428fc48758ed8fea36c8ce0e752a45) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END

ROM_START( galaga3a ) /* Version 2 or 3 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp3-4c.8d", 0xa000, 0x2000, CRC(10d7f64c) SHA1(e39f77af16016d28170e4ac1c2a784b0a7ec5454) )
	ROM_LOAD( "gp3-3c.8c", 0xc000, 0x2000, CRC(962411e8) SHA1(2b6bb2a5d77a837810180391ef6c0ce745bfed64) )
	ROM_LOAD( "gp3-2c.8b", 0xe000, 0x2000, CRC(f72d6fc5) SHA1(7031c4a2c4374fb786fc563cbad3e3de0dbaa8d2) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp3-8b.11d", 0xa000, 0x2000, CRC(f5e056d1) SHA1(bbed2056dc28dc2828e29987c16d89fb16e7059e) )
	ROM_LOAD( "gp2-7.11c",  0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp3-6b.11b", 0xe000, 0x2000, CRC(026491b6) SHA1(a19f2942dafc899d686a42240fc2f7a7a7d3b1f5) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp3-5.8s", 0x0000, 0x2000, CRC(8d4dcebf) SHA1(0a556b45976bc36eb99048b1512c446b472da1d2) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp3-6.6p", 0x0400, 0x0200, CRC(d48c0eef) SHA1(6d0512958bc522d22e69336677369507847f8f6f) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp3-5.6n", 0x0600, 0x0200, CRC(417ba0dc) SHA1(2ba51ccdd0428fc48758ed8fea36c8ce0e752a45) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END

ROM_START( galaga3b ) /* Version 2 or 3 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gp3-4.8d", 0xa000, 0x2000, CRC(58de387c) SHA1(9a2519e345e2599bb9ea28b916cff95c03d7b262) )
	ROM_LOAD( "gp3-3.8c", 0xc000, 0x2000, CRC(94a3fd4e) SHA1(e566b7a76fb8db849c3c1660a1551a7a94caddc2) )
	ROM_LOAD( "gp3-2.8b", 0xe000, 0x2000, CRC(4b1cb589) SHA1(1f016341f8c73a2b379362be091f0a95ef81c2fa) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gp3-8.11d", 0xa000, 0x2000, CRC(d390ef28) SHA1(fa3325ce7b8d29edea467678646ab0e4c1f6d1f8) )
	ROM_LOAD( "gp2-7.11c", 0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gp3-6.11b", 0xe000, 0x2000, CRC(b36a9a2b) SHA1(8d11252c23ca6e10c994a58aa4a48690255e2268) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.4b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gp3-5.8s", 0x0000, 0x2000, CRC(8d4dcebf) SHA1(0a556b45976bc36eb99048b1512c446b472da1d2) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.11p", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.11n", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.11r", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.11m",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(               0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1p", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1n", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2n", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.6s", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "gp3-6.6p", 0x0400, 0x0200, CRC(d48c0eef) SHA1(6d0512958bc522d22e69336677369507847f8f6f) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "gp3-5.6n", 0x0600, 0x0200, CRC(417ba0dc) SHA1(2ba51ccdd0428fc48758ed8fea36c8ce0e752a45) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )

	ROM_REGION( 0x0100, "plds", 0 )
	ROM_LOAD( "pal10l8.8n", 0x0000, 0x002c, CRC(08e5b2fe) SHA1(1aa7fa1a61795703af84ae427d0d8588ef8c4c3f) )
ROM_END


ROM_START( galaga3c ) /* Version (AKA Midway) 1 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "gal3_9e.9e", 0xa000, 0x2000, CRC(f4845e7f) SHA1(7b1377254f594bea4a8ffc7e388d9106e0266b55) )
	ROM_LOAD( "gal3_9d.9d", 0xc000, 0x2000, CRC(86fac687) SHA1(07f76af524dbb3e79de41ef4bf32e7380776d9f5) )
	ROM_LOAD( "gal3_9c.9c", 0xe000, 0x2000, CRC(f1b00073) SHA1(5d998d938251f173cedf742b95d02cc0a2b9d3be) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "gal3_6l.6l", 0xa000, 0x2000, CRC(9ec3dce5) SHA1(196a975aff59be19f55041a44b201aafef083ba7) )
	ROM_LOAD( "gal3_6m.6m", 0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "gal3_6n.6n", 0xe000, 0x2000, CRC(6a2942c5) SHA1(6fb2c4dcb2ad393220917b81f1a42e571d209d76) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.7b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gal3_9l.bin", 0x0000, 0x2000, CRC(8d4dcebf) SHA1(0a556b45976bc36eb99048b1512c446b472da1d2) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.5m", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.5l", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.5k", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.5n",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(              0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1c", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1d", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2d", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.4f", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "g3_3f.3f", 0x0400, 0x0200, CRC(d48c0eef) SHA1(6d0512958bc522d22e69336677369507847f8f6f) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "g3_3e.3e", 0x0600, 0x0200, CRC(417ba0dc) SHA1(2ba51ccdd0428fc48758ed8fea36c8ce0e752a45) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )
ROM_END

ROM_START( galaga3m ) /* Version (AKA Midway) 1 PCB */
	ROM_REGION( 0x10000, "maincpu", 0 ) /* 64k for the MAIN CPU */
	ROM_LOAD( "m1.9e", 0xa000, 0x2000, CRC(e392704e) SHA1(8eebd48dfe8491f491e844d4ad0964e25efb013b) )
	ROM_LOAD( "m2.9d", 0xc000, 0x2000, CRC(86fac687) SHA1(07f76af524dbb3e79de41ef4bf32e7380776d9f5) )
	ROM_LOAD( "m3.9c", 0xe000, 0x2000, CRC(f1b00073) SHA1(5d998d938251f173cedf742b95d02cc0a2b9d3be) )

	ROM_REGION( 0x10000, "sub", 0 ) /* 64k for the SUB CPU */
	ROM_LOAD( "m6.6l", 0xa000, 0x2000, CRC(9ec3dce5) SHA1(196a975aff59be19f55041a44b201aafef083ba7) )
	ROM_LOAD( "m5.6m", 0xc000, 0x2000, CRC(0621f7df) SHA1(b86020f819fefb134cb57e203f7c90b1b29581c8) )
	ROM_LOAD( "m4.6n", 0xe000, 0x2000, CRC(6a2942c5) SHA1(6fb2c4dcb2ad393220917b81f1a42e571d209d76) )

	ROM_REGION( 0x10000, "sub2", 0 ) /* 64k for the SOUND CPU */
	ROM_LOAD( "gp2-1.7b", 0xe000, 0x2000, CRC(ed8aa206) SHA1(4e0a31d84cb7aca497485dbe0240009d58275765) )

	ROM_REGION( 0x4000, "gfx1", 0 )
	ROM_LOAD( "gal3_9l.bin", 0x0000, 0x2000, CRC(8d4dcebf) SHA1(0a556b45976bc36eb99048b1512c446b472da1d2) )    /* characters */
	/* 0x2000-0x3fff  will be unpacked from 0x0000-0x1fff */

	ROM_REGION( 0xc000, "gfx2", 0 )
	ROM_LOAD( "gp2-11.5m", 0x0000, 0x2000, CRC(57740ff9) SHA1(16873e0ac5f975768d596d7d32af7571f4817f2b) )    /* objects */
	ROM_LOAD( "gp2-10.5l", 0x2000, 0x2000, CRC(6cd8ce11) SHA1(fc346e98737c9fc20810e32d4c150ae4b4051979) )    /* objects */
	ROM_LOAD( "gp2-12.5k", 0x4000, 0x2000, CRC(7316a1f1) SHA1(368e4541a5151e906a189712bc05192c2ceec8ae) )    /* objects */
	ROM_LOAD( "gp2-9.5n",  0x6000, 0x2000, CRC(e6a9ae67) SHA1(99c1e67c3b216aa1b63f199e21c73cdedde80e1b) )    /* objects */
	/* 0x8000-0x9fff  will be unpacked from 0x6000-0x7fff */
	ROM_FILL(              0xa000, 0x2000, 0x00 )    // optional ROM, not used

	ROM_REGION( 0x0800, "proms", 0 )
	ROM_LOAD( "gp2-3.1c", 0x0000, 0x0100, CRC(a5091352) SHA1(dcd6dfbfbd5281ba0c7b7c189d6fde23617ed3e3) )    /* red palette ROM (4 bits) */
	ROM_LOAD( "gp2-1.1d", 0x0100, 0x0100, CRC(8bc8022a) SHA1(c76f9d9b066e268621d41a703c5280261234709a) )    /* green palette ROM (4 bits) */
	ROM_LOAD( "gp2-2.2d", 0x0200, 0x0100, CRC(8dabc20b) SHA1(64d7b333f529d3ba66aeefd380fd1cbf9ddf460d) )    /* blue palette ROM (4 bits) */
	ROM_LOAD( "gp2-7.4f", 0x0300, 0x0100, CRC(2faa3e09) SHA1(781ffe9088476798409cb922350eff881590cf35) )    /* char color ROM */
	ROM_LOAD( "g3_3f.3f", 0x0400, 0x0200, CRC(d48c0eef) SHA1(6d0512958bc522d22e69336677369507847f8f6f) )    /* sprite color ROM (lower 4 bits) */
	ROM_LOAD( "g3_3e.3e", 0x0600, 0x0200, CRC(417ba0dc) SHA1(2ba51ccdd0428fc48758ed8fea36c8ce0e752a45) )    /* sprite color ROM (upper 4 bits) */

	ROM_REGION( 0x0100, "namco", 0 ) /* sound prom */
	ROM_LOAD( "gp2-4.3f", 0x0000, 0x0100, CRC(2d9fbdd8) SHA1(e6a23cd5ce3d3e76de3b70c8ab5a3c45b1147af4) )
ROM_END


DRIVER_INIT_MEMBER(gaplus_state,gaplus)
{
	uint8_t *rom;

	rom = memregion("gfx1")->base();
	for (int i = 0;i < 0x2000;i++)
		rom[i + 0x2000] = rom[i] >> 4;

	rom = memregion("gfx2")->base() + 0x6000;
	for (int i = 0;i < 0x2000;i++)
		rom[i + 0x2000] = rom[i] << 4;

	m_type = GAME_GAPLUS;
}


DRIVER_INIT_MEMBER(gaplus_state,gaplusd)
{
	DRIVER_INIT_CALL(gaplus);
	m_type = GAME_GAPLUSD;
}

DRIVER_INIT_MEMBER(gaplus_state,galaga3)
{
	DRIVER_INIT_CALL(gaplus);
	m_type = GAME_GALAGA3;
}


/* These sets are on revision 2 or 3 PCBs AKA "Namco" PCBs */
GAME( 1984, gaplus,   0,        gapluso,  gapluso,  gaplus_state, gaplus,  ROT90, "Namco", "Gaplus (GP2 rev. B)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
GAME( 1984, gaplusa,  gaplus,   gapluso,  gapluso,  gaplus_state, gaplus,  ROT90, "Namco", "Gaplus (GP2)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
GAME( 1984, gaplusd,  gaplus,   gaplusd,  gapluso,  gaplus_state, gaplusd, ROT90, "Namco", "Gaplus (GP2 rev D, alternate hardware)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
GAME( 1984, galaga3,  gaplus,   gaplus,   gaplus,   gaplus_state, galaga3, ROT90, "Namco", "Galaga 3 (GP3 rev. D)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
GAME( 1984, galaga3a, gaplus,   gaplus,   gaplus,   gaplus_state, galaga3, ROT90, "Namco", "Galaga 3 (GP3 rev. C)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
GAME( 1984, galaga3b, gaplus,   gaplus,   gaplus,   gaplus_state, galaga3, ROT90, "Namco", "Galaga 3 (GP3)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )

/* These sets are on older revision (AKA Midway) 1 PCBs */
GAME( 1984, galaga3c, gaplus,   gaplus,   galaga3a, gaplus_state, galaga3, ROT90, "Namco", "Galaga 3 (set 4)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
GAME( 1984, galaga3m, gaplus,   gaplus,   galaga3m, gaplus_state, galaga3, ROT90, "Namco", "Galaga 3 (set 5)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )

/* This is an odd mix of Galaga3 and Gaplus, main code seems closest to galaga3m but still has significant changes, copyright is modified to 1992, has Galaga 3 style high scores, PARSEF spelling error on high score table */
GAME( 1992, gaplust,  gaplus,   gaplus,   galaga3m, gaplus_state, galaga3, ROT90, "bootleg (Tecfri)", "Gaplus (Tecfri PCB)", MACHINE_IMPERFECT_SOUND | MACHINE_IMPERFECT_GRAPHICS | MACHINE_SUPPORTS_SAVE )
