/* martian.c: pci kernel mode driver module
 * 	
 * description:
 *	driver takes control over 164x Agere devices
 *	real work only in pair with martian_modem
 *
 * author(s):
 *	A. Chentsov <chentsov at ukr.net> 
 *
 * copying: GPLv2
 *	
 *
 * provides some inevitable low-level support for the user space modem driver (martian_modem)
 *
 * communication actions: 
 *	- pass device configuration information to the martian_modem
 *	- process interrupts, notify driver through poll/read
 *
 * implementation details:
 *	every device is represented by the node in the vfs
 *	currently proc entries are used
 *	module and martian_modem have the common block of information
 *	the data is mapped into martian_modem process by mmap call
 *	this is forced by isr dependance on run-time core state
 *
 * based on practices and ideas people worked out in linux kernel
 *
 * changes:
 *	06/2008		Added all IO routines. Exported to martian_modem through ioctl service.
 *	12/2006		Smp support with intermode synchronization. { Infidel idea. Abandoned. }
 *	07/2006		All data-related interrupt processing moved to kernel
 *	12/2005		First release. Minimal work in kernel: disabling further interrupt and relaying to martian_modem
*/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/bitops.h>
#include <linux/poll.h>
#include <linux/interrupt.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 32)
#include <linux/sched.h>
#endif

#include "kmartian.h"
#include "../martian.h"
#include "marsio.h"

/***** Driver globals *****/

struct proc_dir_entry *martians_proc_dir = NULL;

/**************************/

// statistics
unsigned int interrupt_counter = 0;
static char irqstring[11] = "|\0\0\0\0\0\0\0\0\0\0";
static int irqptr = 0;

#define MARTIAN_IRQ_ACCOUNT(letter, count_letter) \
	irqstring[irqptr] = letter; mdev->icounts.count_letter++;

#define MARTIAN_IRQ_ACCOUNT_FIN \
	do { \
		if (++irqptr == 10) irqptr = 0; \
		irqstring[irqptr] = '|'; \
	} while (0)


/***** ISR *****/

/* martian_isr
 *	do data processing
 *	relate ring and dcp to the martian_modem
 */
static
irqreturn_t martian_isr (int irq, void *dev)
{
	struct martian *mdev = (struct martian *) dev;
	struct martian_common *mcb = mdev->common;
	u8 rd7;
	irqreturn_t retval = IRQ_NONE;

	interrupt_counter++;

	if (test_bit (KMSTATE_UREADY, &mdev->state) && mcb->dp_bamil_rd7 == 0xff) {
		MARTIAN_IRQ_ACCOUNT('b', b); 
		MARTIAN_IRQ_ACCOUNT_FIN; 
		return IRQ_NONE;
	}

	spin_lock(&mdev->lock);
	rd7 = mars_read_register(mdev, 0xd7);

	if (rd7 == 0) {
		MARTIAN_IRQ_ACCOUNT('n', n); 
	}
	else if (rd7 == 0xff) {
		MARTIAN_IRQ_ACCOUNT('f', f); 
	}
	else if (rd7 & IR_SIGNIFICANT) {
		if (! test_bit (KMSTATE_UREADY, &mdev->state)) {
			mars_write_register(mdev, IR, 0xff);
			retval = IRQ_HANDLED;

			MARTIAN_IRQ_ACCOUNT('i', i); 
			goto fin;
		}

		if (mdev->params.dsp_mars == 1)
			mars_write_word(mdev, 0x3c, 0x101);

		/* disabling */
		mars_write_register(mdev, IR, 0xff);

		/* check not fully served yet by user */
		if (atomic_read (&mcb->UnservedIrqs) > 0) {
			mcb->OverlappedIrqs++;
			atomic_set (&mcb->UnservedIrqs, 0);
		}

		atomic_inc (&mcb->UnservedIrqs);

		/* check not passed yet to user */
		if (atomic_read (&mcb->UnpassedIrqs) > 0)
			mcb->RepeatedIrqs++;
		atomic_inc (&mcb->UnpassedIrqs);


		if (test_bit (KMSETTING_SERVE_IRQS, &mdev->state)) {
			if (rd7 & 2)
				process_ring(mdev);

			if (rd7 & 4) 
				process_stream (mdev) && (rd7 &= ~4);

			if (rd7 & 8) {
				mars_write_register(mdev, 0xd8, 8);

				mcb->byte_c = mars_read_register(mdev, 0x32);
				mcb->byte_d = mars_read_register(mdev, 0x33);
				mcb->byte_e = mars_read_register(mdev, 0x34);
				mcb->dp_byte_f = mars_read_register(mdev, 0x35);
				rd7 &= ~8;
			}
		}

		if ((rd7 & IR_SIGNIFICANT) || test_bit (KMSTATE_FIFO_FAULT, &mdev->state)) {
			mcb->rd7_isr = rd7;
			/* notify readers */
			atomic_inc (& mdev->count);
			wake_up_interruptible (& mdev->wq);
		}
		else {
			mars_write_register(mdev, IR, mcb->dp_bamil_rd7);

			atomic_dec (&mcb->UnservedIrqs);
			atomic_dec (&mcb->UnpassedIrqs);
		}

		retval = IRQ_HANDLED;

		MARTIAN_IRQ_ACCOUNT('i', i); 

	}
	else {
		mars_write_register(mdev, IR, mcb->dp_bamil_rd7);
		
		retval = IRQ_HANDLED;
		MARTIAN_IRQ_ACCOUNT('o', o);
	}

fin:
	spin_unlock(&mdev->lock);

	MARTIAN_IRQ_ACCOUNT_FIN; 

	return retval;
}


/***** file operations on device node *****/

static int martian_open (struct inode *inode, struct file *filp)
{
	// allow single open
	struct proc_dir_entry *entry = PDE (inode);
	struct martian *mdev = (struct martian *) entry->data;

	if (test_and_set_bit (MARTIAN_STATE_OPEN, &mdev->state)) {
		// already opened
		return -EBUSY;
	}

	filp->private_data = mdev;
	
	clear_bit (KMSTATE_FIFO_FAULT, &mdev->state);
	atomic_set (&mdev->count, 0);

	return 0;
}

void dump_core_vars_common (struct martian_common *common);
/* clean up
 */
static int martian_release (struct inode *inode, struct file *filp)
{
	// struct martian *mdev = (struct martian *) PDE(inode)->data;
	struct martian *mdev = (struct martian *) filp->private_data;
	int debug;

	atomic_set (&mdev->common->UnservedIrqs, 0);

	clear_bit (MARTIAN_STATE_OPEN, &mdev->state);
	clear_bit (KMSETTING_SERVE_IRQS, &mdev->state);

	mars_write_register_safe(mdev, IR, 0xff);
	mdev->common->dp_bamil_rd7 = 0xff;

	debug = test_and_clear_bit (KMSETTING_DEBUG, &mdev->state);

	clear_bit (KMSTATE_UREADY, &mdev->state);

	/* reset core related params */
	mdev->params.DCPAudioOn = 0;

	pr_info("martian_modem is detached\n");

	if (debug) {
		dump_core_vars_common (mdev->common);
		MFIFO_DUMP (mdev, dce_rx);
		MFIFO_DUMP (mdev, pdm_rx);
		MFIFO_DUMP (mdev, dce_tx);
	}

	atomic_set (&mdev->count, 0);
	/* check with wq */
	mars_write_register(mdev, IR, 0xff);

	return 0;
}

/* poll&select
 */
static unsigned int martian_poll(struct file *filp, struct poll_table_struct *wait)
{
	struct martian *mdev = (struct martian *) filp->private_data;

	if (atomic_read (& mdev->count) > 0)
		return POLLIN | POLLRDNORM; /* readable */

	poll_wait (filp, & mdev->wq, wait);

	if (atomic_read(& mdev->count) > 0)
		return POLLIN | POLLRDNORM; /* readable */

	return 0;
}

/* ioctl logger(s) */

#define DUMP_CORE_COMMON(fmt, var) pr_info("martian:" fmt, #var, common->var)
void dump_core_vars_common (struct martian_common *common) {
	pr_info("martian: Core vars:\n");
	pr_info("martian: Variable\t\t\tValue\n");

	DUMP_CORE_COMMON ("%s\t\t\t0x%x\n", BaseAddress);
	DUMP_CORE_COMMON ("%s\t\t\t0x%x\n", BaseAddress2);

	DUMP_CORE_COMMON ("%s\t\t0x%x\n", BaseAddressIndex);
	DUMP_CORE_COMMON ("%s\t\t\t0x%x\n", BaseValue);

	DUMP_CORE_COMMON ("%s\t\t\t0x%x\n", BaseAddressData);
	DUMP_CORE_COMMON ("%s\t\t\t0x%x\n", dp_bamil_rd7);
	DUMP_CORE_COMMON ("%s\t\t\t%d\n", x_dsp_mars);
}

/* ioctl getters */

static void fill_martian_conf (struct martian_conf *conf, struct martian *mdev) 
{
	struct martian_metrics *metrics = &mdev->metrics;

	conf->BaseAddress = metrics->BaseAddress;
	conf->CommAddress = metrics->CommAddress;
	conf->irq	  = metrics->irq;
	
	conf->device	  = metrics->device;
	conf->vendor	  = metrics->vendor;
	conf->subsys_vendor  = metrics->subsystem_vendor;
	conf->subsys_device  = metrics->subsystem_device;

	conf->flags	  = metrics->flags;

	conf->index	  = mdev->index;
}

static void 
fill_martian_irq (struct martian_irq_state *state, struct martian *mdev) 
{
	memcpy (state->string, irqstring, sizeof state->string);
	state->count  = interrupt_counter;	
	state->counts = mdev->icounts;
}

/*
 * sys_ioctl:
 * 	unlocked_ioctl
 * 
 * compat_sys_ioctl:
 * 	compat_ioctl
 *
 *	no conventional
 */

typedef long martian_ioctl_result_t;

#define copy_struct_to_user(dst, s) 		\
	copy_to_user ((void __user *) dst, &s, sizeof s) ? -EFAULT : 0

/*  martian_ioctl:
 *	for unlocked and compat ioctl calls
 *	no presumption about BKL
 */
static martian_ioctl_result_t 
martian_ioctl (struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct martian *mdev = (struct martian *) filp->private_data;
	/* void *buf; int size; */

	switch (cmd) {
	case MARTIAN_CONF_GET:
	{
		struct martian_conf conf;

		fill_martian_conf (&conf, mdev);
		return copy_struct_to_user (arg, conf);
	}
	case MARTIAN_IRQ_STATE:
	{
		struct martian_irq_state irq_state;

		fill_martian_irq (&irq_state, mdev);
		return copy_struct_to_user (arg, irq_state);
	}
	case MARTIAN_DUMP_VARS:
		dump_core_vars_common (mdev->common);
		return 0;

	case MARTIAN_MODEM_READY:
		if (mdev->common->write_check != 'm') {
			MERROR("common block failure\n");
			return -EFAULT;
		}

		if (! test_bit (KMSETTING_SERVE_IRQS, &mdev->state)) {
			unsigned long flags;
			
			pr_debug("martian: serving irqs in module\n");

			mars_lock(mdev, flags);
			mfifo_flush (&mdev->dce_rx);
			mfifo_flush (&mdev->dce_tx);
			mfifo_flush (&mdev->pdm_rx);
			mars_unlock(mdev, flags);

			set_bit (KMSETTING_SERVE_IRQS, &mdev->state);
		}

		set_bit (KMSTATE_UREADY, &mdev->state);
		pr_info("martian_modem is attached.\n");
		return 0;


	case MARTIAN_KDEBUG:
		set_bit (KMSETTING_DEBUG, &mdev->state);
		pr_info("martian: debug option is enabled.\n");
		return 0;

	case MARTIAN_TRUE_SMP:
	case MARTIAN_CHECK_SMP:
		pr_info("martian: smp ioctls are obsolete\n");
		return -ENOIOCTLCMD;


	case MARTIAN_IOCTL_CMD_READ_DSP:
		return mars_read_dsp_ram (mdev, (u16 ) arg);
		
	case MARTIAN_IOCTL_CMD_WRITE_DSP:
	{
		struct _mstore a;
		if (copy_from_user (&a, (void __user *) arg, sizeof a)) 
			return -EFAULT;
		
		mars_write_dsp_ram (mdev, a.addr, a.val);
		return 0;
	}

	case MARTIAN_IOCTL_CMD_CLEAR_DSP:
		mars_clear_dsp_ram (mdev);
		return 0;
		
	case MARTIAN_IOCTL_CMD_DLOAD_DSP:
	{
		struct _mdsp bundle;
		int res;

		if (copy_from_user (&bundle, (void __user *) arg, sizeof bundle))
			return -EFAULT;
		
		//mdev->params.DCPAudioOn = bundle.DCPAudioOn;
		res = mars_download_dsp_user (mdev, (u16 __user *) bundle.data, bundle.num); 

		if (copy_to_user ((void __user *) bundle.pdp_byte_f, &mdev->common->dp_byte_f, sizeof mdev->common->dp_byte_f)) {
			MERROR ("DLOAD: Failed to update byte_f\n");
			if (res >= 0)
				res = -EFAULT;
		}

		return res;
	}

	case MARTIAN_IOCTL_CMD_READREG:
		return mars_read_register_safe(mdev, (u8 ) arg);

	case MARTIAN_IOCTL_CMD_WRITEREG:
	{
		struct _mstore a;
		if (copy_from_user (&a, (void __user *) arg, sizeof a)) 
			return -EFAULT;
		
		mars_write_register_safe(mdev, a.addr, a.val);
		return 0;
	}

	case MARTIAN_IOCTL_CMD_WRITE:
	{
		struct _write_params a;
		if (copy_from_user (&a, (void __user *) arg, sizeof a)) 
			return -EFAULT;
		
		switch(a.type) {
		case Byte:
			mars_write_byte(mdev, a.addr, a.val);
			break;

		case Word:
			mars_write_word(mdev, a.addr, a.val);
			break;

		case Dword:
			mars_write_dword(mdev, a.addr, a.val);
			break;
		}
		return 0;
	}

	case MARTIAN_IOCTL_CMD_REGANDOR:
	{
		struct _andor_params a;
		if (copy_from_user (&a, (void __user *) arg, sizeof a)) 
			return -EFAULT;
		
		mars_reg_andor(mdev, a.reg, a.mask_and, a.mask_or);
		return 0;
	}

	case MARTIAN_IOCTL_CMD_COMMAND:
	{
		struct _command_params a;
		if (copy_from_user (&a, (void __user *) arg, sizeof a)) 
			return -EFAULT;
		
		mars_command (mdev, a.cmd, a.a1, a.a2); 
		return 0;
	}
					
	case MARTIAN_IOCTL_CMD_LONGCMD:
	{
		struct _longcmd_params a;
		if (copy_from_user (&a, (void __user *) arg, sizeof a)) 
			return -EFAULT;
		
		if (mars_command_long (mdev, a.cmd, a.a1, a.a2, a.a3, a.a4) == -1) {
			MERROR("long cmd failure\n");
			return 1;
		}
		return 0;
	}
					
	case MARTIAN_IOCTL_CMD_CSUM:
		return mars_dsp_rom_checksum(mdev);
		
	case MARTIAN_IOCTL_CMD_DCPAUDIO:
		mdev->params.DCPAudioOn = (mdev->params.DCPAudioOn == 0) ? 1 : 0;
		return 0;

	case MARTIAN_IOCTL_CMD_OUTINIT:
		mdev->params.dsp_mars  = mdev->common->x_dsp_mars;
		mdev->params.dsp_mars3 = mdev->common->x_dsp_mars3;

		return 0;

		
	default:
		return -ENOIOCTLCMD;
	}
}	

/* martian_mmap: 
 *	maps common data into martian_modem process space
 *
 * 	we prefer vm_insert_page method to remap_pfn_range
 */
static int martian_mmap(struct file *filp, struct vm_area_struct * vma)
{
	struct martian *mdev = (struct martian *) filp->private_data;
	unsigned long uaddr;
	void *kaddr;
	int size;

	size = PAGE_ALIGN(sizeof *mdev->common);
	if (vma->vm_end - vma->vm_start != size)
		return -EINVAL;

	uaddr = vma->vm_start;
	kaddr = mdev->common;
	for (; size > 0; size -= PAGE_SIZE)
	{
		int res = vm_insert_page(vma, uaddr, vmalloc_to_page(kaddr));
		if (res)
			return res;
		uaddr += PAGE_SIZE;
		kaddr += PAGE_SIZE;
	}

	mdev->common->write_check = '\0';

	return 0;
}

/* single reader only; 
 * it won't function correctly with more
 */
static ssize_t 
martian_read (struct file *filp, char __user *buf,
			size_t count, loff_t *ppos)
{
	struct martian *mdev = (struct martian *) filp->private_data;
	unsigned int pulses;
	int ret;

	if (count < sizeof mdev->count)
		return -EINVAL;
	
 	// NONBLOCK to be added

	ret = wait_event_interruptible (
		mdev->wq, 
 		(pulses = atomic_read (&mdev->count)) > 0
		);

	if (ret == 0) {
		atomic_dec (&mdev->common->UnpassedIrqs);
		atomic_sub (pulses, &mdev->count); 
 		if (put_user (pulses, (unsigned int __user *) buf) != 0)
			return -EFAULT;

		*ppos += sizeof pulses;
		return sizeof pulses;
	}

	return ret;
}

struct file_operations martian_fops = {
	.owner		= THIS_MODULE,
	.open		= martian_open,
	.release	= martian_release,
	
	.read		= martian_read,
	.poll		= martian_poll,
	.mmap		= martian_mmap,

	.unlocked_ioctl	= martian_ioctl,
#if   HAVE_COMPAT_IOCTL
	.compat_ioctl	= martian_ioctl,
#endif
};

/***** martian devices management *****/

static int dev_idx = 0;

static void __devinit 
martian_configure_isr (struct martian *mdev) 
{
	struct martian_common *mcb = mdev->common;

	clear_bit (KMSTATE_UREADY, &mdev->state);
	clear_bit (KMSETTING_SERVE_IRQS, &mdev->state);
	init_waitqueue_head (& mdev->wq);
	atomic_set (&mdev->count, 0);

	/* register settings */
	mcb->BaseAddress	= mdev->metrics.BaseAddress;
	mcb->BaseAddressData	= mdev->metrics.BaseAddress + 1;
	mcb->BaseAddress2	= mdev->metrics.BaseAddress + 2;

	mcb->BaseAddressIndex	= mdev->metrics.BaseAddress;
	/* mcb->BaseValue - defined by first nonint reg{read,write} */

	mars_write_register_safe(mdev, IR, 0xff);
	mcb->dp_bamil_rd7 = 0xff;
}

static void __devinit 
martian_configure_params (struct martian *mdev) 
{
	mdev->params.dsp_mars	= mdev->metrics.dsp_mars;
	mdev->params.dsp_mars3	= mdev->metrics.dsp_mars3;

	if (mdev->metrics.flags & MARTIAN_DEV_TOSHIBA)
		mdev->params.flags = MPF_TOSHIBA;


	mdev->params.s82	= mdev->metrics.s82;
	mdev->params.s84	= mdev->metrics.s84;
	mdev->params.s86	= mdev->metrics.s86;
	mdev->params.s88	= mdev->metrics.s88;

	mdev->params.s75	= mdev->metrics.s75;
	mdev->params.s7e	= mdev->metrics.s7e;


	mdev->params.checksum		= 0;
	mdev->params.chip_version	= 0;
	mdev->params.BaseAddress	= mdev->metrics.BaseAddress;
	mdev->params.BaseAddressData 	= mdev->metrics.BaseAddress + 1;
	mdev->params.BaseAddressIndex	= mdev->metrics.BaseAddress;
	mdev->params.BaseAddress2	= mdev->metrics.BaseAddress + 2;

}

static void martian_init_mcb(struct martian *mdev)
{
	struct martian_common *mcb = mdev->common;

	strcpy (mcb->signature, MARTIAN_SIGNATURE);
	strcpy (mcb->tsignature, MARTIAN_TSIGNATURE);
	strcpy (mcb->kstamp, __stringify (KMARTIAN_STAMP));

	MFIFO_INIT (&mdev->dce_rx, mcb->io_dce_rx_buff);
	MFIFO_INIT (&mdev->dce_tx, mcb->io_dce_tx_buff);
	MFIFO_INIT (&mdev->pdm_rx, mcb->io_pdm_rx_buff);

	mcb->tx_incorrespondence = 
	mcb->rx_incorrespondence =
	mcb->pdm_rx_incorrespondence = 0;

	mcb->pdm_rx_overrun	=
	mcb->rx_overrun		= 0;

	/* core variables */
	*(u16 *)&mcb->S[0x82] = mdev->metrics.s82;
	*(u16 *)&mcb->S[0x84] = mdev->metrics.s84;
	*(u16 *)&mcb->S[0x86] = mdev->metrics.s86;
	*(u16 *)&mcb->S[0x88] = mdev->metrics.s88;

	mcb->S[0x75] = mdev->metrics.s75;
	mcb->S[0x7e] = mdev->metrics.s7e;
	mcb->S[0xa6] = 0x12;
	mcb->S[0xa7] = 0x50;

	mcb->x_dsp_mars	 = mdev->params.dsp_mars;
	mcb->x_dsp_mars3 = mdev->params.dsp_mars3;
}

void mars_report_config(struct martian *mdev)
{
	pr_debug("martian: chip version %d, dp version 0x%x, checksum 0x%x\n", 
			mdev->params.chip_version, 
			mdev->params.dp_version,
			mdev->params.checksum);
}

struct martian *martian_add (struct martian_metrics *metrics, int *error)
{
	struct martian *mdev;
	int ret;
	char name[20] = "driver/mars/0";
	struct martian_common *mcb = NULL;

	mdev = kzalloc (sizeof *mdev, GFP_KERNEL);
	if (!mdev) {
		*error = -ENOMEM;
		return NULL; 
	}

	mdev->metrics = *metrics;

	/*** allocate pages for common data ***/
	mcb = vmalloc_32(sizeof *mcb);
	if (! mcb) {
		MERROR ("page allocation failure\n");
		ret = -ENOMEM;
		goto err_free_mdev;
	}
	memset (mcb, 0, sizeof *mcb);

	spin_lock_init (&mdev->lock);

	// as soon interrupt allocated we should be ready to handle it
	mdev->common = mcb;
	martian_configure_isr (mdev);
	martian_configure_params (mdev);

	// claim resource
	ret = request_irq(
		metrics->irq, 
		martian_isr,  
		IRQF_SHARED | IRQF_DISABLED, 
		"164x", 
		mdev
	);

	if (ret < 0) {
		MERROR ("failed to request irq\n");
		goto err_free_common;
	}

	// just to be explicit
	clear_bit (MARTIAN_STATE_OPEN, &mdev->state);
	clear_bit (KMSETTING_DEBUG, &mdev->state);
	
	// create proc entry for this device
	name[ strlen(name) - 1 ] = '0' + dev_idx;

	mdev->entry = create_proc_entry (name, 0400, NULL);
	if (! mdev->entry) {
		MERROR ("martian: failed to create /proc/%s\n", name);
		ret = -ENOMEM;
		goto err_free_irq;
	}

	martian_init_mcb(mdev);

	// set an index
	mdev->index = dev_idx;

	mdev->entry->proc_fops = & martian_fops;
	mdev->entry->data 	 = mdev;

	pr_info("martian: added device %x:%x " 
			"BaseAddress = 0x%x, CommAddres = 0x%x, irq = %d%s\n", 
			metrics->vendor, metrics->device, 
			metrics->BaseAddress, metrics->CommAddress,
			metrics->irq,
			(metrics->flags & MARTIAN_DEV_TOSHIBA) ? ", Toshiba" :""
		);
	mars_report_config(mdev);

	dev_idx++;

	return mdev;

err_free_irq:
	free_irq (metrics->irq, mdev);

err_free_common:
	vfree(mcb);

err_free_mdev:
	kfree (mdev);

	*error = ret;
	return NULL;
}


void martian_del (struct martian *mdev)
{
	char name[20] = "driver/mars/0";
	name[ strlen(name) - 1 ] = '0' + mdev->index;
	remove_proc_entry (name, NULL);
	
	free_irq(mdev->metrics.irq, mdev);
	vfree(mdev->common);
	kfree(mdev);
}


/***** PCI Driver Interface *****/

static int __devinit 
martian_device_gatherinfo (const struct pci_dev *dev, const struct pci_device_id *ent, struct martian_metrics *metrics)
{
	int bar;
	unsigned short id;

	memset(metrics, 0, sizeof *metrics);

	/*** common setting ***/
	metrics->subsystem_device = dev->subsystem_device;
	metrics->subsystem_vendor = dev->subsystem_vendor;
	metrics->device = dev->device;
	metrics->vendor = dev->vendor;

	metrics->irq = dev->irq;

	if (ent && ent->subvendor == PCI_VENDOR_ID_ATT) {
		/* Toshibas */

		switch (dev->class >> 8) {
		case PCI_CLASS_COMMUNICATION_SERIAL:
		case PCI_CLASS_COMMUNICATION_MULTISERIAL:
		case PCI_CLASS_COMMUNICATION_MODEM:
		case PCI_CLASS_COMMUNICATION_OTHER:
			break;

		default:
			return -ENODEV;
		}

		if (PCI_FUNC (dev->devfn) != 0) {
			pr_info ("martian: Toshiba outside the range\n");
			return -ENODEV;
		}

		metrics->device = 0x440;
		metrics->vendor = PCI_VENDOR_ID_ATT;

		metrics->CommAddress = pci_resource_start (dev, 1);
		metrics->BaseAddress = pci_resource_start (dev, 2);
		metrics->flags |= MARTIAN_DEV_TOSHIBA;
		pr_debug ("martian: Toshiba device\n");

		return 0;
	}

	// check range for xircom
	if (ent && ent->vendor == PCI_VENDOR_ID_XIRCOM 
				&& ent->device == PCI_ANY_ID)

		switch (dev->class >> 8) {
		case PCI_CLASS_COMMUNICATION_SERIAL:
		case PCI_CLASS_COMMUNICATION_MULTISERIAL:
		case PCI_CLASS_COMMUNICATION_MODEM:
		case PCI_CLASS_COMMUNICATION_OTHER:
			if (! (dev->device <= 0x3ff)) {
				MERROR ("XIRCOM outside the range\n");
				return -ENODEV;
			}
			break;

		default:
			return -ENODEV;
		}
	
	// all ids added through sysfs come here

	/* Base/Common Addresses */
	for (bar = 5; bar >= 0; bar--) 
		if ( pci_resource_start (dev, bar) != 0 && 
			(pci_resource_flags (dev, bar) & IORESOURCE_IO) ) {
			metrics->BaseAddress =  pci_resource_start (dev, bar);
			break;
		}

	// continue
	for (bar--; bar >= 0; bar--) 
		if ( pci_resource_start (dev, bar) != 0 && 
			(pci_resource_flags (dev, bar) & IORESOURCE_IO) ) {
			metrics->CommAddress =  pci_resource_start (dev, bar);
			break;
		}

	if (! metrics->BaseAddress) {
		MERROR ("base address not found\n");
		return -ENODEV;
	}


	id = metrics->device;

	metrics->dsp_mars  = 1;
	metrics->dsp_mars3 = 0;

	metrics->s88 = metrics->vendor;	/* metrics!!! */
	metrics->s86 = metrics->device;
	metrics->s82 = dev->subsystem_vendor;
	metrics->s84 = dev->subsystem_device;
	metrics->s75 = 0;
	metrics->s7e = metrics->device & 0xf;

	if (dev->subsystem_vendor == PCI_VENDOR_ID_NEC)
		metrics->s75 = 0x70;

	if (dev->vendor == PCI_VENDOR_ID_XIRCOM && dev->subsystem_vendor == PCI_VENDOR_ID_INTEL) {
		metrics->s82 = PCI_VENDOR_ID_XIRCOM;
		metrics->dsp_mars = 0;
		metrics->s7e = 0xa0;
	}

	if (dev->vendor == PCI_VENDOR_ID_XIRCOM ||
	    dev->subsystem_vendor == PCI_VENDOR_ID_XIRCOM) {
		if (dev->subsystem_device == 0x45b) {
			metrics->s86 = id = 0x45b;
		}
	}

	switch (id) {

	case 0x443:
		metrics->s7e = 0;
		break;

	case 0x45b:
		metrics->dsp_mars3 = 4;
	case 0x444:
		metrics->dsp_mars = 0;
		metrics->s7e = 0xa0;
		break;

	case 0x45c:
		metrics->dsp_mars3 = 3;
	case 0x445:
		metrics->dsp_mars = 0;
		metrics->s7e = 0xa0;
		metrics->s75 = 0x50;
		break;

	case 0x446:
		metrics->dsp_mars = 0;
		metrics->s7e = 0;
		break;

	case 0x447:
		metrics->dsp_mars = 0;
		metrics->s7e = 0;
		metrics->s75 = 0x50;
		break;

	case 0x44b:
	case 0x457:
		metrics->s7e = 8;
		break;

	case 0x44d:
	case 0x455:
	case 0x456:
		metrics->dsp_mars3 = 4;
		metrics->s7e = 0;
		break;

	case 0x44e:
		metrics->dsp_mars3 = 3;
		metrics->s7e = 1;
		break;

	case 0x450:
		metrics->dsp_mars3 = 3;
		metrics->s7e = 9;
		break;

	case 0x453:
	case 0x451:
		metrics->dsp_mars3 = 3;
		metrics->s7e = 0;
		break;
	
	case 0x454:
	case 0x452:
		metrics->dsp_mars3 = 3;
		metrics->s7e = 8;
		break;

	case 0x458:
		metrics->dsp_mars3 = 3;
		metrics->s7e = 2;
		break;

	case 0x45a:
	case 0x44c:
		metrics->dsp_mars3 = 4;
		metrics->s7e = 2;
		break;

	case 0x45d:
	case 0x459:
		metrics->dsp_mars3 = 3;
		metrics->s7e = 0xa;
		break;

	}

	return 0;
}

static int __devinit 
mars_device_probe (struct pci_dev *dev, const struct pci_device_id *ent)
{
	int ret;
	struct martian *mdev;
	struct martian_metrics metrics;

	ret = pci_enable_device (dev);
	if (ret != 0) {
		MERROR ("failed to enable device\n");
		pr_debug("martian: res %d\n", ret);
		return -ENODEV;
	}

	// initialize data
	ret = martian_device_gatherinfo (dev, ent, &metrics);
	if (ret < 0) {
		MERROR ("failed to gather device info\n");
		goto err_disable_pci;
	}

	mdev = martian_add (&metrics, &ret);
	if (! mdev) {
		goto err_disable_pci;
	}

	pci_set_drvdata (dev, mdev);

	dev_idx++;

	return 0;

err_disable_pci:
	pci_disable_device (dev);

	return ret;
}


static void __devexit mars_remove (struct pci_dev *dev) 
{
	martian_del (pci_get_drvdata (dev));
	pci_disable_device (dev);
}

#include "martian_ids.c"

// the device
struct pci_driver martian_driver = {
	.name		= "martian",
	.probe		= mars_device_probe,
	.remove		= __devexit_p (mars_remove),
	.id_table	= martian_ids
};

#define for_each_pci_dev_in_class(_class, mask, dev)  \
while ((dev = pci_get_device (PCI_ANY_ID, PCI_ANY_ID, dev)) != NULL) \
	if ((dev->class ^ _class) & mask)  	\
		continue;			\
	else

// ^ pci_get_class won't do cause we need a mask 

#define SERIAL_DRIVER	"serial"

static void __devinit detach_from_serial_class (unsigned int class, unsigned int mask) 
{
	struct pci_dev *dev = NULL;
	const struct pci_device_id *entry;


	for_each_pci_dev_in_class (class, mask, dev) {

		if (! dev->driver || ! dev->driver->name || strcmp (SERIAL_DRIVER, dev->driver->name) ) 
			continue;

		entry = pci_match_id (martian_ids, dev);
		if (entry && entry->vendor != PCI_ANY_ID) {
			pr_debug("martian: detaching %x:%x from %s\n", 
				dev->vendor, dev->device, dev->driver->name);

			device_release_driver (& dev->dev);
		}
	}
}

static void __devinit detach_from_serial(void) 
{
	detach_from_serial_class (PCI_CLASS_COMMUNICATION_SERIAL << 8, 0xffff00);
	detach_from_serial_class (PCI_CLASS_COMMUNICATION_MODEM << 8, 0xffff00);
	detach_from_serial_class (PCI_CLASS_COMMUNICATION_MULTISERIAL << 8, 0xffff00);
	detach_from_serial_class (PCI_CLASS_COMMUNICATION_OTHER << 8, 0xffff00);
}

/***** kernel module interface *****/

static int __init martian_init (void) 
{
	pr_info("martian loaded - " __stringify (KMARTIAN_STAMP) "\n");

	// allocate proc directory for martians
	martians_proc_dir = proc_mkdir ("driver/mars", NULL);
	if (! martians_proc_dir) {
		MERROR ("cannot create dir\n");
		return -ENOMEM;
	}

	detach_from_serial();

	return pci_register_driver (&martian_driver);
}

static void __exit martian_exit(void) 
{
	pci_unregister_driver (&martian_driver);
	remove_proc_entry ("driver/mars", NULL);
}

module_exit (martian_exit);

module_init (martian_init);

/***** module info *****/

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("A. Chentsov");
MODULE_DESCRIPTION ("Agere 164x dsp driver");
MODULE_DEVICE_TABLE( pci, martian_ids );
