/* Copyright (C) 2004,2005,2006 Andi Kleen, SuSE Labs.
   Decode IA32/x86-64 machine check events in /dev/mcelog. 

   mcelog is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; version
   2.

   mcelog is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should find a copy of v2 of the GNU General Public License somewhere
   on your Linux system; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#define _GNU_SOURCE 1
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <asm/types.h>
#include <asm/ioctls.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <stdarg.h>
#include <ctype.h>
#include "mcelog.h"
#include "k8.h"
#include "p4.h"
#include "dmi.h"

enum {
	CPU_GENERIC,
	CPU_K8,
	CPU_P4
} cpu = CPU_GENERIC;	

char *logfn = "/dev/mcelog";

int use_syslog;
int do_dmi;
int ignore_nodev;
int filter_bogus;

void Wprintf(char *fmt, ...)
{
	va_list ap;
	va_start(ap,fmt);
	if (use_syslog) 
		vsyslog(LOG_NOTICE, fmt, ap);
	else
		vprintf(fmt, ap);
	va_end(ap);
}

char *bankname(unsigned bank) 
{ 
	static char numeric[64];
	switch (cpu) { 
	case CPU_K8:
		return k8_bank_name(bank);
	case CPU_P4:
		return p4_bank_name(bank);
	/* add banks of other cpu types here */
	default:
		sprintf(numeric, "BANK %d", bank); 
		return numeric;
	}
} 

void resolveaddr(unsigned long addr)
{
	if (addr && do_dmi)
		dmi_decodeaddr(addr);
	/* Should check for PCI resources here too */
}

int mce_filter_k8(struct mce *m)
{	
	/* Filter out GART errors */
	if (m->bank == 4) { 
		unsigned short exterrcode = (m->status >> 16) & 0x0f;
		if (exterrcode == 5 && (m->status & (1ULL<<61)))
			return 0;
	} 
	return 1;
}

int mce_filter(struct mce *m)
{
	if (!filter_bogus) 
		return 1;
	/* Filter out known broken MCEs */
	switch (cpu) {
	case CPU_K8:
		return mce_filter_k8(m);
		/* add more buggy CPUs here */
	case CPU_P4:
		/* No bugs known */
		return 1;
	default:
	case CPU_GENERIC:
		return 1;
	}	
}

void dump_mce(struct mce *m) 
{
	Wprintf("HARDWARE ERROR. This is *NOT* a software problem!\n");
	Wprintf("Please contact your hardware vendor\n");
	/* should not happen */
	if (!m->finished)
		Wprintf("not finished?\n");
	Wprintf("CPU %d %s ", m->cpu, bankname(m->bank));
	if (m->tsc)
		Wprintf("TSC %Lx %s\n", 
			m->tsc, 
			(m->mcgstatus & MCI_STATUS_UC) ? 
			"(upper bound, found by polled driver)" : "");
	if (m->rip) 
		Wprintf("RIP%s %02x:%Lx ", 
		       !(m->mcgstatus & MCG_STATUS_EIPV) ? " !INEXACT!" : "",
		       m->cs, m->rip);
	if (m->misc)
		Wprintf("MISC %Lx ", m->misc);
	if (m->addr)
		Wprintf("ADDR %Lx ", m->addr);
	if (m->rip | m->misc | m->addr)	
		Wprintf("\n");
	switch (cpu) { 
	case CPU_K8:
		decode_k8_mc(m); 
		break;
	case CPU_P4:
		decode_p4_mc(m);
		break;
	/* add handlers for other CPUs here */
	default:
		break;
	} 
	/* decode all status bits here */
	Wprintf("STATUS %Lx MCGSTATUS %Lx\n", m->status, m->mcgstatus);
	resolveaddr(m->addr);
}

void check_cpu(void)
{ 
	FILE *f;
	f = fopen("/proc/cpuinfo","r");
	if (f != NULL) { 
		int found = 0; 
		int family; 
		char vendor[64];
		char *line = NULL;
		size_t linelen = 0; 
		while (getdelim(&line, &linelen, '\n', f) > 0 && found < 2) { 
			if (sscanf(line, "vendor_id : %63[^\n]", vendor) == 1) 
				found++; 
			if (sscanf(line, "cpu family : %d", &family) == 1)
				found++;
		} 
		if (found == 2) {
			if (!strcmp(vendor,"AuthenticAMD") && family == 15)
				cpu = CPU_K8;
			if (!strcmp(vendor,"GenuineIntel") && family == 15)
				cpu = CPU_P4;
			/* Add checks for other CPUs here */	
		} else {
			fprintf(stderr, "mcelog: warning: Cannot parse /proc/cpuinfo\n"); 
		} 
		fclose(f);
		free(line);
	} else
		fprintf(stderr, "mcelog: warning: Cannot open /proc/cpuinfo\n");
} 

char *skipgunk(char *s)
{
	while (isspace(*s))
		++s; 
	if (*s == '<') { 
		s += strcspn(s, ">"); 
		if (*s == '>') 
			++s; 
	}
	while (isspace(*s))
		++s; 
	return s;
}

void dump_mce_final(struct mce *m, char *symbol, int missing)
{
	m->finished = 1;
	dump_mce(m);
	if (symbol[0])
		Wprintf("RIP: %s\n", symbol);
}

/* Decode ASCII input for fatal messages */
void decodefatal(FILE *inf)
{
	struct mce m;
	char *line = NULL; 
	size_t linelen = 0;
	int k;
	int missing = 0; 
	char symbol[100];
	int data = 0;

	if (do_dmi)
		Wprintf(
 "WARNING: with --dmi mcelog --ascii must run on the same machine with the\n"
 "     same BIOS/memory configuration as where the machine check occurred.\n");

	k = 0;
	memset(&m, 0, sizeof(struct mce));
	symbol[0] = '\0';
	while (getdelim(&line, &linelen, '\n', inf) > 0) { 
		int n = 0;
		char *s = skipgunk(line), *p;

		if (!strncmp(s, "CPU", 3)) { 
			unsigned cpu = 0, bank = 0;
			n = sscanf(s,
	       "CPU %u: Machine Check Exception: %16Lx Bank %d: %016Lx",
			       &cpu,
			       &m.mcgstatus,
			       &bank,
			       &m.status
			       );
			if (n == 1) {
				n = sscanf(s, "CPU %u %u", &cpu, &bank);
				m.cpu = cpu;
				m.bank = bank;
				if (n != 2) 
					missing++;
				
				if ((p = strstr(s, "TSC")) != NULL) {
					if (sscanf(p,"TSC %Lx", &m.tsc)!=1)
						missing++;
				}
			} else { 
				m.cpu = cpu;
				m.bank = bank;
				if (n != 4) 
					missing++; 

			}
		} 
		else if (!strncmp(s, "STATUS", 6)) {
			if (sscanf(s,"STATUS %Lx MCGSTATUS %Lx", 
				   &m.status,  &m.mcgstatus) != 2)
				missing++;			
		} 
		else if (!strncmp(s, "RIP", 3)) { 
			unsigned cs = 0; 

			if (!strncmp(s, "RIP!INEXACT!", 12))
				s += 12; 
			else
				s += 3; 

			n = sscanf(s, "%02x:<%016Lx> {%100s}",
				   &cs,
				   &m.rip, 
				   symbol); 
			m.cs = cs;
			if (n < 2) 
				missing++; 
		} 

		else if (!strncmp(s, "TSC",3)) { 
			if ((n = sscanf(s, "TSC %Lx", &m.tsc)) != 1) 
				missing++; 
		}
		else if (!strncmp(s, "ADDR",4)) { 
			if ((n = sscanf(s, "ADDR %Lx", &m.addr)) != 1) 
				missing++;
		}
		else if (!strncmp(s, "MISC",4)) { 
			if ((n = sscanf(s, "MISC %Lx", &m.misc)) != 1) 
				missing++; 
		}
		else { 
			if (*s && data) { 
				dump_mce_final(&m, symbol, missing); 
				data = 0;
			} 
			Wprintf("%s", line); 
		} 
		if (n > 0) 
			data = 1;
	} 
	free(line);
	if (data)
		dump_mce_final(&m, symbol, missing);
}

void usage(void)
{
	fprintf(stderr, 
		"Usage:\n"
		"  mcelog [--k8|--p4|--generic] [--ignorenodev] [--dmi] [--syslog] [--filter] [mcelogdevice]\n"
		"Decode machine check error records from kernel\n"
		"  mcelog [--k8|--p4|--generic] [--dmi] --ascii < log\n"
		"Decode machine check ASCII output from kernel logs\n");
	exit(1);
}

int modifier(char *s)
{
	if (!strcmp(s, "--k8")) {
		cpu = CPU_K8;
	} else if (!strcmp(s, "--p4")) {
		cpu = CPU_P4;
	} else if (!strcmp(s, "--generic")) { 
		cpu = CPU_GENERIC;
	} else if (!strcmp(s, "--ignorenodev")) { 
		ignore_nodev = 1;
	} else if (!strcmp(s,"--filter")) { 
		filter_bogus = 1;			
	} else if (!strcmp(s, "--dmi")) { 
		do_dmi = 1;
		opendmi();
	} else if (!strcmp(s, "--syslog")) { 
		openlog("mcelog", 0, LOG_DAEMON);
		use_syslog = 1;
	} else
		return 0;
	return 1;
} 

int main(int ac, char **av) 
{ 
	int recordlen = 0;
	int loglen = 0;

	check_cpu();

	while (*++av) { 
		if (modifier(*av)) {
			/* ok */
		} else if (!strcmp(*av, "--ascii")) { 
			while (*++av) {
				if (!modifier(*av))
					usage();
			}
			decodefatal(stdin); 
			exit(0);
		} else if (!strncmp(*av, "--", 2)) { 
			if (av[0][2] == '\0') 
				break;
			usage();
		}
	} 
	if (*av)
		logfn = *av++;
	if (*av)
		usage();
		
	int fd = open(logfn, O_RDONLY); 
	if (fd < 0) {
		if (ignore_nodev) 
			exit(0);
		fprintf(stderr, "Cannot open %s\n", logfn); 
		exit(1);
	}
	
	if (ioctl(fd, MCE_GET_RECORD_LEN, &recordlen) < 0)
		err("MCE_GET_RECORD_LEN");
	if (ioctl(fd, MCE_GET_LOG_LEN, &loglen) < 0)
		err("MCE_GET_LOG_LEN");

	if (recordlen > sizeof(struct mce))
		fprintf(stderr, 
    "mcelog: warning: record length longer than expected. Consider update.\n");

	char *buf = calloc(recordlen, loglen); 
	if (!buf) 
		exit(100);
	
	int len = read(fd, buf, recordlen * loglen); 
	if (len < 0) 
		err("read"); 

	int i; 
	for (i = 0; i < len / recordlen; i++) { 
		struct mce *mce = (struct mce *)(buf + i*recordlen);
		if (!mce_filter(mce)) 
			continue;
		printf("MCE %d\n", i); 
		dump_mce(mce); 
	}

	if (recordlen < sizeof(struct mce))  {
		fprintf(stderr, 
			"mcelog: warning: %lu bytes ignored in each record\n",
				(unsigned long)recordlen - sizeof(struct mce)); 
		fprintf(stderr, "mcelog: consider an update\n"); 
	}	

	exit(0); 
} 
