/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/**
 * \file indirect_api.cpp
 * Validate the generated code in indirect_init.c
 *
 * Tests various apsects of the dispatch table generated by
 * \c __glXNewIndirectAPI.
 *
 *    * No entry in the table should be \c NULL.
 *
 *    * Entries in the table that correspond to "known" functions with GLX
 *      protocol should point to the correct function.
 *
 *    * Entries beyond the end of the "known" part of the table (i.e., entries
 *      that can be allocated by drivers for extensions) should point to a
 *      no-op function.
 *
 *    * Entries in the table that correspond to "known" functions that lack
 *      GLX protocol should point to a no-op function.
 *
 * Very few entries in the last catogory are tests.  See \c OpenGL_20_is_nop.
 */

#include <gtest/gtest.h>
#include <GL/gl.h>
extern "C" {
#include "../indirect_init.h"
#include "glapi/glapi.h"
}

static const void *nil = 0;

#define EXTRA_DISPATCH 111

static bool dispatch_table_size_was_queried = false;

extern "C" GLuint
_glapi_get_dispatch_table_size(void)
{
   dispatch_table_size_was_queried = true;
   return (sizeof(struct _glapi_table) / sizeof(void *)) + EXTRA_DISPATCH;
}

/**
 * \name Indirect-rendering function stubs
 *
 * These are necessary so that indirect_init.c doesn't try to link with
 * indirect.c.  Linking with indirect.c would require linking with various X
 * libraries and a bunch of other stuff.  While this is ugly, it does simplify
 * the build quite a bit.
 */
/*@{*/
extern "C" {
void __indirect_glAccum(void) { }
void __indirect_glAlphaFunc(void) { }
void __indirect_glBegin(void) { }
void __indirect_glBitmap(void) { }
void __indirect_glBlendFunc(void) { }
void __indirect_glCallList(void) { }
void __indirect_glCallLists(void) { }
void __indirect_glClear(void) { }
void __indirect_glClearAccum(void) { }
void __indirect_glClearColor(void) { }
void __indirect_glClearDepth(void) { }
void __indirect_glClearIndex(void) { }
void __indirect_glClearStencil(void) { }
void __indirect_glClipPlane(void) { }
void __indirect_glColor3b(void) { }
void __indirect_glColor3bv(void) { }
void __indirect_glColor3d(void) { }
void __indirect_glColor3dv(void) { }
void __indirect_glColor3f(void) { }
void __indirect_glColor3fv(void) { }
void __indirect_glColor3i(void) { }
void __indirect_glColor3iv(void) { }
void __indirect_glColor3s(void) { }
void __indirect_glColor3sv(void) { }
void __indirect_glColor3ub(void) { }
void __indirect_glColor3ubv(void) { }
void __indirect_glColor3ui(void) { }
void __indirect_glColor3uiv(void) { }
void __indirect_glColor3us(void) { }
void __indirect_glColor3usv(void) { }
void __indirect_glColor4b(void) { }
void __indirect_glColor4bv(void) { }
void __indirect_glColor4d(void) { }
void __indirect_glColor4dv(void) { }
void __indirect_glColor4f(void) { }
void __indirect_glColor4fv(void) { }
void __indirect_glColor4i(void) { }
void __indirect_glColor4iv(void) { }
void __indirect_glColor4s(void) { }
void __indirect_glColor4sv(void) { }
void __indirect_glColor4ub(void) { }
void __indirect_glColor4ubv(void) { }
void __indirect_glColor4ui(void) { }
void __indirect_glColor4uiv(void) { }
void __indirect_glColor4us(void) { }
void __indirect_glColor4usv(void) { }
void __indirect_glColorMask(void) { }
void __indirect_glColorMaterial(void) { }
void __indirect_glCopyPixels(void) { }
void __indirect_glCullFace(void) { }
void __indirect_glDeleteLists(void) { }
void __indirect_glDepthFunc(void) { }
void __indirect_glDepthMask(void) { }
void __indirect_glDepthRange(void) { }
void __indirect_glDisable(void) { }
void __indirect_glDrawBuffer(void) { }
void __indirect_glDrawPixels(void) { }
void __indirect_glEdgeFlag(void) { }
void __indirect_glEdgeFlagv(void) { }
void __indirect_glEnable(void) { }
void __indirect_glEnd(void) { }
void __indirect_glEndList(void) { }
void __indirect_glEvalCoord1d(void) { }
void __indirect_glEvalCoord1dv(void) { }
void __indirect_glEvalCoord1f(void) { }
void __indirect_glEvalCoord1fv(void) { }
void __indirect_glEvalCoord2d(void) { }
void __indirect_glEvalCoord2dv(void) { }
void __indirect_glEvalCoord2f(void) { }
void __indirect_glEvalCoord2fv(void) { }
void __indirect_glEvalMesh1(void) { }
void __indirect_glEvalMesh2(void) { }
void __indirect_glEvalPoint1(void) { }
void __indirect_glEvalPoint2(void) { }
void __indirect_glFeedbackBuffer(void) { }
void __indirect_glFinish(void) { }
void __indirect_glFlush(void) { }
void __indirect_glFogf(void) { }
void __indirect_glFogfv(void) { }
void __indirect_glFogi(void) { }
void __indirect_glFogiv(void) { }
void __indirect_glFrontFace(void) { }
void __indirect_glFrustum(void) { }
void __indirect_glGenLists(void) { }
void __indirect_glGetBooleanv(void) { }
void __indirect_glGetClipPlane(void) { }
void __indirect_glGetDoublev(void) { }
void __indirect_glGetError(void) { }
void __indirect_glGetFloatv(void) { }
void __indirect_glGetIntegerv(void) { }
void __indirect_glGetLightfv(void) { }
void __indirect_glGetLightiv(void) { }
void __indirect_glGetMapdv(void) { }
void __indirect_glGetMapfv(void) { }
void __indirect_glGetMapiv(void) { }
void __indirect_glGetMaterialfv(void) { }
void __indirect_glGetMaterialiv(void) { }
void __indirect_glGetPixelMapfv(void) { }
void __indirect_glGetPixelMapuiv(void) { }
void __indirect_glGetPixelMapusv(void) { }
void __indirect_glGetPolygonStipple(void) { }
void __indirect_glGetString(void) { }
void __indirect_glGetTexEnvfv(void) { }
void __indirect_glGetTexEnviv(void) { }
void __indirect_glGetTexGendv(void) { }
void __indirect_glGetTexGenfv(void) { }
void __indirect_glGetTexGeniv(void) { }
void __indirect_glGetTexImage(void) { }
void __indirect_glGetTexLevelParameterfv(void) { }
void __indirect_glGetTexLevelParameteriv(void) { }
void __indirect_glGetTexParameterfv(void) { }
void __indirect_glGetTexParameteriv(void) { }
void __indirect_glHint(void) { }
void __indirect_glIndexMask(void) { }
void __indirect_glIndexd(void) { }
void __indirect_glIndexdv(void) { }
void __indirect_glIndexf(void) { }
void __indirect_glIndexfv(void) { }
void __indirect_glIndexi(void) { }
void __indirect_glIndexiv(void) { }
void __indirect_glIndexs(void) { }
void __indirect_glIndexsv(void) { }
void __indirect_glInitNames(void) { }
void __indirect_glIsEnabled(void) { }
void __indirect_glIsList(void) { }
void __indirect_glLightModelf(void) { }
void __indirect_glLightModelfv(void) { }
void __indirect_glLightModeli(void) { }
void __indirect_glLightModeliv(void) { }
void __indirect_glLightf(void) { }
void __indirect_glLightfv(void) { }
void __indirect_glLighti(void) { }
void __indirect_glLightiv(void) { }
void __indirect_glLineStipple(void) { }
void __indirect_glLineWidth(void) { }
void __indirect_glListBase(void) { }
void __indirect_glLoadIdentity(void) { }
void __indirect_glLoadMatrixd(void) { }
void __indirect_glLoadMatrixf(void) { }
void __indirect_glLoadName(void) { }
void __indirect_glLogicOp(void) { }
void __indirect_glMap1d(void) { }
void __indirect_glMap1f(void) { }
void __indirect_glMap2d(void) { }
void __indirect_glMap2f(void) { }
void __indirect_glMapGrid1d(void) { }
void __indirect_glMapGrid1f(void) { }
void __indirect_glMapGrid2d(void) { }
void __indirect_glMapGrid2f(void) { }
void __indirect_glMaterialf(void) { }
void __indirect_glMaterialfv(void) { }
void __indirect_glMateriali(void) { }
void __indirect_glMaterialiv(void) { }
void __indirect_glMatrixMode(void) { }
void __indirect_glMultMatrixd(void) { }
void __indirect_glMultMatrixf(void) { }
void __indirect_glNewList(void) { }
void __indirect_glNormal3b(void) { }
void __indirect_glNormal3bv(void) { }
void __indirect_glNormal3d(void) { }
void __indirect_glNormal3dv(void) { }
void __indirect_glNormal3f(void) { }
void __indirect_glNormal3fv(void) { }
void __indirect_glNormal3i(void) { }
void __indirect_glNormal3iv(void) { }
void __indirect_glNormal3s(void) { }
void __indirect_glNormal3sv(void) { }
void __indirect_glOrtho(void) { }
void __indirect_glPassThrough(void) { }
void __indirect_glPixelMapfv(void) { }
void __indirect_glPixelMapuiv(void) { }
void __indirect_glPixelMapusv(void) { }
void __indirect_glPixelStoref(void) { }
void __indirect_glPixelStorei(void) { }
void __indirect_glPixelTransferf(void) { }
void __indirect_glPixelTransferi(void) { }
void __indirect_glPixelZoom(void) { }
void __indirect_glPointSize(void) { }
void __indirect_glPolygonMode(void) { }
void __indirect_glPolygonStipple(void) { }
void __indirect_glPopAttrib(void) { }
void __indirect_glPopMatrix(void) { }
void __indirect_glPopName(void) { }
void __indirect_glPushAttrib(void) { }
void __indirect_glPushMatrix(void) { }
void __indirect_glPushName(void) { }
void __indirect_glRasterPos2d(void) { }
void __indirect_glRasterPos2dv(void) { }
void __indirect_glRasterPos2f(void) { }
void __indirect_glRasterPos2fv(void) { }
void __indirect_glRasterPos2i(void) { }
void __indirect_glRasterPos2iv(void) { }
void __indirect_glRasterPos2s(void) { }
void __indirect_glRasterPos2sv(void) { }
void __indirect_glRasterPos3d(void) { }
void __indirect_glRasterPos3dv(void) { }
void __indirect_glRasterPos3f(void) { }
void __indirect_glRasterPos3fv(void) { }
void __indirect_glRasterPos3i(void) { }
void __indirect_glRasterPos3iv(void) { }
void __indirect_glRasterPos3s(void) { }
void __indirect_glRasterPos3sv(void) { }
void __indirect_glRasterPos4d(void) { }
void __indirect_glRasterPos4dv(void) { }
void __indirect_glRasterPos4f(void) { }
void __indirect_glRasterPos4fv(void) { }
void __indirect_glRasterPos4i(void) { }
void __indirect_glRasterPos4iv(void) { }
void __indirect_glRasterPos4s(void) { }
void __indirect_glRasterPos4sv(void) { }
void __indirect_glReadBuffer(void) { }
void __indirect_glReadPixels(void) { }
void __indirect_glRectd(void) { }
void __indirect_glRectdv(void) { }
void __indirect_glRectf(void) { }
void __indirect_glRectfv(void) { }
void __indirect_glRecti(void) { }
void __indirect_glRectiv(void) { }
void __indirect_glRects(void) { }
void __indirect_glRectsv(void) { }
void __indirect_glRenderMode(void) { }
void __indirect_glRotated(void) { }
void __indirect_glRotatef(void) { }
void __indirect_glScaled(void) { }
void __indirect_glScalef(void) { }
void __indirect_glScissor(void) { }
void __indirect_glSelectBuffer(void) { }
void __indirect_glShadeModel(void) { }
void __indirect_glStencilFunc(void) { }
void __indirect_glStencilMask(void) { }
void __indirect_glStencilOp(void) { }
void __indirect_glTexCoord1d(void) { }
void __indirect_glTexCoord1dv(void) { }
void __indirect_glTexCoord1f(void) { }
void __indirect_glTexCoord1fv(void) { }
void __indirect_glTexCoord1i(void) { }
void __indirect_glTexCoord1iv(void) { }
void __indirect_glTexCoord1s(void) { }
void __indirect_glTexCoord1sv(void) { }
void __indirect_glTexCoord2d(void) { }
void __indirect_glTexCoord2dv(void) { }
void __indirect_glTexCoord2f(void) { }
void __indirect_glTexCoord2fv(void) { }
void __indirect_glTexCoord2i(void) { }
void __indirect_glTexCoord2iv(void) { }
void __indirect_glTexCoord2s(void) { }
void __indirect_glTexCoord2sv(void) { }
void __indirect_glTexCoord3d(void) { }
void __indirect_glTexCoord3dv(void) { }
void __indirect_glTexCoord3f(void) { }
void __indirect_glTexCoord3fv(void) { }
void __indirect_glTexCoord3i(void) { }
void __indirect_glTexCoord3iv(void) { }
void __indirect_glTexCoord3s(void) { }
void __indirect_glTexCoord3sv(void) { }
void __indirect_glTexCoord4d(void) { }
void __indirect_glTexCoord4dv(void) { }
void __indirect_glTexCoord4f(void) { }
void __indirect_glTexCoord4fv(void) { }
void __indirect_glTexCoord4i(void) { }
void __indirect_glTexCoord4iv(void) { }
void __indirect_glTexCoord4s(void) { }
void __indirect_glTexCoord4sv(void) { }
void __indirect_glTexEnvf(void) { }
void __indirect_glTexEnvfv(void) { }
void __indirect_glTexEnvi(void) { }
void __indirect_glTexEnviv(void) { }
void __indirect_glTexGend(void) { }
void __indirect_glTexGendv(void) { }
void __indirect_glTexGenf(void) { }
void __indirect_glTexGenfv(void) { }
void __indirect_glTexGeni(void) { }
void __indirect_glTexGeniv(void) { }
void __indirect_glTexImage1D(void) { }
void __indirect_glTexImage2D(void) { }
void __indirect_glTexParameterf(void) { }
void __indirect_glTexParameterfv(void) { }
void __indirect_glTexParameteri(void) { }
void __indirect_glTexParameteriv(void) { }
void __indirect_glTranslated(void) { }
void __indirect_glTranslatef(void) { }
void __indirect_glVertex2d(void) { }
void __indirect_glVertex2dv(void) { }
void __indirect_glVertex2f(void) { }
void __indirect_glVertex2fv(void) { }
void __indirect_glVertex2i(void) { }
void __indirect_glVertex2iv(void) { }
void __indirect_glVertex2s(void) { }
void __indirect_glVertex2sv(void) { }
void __indirect_glVertex3d(void) { }
void __indirect_glVertex3dv(void) { }
void __indirect_glVertex3f(void) { }
void __indirect_glVertex3fv(void) { }
void __indirect_glVertex3i(void) { }
void __indirect_glVertex3iv(void) { }
void __indirect_glVertex3s(void) { }
void __indirect_glVertex3sv(void) { }
void __indirect_glVertex4d(void) { }
void __indirect_glVertex4dv(void) { }
void __indirect_glVertex4f(void) { }
void __indirect_glVertex4fv(void) { }
void __indirect_glVertex4i(void) { }
void __indirect_glVertex4iv(void) { }
void __indirect_glVertex4s(void) { }
void __indirect_glVertex4sv(void) { }
void __indirect_glViewport(void) { }
void __indirect_glAreTexturesResident(void) { }
void __indirect_glArrayElement(void) { }
void __indirect_glBindTexture(void) { }
void __indirect_glColorPointer(void) { }
void __indirect_glCopyTexImage1D(void) { }
void __indirect_glCopyTexImage2D(void) { }
void __indirect_glCopyTexSubImage1D(void) { }
void __indirect_glCopyTexSubImage2D(void) { }
void __indirect_glDeleteTextures(void) { }
void __indirect_glDisableClientState(void) { }
void __indirect_glDrawArrays(void) { }
void __indirect_glDrawElements(void) { }
void __indirect_glEdgeFlagPointer(void) { }
void __indirect_glEnableClientState(void) { }
void __indirect_glGenTextures(void) { }
void __indirect_glGetPointerv(void) { }
void __indirect_glIndexPointer(void) { }
void __indirect_glIndexub(void) { }
void __indirect_glIndexubv(void) { }
void __indirect_glInterleavedArrays(void) { }
void __indirect_glIsTexture(void) { }
void __indirect_glNormalPointer(void) { }
void __indirect_glPolygonOffset(void) { }
void __indirect_glPopClientAttrib(void) { }
void __indirect_glPrioritizeTextures(void) { }
void __indirect_glPushClientAttrib(void) { }
void __indirect_glTexCoordPointer(void) { }
void __indirect_glTexSubImage1D(void) { }
void __indirect_glTexSubImage2D(void) { }
void __indirect_glVertexPointer(void) { }
void __indirect_glBlendColor(void) { }
void __indirect_glBlendEquation(void) { }
void __indirect_glColorSubTable(void) { }
void __indirect_glColorTable(void) { }
void __indirect_glColorTableParameterfv(void) { }
void __indirect_glColorTableParameteriv(void) { }
void __indirect_glConvolutionFilter1D(void) { }
void __indirect_glConvolutionFilter2D(void) { }
void __indirect_glConvolutionParameterf(void) { }
void __indirect_glConvolutionParameterfv(void) { }
void __indirect_glConvolutionParameteri(void) { }
void __indirect_glConvolutionParameteriv(void) { }
void __indirect_glCopyColorSubTable(void) { }
void __indirect_glCopyColorTable(void) { }
void __indirect_glCopyConvolutionFilter1D(void) { }
void __indirect_glCopyConvolutionFilter2D(void) { }
void __indirect_glCopyTexSubImage3D(void) { }
void __indirect_glDrawRangeElements(void) { }
void __indirect_glGetColorTable(void) { }
void __indirect_glGetColorTableParameterfv(void) { }
void __indirect_glGetColorTableParameteriv(void) { }
void __indirect_glGetConvolutionFilter(void) { }
void __indirect_glGetConvolutionParameterfv(void) { }
void __indirect_glGetConvolutionParameteriv(void) { }
void __indirect_glGetHistogram(void) { }
void __indirect_glGetHistogramParameterfv(void) { }
void __indirect_glGetHistogramParameteriv(void) { }
void __indirect_glGetMinmax(void) { }
void __indirect_glGetMinmaxParameterfv(void) { }
void __indirect_glGetMinmaxParameteriv(void) { }
void __indirect_glGetSeparableFilter(void) { }
void __indirect_glHistogram(void) { }
void __indirect_glMinmax(void) { }
void __indirect_glResetHistogram(void) { }
void __indirect_glResetMinmax(void) { }
void __indirect_glSeparableFilter2D(void) { }
void __indirect_glTexImage3D(void) { }
void __indirect_glTexSubImage3D(void) { }
void __indirect_glActiveTextureARB(void) { }
void __indirect_glClientActiveTextureARB(void) { }
void __indirect_glMultiTexCoord1dARB(void) { }
void __indirect_glMultiTexCoord1dvARB(void) { }
void __indirect_glMultiTexCoord1fARB(void) { }
void __indirect_glMultiTexCoord1fvARB(void) { }
void __indirect_glMultiTexCoord1iARB(void) { }
void __indirect_glMultiTexCoord1ivARB(void) { }
void __indirect_glMultiTexCoord1sARB(void) { }
void __indirect_glMultiTexCoord1svARB(void) { }
void __indirect_glMultiTexCoord2dARB(void) { }
void __indirect_glMultiTexCoord2dvARB(void) { }
void __indirect_glMultiTexCoord2fARB(void) { }
void __indirect_glMultiTexCoord2fvARB(void) { }
void __indirect_glMultiTexCoord2iARB(void) { }
void __indirect_glMultiTexCoord2ivARB(void) { }
void __indirect_glMultiTexCoord2sARB(void) { }
void __indirect_glMultiTexCoord2svARB(void) { }
void __indirect_glMultiTexCoord3dARB(void) { }
void __indirect_glMultiTexCoord3dvARB(void) { }
void __indirect_glMultiTexCoord3fARB(void) { }
void __indirect_glMultiTexCoord3fvARB(void) { }
void __indirect_glMultiTexCoord3iARB(void) { }
void __indirect_glMultiTexCoord3ivARB(void) { }
void __indirect_glMultiTexCoord3sARB(void) { }
void __indirect_glMultiTexCoord3svARB(void) { }
void __indirect_glMultiTexCoord4dARB(void) { }
void __indirect_glMultiTexCoord4dvARB(void) { }
void __indirect_glMultiTexCoord4fARB(void) { }
void __indirect_glMultiTexCoord4fvARB(void) { }
void __indirect_glMultiTexCoord4iARB(void) { }
void __indirect_glMultiTexCoord4ivARB(void) { }
void __indirect_glMultiTexCoord4sARB(void) { }
void __indirect_glMultiTexCoord4svARB(void) { }
void __indirect_glLoadTransposeMatrixdARB(void) { }
void __indirect_glLoadTransposeMatrixfARB(void) { }
void __indirect_glMultTransposeMatrixdARB(void) { }
void __indirect_glMultTransposeMatrixfARB(void) { }
void __indirect_glSampleCoverageARB(void) { }
void __indirect_glCompressedTexImage1DARB(void) { }
void __indirect_glCompressedTexImage2DARB(void) { }
void __indirect_glCompressedTexImage3DARB(void) { }
void __indirect_glCompressedTexSubImage1DARB(void) { }
void __indirect_glCompressedTexSubImage2DARB(void) { }
void __indirect_glCompressedTexSubImage3DARB(void) { }
void __indirect_glGetCompressedTexImageARB(void) { }
void __indirect_glDisableVertexAttribArrayARB(void) { }
void __indirect_glEnableVertexAttribArrayARB(void) { }
void __indirect_glGetProgramEnvParameterdvARB(void) { }
void __indirect_glGetProgramEnvParameterfvARB(void) { }
void __indirect_glGetProgramLocalParameterdvARB(void) { }
void __indirect_glGetProgramLocalParameterfvARB(void) { }
void __indirect_glGetProgramStringARB(void) { }
void __indirect_glGetProgramivARB(void) { }
void __indirect_glGetVertexAttribdvARB(void) { }
void __indirect_glGetVertexAttribfvARB(void) { }
void __indirect_glGetVertexAttribivARB(void) { }
void __indirect_glProgramEnvParameter4dARB(void) { }
void __indirect_glProgramEnvParameter4dvARB(void) { }
void __indirect_glProgramEnvParameter4fARB(void) { }
void __indirect_glProgramEnvParameter4fvARB(void) { }
void __indirect_glProgramLocalParameter4dARB(void) { }
void __indirect_glProgramLocalParameter4dvARB(void) { }
void __indirect_glProgramLocalParameter4fARB(void) { }
void __indirect_glProgramLocalParameter4fvARB(void) { }
void __indirect_glProgramStringARB(void) { }
void __indirect_glVertexAttrib1dARB(void) { }
void __indirect_glVertexAttrib1dvARB(void) { }
void __indirect_glVertexAttrib1fARB(void) { }
void __indirect_glVertexAttrib1fvARB(void) { }
void __indirect_glVertexAttrib1sARB(void) { }
void __indirect_glVertexAttrib1svARB(void) { }
void __indirect_glVertexAttrib2dARB(void) { }
void __indirect_glVertexAttrib2dvARB(void) { }
void __indirect_glVertexAttrib2fARB(void) { }
void __indirect_glVertexAttrib2fvARB(void) { }
void __indirect_glVertexAttrib2sARB(void) { }
void __indirect_glVertexAttrib2svARB(void) { }
void __indirect_glVertexAttrib3dARB(void) { }
void __indirect_glVertexAttrib3dvARB(void) { }
void __indirect_glVertexAttrib3fARB(void) { }
void __indirect_glVertexAttrib3fvARB(void) { }
void __indirect_glVertexAttrib3sARB(void) { }
void __indirect_glVertexAttrib3svARB(void) { }
void __indirect_glVertexAttrib4NbvARB(void) { }
void __indirect_glVertexAttrib4NivARB(void) { }
void __indirect_glVertexAttrib4NsvARB(void) { }
void __indirect_glVertexAttrib4NubARB(void) { }
void __indirect_glVertexAttrib4NubvARB(void) { }
void __indirect_glVertexAttrib4NuivARB(void) { }
void __indirect_glVertexAttrib4NusvARB(void) { }
void __indirect_glVertexAttrib4bvARB(void) { }
void __indirect_glVertexAttrib4dARB(void) { }
void __indirect_glVertexAttrib4dvARB(void) { }
void __indirect_glVertexAttrib4fARB(void) { }
void __indirect_glVertexAttrib4fvARB(void) { }
void __indirect_glVertexAttrib4ivARB(void) { }
void __indirect_glVertexAttrib4sARB(void) { }
void __indirect_glVertexAttrib4svARB(void) { }
void __indirect_glVertexAttrib4ubvARB(void) { }
void __indirect_glVertexAttrib4uivARB(void) { }
void __indirect_glVertexAttrib4usvARB(void) { }
void __indirect_glVertexAttribPointerARB(void) { }
void __indirect_glBeginQueryARB(void) { }
void __indirect_glDeleteQueriesARB(void) { }
void __indirect_glEndQueryARB(void) { }
void __indirect_glGenQueriesARB(void) { }
void __indirect_glGetQueryObjectivARB(void) { }
void __indirect_glGetQueryObjectuivARB(void) { }
void __indirect_glGetQueryivARB(void) { }
void __indirect_glIsQueryARB(void) { }
void __indirect_glDrawBuffersARB(void) { }
void __indirect_glClampColorARB(void) { }
void __indirect_glRenderbufferStorageMultisample(void) { }
void __indirect_glSampleMaskSGIS(void) { }
void __indirect_glSamplePatternSGIS(void) { }
void __indirect_glColorPointerEXT(void) { }
void __indirect_glEdgeFlagPointerEXT(void) { }
void __indirect_glIndexPointerEXT(void) { }
void __indirect_glNormalPointerEXT(void) { }
void __indirect_glTexCoordPointerEXT(void) { }
void __indirect_glVertexPointerEXT(void) { }
void __indirect_glPointParameterfEXT(void) { }
void __indirect_glPointParameterfvEXT(void) { }
void __indirect_glSecondaryColor3bEXT(void) { }
void __indirect_glSecondaryColor3bvEXT(void) { }
void __indirect_glSecondaryColor3dEXT(void) { }
void __indirect_glSecondaryColor3dvEXT(void) { }
void __indirect_glSecondaryColor3fEXT(void) { }
void __indirect_glSecondaryColor3fvEXT(void) { }
void __indirect_glSecondaryColor3iEXT(void) { }
void __indirect_glSecondaryColor3ivEXT(void) { }
void __indirect_glSecondaryColor3sEXT(void) { }
void __indirect_glSecondaryColor3svEXT(void) { }
void __indirect_glSecondaryColor3ubEXT(void) { }
void __indirect_glSecondaryColor3ubvEXT(void) { }
void __indirect_glSecondaryColor3uiEXT(void) { }
void __indirect_glSecondaryColor3uivEXT(void) { }
void __indirect_glSecondaryColor3usEXT(void) { }
void __indirect_glSecondaryColor3usvEXT(void) { }
void __indirect_glSecondaryColorPointerEXT(void) { }
void __indirect_glMultiDrawArraysEXT(void) { }
void __indirect_glMultiDrawElementsEXT(void) { }
void __indirect_glFogCoordPointerEXT(void) { }
void __indirect_glFogCoorddEXT(void) { }
void __indirect_glFogCoorddvEXT(void) { }
void __indirect_glFogCoordfEXT(void) { }
void __indirect_glFogCoordfvEXT(void) { }
void __indirect_glBlendFuncSeparateEXT(void) { }
void __indirect_glWindowPos2dMESA(void) { }
void __indirect_glWindowPos2dvMESA(void) { }
void __indirect_glWindowPos2fMESA(void) { }
void __indirect_glWindowPos2fvMESA(void) { }
void __indirect_glWindowPos2iMESA(void) { }
void __indirect_glWindowPos2ivMESA(void) { }
void __indirect_glWindowPos2sMESA(void) { }
void __indirect_glWindowPos2svMESA(void) { }
void __indirect_glWindowPos3dMESA(void) { }
void __indirect_glWindowPos3dvMESA(void) { }
void __indirect_glWindowPos3fMESA(void) { }
void __indirect_glWindowPos3fvMESA(void) { }
void __indirect_glWindowPos3iMESA(void) { }
void __indirect_glWindowPos3ivMESA(void) { }
void __indirect_glWindowPos3sMESA(void) { }
void __indirect_glWindowPos3svMESA(void) { }
void __indirect_glAreProgramsResidentNV(void) { }
void __indirect_glBindProgramNV(void) { }
void __indirect_glDeleteProgramsNV(void) { }
void __indirect_glExecuteProgramNV(void) { }
void __indirect_glGenProgramsNV(void) { }
void __indirect_glGetProgramParameterdvNV(void) { }
void __indirect_glGetProgramParameterfvNV(void) { }
void __indirect_glGetProgramStringNV(void) { }
void __indirect_glGetProgramivNV(void) { }
void __indirect_glGetTrackMatrixivNV(void) { }
void __indirect_glGetVertexAttribPointervNV(void) { }
void __indirect_glGetVertexAttribdvNV(void) { }
void __indirect_glGetVertexAttribfvNV(void) { }
void __indirect_glGetVertexAttribivNV(void) { }
void __indirect_glIsProgramNV(void) { }
void __indirect_glLoadProgramNV(void) { }
void __indirect_glProgramParameters4dvNV(void) { }
void __indirect_glProgramParameters4fvNV(void) { }
void __indirect_glRequestResidentProgramsNV(void) { }
void __indirect_glTrackMatrixNV(void) { }
void __indirect_glVertexAttrib1dNV(void) { }
void __indirect_glVertexAttrib1dvNV(void) { }
void __indirect_glVertexAttrib1fNV(void) { }
void __indirect_glVertexAttrib1fvNV(void) { }
void __indirect_glVertexAttrib1sNV(void) { }
void __indirect_glVertexAttrib1svNV(void) { }
void __indirect_glVertexAttrib2dNV(void) { }
void __indirect_glVertexAttrib2dvNV(void) { }
void __indirect_glVertexAttrib2fNV(void) { }
void __indirect_glVertexAttrib2fvNV(void) { }
void __indirect_glVertexAttrib2sNV(void) { }
void __indirect_glVertexAttrib2svNV(void) { }
void __indirect_glVertexAttrib3dNV(void) { }
void __indirect_glVertexAttrib3dvNV(void) { }
void __indirect_glVertexAttrib3fNV(void) { }
void __indirect_glVertexAttrib3fvNV(void) { }
void __indirect_glVertexAttrib3sNV(void) { }
void __indirect_glVertexAttrib3svNV(void) { }
void __indirect_glVertexAttrib4dNV(void) { }
void __indirect_glVertexAttrib4dvNV(void) { }
void __indirect_glVertexAttrib4fNV(void) { }
void __indirect_glVertexAttrib4fvNV(void) { }
void __indirect_glVertexAttrib4sNV(void) { }
void __indirect_glVertexAttrib4svNV(void) { }
void __indirect_glVertexAttrib4ubNV(void) { }
void __indirect_glVertexAttrib4ubvNV(void) { }
void __indirect_glVertexAttribPointerNV(void) { }
void __indirect_glVertexAttribs1dvNV(void) { }
void __indirect_glVertexAttribs1fvNV(void) { }
void __indirect_glVertexAttribs1svNV(void) { }
void __indirect_glVertexAttribs2dvNV(void) { }
void __indirect_glVertexAttribs2fvNV(void) { }
void __indirect_glVertexAttribs2svNV(void) { }
void __indirect_glVertexAttribs3dvNV(void) { }
void __indirect_glVertexAttribs3fvNV(void) { }
void __indirect_glVertexAttribs3svNV(void) { }
void __indirect_glVertexAttribs4dvNV(void) { }
void __indirect_glVertexAttribs4fvNV(void) { }
void __indirect_glVertexAttribs4svNV(void) { }
void __indirect_glVertexAttribs4ubvNV(void) { }
void __indirect_glPointParameteriNV(void) { }
void __indirect_glPointParameterivNV(void) { }
void __indirect_glActiveStencilFaceEXT(void) { }
void __indirect_glGetProgramNamedParameterdvNV(void) { }
void __indirect_glGetProgramNamedParameterfvNV(void) { }
void __indirect_glProgramNamedParameter4dNV(void) { }
void __indirect_glProgramNamedParameter4dvNV(void) { }
void __indirect_glProgramNamedParameter4fNV(void) { }
void __indirect_glProgramNamedParameter4fvNV(void) { }
void __indirect_glBlendEquationSeparateEXT(void) { }
void __indirect_glBindFramebufferEXT(void) { }
void __indirect_glBindRenderbufferEXT(void) { }
void __indirect_glCheckFramebufferStatusEXT(void) { }
void __indirect_glDeleteFramebuffersEXT(void) { }
void __indirect_glDeleteRenderbuffersEXT(void) { }
void __indirect_glFramebufferRenderbufferEXT(void) { }
void __indirect_glFramebufferTexture1DEXT(void) { }
void __indirect_glFramebufferTexture2DEXT(void) { }
void __indirect_glFramebufferTexture3DEXT(void) { }
void __indirect_glGenFramebuffersEXT(void) { }
void __indirect_glGenRenderbuffersEXT(void) { }
void __indirect_glGenerateMipmapEXT(void) { }
void __indirect_glGetFramebufferAttachmentParameterivEXT(void) { }
void __indirect_glGetRenderbufferParameterivEXT(void) { }
void __indirect_glIsFramebufferEXT(void) { }
void __indirect_glIsRenderbufferEXT(void) { }
void __indirect_glRenderbufferStorageEXT(void) { }
void __indirect_glBlitFramebufferEXT(void) { }
void __indirect_glFramebufferTextureLayerEXT(void) { }
}
/*@}*/

class IndirectAPI : public ::testing::Test {
public:
   virtual void SetUp();
   virtual void TearDown();

   _glapi_table *table;
};

void
IndirectAPI::SetUp()
{
   this->table = __glXNewIndirectAPI();
}

void
IndirectAPI::TearDown()
{
   free(this->table);
   this->table = NULL;
}

TEST_F(IndirectAPI, DispatchTableSizeWasQueried)
{
   dispatch_table_size_was_queried = false;

   /* __glXNewIndirectAPI determines the size of the dispatch table by calling
    * _glapi_get_dispatch_table_size.  Our stub version of this function sets
    * a flag so that we can detect that __glXNewIndirectAPI called it.
    */
   EXPECT_FALSE(dispatch_table_size_was_queried);
   _glapi_table *table = __glXNewIndirectAPI();
   EXPECT_TRUE(dispatch_table_size_was_queried);

   /* Since a second dispatch table was created, it must be freed by hand.
    */
   free(table);
}

TEST_F(IndirectAPI, NoNullEntries)
{
   const unsigned entries = _glapi_get_dispatch_table_size();

   void **dispatch = (void **) table;

   /* There should not be any NULL entries in the dispatch table.  All entires
    * should either point to a real implementation or to a no-op function.
    */
   for (unsigned i = 0; i < entries; i++) {
      EXPECT_NE(nil, dispatch[i]);
   }
}

TEST_F(IndirectAPI, ExtensionNopDispatch)
{
   /* Entries in the dispatch table past the "known" range canot possibly have
    * an indirect-rendering implementation.  Verify that all of the pointers
    * in that range point to the same function.  We can only assume that
    * function is the no-op dispatch.
    */
   void **dispatch = (void **) (table + 1);
   const void *const nop = dispatch[0];

   for (unsigned i = 0; i < EXTRA_DISPATCH; i++) {
      EXPECT_EQ(nop, dispatch[i]);
   }
}

TEST_F(IndirectAPI, OpenGL_10)
{
   EXPECT_EQ((void *) __indirect_glAccum, table->Accum);
   EXPECT_EQ((void *) __indirect_glAlphaFunc, table->AlphaFunc);
   EXPECT_EQ((void *) __indirect_glBegin, table->Begin);
   EXPECT_EQ((void *) __indirect_glBitmap, table->Bitmap);
   EXPECT_EQ((void *) __indirect_glBlendFunc, table->BlendFunc);
   EXPECT_EQ((void *) __indirect_glCallList, table->CallList);
   EXPECT_EQ((void *) __indirect_glCallLists, table->CallLists);
   EXPECT_EQ((void *) __indirect_glClear, table->Clear);
   EXPECT_EQ((void *) __indirect_glClearAccum, table->ClearAccum);
   EXPECT_EQ((void *) __indirect_glClearColor, table->ClearColor);
   EXPECT_EQ((void *) __indirect_glClearDepth, table->ClearDepth);
   EXPECT_EQ((void *) __indirect_glClearIndex, table->ClearIndex);
   EXPECT_EQ((void *) __indirect_glClearStencil, table->ClearStencil);
   EXPECT_EQ((void *) __indirect_glClipPlane, table->ClipPlane);
   EXPECT_EQ((void *) __indirect_glColor3b, table->Color3b);
   EXPECT_EQ((void *) __indirect_glColor3bv, table->Color3bv);
   EXPECT_EQ((void *) __indirect_glColor3d, table->Color3d);
   EXPECT_EQ((void *) __indirect_glColor3dv, table->Color3dv);
   EXPECT_EQ((void *) __indirect_glColor3f, table->Color3f);
   EXPECT_EQ((void *) __indirect_glColor3fv, table->Color3fv);
   EXPECT_EQ((void *) __indirect_glColor3i, table->Color3i);
   EXPECT_EQ((void *) __indirect_glColor3iv, table->Color3iv);
   EXPECT_EQ((void *) __indirect_glColor3s, table->Color3s);
   EXPECT_EQ((void *) __indirect_glColor3sv, table->Color3sv);
   EXPECT_EQ((void *) __indirect_glColor3ub, table->Color3ub);
   EXPECT_EQ((void *) __indirect_glColor3ubv, table->Color3ubv);
   EXPECT_EQ((void *) __indirect_glColor3ui, table->Color3ui);
   EXPECT_EQ((void *) __indirect_glColor3uiv, table->Color3uiv);
   EXPECT_EQ((void *) __indirect_glColor3us, table->Color3us);
   EXPECT_EQ((void *) __indirect_glColor3usv, table->Color3usv);
   EXPECT_EQ((void *) __indirect_glColor4b, table->Color4b);
   EXPECT_EQ((void *) __indirect_glColor4bv, table->Color4bv);
   EXPECT_EQ((void *) __indirect_glColor4d, table->Color4d);
   EXPECT_EQ((void *) __indirect_glColor4dv, table->Color4dv);
   EXPECT_EQ((void *) __indirect_glColor4f, table->Color4f);
   EXPECT_EQ((void *) __indirect_glColor4fv, table->Color4fv);
   EXPECT_EQ((void *) __indirect_glColor4i, table->Color4i);
   EXPECT_EQ((void *) __indirect_glColor4iv, table->Color4iv);
   EXPECT_EQ((void *) __indirect_glColor4s, table->Color4s);
   EXPECT_EQ((void *) __indirect_glColor4sv, table->Color4sv);
   EXPECT_EQ((void *) __indirect_glColor4ub, table->Color4ub);
   EXPECT_EQ((void *) __indirect_glColor4ubv, table->Color4ubv);
   EXPECT_EQ((void *) __indirect_glColor4ui, table->Color4ui);
   EXPECT_EQ((void *) __indirect_glColor4uiv, table->Color4uiv);
   EXPECT_EQ((void *) __indirect_glColor4us, table->Color4us);
   EXPECT_EQ((void *) __indirect_glColor4usv, table->Color4usv);
   EXPECT_EQ((void *) __indirect_glColorMask, table->ColorMask);
   EXPECT_EQ((void *) __indirect_glColorMaterial, table->ColorMaterial);
   EXPECT_EQ((void *) __indirect_glCopyPixels, table->CopyPixels);
   EXPECT_EQ((void *) __indirect_glCullFace, table->CullFace);
   EXPECT_EQ((void *) __indirect_glDeleteLists, table->DeleteLists);
   EXPECT_EQ((void *) __indirect_glDepthFunc, table->DepthFunc);
   EXPECT_EQ((void *) __indirect_glDepthMask, table->DepthMask);
   EXPECT_EQ((void *) __indirect_glDepthRange, table->DepthRange);
   EXPECT_EQ((void *) __indirect_glDisable, table->Disable);
   EXPECT_EQ((void *) __indirect_glDrawBuffer, table->DrawBuffer);
   EXPECT_EQ((void *) __indirect_glDrawPixels, table->DrawPixels);
   EXPECT_EQ((void *) __indirect_glEdgeFlag, table->EdgeFlag);
   EXPECT_EQ((void *) __indirect_glEdgeFlagv, table->EdgeFlagv);
   EXPECT_EQ((void *) __indirect_glEnable, table->Enable);
   EXPECT_EQ((void *) __indirect_glEnd, table->End);
   EXPECT_EQ((void *) __indirect_glEndList, table->EndList);
   EXPECT_EQ((void *) __indirect_glEvalCoord1d, table->EvalCoord1d);
   EXPECT_EQ((void *) __indirect_glEvalCoord1dv, table->EvalCoord1dv);
   EXPECT_EQ((void *) __indirect_glEvalCoord1f, table->EvalCoord1f);
   EXPECT_EQ((void *) __indirect_glEvalCoord1fv, table->EvalCoord1fv);
   EXPECT_EQ((void *) __indirect_glEvalCoord2d, table->EvalCoord2d);
   EXPECT_EQ((void *) __indirect_glEvalCoord2dv, table->EvalCoord2dv);
   EXPECT_EQ((void *) __indirect_glEvalCoord2f, table->EvalCoord2f);
   EXPECT_EQ((void *) __indirect_glEvalCoord2fv, table->EvalCoord2fv);
   EXPECT_EQ((void *) __indirect_glEvalMesh1, table->EvalMesh1);
   EXPECT_EQ((void *) __indirect_glEvalMesh2, table->EvalMesh2);
   EXPECT_EQ((void *) __indirect_glEvalPoint1, table->EvalPoint1);
   EXPECT_EQ((void *) __indirect_glEvalPoint2, table->EvalPoint2);
   EXPECT_EQ((void *) __indirect_glFeedbackBuffer, table->FeedbackBuffer);
   EXPECT_EQ((void *) __indirect_glFinish, table->Finish);
   EXPECT_EQ((void *) __indirect_glFlush, table->Flush);
   EXPECT_EQ((void *) __indirect_glFogf, table->Fogf);
   EXPECT_EQ((void *) __indirect_glFogfv, table->Fogfv);
   EXPECT_EQ((void *) __indirect_glFogi, table->Fogi);
   EXPECT_EQ((void *) __indirect_glFogiv, table->Fogiv);
   EXPECT_EQ((void *) __indirect_glFrontFace, table->FrontFace);
   EXPECT_EQ((void *) __indirect_glFrustum, table->Frustum);
   EXPECT_EQ((void *) __indirect_glGenLists, table->GenLists);
   EXPECT_EQ((void *) __indirect_glGetBooleanv, table->GetBooleanv);
   EXPECT_EQ((void *) __indirect_glGetClipPlane, table->GetClipPlane);
   EXPECT_EQ((void *) __indirect_glGetDoublev, table->GetDoublev);
   EXPECT_EQ((void *) __indirect_glGetError, table->GetError);
   EXPECT_EQ((void *) __indirect_glGetFloatv, table->GetFloatv);
   EXPECT_EQ((void *) __indirect_glGetIntegerv, table->GetIntegerv);
   EXPECT_EQ((void *) __indirect_glGetLightfv, table->GetLightfv);
   EXPECT_EQ((void *) __indirect_glGetLightiv, table->GetLightiv);
   EXPECT_EQ((void *) __indirect_glGetMapdv, table->GetMapdv);
   EXPECT_EQ((void *) __indirect_glGetMapfv, table->GetMapfv);
   EXPECT_EQ((void *) __indirect_glGetMapiv, table->GetMapiv);
   EXPECT_EQ((void *) __indirect_glGetMaterialfv, table->GetMaterialfv);
   EXPECT_EQ((void *) __indirect_glGetMaterialiv, table->GetMaterialiv);
   EXPECT_EQ((void *) __indirect_glGetPixelMapfv, table->GetPixelMapfv);
   EXPECT_EQ((void *) __indirect_glGetPixelMapuiv, table->GetPixelMapuiv);
   EXPECT_EQ((void *) __indirect_glGetPixelMapusv, table->GetPixelMapusv);
   EXPECT_EQ((void *) __indirect_glGetPolygonStipple, table->GetPolygonStipple);
   EXPECT_EQ((void *) __indirect_glGetString, table->GetString);
   EXPECT_EQ((void *) __indirect_glGetTexEnvfv, table->GetTexEnvfv);
   EXPECT_EQ((void *) __indirect_glGetTexEnviv, table->GetTexEnviv);
   EXPECT_EQ((void *) __indirect_glGetTexGendv, table->GetTexGendv);
   EXPECT_EQ((void *) __indirect_glGetTexGenfv, table->GetTexGenfv);
   EXPECT_EQ((void *) __indirect_glGetTexGeniv, table->GetTexGeniv);
   EXPECT_EQ((void *) __indirect_glGetTexImage, table->GetTexImage);
   EXPECT_EQ((void *) __indirect_glGetTexLevelParameterfv, table->GetTexLevelParameterfv);
   EXPECT_EQ((void *) __indirect_glGetTexLevelParameteriv, table->GetTexLevelParameteriv);
   EXPECT_EQ((void *) __indirect_glGetTexParameterfv, table->GetTexParameterfv);
   EXPECT_EQ((void *) __indirect_glGetTexParameteriv, table->GetTexParameteriv);
   EXPECT_EQ((void *) __indirect_glHint, table->Hint);
   EXPECT_EQ((void *) __indirect_glIndexMask, table->IndexMask);
   EXPECT_EQ((void *) __indirect_glIndexd, table->Indexd);
   EXPECT_EQ((void *) __indirect_glIndexdv, table->Indexdv);
   EXPECT_EQ((void *) __indirect_glIndexf, table->Indexf);
   EXPECT_EQ((void *) __indirect_glIndexfv, table->Indexfv);
   EXPECT_EQ((void *) __indirect_glIndexi, table->Indexi);
   EXPECT_EQ((void *) __indirect_glIndexiv, table->Indexiv);
   EXPECT_EQ((void *) __indirect_glIndexs, table->Indexs);
   EXPECT_EQ((void *) __indirect_glIndexsv, table->Indexsv);
   EXPECT_EQ((void *) __indirect_glInitNames, table->InitNames);
   EXPECT_EQ((void *) __indirect_glIsEnabled, table->IsEnabled);
   EXPECT_EQ((void *) __indirect_glIsList, table->IsList);
   EXPECT_EQ((void *) __indirect_glLightModelf, table->LightModelf);
   EXPECT_EQ((void *) __indirect_glLightModelfv, table->LightModelfv);
   EXPECT_EQ((void *) __indirect_glLightModeli, table->LightModeli);
   EXPECT_EQ((void *) __indirect_glLightModeliv, table->LightModeliv);
   EXPECT_EQ((void *) __indirect_glLightf, table->Lightf);
   EXPECT_EQ((void *) __indirect_glLightfv, table->Lightfv);
   EXPECT_EQ((void *) __indirect_glLighti, table->Lighti);
   EXPECT_EQ((void *) __indirect_glLightiv, table->Lightiv);
   EXPECT_EQ((void *) __indirect_glLineStipple, table->LineStipple);
   EXPECT_EQ((void *) __indirect_glLineWidth, table->LineWidth);
   EXPECT_EQ((void *) __indirect_glListBase, table->ListBase);
   EXPECT_EQ((void *) __indirect_glLoadIdentity, table->LoadIdentity);
   EXPECT_EQ((void *) __indirect_glLoadMatrixd, table->LoadMatrixd);
   EXPECT_EQ((void *) __indirect_glLoadMatrixf, table->LoadMatrixf);
   EXPECT_EQ((void *) __indirect_glLoadName, table->LoadName);
   EXPECT_EQ((void *) __indirect_glLogicOp, table->LogicOp);
   EXPECT_EQ((void *) __indirect_glMap1d, table->Map1d);
   EXPECT_EQ((void *) __indirect_glMap1f, table->Map1f);
   EXPECT_EQ((void *) __indirect_glMap2d, table->Map2d);
   EXPECT_EQ((void *) __indirect_glMap2f, table->Map2f);
   EXPECT_EQ((void *) __indirect_glMapGrid1d, table->MapGrid1d);
   EXPECT_EQ((void *) __indirect_glMapGrid1f, table->MapGrid1f);
   EXPECT_EQ((void *) __indirect_glMapGrid2d, table->MapGrid2d);
   EXPECT_EQ((void *) __indirect_glMapGrid2f, table->MapGrid2f);
   EXPECT_EQ((void *) __indirect_glMaterialf, table->Materialf);
   EXPECT_EQ((void *) __indirect_glMaterialfv, table->Materialfv);
   EXPECT_EQ((void *) __indirect_glMateriali, table->Materiali);
   EXPECT_EQ((void *) __indirect_glMaterialiv, table->Materialiv);
   EXPECT_EQ((void *) __indirect_glMatrixMode, table->MatrixMode);
   EXPECT_EQ((void *) __indirect_glMultMatrixd, table->MultMatrixd);
   EXPECT_EQ((void *) __indirect_glMultMatrixf, table->MultMatrixf);
   EXPECT_EQ((void *) __indirect_glNewList, table->NewList);
   EXPECT_EQ((void *) __indirect_glNormal3b, table->Normal3b);
   EXPECT_EQ((void *) __indirect_glNormal3bv, table->Normal3bv);
   EXPECT_EQ((void *) __indirect_glNormal3d, table->Normal3d);
   EXPECT_EQ((void *) __indirect_glNormal3dv, table->Normal3dv);
   EXPECT_EQ((void *) __indirect_glNormal3f, table->Normal3f);
   EXPECT_EQ((void *) __indirect_glNormal3fv, table->Normal3fv);
   EXPECT_EQ((void *) __indirect_glNormal3i, table->Normal3i);
   EXPECT_EQ((void *) __indirect_glNormal3iv, table->Normal3iv);
   EXPECT_EQ((void *) __indirect_glNormal3s, table->Normal3s);
   EXPECT_EQ((void *) __indirect_glNormal3sv, table->Normal3sv);
   EXPECT_EQ((void *) __indirect_glOrtho, table->Ortho);
   EXPECT_EQ((void *) __indirect_glPassThrough, table->PassThrough);
   EXPECT_EQ((void *) __indirect_glPixelMapfv, table->PixelMapfv);
   EXPECT_EQ((void *) __indirect_glPixelMapuiv, table->PixelMapuiv);
   EXPECT_EQ((void *) __indirect_glPixelMapusv, table->PixelMapusv);
   EXPECT_EQ((void *) __indirect_glPixelStoref, table->PixelStoref);
   EXPECT_EQ((void *) __indirect_glPixelStorei, table->PixelStorei);
   EXPECT_EQ((void *) __indirect_glPixelTransferf, table->PixelTransferf);
   EXPECT_EQ((void *) __indirect_glPixelTransferi, table->PixelTransferi);
   EXPECT_EQ((void *) __indirect_glPixelZoom, table->PixelZoom);
   EXPECT_EQ((void *) __indirect_glPointSize, table->PointSize);
   EXPECT_EQ((void *) __indirect_glPolygonMode, table->PolygonMode);
   EXPECT_EQ((void *) __indirect_glPolygonStipple, table->PolygonStipple);
   EXPECT_EQ((void *) __indirect_glPopAttrib, table->PopAttrib);
   EXPECT_EQ((void *) __indirect_glPopMatrix, table->PopMatrix);
   EXPECT_EQ((void *) __indirect_glPopName, table->PopName);
   EXPECT_EQ((void *) __indirect_glPushAttrib, table->PushAttrib);
   EXPECT_EQ((void *) __indirect_glPushMatrix, table->PushMatrix);
   EXPECT_EQ((void *) __indirect_glPushName, table->PushName);
   EXPECT_EQ((void *) __indirect_glRasterPos2d, table->RasterPos2d);
   EXPECT_EQ((void *) __indirect_glRasterPos2dv, table->RasterPos2dv);
   EXPECT_EQ((void *) __indirect_glRasterPos2f, table->RasterPos2f);
   EXPECT_EQ((void *) __indirect_glRasterPos2fv, table->RasterPos2fv);
   EXPECT_EQ((void *) __indirect_glRasterPos2i, table->RasterPos2i);
   EXPECT_EQ((void *) __indirect_glRasterPos2iv, table->RasterPos2iv);
   EXPECT_EQ((void *) __indirect_glRasterPos2s, table->RasterPos2s);
   EXPECT_EQ((void *) __indirect_glRasterPos2sv, table->RasterPos2sv);
   EXPECT_EQ((void *) __indirect_glRasterPos3d, table->RasterPos3d);
   EXPECT_EQ((void *) __indirect_glRasterPos3dv, table->RasterPos3dv);
   EXPECT_EQ((void *) __indirect_glRasterPos3f, table->RasterPos3f);
   EXPECT_EQ((void *) __indirect_glRasterPos3fv, table->RasterPos3fv);
   EXPECT_EQ((void *) __indirect_glRasterPos3i, table->RasterPos3i);
   EXPECT_EQ((void *) __indirect_glRasterPos3iv, table->RasterPos3iv);
   EXPECT_EQ((void *) __indirect_glRasterPos3s, table->RasterPos3s);
   EXPECT_EQ((void *) __indirect_glRasterPos3sv, table->RasterPos3sv);
   EXPECT_EQ((void *) __indirect_glRasterPos4d, table->RasterPos4d);
   EXPECT_EQ((void *) __indirect_glRasterPos4dv, table->RasterPos4dv);
   EXPECT_EQ((void *) __indirect_glRasterPos4f, table->RasterPos4f);
   EXPECT_EQ((void *) __indirect_glRasterPos4fv, table->RasterPos4fv);
   EXPECT_EQ((void *) __indirect_glRasterPos4i, table->RasterPos4i);
   EXPECT_EQ((void *) __indirect_glRasterPos4iv, table->RasterPos4iv);
   EXPECT_EQ((void *) __indirect_glRasterPos4s, table->RasterPos4s);
   EXPECT_EQ((void *) __indirect_glRasterPos4sv, table->RasterPos4sv);
   EXPECT_EQ((void *) __indirect_glReadBuffer, table->ReadBuffer);
   EXPECT_EQ((void *) __indirect_glReadPixels, table->ReadPixels);
   EXPECT_EQ((void *) __indirect_glRectd, table->Rectd);
   EXPECT_EQ((void *) __indirect_glRectdv, table->Rectdv);
   EXPECT_EQ((void *) __indirect_glRectf, table->Rectf);
   EXPECT_EQ((void *) __indirect_glRectfv, table->Rectfv);
   EXPECT_EQ((void *) __indirect_glRecti, table->Recti);
   EXPECT_EQ((void *) __indirect_glRectiv, table->Rectiv);
   EXPECT_EQ((void *) __indirect_glRects, table->Rects);
   EXPECT_EQ((void *) __indirect_glRectsv, table->Rectsv);
   EXPECT_EQ((void *) __indirect_glRenderMode, table->RenderMode);
   EXPECT_EQ((void *) __indirect_glRotated, table->Rotated);
   EXPECT_EQ((void *) __indirect_glRotatef, table->Rotatef);
   EXPECT_EQ((void *) __indirect_glScaled, table->Scaled);
   EXPECT_EQ((void *) __indirect_glScalef, table->Scalef);
   EXPECT_EQ((void *) __indirect_glScissor, table->Scissor);
   EXPECT_EQ((void *) __indirect_glSelectBuffer, table->SelectBuffer);
   EXPECT_EQ((void *) __indirect_glShadeModel, table->ShadeModel);
   EXPECT_EQ((void *) __indirect_glStencilFunc, table->StencilFunc);
   EXPECT_EQ((void *) __indirect_glStencilMask, table->StencilMask);
   EXPECT_EQ((void *) __indirect_glStencilOp, table->StencilOp);
   EXPECT_EQ((void *) __indirect_glTexCoord1d, table->TexCoord1d);
   EXPECT_EQ((void *) __indirect_glTexCoord1dv, table->TexCoord1dv);
   EXPECT_EQ((void *) __indirect_glTexCoord1f, table->TexCoord1f);
   EXPECT_EQ((void *) __indirect_glTexCoord1fv, table->TexCoord1fv);
   EXPECT_EQ((void *) __indirect_glTexCoord1i, table->TexCoord1i);
   EXPECT_EQ((void *) __indirect_glTexCoord1iv, table->TexCoord1iv);
   EXPECT_EQ((void *) __indirect_glTexCoord1s, table->TexCoord1s);
   EXPECT_EQ((void *) __indirect_glTexCoord1sv, table->TexCoord1sv);
   EXPECT_EQ((void *) __indirect_glTexCoord2d, table->TexCoord2d);
   EXPECT_EQ((void *) __indirect_glTexCoord2dv, table->TexCoord2dv);
   EXPECT_EQ((void *) __indirect_glTexCoord2f, table->TexCoord2f);
   EXPECT_EQ((void *) __indirect_glTexCoord2fv, table->TexCoord2fv);
   EXPECT_EQ((void *) __indirect_glTexCoord2i, table->TexCoord2i);
   EXPECT_EQ((void *) __indirect_glTexCoord2iv, table->TexCoord2iv);
   EXPECT_EQ((void *) __indirect_glTexCoord2s, table->TexCoord2s);
   EXPECT_EQ((void *) __indirect_glTexCoord2sv, table->TexCoord2sv);
   EXPECT_EQ((void *) __indirect_glTexCoord3d, table->TexCoord3d);
   EXPECT_EQ((void *) __indirect_glTexCoord3dv, table->TexCoord3dv);
   EXPECT_EQ((void *) __indirect_glTexCoord3f, table->TexCoord3f);
   EXPECT_EQ((void *) __indirect_glTexCoord3fv, table->TexCoord3fv);
   EXPECT_EQ((void *) __indirect_glTexCoord3i, table->TexCoord3i);
   EXPECT_EQ((void *) __indirect_glTexCoord3iv, table->TexCoord3iv);
   EXPECT_EQ((void *) __indirect_glTexCoord3s, table->TexCoord3s);
   EXPECT_EQ((void *) __indirect_glTexCoord3sv, table->TexCoord3sv);
   EXPECT_EQ((void *) __indirect_glTexCoord4d, table->TexCoord4d);
   EXPECT_EQ((void *) __indirect_glTexCoord4dv, table->TexCoord4dv);
   EXPECT_EQ((void *) __indirect_glTexCoord4f, table->TexCoord4f);
   EXPECT_EQ((void *) __indirect_glTexCoord4fv, table->TexCoord4fv);
   EXPECT_EQ((void *) __indirect_glTexCoord4i, table->TexCoord4i);
   EXPECT_EQ((void *) __indirect_glTexCoord4iv, table->TexCoord4iv);
   EXPECT_EQ((void *) __indirect_glTexCoord4s, table->TexCoord4s);
   EXPECT_EQ((void *) __indirect_glTexCoord4sv, table->TexCoord4sv);
   EXPECT_EQ((void *) __indirect_glTexEnvf, table->TexEnvf);
   EXPECT_EQ((void *) __indirect_glTexEnvfv, table->TexEnvfv);
   EXPECT_EQ((void *) __indirect_glTexEnvi, table->TexEnvi);
   EXPECT_EQ((void *) __indirect_glTexEnviv, table->TexEnviv);
   EXPECT_EQ((void *) __indirect_glTexGend, table->TexGend);
   EXPECT_EQ((void *) __indirect_glTexGendv, table->TexGendv);
   EXPECT_EQ((void *) __indirect_glTexGenf, table->TexGenf);
   EXPECT_EQ((void *) __indirect_glTexGenfv, table->TexGenfv);
   EXPECT_EQ((void *) __indirect_glTexGeni, table->TexGeni);
   EXPECT_EQ((void *) __indirect_glTexGeniv, table->TexGeniv);
   EXPECT_EQ((void *) __indirect_glTexImage1D, table->TexImage1D);
   EXPECT_EQ((void *) __indirect_glTexImage2D, table->TexImage2D);
   EXPECT_EQ((void *) __indirect_glTexParameterf, table->TexParameterf);
   EXPECT_EQ((void *) __indirect_glTexParameterfv, table->TexParameterfv);
   EXPECT_EQ((void *) __indirect_glTexParameteri, table->TexParameteri);
   EXPECT_EQ((void *) __indirect_glTexParameteriv, table->TexParameteriv);
   EXPECT_EQ((void *) __indirect_glTranslated, table->Translated);
   EXPECT_EQ((void *) __indirect_glTranslatef, table->Translatef);
   EXPECT_EQ((void *) __indirect_glVertex2d, table->Vertex2d);
   EXPECT_EQ((void *) __indirect_glVertex2dv, table->Vertex2dv);
   EXPECT_EQ((void *) __indirect_glVertex2f, table->Vertex2f);
   EXPECT_EQ((void *) __indirect_glVertex2fv, table->Vertex2fv);
   EXPECT_EQ((void *) __indirect_glVertex2i, table->Vertex2i);
   EXPECT_EQ((void *) __indirect_glVertex2iv, table->Vertex2iv);
   EXPECT_EQ((void *) __indirect_glVertex2s, table->Vertex2s);
   EXPECT_EQ((void *) __indirect_glVertex2sv, table->Vertex2sv);
   EXPECT_EQ((void *) __indirect_glVertex3d, table->Vertex3d);
   EXPECT_EQ((void *) __indirect_glVertex3dv, table->Vertex3dv);
   EXPECT_EQ((void *) __indirect_glVertex3f, table->Vertex3f);
   EXPECT_EQ((void *) __indirect_glVertex3fv, table->Vertex3fv);
   EXPECT_EQ((void *) __indirect_glVertex3i, table->Vertex3i);
   EXPECT_EQ((void *) __indirect_glVertex3iv, table->Vertex3iv);
   EXPECT_EQ((void *) __indirect_glVertex3s, table->Vertex3s);
   EXPECT_EQ((void *) __indirect_glVertex3sv, table->Vertex3sv);
   EXPECT_EQ((void *) __indirect_glVertex4d, table->Vertex4d);
   EXPECT_EQ((void *) __indirect_glVertex4dv, table->Vertex4dv);
   EXPECT_EQ((void *) __indirect_glVertex4f, table->Vertex4f);
   EXPECT_EQ((void *) __indirect_glVertex4fv, table->Vertex4fv);
   EXPECT_EQ((void *) __indirect_glVertex4i, table->Vertex4i);
   EXPECT_EQ((void *) __indirect_glVertex4iv, table->Vertex4iv);
   EXPECT_EQ((void *) __indirect_glVertex4s, table->Vertex4s);
   EXPECT_EQ((void *) __indirect_glVertex4sv, table->Vertex4sv);
   EXPECT_EQ((void *) __indirect_glViewport, table->Viewport);
}

TEST_F(IndirectAPI, OpenGL_11)
{
   EXPECT_EQ((void *) __indirect_glAreTexturesResident, table->AreTexturesResident);
   EXPECT_EQ((void *) __indirect_glArrayElement, table->ArrayElement);
   EXPECT_EQ((void *) __indirect_glBindTexture, table->BindTexture);
   EXPECT_EQ((void *) __indirect_glColorPointer, table->ColorPointer);
   EXPECT_EQ((void *) __indirect_glCopyTexImage1D, table->CopyTexImage1D);
   EXPECT_EQ((void *) __indirect_glCopyTexImage2D, table->CopyTexImage2D);
   EXPECT_EQ((void *) __indirect_glCopyTexSubImage1D, table->CopyTexSubImage1D);
   EXPECT_EQ((void *) __indirect_glCopyTexSubImage2D, table->CopyTexSubImage2D);
   EXPECT_EQ((void *) __indirect_glDeleteTextures, table->DeleteTextures);
   EXPECT_EQ((void *) __indirect_glDisableClientState, table->DisableClientState);
   EXPECT_EQ((void *) __indirect_glDrawArrays, table->DrawArrays);
   EXPECT_EQ((void *) __indirect_glDrawElements, table->DrawElements);
   EXPECT_EQ((void *) __indirect_glEdgeFlagPointer, table->EdgeFlagPointer);
   EXPECT_EQ((void *) __indirect_glEnableClientState, table->EnableClientState);
   EXPECT_EQ((void *) __indirect_glGenTextures, table->GenTextures);
   EXPECT_EQ((void *) __indirect_glGetPointerv, table->GetPointerv);
   EXPECT_EQ((void *) __indirect_glIndexPointer, table->IndexPointer);
   EXPECT_EQ((void *) __indirect_glIndexub, table->Indexub);
   EXPECT_EQ((void *) __indirect_glIndexubv, table->Indexubv);
   EXPECT_EQ((void *) __indirect_glInterleavedArrays, table->InterleavedArrays);
   EXPECT_EQ((void *) __indirect_glIsTexture, table->IsTexture);
   EXPECT_EQ((void *) __indirect_glNormalPointer, table->NormalPointer);
   EXPECT_EQ((void *) __indirect_glPolygonOffset, table->PolygonOffset);
   EXPECT_EQ((void *) __indirect_glPopClientAttrib, table->PopClientAttrib);
   EXPECT_EQ((void *) __indirect_glPrioritizeTextures, table->PrioritizeTextures);
   EXPECT_EQ((void *) __indirect_glPushClientAttrib, table->PushClientAttrib);
   EXPECT_EQ((void *) __indirect_glTexCoordPointer, table->TexCoordPointer);
   EXPECT_EQ((void *) __indirect_glTexSubImage1D, table->TexSubImage1D);
   EXPECT_EQ((void *) __indirect_glTexSubImage2D, table->TexSubImage2D);
   EXPECT_EQ((void *) __indirect_glVertexPointer, table->VertexPointer);
}

TEST_F(IndirectAPI, OpenGL_12)
{
   EXPECT_EQ((void *) __indirect_glBlendColor, table->BlendColor);
   EXPECT_EQ((void *) __indirect_glBlendEquation, table->BlendEquation);
   EXPECT_EQ((void *) __indirect_glColorSubTable, table->ColorSubTable);
   EXPECT_EQ((void *) __indirect_glColorTable, table->ColorTable);
   EXPECT_EQ((void *) __indirect_glColorTableParameterfv, table->ColorTableParameterfv);
   EXPECT_EQ((void *) __indirect_glColorTableParameteriv, table->ColorTableParameteriv);
   EXPECT_EQ((void *) __indirect_glConvolutionFilter1D, table->ConvolutionFilter1D);
   EXPECT_EQ((void *) __indirect_glConvolutionFilter2D, table->ConvolutionFilter2D);
   EXPECT_EQ((void *) __indirect_glConvolutionParameterf, table->ConvolutionParameterf);
   EXPECT_EQ((void *) __indirect_glConvolutionParameterfv, table->ConvolutionParameterfv);
   EXPECT_EQ((void *) __indirect_glConvolutionParameteri, table->ConvolutionParameteri);
   EXPECT_EQ((void *) __indirect_glConvolutionParameteriv, table->ConvolutionParameteriv);
   EXPECT_EQ((void *) __indirect_glCopyColorSubTable, table->CopyColorSubTable);
   EXPECT_EQ((void *) __indirect_glCopyColorTable, table->CopyColorTable);
   EXPECT_EQ((void *) __indirect_glCopyConvolutionFilter1D, table->CopyConvolutionFilter1D);
   EXPECT_EQ((void *) __indirect_glCopyConvolutionFilter2D, table->CopyConvolutionFilter2D);
   EXPECT_EQ((void *) __indirect_glCopyTexSubImage3D, table->CopyTexSubImage3D);
   EXPECT_EQ((void *) __indirect_glDrawRangeElements, table->DrawRangeElements);
   EXPECT_EQ((void *) __indirect_glGetColorTable, table->GetColorTable);
   EXPECT_EQ((void *) __indirect_glGetColorTableParameterfv, table->GetColorTableParameterfv);
   EXPECT_EQ((void *) __indirect_glGetColorTableParameteriv, table->GetColorTableParameteriv);
   EXPECT_EQ((void *) __indirect_glGetConvolutionFilter, table->GetConvolutionFilter);
   EXPECT_EQ((void *) __indirect_glGetConvolutionParameterfv, table->GetConvolutionParameterfv);
   EXPECT_EQ((void *) __indirect_glGetConvolutionParameteriv, table->GetConvolutionParameteriv);
   EXPECT_EQ((void *) __indirect_glGetHistogram, table->GetHistogram);
   EXPECT_EQ((void *) __indirect_glGetHistogramParameterfv, table->GetHistogramParameterfv);
   EXPECT_EQ((void *) __indirect_glGetHistogramParameteriv, table->GetHistogramParameteriv);
   EXPECT_EQ((void *) __indirect_glGetMinmax, table->GetMinmax);
   EXPECT_EQ((void *) __indirect_glGetMinmaxParameterfv, table->GetMinmaxParameterfv);
   EXPECT_EQ((void *) __indirect_glGetMinmaxParameteriv, table->GetMinmaxParameteriv);
   EXPECT_EQ((void *) __indirect_glGetSeparableFilter, table->GetSeparableFilter);
   EXPECT_EQ((void *) __indirect_glHistogram, table->Histogram);
   EXPECT_EQ((void *) __indirect_glMinmax, table->Minmax);
   EXPECT_EQ((void *) __indirect_glResetHistogram, table->ResetHistogram);
   EXPECT_EQ((void *) __indirect_glResetMinmax, table->ResetMinmax);
   EXPECT_EQ((void *) __indirect_glSeparableFilter2D, table->SeparableFilter2D);
   EXPECT_EQ((void *) __indirect_glTexImage3D, table->TexImage3D);
   EXPECT_EQ((void *) __indirect_glTexSubImage3D, table->TexSubImage3D);
}

TEST_F(IndirectAPI, OpenGL_20_is_nop)
{
   const void *const nop = ((void **)(table + 1))[0];

   EXPECT_EQ(nop, table->AttachShader);
   EXPECT_EQ(nop, table->CreateProgram);
   EXPECT_EQ(nop, table->CreateShader);
   EXPECT_EQ(nop, table->DeleteProgram);
   EXPECT_EQ(nop, table->DeleteShader);
   EXPECT_EQ(nop, table->DetachShader);
   EXPECT_EQ(nop, table->GetAttachedShaders);
   EXPECT_EQ(nop, table->GetProgramInfoLog);
   EXPECT_EQ(nop, table->GetProgramiv);
   EXPECT_EQ(nop, table->GetShaderInfoLog);
   EXPECT_EQ(nop, table->GetShaderiv);
   EXPECT_EQ(nop, table->IsProgram);
   EXPECT_EQ(nop, table->IsShader);
   EXPECT_EQ(nop, table->StencilFuncSeparate);
   EXPECT_EQ(nop, table->StencilMaskSeparate);
   EXPECT_EQ(nop, table->StencilOpSeparate);
}

TEST_F(IndirectAPI, ARB_multitexture)
{
   EXPECT_EQ((void *) __indirect_glActiveTextureARB, table->ActiveTextureARB);
   EXPECT_EQ((void *) __indirect_glClientActiveTextureARB, table->ClientActiveTextureARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1dARB, table->MultiTexCoord1dARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1dvARB, table->MultiTexCoord1dvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1fARB, table->MultiTexCoord1fARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1fvARB, table->MultiTexCoord1fvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1iARB, table->MultiTexCoord1iARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1ivARB, table->MultiTexCoord1ivARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1sARB, table->MultiTexCoord1sARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord1svARB, table->MultiTexCoord1svARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2dARB, table->MultiTexCoord2dARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2dvARB, table->MultiTexCoord2dvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2fARB, table->MultiTexCoord2fARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2fvARB, table->MultiTexCoord2fvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2iARB, table->MultiTexCoord2iARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2ivARB, table->MultiTexCoord2ivARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2sARB, table->MultiTexCoord2sARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord2svARB, table->MultiTexCoord2svARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3dARB, table->MultiTexCoord3dARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3dvARB, table->MultiTexCoord3dvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3fARB, table->MultiTexCoord3fARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3fvARB, table->MultiTexCoord3fvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3iARB, table->MultiTexCoord3iARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3ivARB, table->MultiTexCoord3ivARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3sARB, table->MultiTexCoord3sARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord3svARB, table->MultiTexCoord3svARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4dARB, table->MultiTexCoord4dARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4dvARB, table->MultiTexCoord4dvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4fARB, table->MultiTexCoord4fARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4fvARB, table->MultiTexCoord4fvARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4iARB, table->MultiTexCoord4iARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4ivARB, table->MultiTexCoord4ivARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4sARB, table->MultiTexCoord4sARB);
   EXPECT_EQ((void *) __indirect_glMultiTexCoord4svARB, table->MultiTexCoord4svARB);
}

TEST_F(IndirectAPI, ARB_transpose_matrix)
{
   EXPECT_EQ((void *) __indirect_glLoadTransposeMatrixdARB, table->LoadTransposeMatrixdARB);
   EXPECT_EQ((void *) __indirect_glLoadTransposeMatrixfARB, table->LoadTransposeMatrixfARB);
   EXPECT_EQ((void *) __indirect_glMultTransposeMatrixdARB, table->MultTransposeMatrixdARB);
   EXPECT_EQ((void *) __indirect_glMultTransposeMatrixfARB, table->MultTransposeMatrixfARB);
}

TEST_F(IndirectAPI, ARB_multisample)
{
   EXPECT_EQ((void *) __indirect_glSampleCoverageARB, table->SampleCoverageARB);
}

TEST_F(IndirectAPI, ARB_texture_compression)
{
   EXPECT_EQ((void *) __indirect_glCompressedTexImage1DARB, table->CompressedTexImage1DARB);
   EXPECT_EQ((void *) __indirect_glCompressedTexImage2DARB, table->CompressedTexImage2DARB);
   EXPECT_EQ((void *) __indirect_glCompressedTexImage3DARB, table->CompressedTexImage3DARB);
   EXPECT_EQ((void *) __indirect_glCompressedTexSubImage1DARB, table->CompressedTexSubImage1DARB);
   EXPECT_EQ((void *) __indirect_glCompressedTexSubImage2DARB, table->CompressedTexSubImage2DARB);
   EXPECT_EQ((void *) __indirect_glCompressedTexSubImage3DARB, table->CompressedTexSubImage3DARB);
   EXPECT_EQ((void *) __indirect_glGetCompressedTexImageARB, table->GetCompressedTexImageARB);
}

TEST_F(IndirectAPI, ARB_vertex_program)
{
   EXPECT_EQ((void *) __indirect_glDisableVertexAttribArrayARB, table->DisableVertexAttribArrayARB);
   EXPECT_EQ((void *) __indirect_glEnableVertexAttribArrayARB, table->EnableVertexAttribArrayARB);
   EXPECT_EQ((void *) __indirect_glGetProgramEnvParameterdvARB, table->GetProgramEnvParameterdvARB);
   EXPECT_EQ((void *) __indirect_glGetProgramEnvParameterfvARB, table->GetProgramEnvParameterfvARB);
   EXPECT_EQ((void *) __indirect_glGetProgramLocalParameterdvARB, table->GetProgramLocalParameterdvARB);
   EXPECT_EQ((void *) __indirect_glGetProgramLocalParameterfvARB, table->GetProgramLocalParameterfvARB);
   EXPECT_EQ((void *) __indirect_glGetProgramStringARB, table->GetProgramStringARB);
   EXPECT_EQ((void *) __indirect_glGetProgramivARB, table->GetProgramivARB);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribdvARB, table->GetVertexAttribdvARB);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribfvARB, table->GetVertexAttribfvARB);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribivARB, table->GetVertexAttribivARB);
   EXPECT_EQ((void *) __indirect_glProgramEnvParameter4dARB, table->ProgramEnvParameter4dARB);
   EXPECT_EQ((void *) __indirect_glProgramEnvParameter4dvARB, table->ProgramEnvParameter4dvARB);
   EXPECT_EQ((void *) __indirect_glProgramEnvParameter4fARB, table->ProgramEnvParameter4fARB);
   EXPECT_EQ((void *) __indirect_glProgramEnvParameter4fvARB, table->ProgramEnvParameter4fvARB);
   EXPECT_EQ((void *) __indirect_glProgramLocalParameter4dARB, table->ProgramLocalParameter4dARB);
   EXPECT_EQ((void *) __indirect_glProgramLocalParameter4dvARB, table->ProgramLocalParameter4dvARB);
   EXPECT_EQ((void *) __indirect_glProgramLocalParameter4fARB, table->ProgramLocalParameter4fARB);
   EXPECT_EQ((void *) __indirect_glProgramLocalParameter4fvARB, table->ProgramLocalParameter4fvARB);
   EXPECT_EQ((void *) __indirect_glProgramStringARB, table->ProgramStringARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1dARB, table->VertexAttrib1dARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1dvARB, table->VertexAttrib1dvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1fARB, table->VertexAttrib1fARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1fvARB, table->VertexAttrib1fvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1sARB, table->VertexAttrib1sARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1svARB, table->VertexAttrib1svARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2dARB, table->VertexAttrib2dARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2dvARB, table->VertexAttrib2dvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2fARB, table->VertexAttrib2fARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2fvARB, table->VertexAttrib2fvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2sARB, table->VertexAttrib2sARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2svARB, table->VertexAttrib2svARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3dARB, table->VertexAttrib3dARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3dvARB, table->VertexAttrib3dvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3fARB, table->VertexAttrib3fARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3fvARB, table->VertexAttrib3fvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3sARB, table->VertexAttrib3sARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3svARB, table->VertexAttrib3svARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NbvARB, table->VertexAttrib4NbvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NivARB, table->VertexAttrib4NivARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NsvARB, table->VertexAttrib4NsvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NubARB, table->VertexAttrib4NubARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NubvARB, table->VertexAttrib4NubvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NuivARB, table->VertexAttrib4NuivARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4NusvARB, table->VertexAttrib4NusvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4bvARB, table->VertexAttrib4bvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4dARB, table->VertexAttrib4dARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4dvARB, table->VertexAttrib4dvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4fARB, table->VertexAttrib4fARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4fvARB, table->VertexAttrib4fvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4ivARB, table->VertexAttrib4ivARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4sARB, table->VertexAttrib4sARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4svARB, table->VertexAttrib4svARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4ubvARB, table->VertexAttrib4ubvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4uivARB, table->VertexAttrib4uivARB);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4usvARB, table->VertexAttrib4usvARB);
   EXPECT_EQ((void *) __indirect_glVertexAttribPointerARB, table->VertexAttribPointerARB);
}

TEST_F(IndirectAPI, ARB_occlusion_query)
{
   EXPECT_EQ((void *) __indirect_glBeginQueryARB, table->BeginQueryARB);
   EXPECT_EQ((void *) __indirect_glDeleteQueriesARB, table->DeleteQueriesARB);
   EXPECT_EQ((void *) __indirect_glEndQueryARB, table->EndQueryARB);
   EXPECT_EQ((void *) __indirect_glGenQueriesARB, table->GenQueriesARB);
   EXPECT_EQ((void *) __indirect_glGetQueryObjectivARB, table->GetQueryObjectivARB);
   EXPECT_EQ((void *) __indirect_glGetQueryObjectuivARB, table->GetQueryObjectuivARB);
   EXPECT_EQ((void *) __indirect_glGetQueryivARB, table->GetQueryivARB);
   EXPECT_EQ((void *) __indirect_glIsQueryARB, table->IsQueryARB);
}

TEST_F(IndirectAPI, ARB_draw_buffers)
{
   EXPECT_EQ((void *) __indirect_glDrawBuffersARB, table->DrawBuffersARB);
}

TEST_F(IndirectAPI, ARB_color_buffer_float)
{
   EXPECT_EQ((void *) __indirect_glClampColorARB, table->ClampColorARB);
}

TEST_F(IndirectAPI, ARB_framebuffer_object)
{
   EXPECT_EQ((void *) __indirect_glRenderbufferStorageMultisample, table->RenderbufferStorageMultisample);
}

TEST_F(IndirectAPI, SGIS_multisample)
{
   EXPECT_EQ((void *) __indirect_glSampleMaskSGIS, table->SampleMaskSGIS);
   EXPECT_EQ((void *) __indirect_glSamplePatternSGIS, table->SamplePatternSGIS);
}

TEST_F(IndirectAPI, EXT_vertex_array)
{
   EXPECT_EQ((void *) __indirect_glColorPointerEXT, table->ColorPointerEXT);
   EXPECT_EQ((void *) __indirect_glEdgeFlagPointerEXT, table->EdgeFlagPointerEXT);
   EXPECT_EQ((void *) __indirect_glIndexPointerEXT, table->IndexPointerEXT);
   EXPECT_EQ((void *) __indirect_glNormalPointerEXT, table->NormalPointerEXT);
   EXPECT_EQ((void *) __indirect_glTexCoordPointerEXT, table->TexCoordPointerEXT);
   EXPECT_EQ((void *) __indirect_glVertexPointerEXT, table->VertexPointerEXT);
}

TEST_F(IndirectAPI, EXT_point_parameters)
{
   EXPECT_EQ((void *) __indirect_glPointParameterfEXT, table->PointParameterfEXT);
   EXPECT_EQ((void *) __indirect_glPointParameterfvEXT, table->PointParameterfvEXT);
}

TEST_F(IndirectAPI, EXT_secondary_color)
{
   EXPECT_EQ((void *) __indirect_glSecondaryColor3bEXT, table->SecondaryColor3bEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3bvEXT, table->SecondaryColor3bvEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3dEXT, table->SecondaryColor3dEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3dvEXT, table->SecondaryColor3dvEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3fEXT, table->SecondaryColor3fEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3fvEXT, table->SecondaryColor3fvEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3iEXT, table->SecondaryColor3iEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3ivEXT, table->SecondaryColor3ivEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3sEXT, table->SecondaryColor3sEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3svEXT, table->SecondaryColor3svEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3ubEXT, table->SecondaryColor3ubEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3ubvEXT, table->SecondaryColor3ubvEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3uiEXT, table->SecondaryColor3uiEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3uivEXT, table->SecondaryColor3uivEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3usEXT, table->SecondaryColor3usEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColor3usvEXT, table->SecondaryColor3usvEXT);
   EXPECT_EQ((void *) __indirect_glSecondaryColorPointerEXT, table->SecondaryColorPointerEXT);
}

TEST_F(IndirectAPI, EXT_multi_draw_arrays)
{
   EXPECT_EQ((void *) __indirect_glMultiDrawArraysEXT, table->MultiDrawArraysEXT);
   EXPECT_EQ((void *) __indirect_glMultiDrawElementsEXT, table->MultiDrawElementsEXT);
}

TEST_F(IndirectAPI, EXT_fog_coord)
{
   EXPECT_EQ((void *) __indirect_glFogCoordPointerEXT, table->FogCoordPointerEXT);
   EXPECT_EQ((void *) __indirect_glFogCoorddEXT, table->FogCoorddEXT);
   EXPECT_EQ((void *) __indirect_glFogCoorddvEXT, table->FogCoorddvEXT);
   EXPECT_EQ((void *) __indirect_glFogCoordfEXT, table->FogCoordfEXT);
   EXPECT_EQ((void *) __indirect_glFogCoordfvEXT, table->FogCoordfvEXT);
}

TEST_F(IndirectAPI, EXT_blend_func_separate)
{
   EXPECT_EQ((void *) __indirect_glBlendFuncSeparateEXT, table->BlendFuncSeparateEXT);
}

TEST_F(IndirectAPI, MESA_window_pos)
{
   EXPECT_EQ((void *) __indirect_glWindowPos2dMESA, table->WindowPos2dMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2dvMESA, table->WindowPos2dvMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2fMESA, table->WindowPos2fMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2fvMESA, table->WindowPos2fvMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2iMESA, table->WindowPos2iMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2ivMESA, table->WindowPos2ivMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2sMESA, table->WindowPos2sMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos2svMESA, table->WindowPos2svMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3dMESA, table->WindowPos3dMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3dvMESA, table->WindowPos3dvMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3fMESA, table->WindowPos3fMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3fvMESA, table->WindowPos3fvMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3iMESA, table->WindowPos3iMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3ivMESA, table->WindowPos3ivMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3sMESA, table->WindowPos3sMESA);
   EXPECT_EQ((void *) __indirect_glWindowPos3svMESA, table->WindowPos3svMESA);
}

TEST_F(IndirectAPI, NV_vertex_program)
{
   EXPECT_EQ((void *) __indirect_glAreProgramsResidentNV, table->AreProgramsResidentNV);
   EXPECT_EQ((void *) __indirect_glBindProgramNV, table->BindProgramNV);
   EXPECT_EQ((void *) __indirect_glDeleteProgramsNV, table->DeleteProgramsNV);
   EXPECT_EQ((void *) __indirect_glExecuteProgramNV, table->ExecuteProgramNV);
   EXPECT_EQ((void *) __indirect_glGenProgramsNV, table->GenProgramsNV);
   EXPECT_EQ((void *) __indirect_glGetProgramParameterdvNV, table->GetProgramParameterdvNV);
   EXPECT_EQ((void *) __indirect_glGetProgramParameterfvNV, table->GetProgramParameterfvNV);
   EXPECT_EQ((void *) __indirect_glGetProgramStringNV, table->GetProgramStringNV);
   EXPECT_EQ((void *) __indirect_glGetProgramivNV, table->GetProgramivNV);
   EXPECT_EQ((void *) __indirect_glGetTrackMatrixivNV, table->GetTrackMatrixivNV);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribPointervNV, table->GetVertexAttribPointervNV);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribdvNV, table->GetVertexAttribdvNV);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribfvNV, table->GetVertexAttribfvNV);
   EXPECT_EQ((void *) __indirect_glGetVertexAttribivNV, table->GetVertexAttribivNV);
   EXPECT_EQ((void *) __indirect_glIsProgramNV, table->IsProgramNV);
   EXPECT_EQ((void *) __indirect_glLoadProgramNV, table->LoadProgramNV);
   EXPECT_EQ((void *) __indirect_glProgramParameters4dvNV, table->ProgramParameters4dvNV);
   EXPECT_EQ((void *) __indirect_glProgramParameters4fvNV, table->ProgramParameters4fvNV);
   EXPECT_EQ((void *) __indirect_glRequestResidentProgramsNV, table->RequestResidentProgramsNV);
   EXPECT_EQ((void *) __indirect_glTrackMatrixNV, table->TrackMatrixNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1dNV, table->VertexAttrib1dNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1dvNV, table->VertexAttrib1dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1fNV, table->VertexAttrib1fNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1fvNV, table->VertexAttrib1fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1sNV, table->VertexAttrib1sNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib1svNV, table->VertexAttrib1svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2dNV, table->VertexAttrib2dNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2dvNV, table->VertexAttrib2dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2fNV, table->VertexAttrib2fNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2fvNV, table->VertexAttrib2fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2sNV, table->VertexAttrib2sNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib2svNV, table->VertexAttrib2svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3dNV, table->VertexAttrib3dNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3dvNV, table->VertexAttrib3dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3fNV, table->VertexAttrib3fNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3fvNV, table->VertexAttrib3fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3sNV, table->VertexAttrib3sNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib3svNV, table->VertexAttrib3svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4dNV, table->VertexAttrib4dNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4dvNV, table->VertexAttrib4dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4fNV, table->VertexAttrib4fNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4fvNV, table->VertexAttrib4fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4sNV, table->VertexAttrib4sNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4svNV, table->VertexAttrib4svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4ubNV, table->VertexAttrib4ubNV);
   EXPECT_EQ((void *) __indirect_glVertexAttrib4ubvNV, table->VertexAttrib4ubvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribPointerNV, table->VertexAttribPointerNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs1dvNV, table->VertexAttribs1dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs1fvNV, table->VertexAttribs1fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs1svNV, table->VertexAttribs1svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs2dvNV, table->VertexAttribs2dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs2fvNV, table->VertexAttribs2fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs2svNV, table->VertexAttribs2svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs3dvNV, table->VertexAttribs3dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs3fvNV, table->VertexAttribs3fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs3svNV, table->VertexAttribs3svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs4dvNV, table->VertexAttribs4dvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs4fvNV, table->VertexAttribs4fvNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs4svNV, table->VertexAttribs4svNV);
   EXPECT_EQ((void *) __indirect_glVertexAttribs4ubvNV, table->VertexAttribs4ubvNV);
}

TEST_F(IndirectAPI, NV_point_sprite)
{
   EXPECT_EQ((void *) __indirect_glPointParameteriNV, table->PointParameteriNV);
   EXPECT_EQ((void *) __indirect_glPointParameterivNV, table->PointParameterivNV);
}

TEST_F(IndirectAPI, EXT_stencil_two_side)
{
   EXPECT_EQ((void *) __indirect_glActiveStencilFaceEXT, table->ActiveStencilFaceEXT);
}

TEST_F(IndirectAPI, NV_fragment_program)
{
   EXPECT_EQ((void *) __indirect_glGetProgramNamedParameterdvNV, table->GetProgramNamedParameterdvNV);
   EXPECT_EQ((void *) __indirect_glGetProgramNamedParameterfvNV, table->GetProgramNamedParameterfvNV);
   EXPECT_EQ((void *) __indirect_glProgramNamedParameter4dNV, table->ProgramNamedParameter4dNV);
   EXPECT_EQ((void *) __indirect_glProgramNamedParameter4dvNV, table->ProgramNamedParameter4dvNV);
   EXPECT_EQ((void *) __indirect_glProgramNamedParameter4fNV, table->ProgramNamedParameter4fNV);
   EXPECT_EQ((void *) __indirect_glProgramNamedParameter4fvNV, table->ProgramNamedParameter4fvNV);
}

TEST_F(IndirectAPI, EXT_blend_equation_separate)
{
   EXPECT_EQ((void *) __indirect_glBlendEquationSeparateEXT, table->BlendEquationSeparateEXT);
}

TEST_F(IndirectAPI, EXT_framebuffer_object)
{
   EXPECT_EQ((void *) __indirect_glBindFramebufferEXT, table->BindFramebufferEXT);
   EXPECT_EQ((void *) __indirect_glBindRenderbufferEXT, table->BindRenderbufferEXT);
   EXPECT_EQ((void *) __indirect_glCheckFramebufferStatusEXT, table->CheckFramebufferStatusEXT);
   EXPECT_EQ((void *) __indirect_glDeleteFramebuffersEXT, table->DeleteFramebuffersEXT);
   EXPECT_EQ((void *) __indirect_glDeleteRenderbuffersEXT, table->DeleteRenderbuffersEXT);
   EXPECT_EQ((void *) __indirect_glFramebufferRenderbufferEXT, table->FramebufferRenderbufferEXT);
   EXPECT_EQ((void *) __indirect_glFramebufferTexture1DEXT, table->FramebufferTexture1DEXT);
   EXPECT_EQ((void *) __indirect_glFramebufferTexture2DEXT, table->FramebufferTexture2DEXT);
   EXPECT_EQ((void *) __indirect_glFramebufferTexture3DEXT, table->FramebufferTexture3DEXT);
   EXPECT_EQ((void *) __indirect_glGenFramebuffersEXT, table->GenFramebuffersEXT);
   EXPECT_EQ((void *) __indirect_glGenRenderbuffersEXT, table->GenRenderbuffersEXT);
   EXPECT_EQ((void *) __indirect_glGenerateMipmapEXT, table->GenerateMipmapEXT);
   EXPECT_EQ((void *) __indirect_glGetFramebufferAttachmentParameterivEXT, table->GetFramebufferAttachmentParameterivEXT);
   EXPECT_EQ((void *) __indirect_glGetRenderbufferParameterivEXT, table->GetRenderbufferParameterivEXT);
   EXPECT_EQ((void *) __indirect_glIsFramebufferEXT, table->IsFramebufferEXT);
   EXPECT_EQ((void *) __indirect_glIsRenderbufferEXT, table->IsRenderbufferEXT);
   EXPECT_EQ((void *) __indirect_glRenderbufferStorageEXT, table->RenderbufferStorageEXT);
}

TEST_F(IndirectAPI, EXT_framebuffer_blit)
{
   EXPECT_EQ((void *) __indirect_glBlitFramebufferEXT, table->BlitFramebufferEXT);
}

TEST_F(IndirectAPI, EXT_texture_array)
{
   EXPECT_EQ((void *) __indirect_glFramebufferTextureLayerEXT, table->FramebufferTextureLayerEXT);
}
