; =================================================================
; MikeOS -- The Mike Operating System kernel
; Copyright (C) 2006, 2007 MikeOS Developers -- see LICENSE.TXT
;
; SYSTEM CALL SECTION -- Accessible to user programs
; =================================================================


; -----------------------------------------------------------------
; os_print_string -- Displays text
; IN: SI = message location (zero-terminated string)
; OUT: Nothing (registers preserved)

os_print_string:
	pusha

	mov ah, 0Eh		; int 10h teletype function

.repeat:
	lodsb			; Get char from string
	cmp al, 0
	je .done		; If char is zero, end of string
	int 10h			; Otherwise, print it
	jmp .repeat

.done:
	popa
	ret


; -----------------------------------------------------------------
; os_move_cursor -- Moves cursor in text mode
; IN: DH, DL = row, column, OUT: Nothing (registers preserved)

os_move_cursor:
	pusha

	xor bh, bh
	mov ah, 2
	int 10h

	popa
	ret


; -----------------------------------------------------------------
; os_show_cursor -- Turns on cursor in text mode
; IN/OUT: Nothing!

os_show_cursor:
	pusha

	mov ch, 0			; Set cursor to solid block
	mov cl, 7
	mov ah, 1
	mov al, 3			; Must be video mode for buggy BIOSes!
	int 10h

	popa
	ret


; -----------------------------------------------------------------
; os_hide_cursor -- Turns off cursor in text mode
; IN/OUT: Nothing!

os_hide_cursor:
	pusha

	mov ch, 32
	mov ah, 1
	mov al, 3			; Must be video mode for buggy BIOSes!
	int 10h

	popa
	ret


; -----------------------------------------------------------------
; os_draw_block -- Render block of specified colour
; IN: BL/DL/DH/SI/DI = colour/start X pos/start Y pos/width/finish Y pos

os_draw_block:
	pusha

.more:
	call os_move_cursor		; Move to block starting position

	mov ah, 09h			; Draw colour section
	xor bh, bh
	mov cx, si
	mov al, ' '
	int 10h

	inc dh				; Get ready for next line

	xor ax, ax
	mov al, dh			; Get current Y position into DL
	cmp ax, di			; Reached finishing point (DI)?
	jne .more			; If not, keep drawing

	popa
	ret


; -----------------------------------------------------------------
; os_file_selector -- Show a file selection dialog
; IN: Nothing, OUT: AX = location of zero-terminated filename string

os_file_selector:
	pusha

	call os_hide_cursor

	mov bl, 01001111b		; White on red
	mov dl, 20			; Start X position
	mov dh, 2			; Start Y position
	mov si, 40			; Width
	mov di, 23			; Finish Y position
	call os_draw_block

	mov dl, 21			; Show first line of help text...
	mov dh, 3
	call os_move_cursor
	mov si, .title_string_1
	call os_print_string

	inc dh				; ...and the second
	call os_move_cursor
	mov si, .title_string_2
	call os_print_string

	mov bl, 01110000b		; Black on grey for file list box
	mov dl, 21
	mov dh, 6
	mov si, 38
	mov di, 22
	call os_draw_block

	mov dl, 33			; Get into position for file list text
	mov dh, 7
	call os_move_cursor

	mov ax, .buffer
	call os_get_file_list

	mov si, ax			; SI = location of file list string


	mov ah, 0Eh			; For int 10h printing in a sec
	mov cx, 0			; Counter for dot in filename
	mov bx, 1			; Counter for total number of files
.more:
	inc cx
	cmp cx, 9			; Up to filename dot position?
	je .print_dot

	mov byte al, [si]		; Get character from file list string
	inc si				; And move along in string

	cmp al, 0			; End of string?
	je .done_list
	cmp al, ' '
	je .more
	cmp al, ','			; Next filename? (String is comma-separated)
	je .newline
	int 10h				; Otherwise print it!
	jmp .more

.newline:
	inc bx				; Update the number-of-files counter
	mov dl, 33			; Go back to starting X position
	inc dh				; But jump down a line
	call os_move_cursor
	mov cx, 0			; Reset dot-in-filename counter
	jmp .more

.print_dot:
	mov al, '.'			; Print dot in filename
	int 10h
	jmp .more



.done_list:
	mov dl, 25			; Set up starting position for selector
	mov dh, 7

	add bx, 6			; BX is our number-of-files counter, and we add
					; 6 to determine position (where list starts - 1)


.more_select:
	call os_move_cursor

	mov si, .position_string	; Show '>>>>>' next to filename
	call os_print_string


	call os_wait_for_key		; Move / select filename
	cmp ah, 48h			; Up pressed?
	je .go_up
	cmp ah, 50h			; Down pressed?
	je .go_down
	cmp al, 13			; Enter pressed?
	je .file_selected
	jmp .more_select		; If not, wait for another key


.go_up:
	cmp dh, 7			; Already at top?
	je .more_select

	mov dl, 25
	call os_move_cursor

	mov si, .position_string_blank	; Otherwise overwrite '>>>>>'
	call os_print_string

	dec dh
	jmp .more_select


.go_down:				; Already at bottom?
	cmp dh, bl
	je .more_select

	mov dl, 25
	call os_move_cursor

	mov si, .position_string_blank	; Otherwise overwrite '>>>>>'
	call os_print_string

	inc dh
	jmp .more_select


.file_selected:
	sub dh, 7			; Started printing list at 7 chars
					; down, so remove that to get the
					; starting point of the file list

	mov ax, 12			; Then multiply that by 12 to get position
	mul dh				; in file list (filenames are 12 chars with
					; commas in the list)

	mov si, .buffer			; Going to put selected filename into
	add si, ax			; the .filename string
	mov di, .filename

	mov cx, 0

.more_copy:
	mov ax, [si]
	mov [di], ax
	inc cx
	cmp cx, 11			; Copy string for 11 characters
	je .done
	inc si
	inc di
	jmp .more_copy

.done:
	inc di
	mov byte [di], 0		; And zero-terminate it!

	popa

	mov ax, .filename		; Filename string location in AX

	ret


	.title_string_1	db 'Please select a file with the cursor', 0
	.title_string_2	db 'keys, and press enter to choose...', 0

	.position_string_blank	db '     ', 0
	.position_string	db '>>>>>', 0

	.buffer	times 255 db 0
	.filename	times 12 db 0


; -----------------------------------------------------------------
; os_draw_background -- Clear screen with white top and bottom bars,
; containing text, and a coloured middle section.
; IN: AX/BX = top/bottom string locations, CX = colour

os_draw_background:
	pusha

	push ax				; Store params to pop out later
	push bx
	push cx

	call os_clear_screen

	mov ah, 09h			; Draw white bar at top
	xor bh, bh
	mov cx, 80
	mov bl, 01110000b
	mov al, ' '
	int 10h

	mov dh, 1
	mov dl, 0
	call os_move_cursor

	mov ah, 09h			; Draw colour section
	mov cx, 1840
	pop bx				; Get colour param (originally in CX)
	xor bh, bh
	mov al, ' '
	int 10h

	mov dh, 24
	mov dl, 0
	call os_move_cursor

	mov ah, 09h			; Draw white bar at bottom
	xor bh, bh
	mov cx, 80
	mov bl, 01110000b
	mov al, ' '
	int 10h

	mov dh, 24
	mov dl, 1
	call os_move_cursor
	pop bx				; Get bottom string param
	mov si, bx
	call os_print_string

	mov dh, 0
	mov dl, 1
	call os_move_cursor
	pop ax				; Get top string param
	mov si, ax
	call os_print_string

	mov dh, 1			; Ready for app text
	mov dl, 0
	call os_move_cursor

	popa
	ret


; -----------------------------------------------------------------
; os_clear_screen -- Clears the screen
; IN/OUT: Nothing (registers preserved)

os_clear_screen:
	pusha

	mov dx, 0			; Position cursor at top-left
	call os_move_cursor

	mov ah, 6			; Scroll full-screen
	mov al, 0			; Normal white on black
	mov bh, 7			; Upper-left corner of screen
	mov cx, 0			; Bottom-right
	mov dh, 24
	mov dl, 79
	int 10h

	popa
	ret


; -----------------------------------------------------------------
; os_bcd_to_int -- Converts binary coded decimal number to an integer
; IN: AL = BCD number, OUT: AX = integer value

os_bcd_to_int:
	pusha

	xor bx, bx
	mov bl, al			; Store for now

	and ax, 0xF			; Zero-out high bits
	mov cl, al			; CL = lower BCD number

	mov al, bl			; Get original number back
	shr ax, 4			; Rotate higher BCD number into lower bits
	and ax, 0xF			; Zero-out high bits

	mov dx, 10			; Multiply by 10 (as it's the higher BCD)
	mul dx

	add ax, cx			; Add it to the lower BCD
	mov [.tmp], ax

	popa
	mov ax, [.tmp]			; And return it in AX!
	ret


	.tmp	dw 0


; -----------------------------------------------------------------
; os_get_time_string -- Get current time in a string (eg '20:25')
; IN/OUT: BX = string location

os_get_time_string:
	pusha

	push bx				; Store string location for now

        mov ax,0x0200
        int 0x1A			; Get time data from BIOS in BCD format

        mov al,	ch			; Hour
	call os_bcd_to_int
        mov ch, al

        mov al, cl			; Minute
	call os_bcd_to_int
        mov cl, al

	xor ax, ax

	mov al, ch			; Convert hour to string
	mov bx, .tmp_string1
	call os_int_to_string

	mov al, cl
	mov bx, .tmp_string2		; Convert minute to string
	call os_int_to_string

	mov ax, .tmp_string2		; Just one number in minutes string?
	call os_string_length
	cmp ax, 1
	jne .twochars

	mov ax, .zero_string		; If so, add '0' char before it
	mov bx, .tmp_string2
	mov cx, .tmp_string3
	call os_string_join

	mov si, .tmp_string3		; And copy string back into old minutes string
	mov di, .tmp_string2
	call os_string_copy

.twochars:
	pop bx				; Get original string location back

	mov cx, bx			; Add hour and separator character to it
	mov ax, .tmp_string1
	mov bx, .separator
	call os_string_join

	mov ax, cx			; And add the minutes
	mov bx, .tmp_string2
	call os_string_join

	popa
	ret


	.tmp_string1 	times 3 db 0
	.tmp_string2 	times 3 db 0
	.tmp_string3	times 3 db 0
	.zero_string	db '0', 0
	.separator	db ':', 0


; -----------------------------------------------------------------
; os_print_horiz_line -- Draw a horizontal line on the screen
; IN: AX = line type (1 for double, otherwise single)
; OUT: Nothing (registers preserved)

os_print_horiz_line:
	pusha

	mov cx, ax			; Store line type param
	mov al, 196			; Default is single-line code

	cmp cx, 1			; Was double-line specified in AX?
	jne .ready
	mov al, 205			; If so, here's the code

.ready:
	mov cx, 0			; Counter
	mov ah, 0Eh			; BIOS output char routine

.restart:
	int 10h
	inc cx
	cmp cx, 80			; Drawn 80 chars yet?
	je .done
	jmp .restart

.done:
	popa
	ret


; -----------------------------------------------------------------
; os_print_newline -- Reset cursor to start of next line
; IN/OUT: Nothing (registers preserved)

os_print_newline:
	pusha

	mov ah, 0Eh			; BIOS output char code

	mov al, 13
	int 10h
	mov al, 10
	int 10h

	popa
	ret


; -----------------------------------------------------------------
; os_wait_for_key -- Waits for keypress and returns key
; IN: Nothing, OUT: AX = key pressed, other regs preserved

os_wait_for_key:
	pusha

	xor ax, ax
	mov ah, 00			; BIOS call to wait for key
	int 16h

	mov [.tmp_buf], ax		; Store resulting keypress

	popa				; But restore all other regs
	mov ax, [.tmp_buf]
	ret


	.tmp_buf	dw 0


; -----------------------------------------------------------------
; os_check_for_key -- Scans keyboard for input, but doesn't wait
; IN: Nothing, OUT: AL = 0 if no key pressed, otherwise ASCII code

os_check_for_key:
	pusha

	xor ax, ax
	mov ah, 01			; BIOS call to check for key
	int 16h

	jz .nokey			; If no key, skip to end

	xor ax, ax			; Otherwise get it from buffer
	mov ah, 00
	int 16h

	mov [.tmp_buf], al		; Store resulting keypress

	popa				; But restore all other regs
	mov al, [.tmp_buf]
	ret

.nokey:
	popa
	mov al, 0			; Zero result if no key pressed
	ret


	.tmp_buf	db 0


; -----------------------------------------------------------------
; os_int_to_string -- Convert value in AX to string
; IN: AX = integer, BX = location of string
; OUT: BX = location of converted string (other regs preserved)
;
; NOTE: Based on public domain code

os_int_to_string:
	pusha

	mov di, bx

	mov byte [.zerow], 0x00
	mov word [.varbuff], ax
	xor ax, ax
	xor cx, cx
	xor dx, dx
 	mov bx, 10000
	mov word [.deel], bx

.mainl:
	mov bx, word [.deel]
	mov ax, word [.varbuff]
	xor dx, dx
	xor cx, cx
	div bx
	mov word [.varbuff], dx

.vdisp:
	cmp ax, 0
	je .firstzero
	jmp .ydisp

.firstzero:
	cmp byte [.zerow], 0x00
	je .nodisp

.ydisp:
	add al, 48                              ; Make it numeric (0123456789)
	mov [di], al
	inc di
	mov byte [.zerow], 0x01
	jmp .yydis

.nodisp:
.yydis:
	xor dx, dx
	xor cx, cx
	xor bx, bx
	mov ax, word [.deel]
	cmp ax, 1
	je .bver
	cmp ax, 0
	je .bver
	mov bx, 10
	div bx
	mov word [.deel], ax
	jmp .mainl

.bver:
	mov byte [di], 0

	popa
	ret


	.deel		dw 0x0000
	.varbuff	dw 0x0000
	.zerow		db 0x00


; -----------------------------------------------------------------
; os_speaker_tone -- Generate PC speaker tone (call os_speaker_off after)
; IN: AX = note frequency, OUT: Nothing (registers preserved)

os_speaker_tone:
	pusha

	mov cx, ax		; Store note value for now

	mov al, 182
	out 43h, al
	mov ax, cx		; Set up frequency
	out 42h, al
	mov al, ah
	out 42h, al

	in al, 61h		; Switch PC speaker on
	or al, 03h
	out 61h, al

	popa
	ret


; -----------------------------------------------------------------
; os_speaker_off -- Turn off PC speaker
; IN/OUT: Nothing (registers preserved)

os_speaker_off:
	pusha

	in al, 61h		; Switch PC speaker off
	and al, 0FCh
	out 61h, al

	popa
	ret


; -----------------------------------------------------------------
; os_dialog_box -- Print dialog box in middle of screen, with button(s)
; IN: AX, BX, CX = string locations (set registers to 0 for no display)
; IN: DX = 0 for single 'OK' dialog, 1 for two-button 'OK' and 'Cancel'
; OUT: If two-button mode, AX = 0 for OK and 1 for cancel
; NOTE: Each string is limited to 40 characters

os_dialog_box:
	pusha

	mov [.tmp], dx

	push ax				; Store first string location...
	call os_string_length		; ...because this converts AX to a number
	cmp ax, 40			; Check to see if it's less than 30 chars
	jg .string_too_long

	mov ax, bx			; Check second string length
	call os_string_length
	cmp ax, 40
	jg .string_too_long

	mov ax, cx			; Check third string length
	call os_string_length
	cmp ax, 40
	jg .string_too_long

	pop ax				; Get first string location back
	jmp .strings_ok			; All string lengths OK, so let's move on


.string_too_long:
	pop ax				; We pushed this before
	mov ax, .err_msg_string_length
	call os_fatal_error


.strings_ok:
	call os_hide_cursor

	mov dh, 9			; First, draw red background box
	mov dl, 19

.redbox:				; Loop to draw all lines of box
	call os_move_cursor

	pusha
	mov ah, 09h
	xor bh, bh
	mov cx, 42
	mov bl, 01001111b		; White on red
	mov al, ' '
	int 10h
	popa

	inc dh
	cmp dh, 16
	je .boxdone
	jmp .redbox


.boxdone:
	cmp ax, 0			; Skip string params if zero
	je .no_first_string
	mov dl, 20
	mov dh, 10
	call os_move_cursor

	mov si, ax			; First string
	call os_print_string

.no_first_string:
	cmp bx, 0
	je .no_second_string
	mov dl, 20
	mov dh, 11
	call os_move_cursor

	mov si, bx			; Second string
	call os_print_string

.no_second_string:
	cmp cx, 0
	je .no_third_string
	mov dl, 20
	mov dh, 12
	call os_move_cursor

	mov si, cx			; Third string
	call os_print_string

.no_third_string:
	mov dx, [.tmp]
	cmp dx, 0
	je .one_button
	cmp dx, 1
	je .two_button


.one_button:
	mov dl, 35			; OK button, centered at bottom of box
	mov dh, 14
	call os_move_cursor
	mov si, .ok_button_string
	call os_print_string

	jmp .one_button_wait


.two_button:
	mov dl, 27			; OK button
	mov dh, 14
	call os_move_cursor
	mov si, .ok_button_string
	call os_print_string

	mov dl, 42			; Cancel button
	mov dh, 14
	call os_move_cursor
	mov si, .cancel_button_noselect
	call os_print_string

	mov cx, 0			; Default button = 0
	jmp .two_button_wait



.one_button_wait:
	call os_wait_for_key
	cmp al, 13			; Wait for enter key (13) to be pressed
	jne .one_button_wait

	call os_show_cursor

	popa
	ret


.two_button_wait:
	call os_wait_for_key

	cmp ah, 75			; Left cursor key pressed?
	jne .noleft

	mov dl, 27			; If so, change printed buttons
	mov dh, 14
	call os_move_cursor
	mov si, .ok_button_string
	call os_print_string

	mov dl, 42			; Cancel button
	mov dh, 14
	call os_move_cursor
	mov si, .cancel_button_noselect
	call os_print_string

	mov cx, 0			; And update result we'll return
	jmp .two_button_wait


.noleft:
	cmp ah, 77			; Right cursor key pressed?
	jne .noright

	mov dl, 27			; If so, change printed buttons
	mov dh, 14
	call os_move_cursor
	mov si, .ok_button_noselect
	call os_print_string

	mov dl, 42			; Cancel button
	mov dh, 14
	call os_move_cursor
	mov si, .cancel_button_string
	call os_print_string

	mov cx, 1			; And update result we'll return
	jmp .two_button_wait


.noright:
	cmp al, 13			; Wait for enter key (13) to be pressed
	jne .two_button_wait

	call os_show_cursor

	mov [.tmp], cx			; Keep result after restoring all regs
	popa
	mov ax, [.tmp]

	ret


	.err_msg_string_length	db 'os_dialog_box: Supplied string too long', 0
	.ok_button_string	db '[= OK =]', 0
	.cancel_button_string	db '[= Cancel =]', 0
	.ok_button_noselect	db '   OK   ', 0
	.cancel_button_noselect	db '   Cancel   ', 0

	.tmp dw 0


; -----------------------------------------------------------------
; os_input_string -- Take string from keyboard entry
; IN/OUT: AX = location of string, other regs preserved

os_input_string:
	pusha

	mov bx, ax			; Store location into BX

	mov ax, 0x2000			; Segment
	mov ds, ax
	mov es, ax

	mov di, bx			; DI is where we'll store input
	mov cx, 0			; Counter for chars entered

	mov ax, 0                       ; First, clear string to 0s

.loop_blank:
	stosb

	cmp byte [di], 0		; Reached zero (end of line) in string?
	je .blank_done

	jmp .loop_blank

.blank_done:
	mov di, bx
	mov cx, 0


.more:					; Now onto string getting
	call os_wait_for_key

	cmp al, 13			; If Enter key pressed, finish
	je .done

	cmp al, 8			; Backspace pressed?
	je .backspace			; If not, skip following checks

	cmp al, 32			; In ASCII range (32 - 126)?
	jl .more

	cmp al, 126
	jg .more

	jmp .nobackspace


.backspace:
	cmp cx, 0			; Backspaced at start of line?
	je .more

	pusha
	mov ah, 0Eh			; If not, write space and move cursor back
	mov al, 8
	int 10h				; Backspace twice, to clear space
	mov al, 32
	int 10h
	mov al, 8
	int 10h
	popa

	dec di
	mov byte [di], 0		; Zero-out the previous character, as we're
					; going back one char and need to
					; zero-terminate the string!

	dec cx				; Step back in counter

	jmp .more


.nobackspace:
	pusha
	mov ah, 0Eh			; Output entered char
	int 10h
	popa

	inc cx
	cmp cx, 250			; Make sure we don't exhaust buffer
	je near .done

	stosb				; Store

	jmp near .more


.done:
	popa
	ret


; -----------------------------------------------------------------
; os_string_length -- Return length of a string
; IN: AX = string location, OUT AX = length (other regs preserved)

os_string_length:
	pusha

	mov bx, ax		; Location of string now in BX
	mov cx, 0

.more:
	cmp byte [bx], 0	; Zero (end of string) yet?
	je .done
	inc bx			; If not, keep adding
	inc cx
	jmp .more


.done:
	mov word [.tmp_counter], cx
	popa
	mov ax, [.tmp_counter]
	ret


	.tmp_counter	dw 0


; -----------------------------------------------------------------
; os_string_uppercase -- Convert zero-terminated string to uppercase
; IN/OUT: AX = string location

os_string_uppercase:
	pusha

.more:
	cmp byte [si], 0		; Zero-termination of string?
	je .done			; If so, quit

	cmp byte [si], 97		; In the uppercase A to Z range?
	jl .noatoz
	cmp byte [si], 122
	jg .noatoz

	sub byte [si], 20h		; If so, convert input char to lowercase

	inc si
	jmp .more

.noatoz:
	inc si
	jmp .more

.done:
	popa
	ret


; -----------------------------------------------------------------
; os_string_lowercase -- Convert zero-terminated string to lowercase
; IN/OUT: AX = string location

os_string_lowercase:
	pusha

.more:
	cmp byte [si], 0		; Zero-termination of string?
	je .done			; If so, quit

	cmp byte [si], 65		; In the lowercase A to Z range?
	jl .noatoz
	cmp byte [si], 90
	jg .noatoz

	add byte [si], 20h		; If so, convert input char to uppercase

	inc si
	jmp .more

.noatoz:
	inc si
	jmp .more

.done:
	popa
	ret


; -----------------------------------------------------------------
; os_string_copy -- Copy one string into another
; IN/OUT: SI = source, DI = destination

os_string_copy:
	pusha

.more:
	cmp byte [si], 0	; If source string is empty, quit out
	je .done
	mov al, [si]		; Transfer contents
	mov [di], al
	inc si
	inc di
	jmp .more

.done:
	mov byte [di], 0	; Write terminating zero

	popa
	ret


; -----------------------------------------------------------------
; os_string_truncate -- Chop string down to specified number of characters
; IN: SI = string location, AX = number of characters
; OUT: SI = string location

os_string_truncate:
	pusha

	add si, ax
	mov byte [si], 0

	popa
	ret


; -----------------------------------------------------------------
; os_string_join -- Join two strings into a third string
; IN/OUT: AX = string one, BX = string two, CX = destination string

os_string_join:
	pusha

	mov si, ax		; Put first string into CX
	mov di, cx
	call os_string_copy

	call os_string_length	; Get length of first string

	add cx, ax		; Position at end of first string

	mov si, bx		; Add second string onto it
	mov di, cx
	call os_string_copy

	popa
	ret


; -----------------------------------------------------------------
; os_string_chomp -- Strip leading and trailing spaces from a string
; IN: AX = string location

os_string_chomp:
	pusha

	push ax				; Store string location

	mov di, ax			; Put location into DI
	xor bx, bx

.keepcounting:				; Get number of leading spaces into BX
	cmp byte [di], ' '
	jne .counted
	inc bx
	inc di
	jmp .keepcounting

.counted:
	pop ax				; Get string location back
	push ax

	mov di, ax			; DI = original string start

	mov si, ax
	add si, bx			; SI = first non-space char in string

.keep_copying:
	mov ax, [si]			; Copy SI into DI (original string start)
	mov [di], ax
	cmp ax, 0
	je .finished_copy
	inc si
	inc di
	jmp .keep_copying

.finished_copy:
	pop ax
	mov si, ax

	call os_string_length

	add si, ax			; Move to end of string

.more:
	dec si
	cmp byte [si], ' '
	jne .done
	mov byte [si], 0
	jmp .more

.done:
	popa
	ret


; -----------------------------------------------------------------
; os_string_strip -- Removes specified character from a string
; IN: SI = string location, AX = character to remove

os_string_strip:
	pusha

	mov dx, si			; Store string location for later

	mov di, os_buffer

.more:
	mov bx, [si]

	cmp bx, 0
	je .done

	inc si
	cmp bl, al
	je .more

	mov [di], bl
	inc di
	jmp .more

.done:
	mov [di], bl

	mov si, os_buffer
	mov di, dx
	call os_string_copy

	popa
	ret


; -----------------------------------------------------------------
; os_string_compare -- See if two strings match
; IN: SI = string one, DI = string two; OUT: carry set if same

os_string_compare:
	pusha

.more:
	mov ax, [si]			; Store string contents
	mov bx, [di]

	cmp byte [si], 0		; End of first string?
	je .terminated

	cmp ax, bx
	jne .not_same

	inc si
	inc di
	jmp .more


.not_same:
	popa
	clc
	ret


.terminated:
	cmp byte [di], 0		; End of second string?
	jne .not_same

	popa
	stc
	ret


; -----------------------------------------------------------------
; os_pause -- Delay execution for specified microseconds
; IN: CX:DX = number of microseconds to wait

os_pause:
	pusha

	mov ah, 86h
	int 15h

	popa
	ret


; -----------------------------------------------------------------
; os_get_api_version -- Return current version of MikeOS API
; IN: Nothing; OUT: AL = API version number

os_get_api_version:
	mov al, MIKEOS_API_VER
	ret


; -----------------------------------------------------------------
; os_modify_int_handler -- Change location of interrupt handler
; IN: CX = int number, SI = handler location

os_modify_int_handler:
	pusha

	mov dx, es                      ; Store original ES

	xor ax, ax                      ; Clear AX for new ES value
	mov es, ax

	mov al, cl                      ; Move supplied int into AL

	mov bl, 4h                      ; Multiply by four to get position
	mul bl                          ; (Interrupt table = 4 byte sections)
	mov bx, ax

	mov [es:bx], si                 ; First store offset
	add bx, 2

	mov ax, 0x2000                  ; Then segment of our handler
	mov [es:bx], ax

	mov es, dx                      ; Finally, restore data segment

	popa
	ret


; -----------------------------------------------------------------
; os_fatal_error -- Display error message, take keypress, and restart OS
; IN: AX = error message string location

os_fatal_error:
	mov bx, ax			; Store string location for now

	mov dh, 0
	mov dl, 0
	call os_move_cursor

	pusha
	mov ah, 09h			; Draw red bar at top
	xor bh, bh
	mov cx, 240
	mov bl, 01001111b
	mov al, ' '
	int 10h
	popa

	mov dh, 0
	mov dl, 0
	call os_move_cursor

	mov si, .msg_inform		; Inform of fatal error
	call os_print_string

	mov si, bx			; Program-supplied error message
	call os_print_string

	call os_print_newline

	mov si, .msg_prompt		; Restart prompt
	call os_print_string

	xor ax, ax
	mov ah, 00			; BIOS call to wait for key
	int 16h

	jmp os_int_reboot


	.msg_inform		db '>>> FATAL OPERATING SYSTEM ERROR', 13, 10, 0
	.msg_prompt		db 'Press a key to restart MikeOS...', 0


; -----------------------------------------------------------------
; os_get_file_list -- Generate comma-separated string of files on floppy
; IN/OUT: AX = location of string to store filenames

os_get_file_list:
	pusha

	mov word [.file_list_tmp], ax	; Store string location

	xor eax, eax			; Needed for some older BIOSes

	call os_int_reset_floppy
	jnc .floppy_ok			; Did the floppy reset OK?

	mov ax, .err_msg_floppy_reset	; If not, bail out
	jmp os_fatal_error


.floppy_ok:				; Ready to read first block of data
	mov ax, 19			; Root dir starts at logical sector 19
	call os_int_l2hts

	lea si, [os_buffer]		; ES:BX should point to our buffer
	mov bx, ds
	mov es, bx
	mov bx, si

	mov ah, 0x02			; Params for int 13h: read floppy sectors
	mov al, 14			; And read 14 of them

	clc				; Prepare to enter loop
	pusha


.read_first_sector:
	popa

	pusha
	int 13h				; Read sectors
	call os_int_reset_floppy	; Check we've read them OK
	jc .read_first_sector		; If not, just keep trying
	popa

	xor ax, ax
	mov cx, 0
	mov bx, os_buffer+64		; Start of filenames
	mov di, os_buffer+64		; Data reader from start of filenames

	mov word dx, [.file_list_tmp]


.showdir:
	mov ax, [di]
	cmp al, 229			; If we read 229 = deleted filename
	je .skip
	cmp al, 0
	je .done

	inc di

	push di
	mov di, dx			; DX = where we're storing string
	stosb
	inc dx
	pop di

	inc cx
	cmp cx, 11			; Done 11 char filename?
	je .gotfilename
	jmp .showdir


.gotfilename:				; Got a filename
	push di
	mov di, dx			; DX = where we're storing string
	mov ax, ','			; Use comma to separate for next file
	stosb
	inc dx
	pop di


.skip:
	mov cx, 0			; Reset char counter

	add bx, 64			; Shift to next 64 bytes (next filename)
	mov di, bx			; And update our DI with that
	jmp .showdir


.done:
	mov di, dx			; Zero-terminate string
	dec di				; Don't want to store last comma!
	mov ax, 0
	stosb

	popa
	ret


	.file_list_tmp	dw 0
	.err_msg_floppy_reset	db 'os_get_file_list: Floppy failed to reset', 0


; -----------------------------------------------------------------
; os_program_load -- Load and execute program (must be 32K or less)
; IN: AX = location of filename, BX = 1 if loader should clear screen
; OUT: Carry set if program not found on the disk
; NOTE: Based on free bootloader code by E Dehling.

os_program_load:
	mov [.filename_loc], ax		; Store filename location
	mov [.clear_screen], bx

	xor eax, eax			; Needed for some older BIOSes

	call os_int_reset_floppy
	jnc .floppy_ok			; Did the floppy reset OK?

	mov ax, .err_msg_floppy_reset	; If not, bail out
	jmp os_fatal_error


.floppy_ok:				; Ready to read first block of data
	mov ax, 19			; Root dir starts at logical sector 19
	call os_int_l2hts

	lea si, [os_buffer]		; ES:BX should point to our buffer
	mov bx, ds
	mov es, bx
	mov bx, si

	mov ah, 0x02			; Params for int 13h: read floppy sectors
	mov al, 14			; And read 14 of them

	clc				; Prepare to enter loop
	pusha


.read_first_sector:
	popa
	pusha

	int 13h				; Read sectors

	call os_int_reset_floppy	; Check we've read them OK
	jc .read_first_sector		; If not, just keep trying

	popa

	mov ax, ds			; ES:DI = root directory
	mov es, ax
	lea di, [os_buffer]
	mov cx, word 244		; Search all entries in root dir
	mov ax, 0			; Searching offset 0 in root dir


.next_root_entry:
	xchg cx, dx			; We use CX in the inner loop...

	mov si, [.filename_loc]		; DS:SI = location of filename to load

	mov cx, 11			; Length of filename, for comparison
	rep cmpsb
	je .found_file_to_load

	add ax, 32			; Bump searched entries by 1 (offset + 32 bytes)

	lea di, [os_buffer]		; Point root-dir at next entry
	add di, ax

	xchg dx, cx			; Swap, as we need the 'outer' CX
	loop .next_root_entry

	stc				; If file never found, return with carry set
	ret


.found_file_to_load:			; Now fetch cluster and load FAT into RAM
	mov ax, word [es:di+0x0F]
	mov word [.cluster], ax

	mov ax, 1			; Sector 1 = first sector of first FAT
	call os_int_l2hts

	lea di, [os_buffer]		; ES:BX points to our buffer
	mov bx, di

	mov ah, 0x02			; int 13h params: read sectors
	mov al, 0x09			; And read 9 of them :-)

	clc				; In case not cleared in reading sector
	pusha

.read_fat:
	popa				; In case regs altered by int 13h
	pusha

	int 13h

	jnc .read_fat_ok

	call os_int_reset_floppy

	jmp .read_fat


.read_fat_ok:
	popa

	mov ax, 0x2000			; Where we'll load the file
	mov es, ax
	xor bx, bx

	mov ah, 0x02			; int 13h floppy read params
	mov al, 0x01

	push ax				; Save in case we (or int calls) lose it


	mov word [.pointer], 100h	; Entry point for apps


.load_file_sector:
	mov ax, word [.cluster]		; Convert sector to logical
	add ax, 31

	call os_int_l2hts		; Make appropriate params for int 13h

	mov ax, 0x2000			; Set buffer past what we've already read
	mov es, ax
	mov bx, word [.pointer]

	pop ax				; Save in case we (or int calls) lose it
	push ax

	int 13h

	jnc .calculate_next_cluster	; If there's no error...

	call os_int_reset_floppy	; Otherwise, reset floppy and retry
	jmp .load_file_sector


.calculate_next_cluster:
	mov ax, [.cluster]
	xor dx, dx
	mov bx, 2
	div bx				; DX = [CLUSTER] mod 2

	or dx, dx			; If DX = 0 [CLUSTER] = even, if DX = 1 then odd

	jz .even			; If [CLUSTER] = even, drop last 4 bits of word
					; with next cluster; if odd, drop first 4 bits

.odd:
	mov ax, [.cluster]
	mov bx, 3
	mul bx
	mov bx, 2
	div bx				; AX = address of word in FAT for the 12 bits

	lea si, [os_buffer]
	add si, ax
	mov ax, word [ds:si]

	shr ax, 4			; Shift out first 4 bits (belong to another entry)

	mov word [.cluster], ax

	cmp ax, 0x0FF8
	jae .end

	add word [.pointer], 512
	jmp .load_file_sector		; Onto next sector!

.even:
	mov ax, [.cluster]
	mov bx, 3
	mul bx
	mov bx, 2
	div bx

	lea si, [os_buffer]		; AX = word in FAT for the 12 bits
	add si, ax
	mov ax, word [ds:si]
	and ax, 0x0FFF			; Mask out last 4 bits
	mov word [.cluster], ax		; Store cluster

	cmp ax, 0x0FF8
	jae .end

	add word [.pointer], 512	; Increase buffer pointer
	jmp .load_file_sector


.end:
	pop ax				; Clear stack

	mov bx, [.clear_screen]
	cmp bx, 1
	jne .run_program

	call os_clear_screen

.run_program:
        pusha                           ; Save all register,
        push ds                         ; segment register
        push es                         ; es, ds and the
        mov [.mainstack],sp             ; save stack pointer
        xor ax,ax                       ; Clear registers
        xor bx,bx                       ; DOS cleared this
        xor cx,cx                       ; and we want be DOS
        xor dx,dx                       ; compatible
        xor si,si                       ;
        xor di,di                       ;
        xor bp,bp                       ;
        mov byte [now_run_a_program],1

	call 0x0100			; Jump to newly-loaded program!

.end_the_program:			; End of the program run
	mov byte [now_run_a_program], 0
	mov sp, [.mainstack]		; Restore stack, segment and
	pop es                          ; common registers
	pop ds
	popa
	clc

	ret


.notvalidfile:
	mov ax, .err_msg_invalid_file
	call os_fatal_error


	.bootd		db 0 		; Boot device number
	.cluster	dw 0 		; Cluster of the file we want to load
	.pointer	dw 0 		; Pointer into os_buffer, for loading 'file2load'

	.filename_loc	dw 0		; Temporary store of filename location
	.clear_screen	dw 0		; Setting for whether we clear the screen

	.err_msg_invalid_file	db 'os_program_load: Not a MikeOS executable (no MOS header bytes)', 0
	.err_msg_floppy_reset	db 'os_program_load: Floppy failed to reset', 0
        .mainstack dw 0

         now_run_a_program db 0



; =================================================================
; INTERNAL OS ROUTINES -- Not accessible to user programs

; -----------------------------------------------------------------
; Reboot machine via keyboard controller

os_int_reboot:
	; XXX -- We should check that keyboard buffer is empty first
	mov al, 0xFE
	out 0x64, al


; -----------------------------------------------------------------
; Reset floppy drive

os_int_reset_floppy:
	push ax
	push dx
	xor ax, ax
	mov dl, 0
	int 13h
	pop dx
	pop ax
	ret


; -----------------------------------------------------------------
; Convert floppy sector from logical to physical

os_int_l2hts:		; Calculate head, track and sector settings for int 13h
			; IN: AX = logical sector, OUT: correct regs for int 13h
	push bx
	push ax

	mov bx, ax			; Save logical sector

	xor dx, dx			; First the sector
	div word [.sectors_per_track]
	add dl, 01h			; Physical sectors start at 1
	mov cl, dl			; Sectors belong in CL for int 13h
	mov ax, bx

	xor dx, dx			; Now calculate the head
	div word [.sectors_per_track]
	xor dx, dx
	div word [.sides]
	mov dh, dl			; Head/side
	mov ch, al			; Track

	pop ax
	pop bx

	mov dl, 0			; Boot device = 0

	ret


	.sectors_per_track	dw 18	; Floppy disc info
	.sides			dw 2


; =================================================================

