/* insmod.c: insert a module into the kernel.
    Copyright (C) 2001  Rusty Russell.
    Copyright (C) 2002  Rusty Russell, IBM Corporation.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <asm/unistd.h>

#include "backwards_compat.c"

#define streq(a,b) (strcmp((a),(b)) == 0)

static void print_usage(const char *progname)
{
	fprintf(stderr, "Usage: %s filename [args]\n", progname);
	exit(1);
}

/* We use error numbers in a loose translation... */
static const char *moderror(int err)
{
	switch (err) {
	case ENOEXEC:
		return "Invalid module format";
	case ENOENT:
		return "Unknown symbol in module";
	case ESRCH:
		return "Module has wrong symbol version";
	case EINVAL:
		return "Invalid parameters";
	default:
		return strerror(err);
	}
}

int main(int argc, char *argv[])
{
	unsigned int i;
	int fd;
	long int ret;
	struct stat st;
	unsigned long len;
	void *map;
	char *filename, *options = strdup("");
	char *progname = argv[0];

	try_old_version("insmod", argv);

	if (argv[1] && (streq(argv[1], "--version") || streq(argv[1], "-V"))) {
		puts(PACKAGE " version " VERSION);
		exit(0);
	}

	/* Ignore old options, for backwards compat. */
	while (argv[1] && (streq(argv[1], "-p")
			   || streq(argv[1], "-s")
			   || streq(argv[1], "-f"))) {
		argv++;
		argc--;
	}

	filename = argv[1];
	if (!filename)
		print_usage(progname);

	/* Rest is options */
	for (i = 2; i < argc; i++) {
		options = realloc(options,
				  strlen(options) + 2 + strlen(argv[i]) + 2);
		/* Spaces handled by "" pairs, but no way of escaping
                   quotes */
		if (strchr(argv[i], ' '))
			strcat(options, "\"");
		strcat(options, argv[i]);
		if (strchr(argv[i], ' '))
			strcat(options, "\"");
		strcat(options, " ");
	}

	fd = open(filename, O_RDONLY, 0);
	if (fd < 0) {
		fprintf(stderr, "Can't open '%s': %s\n",
			filename, strerror(errno));
		exit(1);
	}

	fstat(fd, &st);
	len = st.st_size;
	map = mmap(NULL, len, PROT_READ, MAP_SHARED, fd, 0);
	if (map == MAP_FAILED) {
		fprintf(stderr, "Can't map '%s': %s\n",
			filename, strerror(errno));
		exit(1);
	}

	ret = syscall(__NR_init_module, map, len, options);
	if (ret != 0) {
		fprintf(stderr, "Error inserting '%s': %li %s\n",
			filename, ret, moderror(errno));
		exit(1);
	}
	exit(0);
}
