//
//	SubjectAltNameTest.cs - NUnit Test Cases for 
//	Mono.Security.X509.Extensions.SubjectAltName
//
// Authors:
//	David Wolinsky <davidiw@ufl.edu>
//
// Copyright  (C) 2008 
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Security.Cryptography;
using System.IO;

using Mono.Security;
using Mono.Security.X509;
using Mono.Security.X509.Extensions;

using NUnit.Framework;

namespace MonoTests.Mono.Security.X509.Extensions {

	[TestFixture]
	public class SubjectAltNameTest {

		[Test]
		public void SubjectAltNameGenerator ()
		{
			RSACryptoServiceProvider rsa = new RSACryptoServiceProvider ();
			X509CertificateBuilder x509 = new X509CertificateBuilder ();
			x509.IssuerName = "C=ZA, ST=Western Cape, L=Cape Town, O=Thawte Consulting cc, OU=Certification Services Division, CN=Thawte Server";
			x509.NotAfter = DateTime.MaxValue;
			x509.NotBefore = DateTime.MinValue;
			x509.SubjectName = "C=US, ST=Maryland, L=Pasadena, O=Brent Baccala, OU=FreeSoft, CN=www.freesoft.org";
			x509.SerialNumber = new byte[] {12, 34, 56, 78, 90};
			x509.Version = 3;
			x509.SubjectPublicKey = rsa;

			string[] dns = new string[2];
			dns[0] = "one";
			dns[1] = "two";
			string[] uris = new string[3];
			uris[0] = "one:two://three";
			uris[1] = "Here:I:AM://12345";
			uris[2] = "last:one";
			SubjectAltNameExtension sane = new SubjectAltNameExtension (null, dns, null, uris);
			x509.Extensions.Add (sane);
			byte[] data = x509.Sign (rsa);

			X509Certificate x509_loaded = new X509Certificate (data);
			SubjectAltNameExtension	sane_test = new SubjectAltNameExtension (x509_loaded.Extensions[0]);
			Assert.AreEqual (sane_test.RFC822.Length, 0, "RFC822 count");
			Assert.AreEqual (sane_test.DNSNames.Length, 2, "DNSName count");
			Assert.AreEqual (sane_test.IPAddresses.Length, 0, "IPAddresses count");
			Assert.AreEqual (sane_test.UniformResourceIdentifiers.Length, 3, "URI Count");
			Assert.AreEqual (sane_test.DNSNames[1], "two", "DNSName test");
			Assert.AreEqual (sane_test.UniformResourceIdentifiers[2], "last:one", "URI Test");
		}

		[Test]
		public void ThirdPartyCertificateParse  ()
		{
			byte[] certificate_with_ip_subjectaltname = new byte[] {0x30, 0x82, 0x02, 0x78, 0x30, 0x82, 0x01, 0xE1, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0xEE, 0xF3, 0xC0, 0x32, 0x13, 0x12, 0x66, 0x06, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x78, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x18, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4C, 0x74, 0x64, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x31, 0x0D, 0x30, 0x0B, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x31, 0x13, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x04, 0x6E, 0x6F, 0x6E, 0x65, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x38, 0x30, 0x36, 0x31, 0x33, 0x30, 0x34, 0x35, 0x39, 0x34, 0x36, 0x5A, 0x17, 0x0D, 0x30, 0x39, 0x30, 0x36, 0x31, 0x33, 0x30, 0x34, 0x35, 0x39, 0x34, 0x36, 0x5A, 0x30, 0x6F, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0A, 0x53, 0x6F, 0x6D, 0x65, 0x2D, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x18, 0x49, 0x6E, 0x74, 0x65, 0x72, 0x6E, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4C, 0x74, 0x64, 0x31, 0x0A, 0x30, 0x08, 0x06, 0x03, 0x55, 0x04, 0x0B, 0x13, 0x01, 0x61, 0x31, 0x0A, 0x30, 0x08, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x01, 0x61, 0x31, 0x10, 0x30, 0x0E, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x01, 0x16, 0x01, 0x61, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xA6, 0xD2, 0x03, 0x5D, 0x91, 0x3D, 0xF3, 0x04, 0x4F, 0xB9, 0xF0, 0xE6, 0x40, 0xD0, 0x4E, 0xDF, 0xF7, 0xCE, 0x35, 0x87, 0xC1, 0x95, 0xEA, 0xFD, 0xDF, 0x44, 0x46, 0x20, 0xE4, 0xAF, 0x69, 0xC8, 0x1C, 0xF1, 0x06, 0x6C, 0x46, 0x20, 0x4D, 0xAA, 0xCC, 0x86, 0x40, 0xBB, 0x79, 0xF8, 0x71, 0x22, 0x87, 0x65, 0xBD, 0x20, 0x1D, 0xAD, 0xC6, 0xB0, 0x7C, 0x17, 0xE6, 0x57, 0xE4, 0x3C, 0x55, 0xD7, 0x7C, 0x8A, 0x98, 0xA2, 0xCD, 0x22, 0x85, 0x0E, 0xA2, 0x90, 0x18, 0x44, 0xA9, 0x7F, 0xA6, 0xD8, 0xDB, 0x6D, 0x09, 0x6D, 0x48, 0x6D, 0xD0, 0xDF, 0x94, 0xBF, 0x3B, 0xE5, 0xDA, 0x5F, 0xA2, 0x6F, 0x3C, 0xE5, 0xCE, 0xF3, 0x53, 0x8B, 0x16, 0x39, 0xDD, 0x3B, 0xC7, 0x0E, 0xA5, 0x75, 0xAA, 0x5A, 0x51, 0xD5, 0x7F, 0x31, 0x44, 0xC8, 0x6A, 0x9B, 0x60, 0xEC, 0xA2, 0xB6, 0x42, 0xCA, 0xA3, 0x43, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x13, 0x30, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x08, 0x30, 0x06, 0x87, 0x04, 0xC0, 0xA8, 0x6F, 0x6F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x87, 0x23, 0x47, 0x07, 0x99, 0x69, 0x90, 0x8D, 0x65, 0xF9, 0xE4, 0xF3, 0x03, 0xBB, 0x08, 0x67, 0x29, 0x38, 0x0E, 0xF4, 0x2E, 0x23, 0xFF, 0xC2, 0x05, 0x1C, 0x7B, 0xDD, 0xBD, 0xA6, 0x26, 0x46, 0x99, 0x26, 0xC4, 0x8C, 0xD5, 0xFC, 0x4A, 0x39, 0xE6, 0xF5, 0xD7, 0x9F, 0xE5, 0x80, 0x60, 0x01, 0x30, 0x32, 0xC1, 0x86, 0x5C, 0x2E, 0x7F, 0x01, 0xB2, 0xAE, 0x4D, 0x15, 0xBB, 0x9C, 0xB8, 0xC4, 0xF6, 0x18, 0x48, 0x5F, 0xEF, 0x35, 0x78, 0xE5, 0x7F, 0x35, 0x10, 0xA1, 0xDD, 0x8E, 0x69, 0xCA, 0x52, 0x99, 0xBC, 0x89, 0x42, 0x2F, 0xC3, 0xEF, 0xB6, 0xD1, 0x37, 0xE2, 0xF9, 0x68, 0xC1, 0x3C, 0x10, 0x8C, 0xDE, 0x7A, 0xC9, 0x31, 0x3B, 0x4E, 0xAC, 0x44, 0xA1, 0x9F, 0x7C, 0xA7, 0x41, 0x59, 0xE5, 0x77, 0x12, 0xCB, 0xBE, 0xBB, 0x0F, 0x50, 0x5A, 0x14, 0x3B, 0xA7, 0x86, 0x15, 0x5C, 0x61, 0x0A};

			X509Certificate cert = new X509Certificate (certificate_with_ip_subjectaltname);
			SubjectAltNameExtension	sane_test = new SubjectAltNameExtension (cert.Extensions[0]);
			Assert.AreEqual (sane_test.RFC822.Length, 0, "RFC822 count");
			Assert.AreEqual (sane_test.DNSNames.Length, 0, "DNSName count");
			Assert.AreEqual (sane_test.IPAddresses.Length, 1, "IPAddresses count");
			Assert.AreEqual (sane_test.UniformResourceIdentifiers.Length, 0, "URI Count");
			Assert.AreEqual (sane_test.IPAddresses[0], "192.168.111.111", "IPAddress Test");
		}
	}
}
