/*
 * AC-3 and E-AC-3 decoder tables
 * Copyright (c) 2007 Bartlomiej Wolowiec <bartek.wolowiec@gmail.com>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file ac3dec_data.c
 * tables taken directly from the AC-3 spec.
 */

#include "ac3dec_data.h"
#include "ac3.h"

/**
 * table used to ungroup 3 values stored in 5 bits.
 * used by bap=1 mantissas and GAQ.
 * ff_ac3_ungroup_3_in_5_bits_tab[i] = { i/9, (i%9)/3, (i%9)%3 }
 */
const uint8_t ff_ac3_ungroup_3_in_5_bits_tab[32][3] = {
    { 0, 0, 0 }, { 0, 0, 1 }, { 0, 0, 2 }, { 0, 1, 0 },
    { 0, 1, 1 }, { 0, 1, 2 }, { 0, 2, 0 }, { 0, 2, 1 },
    { 0, 2, 2 }, { 1, 0, 0 }, { 1, 0, 1 }, { 1, 0, 2 },
    { 1, 1, 0 }, { 1, 1, 1 }, { 1, 1, 2 }, { 1, 2, 0 },
    { 1, 2, 1 }, { 1, 2, 2 }, { 2, 0, 0 }, { 2, 0, 1 },
    { 2, 0, 2 }, { 2, 1, 0 }, { 2, 1, 1 }, { 2, 1, 2 },
    { 2, 2, 0 }, { 2, 2, 1 }, { 2, 2, 2 }, { 3, 0, 0 },
    { 3, 0, 1 }, { 3, 0, 2 }, { 3, 1, 0 }, { 3, 1, 1 }
};

const uint8_t ff_eac3_hebap_tab[64] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 8,
    8, 8, 9, 9, 9, 10, 10, 10, 10, 11,
    11, 11, 11, 12, 12, 12, 12, 13, 13, 13,
    13, 14, 14, 14, 14, 15, 15, 15, 15, 16,
    16, 16, 16, 17, 17, 17, 17, 18, 18, 18,
    18, 18, 18, 18, 18, 19, 19, 19, 19, 19,
    19, 19, 19, 19,
};

const uint8_t ff_eac3_bits_vs_hebap[20] = {
    0,  2,  3,  4,  5,  7,  8,  9,  3,  4,
    5,  6,  7,  8,  9, 10, 11, 12, 14, 16,
};

/**
 * Table E3.6, Gk=1
 * No gain (Gk=1) inverse quantization, remapping scale factors
 * ff_eac3_gaq_remap[hebap+8]
 */
const int16_t ff_eac3_gaq_remap_1[12] = {
    4681, 2185, 1057, 520, 258, 129, 64, 32, 16, 8, 2, 0
};

/**
 * Table E3.6, Gk=2 & Gk=4, A
 * Large mantissa inverse quantization, remapping scale factors
 * ff_eac3_gaq_remap_2_4_a[hebap-8][Gk=2,4]
 */
const int16_t ff_eac3_gaq_remap_2_4_a[9][2] = {
    { -10923, -4681 },
    { -14043, -6554 },
    { -15292, -7399 },
    { -15855, -7802 },
    { -16124, -7998 },
    { -16255, -8096 },
    { -16320, -8144 },
    { -16352, -8168 },
    { -16368, -8180 }
};

/**
 * Table E3.6, Gk=2 & Gk=4, B
 * Large mantissa inverse quantization, negative mantissa remapping offsets
 * Table values from the spec are right-shifted by 8 to simplify calculations.
 * ff_eac3_gaq_remap_3_4_b[hebap-8][Gk=2,4]
 */
const int8_t ff_eac3_gaq_remap_2_4_b[9][2] = {
    { -22,  -5 },
    { -46, -20 },
    { -56, -26 },
    { -60, -29 },
    { -62, -31 },
    { -63, -32 },
    { -64, -32 },
    { -64, -32 },
    { -64, -32 },
};

static const int16_t vq_hebap1[4][6] = {
{   7167,    4739,    1106,    4269,   10412,    4820},
{  -5702,   -3187,  -14483,   -1392,   -2027,     849},
{    633,    6199,    7009,  -12779,   -2306,   -2636},
{  -1468,   -7031,    7592,   10617,   -5946,   -3062},
};
static const int16_t vq_hebap2[8][6] = {
{ -12073,     608,   -7019,     590,    4000,     869},
{   6692,   15689,   -6178,   -9239,     -74,     133},
{   1855,    -989,   20596,   -2920,   -4475,     225},
{  -1194,   -3901,    -821,   -6566,    -875,  -20298},
{  -2762,   -3181,   -4094,   -5623,  -16945,    9765},
{   1547,    6839,    1980,   20233,   -1071,   -4986},
{   6221,  -17915,   -5516,    6266,     358,    1162},
{   3753,   -1066,    4283,   -3227,   15928,   10186},
};
static const int16_t vq_hebap3[16][6] = {
{ -10028,   20779,   10982,   -4560,     798,     -68},
{  11050,   20490,   -6617,   -5342,   -1797,   -1631},
{   3977,    -542,    7118,   -1166,   18844,   14678},
{  -4320,     -96,   -7295,    -492,  -22050,   -4277},
{   2692,    5856,    5530,   21862,   -7212,   -5325},
{   -135,  -23391,     962,    8115,    -644,     382},
{  -1563,    3400,   -3299,    4693,   -6892,   22398},
{   3535,    3030,    7296,    6214,   20476,  -12099},
{     57,   -6823,    1848,  -22349,   -5919,    6823},
{   -821,   -3655,    -387,   -6253,   -1735,  -22373},
{  -6046,    1586,  -18890,  -14392,    9214,     705},
{  -5716,     264,  -17964,   14618,    7921,    -337},
{   -110,     108,       8,      74,     -89,     -50},
{   6612,   -1517,   21687,   -1658,   -7949,    -246},
{  21667,   -6335,   -8290,    -101,   -1349,     -22},
{ -22003,   -6476,    7974,     648,    2054,    -331},
};
static const int16_t vq_hebap4[32][6] = {
{   6636,   -4593,   14173,  -17297,  -16523,     864},
{   3658,   22540,     104,   -1763,     -84,       6},
{  21580,  -17815,   -7282,   -1575,   -2078,    -320},
{  -2233,   10017,   -2728,   14938,  -13640,  -17659},
{  -1564,  -17738,  -19161,   13735,    2757,    2951},
{   4520,    5510,    7393,   10799,   19231,  -13770},
{    399,    2976,   -1099,    5013,   -1159,   22095},
{   3624,   -2359,    4680,   -2238,   22702,    3765},
{  -4201,   -8285,   -6810,  -12390,  -18414,   15382},
{  -5198,   -6869,  -10047,   -8364,  -16022,  -20562},
{   -142,  -22671,    -368,    4391,    -464,     -13},
{    814,   -1118,   -1089,  -22019,      74,    1553},
{  -1618,   19222,  -17642,  -13490,     842,   -2309},
{   4689,   16490,   20813,  -15387,   -4164,   -3968},
{  -3308,   11214,  -13542,   13599,  -19473,   13770},
{   1817,     854,   21225,    -966,   -1643,    -268},
{  -2587,    -107,  -20154,     376,    1174,    -304},
{  -2919,     453,   -5390,     750,  -22034,    -978},
{ -19012,   16839,   10000,   -3580,    2211,    1459},
{   1363,   -2658,     -33,   -4067,    1165,  -21985},
{  -8592,   -2760,  -17520,  -15985,   14897,    1323},
{    652,   -9331,    3253,  -14622,   12181,   19692},
{  -6361,    5773,  -15395,   17291,   16590,   -2922},
{   -661,    -601,    1609,   22610,     992,   -1045},
{   4961,    9107,   11225,    7829,   16320,   18627},
{ -21872,   -1433,     138,    1470,   -1891,    -196},
{ -19499,  -18203,   11056,    -516,    2543,   -2249},
{  -1196,  -17574,   20150,   11462,    -401,    2619},
{   4638,   -8154,   11891,  -15759,   17615,  -14955},
{    -83,     278,     323,      55,    -154,     232},
{   7788,    1462,   18395,   15296,  -15763,   -1131},
};
static const int16_t vq_hebap5[128][6] = {
{  -3394,  -19730,    2963,    9590,    4660,   19673},
{ -15665,   -6405,   17671,    3860,   -8232,  -19429},
{   4467,     412,  -17873,   -8037,     691,  -17307},
{   3580,    2363,    6886,    3763,    6379,  -20522},
{ -17230,  -14133,   -1396,  -23939,    8373,  -12537},
{  -8073,  -21469,  -15638,    3214,    8105,   -5965},
{   4343,    5169,    2683,  -16822,   -5146,  -16558},
{   6348,  -10668,   12995,  -25500,  -22090,    4091},
{  -2880,   -8366,   -5968,  -17158,   -2638,   23132},
{  -5095,  -14281,  -22371,   21741,    3689,    2961},
{  -2443,  -17739,   25155,    2707,    1594,       7},
{ -18379,    9010,    4270,     731,    -426,    -640},
{ -23695,   24732,    5642,     612,    -308,    -964},
{   -767,    1268,     225,    1635,     173,     916},
{   5455,    6493,    4902,   10560,   23041,  -17140},
{  17219,  -21054,  -18716,    4936,   -3420,    3357},
{  -1390,   15488,  -21946,  -14611,    1339,     542},
{  -6866,   -2254,  -12070,   -3075,  -19981,  -20622},
{  -1803,   11775,    1343,    8917,     693,   24497},
{ -21610,    9462,    4681,    9254,   -7815,   15904},
{  -5559,   -3018,   -9169,   -1347,  -22547,   12868},
{   -366,    5076,   -1727,   20427,    -283,   -2923},
{  -1886,   -6313,    -939,   -2081,   -1399,    3513},
{  -3161,    -537,   -5075,   11268,   19396,     989},
{   2345,    4153,    5769,   -4273,     233,    -399},
{ -21894,   -1138,  -16474,    5902,    5488,   -3211},
{  10007,  -12530,   18829,   20932,   -1158,    1790},
{  -1165,    5014,   -1199,    6415,   -8418,  -21038},
{   1892,   -3534,    3815,   -5846,   16427,   20288},
{  -2664,  -11627,   -4147,  -18311,  -22710,   14848},
{  17256,   10419,    7764,   12040,   18956,    2525},
{ -21419,  -18685,  -10897,    4368,   -7051,    4539},
{  -1574,    2050,    5760,   24756,   15983,   17678},
{   -538,  -22867,   11067,   10301,     385,     528},
{  -8465,   -3025,  -16357,  -23237,   16491,    3654},
{   5840,     575,   11890,    1947,   25157,    6653},
{   6625,   -3516,   -1964,    3850,    -390,    -116},
{  18005,   20900,   14323,   -7621,  -10922,   11802},
{  -4857,   -2932,  -13334,   -7815,   21622,    2267},
{   -579,   -9431,    -748,  -21321,   12367,    8265},
{  -8317,    1375,  -17847,    2921,    9062,   22046},
{  18398,    8635,   -1503,   -2418,  -18295,  -14734},
{  -2987,   15129,   -3331,   22300,   13878,  -13639},
{   5874,  -19026,   15587,   11350,  -20738,    1971},
{   1581,   -6955,  -21440,    2455,      65,     414},
{    515,   -4468,    -665,   -4672,     125,  -19222},
{  21495,  -20301,   -1872,   -1926,    -211,   -1022},
{   5189,  -12250,   -1775,  -23550,   -4546,    5813},
{    321,   -6331,   14646,    6975,   -1773,     867},
{ -13814,    3180,    7927,     444,   19552,    3146},
{  -6660,   12252,   -1972,   17408,  -24280,  -12956},
{   -745,   14356,   -1107,   23742,   -9631,  -18344},
{  18284,   -7909,   -7531,   19118,    7721,  -12659},
{   1926,   15101,  -12848,    2153,   21631,    1864},
{  -2130,   23416,   17056,  -15597,   -1544,      87},
{   8314,  -11824,   14581,  -20591,    7891,   -2099},
{  19600,   22814,  -17304,   -2040,     285,   -3863},
{  -8214,  -18322,   10724,  -13744,  -13469,   -1666},
{  14351,    4880,  -20034,     964,   -4221,    -180},
{ -24598,  -16635,   19724,    5925,    4777,    4414},
{  -2495,   23493,  -16141,    2918,   -1038,   -2010},
{  18974,   -2540,   13343,    1405,   -6194,   -1136},
{   2489,   13670,   22638,   -7311,    -129,   -2792},
{ -13962,   16775,   23012,     728,    3397,     162},
{   3038,     993,    8774,  -21969,   -6609,     910},
{ -12444,  -22386,   -2626,   -5295,   19520,    9872},
{  -1911,  -18274,  -18506,  -14962,    4760,    7119},
{   8298,   -2978,   25886,    7660,   -7897,    1020},
{   6132,   15127,   18757,  -24370,   -6529,   -6627},
{   7924,   12125,   -9459,  -23962,    5502,     937},
{ -17056,   -5373,    2522,     327,    1129,    -390},
{  15774,   19955,  -10380,   11172,   -3107,   14853},
{ -11904,   -8091,  -17928,  -22287,  -17237,   -6803},
{ -12862,   -2172,   -6509,    5927,   12458,  -22355},
{   -497,     322,    1038,   -6643,   -5404,   20311},
{   1083,  -22984,   -8494,   12130,    -762,    2623},
{   5067,   19712,   -1901,     -30,    -325,      85},
{    987,   -5830,    4212,   -9030,    9121,  -25038},
{  -7868,    7284,  -12292,   12914,  -21592,   20941},
{  -1630,   -7694,   -2187,   -8525,   -5604,  -25196},
{  -6668,     388,  -22535,    1526,    9082,     193},
{  -7867,  -22308,    5163,     362,     944,    -259},
{   3824,  -11850,    7591,  -23176,   25342,   23771},
{ -10504,    4123,  -21111,   21173,   22439,    -838},
{  -4723,   21795,    6184,    -122,    1642,    -717},
{  24504,   19887,   -2043,     986,       7,     -55},
{ -27313,    -135,    2437,     259,      89,     307},
{  24446,   -3873,   -5391,    -820,   -2387,     361},
{   5529,    5784,   18682,     242,  -21896,   -4003},
{  22304,    4483,     722,  -12242,    7570,   15448},
{   8673,    3009,   20437,   21108,  -21100,   -3080},
{  -1132,    2705,   -1825,    5420,    -785,   18532},
{  16932,  -13517,  -16509,  -14858,  -20327,  -14221},
{   2219,    1380,   21474,   -1128,     327,      83},
{  -2177,   21517,   -3856,  -14180,    -204,   -2191},
{    953,   -9426,   15874,  -10710,   -3231,   21030},
{   -421,   -1377,     640,   -8239,  -20976,    2174},
{   4309,   18514,   -9100,  -18319,  -15518,    3704},
{  -5943,     449,   -8387,    1075,  -22210,   -4992},
{   2953,   12788,   18285,    1430,   14937,   21731},
{  -2913,     401,   -4739,  -20105,    1699,   -1147},
{   3449,    5241,    8853,   22134,   -7547,    1451},
{  -2154,    8584,   18120,  -15614,   19319,   -5991},
{   3501,    2841,    5897,    6397,    8630,   23018},
{   2467,    2956,     379,    5703,  -22047,   -2189},
{ -16963,    -594,   18822,   -5295,    1640,     774},
{   2896,   -1424,    3586,   -2292,   19910,   -1822},
{ -18575,   21219,  -14001,  -12573,   16466,     635},
{  -1998,  -19314,  -16527,   12208,  -16576,   -7854},
{  -9674,    1012,  -21645,    2883,  -12712,    2321},
{  -1005,     471,   -3629,    8045,  -11087,   25533},
{   4141,  -21472,   -2673,     756,    -663,    -523},
{   6490,    8531,   19289,   18949,    6092,   -9347},
{  16965,   24599,   14024,   10072,    -536,  -10438},
{  -8147,    2145,  -23028,  -17073,    5451,   -4401},
{ -14873,   20520,  -18303,   -9717,  -11885,  -17831},
{  -2290,  -14120,    2070,   22467,    1671,     725},
{  -8538,   14629,    3521,  -20577,    6673,    8200},
{  20248,    4410,   -1366,    -585,    1229,   -2449},
{   7467,   -7148,   13667,   -8246,   22392,  -17320},
{  -1932,    3875,   -9064,   -3812,     958,     265},
{  -4399,    2959,  -15911,   19598,    4954,   -1105},
{  18009,   -9923,  -18137,   -3862,   11178,    5821},
{ -14596,   -1227,    9660,   21619,   11228,  -11721},
{   -721,   -1700,     109,   -2142,      61,   -6772},
{ -24619,  -22520,    5608,   -1957,   -1761,   -1012},
{ -23728,   -4451,   -2688,  -14679,   -4266,    9919},
{   8495,    -894,   20438,  -13820,  -17267,     139},
};
static const int16_t vq_hebap6[256][6] = {
{  10154,    7365,   16861,   18681,  -22893,   -3636},
{  -2619,   -3788,   -5529,   -5192,   -9009,  -20298},
{  -5583,  -22800,   21297,    7012,     745,     720},
{    428,   -1459,     109,   -3082,     361,   -8403},
{   8161,   22401,     241,    1755,    -874,   -2824},
{   1140,   12643,    2306,   22263,  -25146,  -17557},
{  -2609,    3379,   10337,  -19730,  -15468,  -23944},
{  -4040,  -12796,  -25772,   13096,    3905,    1315},
{   4624,  -23799,   13608,   25317,   -1175,    2173},
{    -97,   13747,   -5122,   23255,    4214,  -22145},
{   6878,    -322,   18264,    -854,  -11916,    -733},
{  17280,  -12669,   -9693,   23563,  -16240,   -1309},
{   5802,   -4968,   19526,  -21194,  -24622,    -183},
{   5851,  -16137,   15229,   -9496,   -1538,     377},
{  14096,   25057,   13419,    8290,   23320,   16818},
{  -7261,     118,  -15867,   19097,    9781,    -277},
{  -4288,   21589,  -13288,  -16259,   16633,   -4862},
{   4909,  -19217,   23411,   14705,    -722,     125},
{  19462,   -4732,   -1928,  -11527,   20770,    5425},
{ -27562,   -2881,   -4331,     384,   -2103,    1367},
{   -266,   -9175,    5441,   26333,   -1924,    4221},
{  -2970,  -20170,  -21816,    5450,   -7426,    5344},
{   -221,   -6696,     603,   -9140,    1308,  -27506},
{   9621,   -8380,   -1967,    9403,   -1651,   22817},
{   7566,   -5250,   -4165,    1385,    -990,     560},
{  -1262,   24738,  -19057,   10741,    7585,   -7098},
{    451,   20130,   -9949,   -6015,   -2188,   -1458},
{  22249,    9380,    9096,   10959,   -2365,   -3724},
{  18668,    -650,   -1234,   11092,    7678,    5969},
{  19207,   -1485,   -1076,    -731,    -684,      43},
{  -4973,   13430,   20139,      60,     476,    -935},
{ -20029,    8710,    2499,    1016,   -1158,     335},
{ -26413,   18598,   -2201,    -669,    3409,     793},
{  -4726,    8875,  -24607,   -9646,    3643,    -283},
{  13303,  -21404,   -3691,   -1184,   -1970,    1612},
{    173,      60,     919,    1229,    6942,    -665},
{  16377,   16991,    5341,  -14015,   -2304,  -20390},
{  25334,  -10609,   11947,   -7653,   -6363,   14058},
{  23929,  -13259,   -7226,    -937,     234,    -187},
{   6311,   -1877,   12506,   -1879,   18751,  -23341},
{    621,    6445,    3354,  -24274,    8406,    5315},
{  -3297,   -5034,   -4704,   -5080,  -25730,    5347},
{  -1275,  -13295,    -965,  -23318,    1214,   26259},
{  -6252,   10035,  -20105,   15301,  -16073,    5136},
{   9562,   -3911,  -19510,    4745,   22270,   -4171},
{   7978,  -19600,   14024,   -5745,  -20855,    8939},
{      7,   -4039,     991,   -6065,      52,  -19423},
{   3485,    2969,    7732,    7786,   25312,    6206},
{   -959,  -12812,   -1840,  -22743,    7324,   10830},
{  -4686,    1678,  -10172,   -5205,    4294,   -1271},
{   3889,    1302,    7450,     638,   20374,   -3133},
{ -12496,   -9123,   18463,  -12343,   -7238,   18552},
{  -6185,    8649,   -6903,    -895,   17109,   16604},
{  -9896,   28579,    2845,    1640,    2925,    -298},
{  14968,  -25988,   14878,  -24012,    1815,   -6474},
{  26107,    5166,   21225,   15873,   21617,   14825},
{ -21684,   16438,   20504,  -14346,   -7114,   -4162},
{  28647,      90,   -1572,     789,    -902,     -75},
{  -1479,    2471,   -4061,    3612,   -2240,   10914},
{   8616,   17491,   17255,  -17456,   17022,  -16357},
{ -20722,  -18597,   25274,   17720,   -3573,    1695},
{   -997,    6129,   -6303,   11250,  -11359,  -19739},
{    -74,   -4001,   -1584,   13384,     162,    -144},
{   -529,   21068,    7923,  -11396,     422,     -26},
{   7102,  -13531,  -20055,    2629,    -178,    -429},
{   9201,    1368,  -22238,    2623,  -20499,   24889},
{   -432,    6675,    -266,    8723,      80,   28024},
{  19493,   -3108,   -9261,    1910,  -21777,    5345},
{  14079,  -11489,   12604,    6079,   19877,    1315},
{  10947,    9837,  -18612,   15742,    4792,     605},
{  -1777,    3758,   -4087,   21696,    6024,    -576},
{   3567,   -3578,   16379,    2680,   -1752,     716},
{  -5049,   -1399,   -4550,    -652,  -17721,   -3366},
{  -3635,   -4372,   -6522,  -22152,    7382,    1458},
{  12242,   19190,    5646,   -7815,  -20289,   21344},
{  -7508,   19952,   23542,   -9753,    5669,   -1990},
{  -2275,   15438,   10907,  -17879,    6497,   13582},
{ -15894,  -15646,   -4716,    6019,   24250,   -6179},
{  -2049,   -6856,   -1208,     918,   17735,     -69},
{  -3721,    9099,  -16065,  -23621,    5981,   -2344},
{   7862,   -8918,   24033,   25508,  -11033,    -741},
{ -12588,   19468,   14649,   15451,  -21226,    1171},
{   2102,    1147,    2789,    4096,    2179,    8750},
{ -18214,  -17758,  -10366,   -5203,   -1066,   -3541},
{  -2819,  -19958,  -11921,    6032,    8315,   10374},
{  -9078,   -2100,   19431,     -17,     732,    -689},
{ -14512,  -19224,   -7095,   18727,    1870,   22906},
{   3912,     659,   25597,   -4006,    9619,     877},
{   2616,   22695,   -5770,   17920,    3812,   20220},
{   2561,   26847,   -5245,  -10908,    2256,    -517},
{  -4974,     198,  -21983,   -3608,   22174,  -18924},
{  21308,   -1211,   19144,   16691,   -1588,   11390},
{  -1790,    3959,   -3488,    7003,   -7107,   20877},
{  -6108,  -17955,  -18722,   24763,   16508,    3211},
{  20462,  -24987,  -20361,    4484,   -5111,    -478},
{  -6378,   -1998,  -10229,    -561,  -22039,  -22339},
{   3047,  -18850,    7586,   14743,  -19862,    6351},
{  -5047,    1405,   -9672,    1055,  -21881,   11170},
{   3481,   -9699,    6526,  -16655,   22813,   21907},
{ -18570,   17501,   14664,    1291,    5026,   19676},
{  16134,  -19810,  -16956,  -17939,  -16933,    5800},
{  -8224,    4908,    8935,    2272,   -1140,  -23217},
{   1572,    2753,   -1598,    2143,   -3346,  -21926},
{  -9832,   -1060,  -27818,    1214,    7289,     150},
{     98,    1538,     535,   17429,  -23198,    -901},
{  21340,  -20146,    3297,   -1744,   -8207,  -21462},
{  -4166,   -4633,  -17902,    5478,    1285,     136},
{  18713,   21003,   24818,   11421,    1282,   -4618},
{  -3535,    7636,    -265,    2141,    -829,   -2035},
{  -3184,   19713,    2775,      -2,    1090,     104},
{  -6771,  -20185,    2938,   -2125,     -36,    1268},
{   9560,    9430,    9586,   22100,   13827,    6296},
{   -535,  -20018,    4276,   -1868,    -448,  -17183},
{ -24352,   14244,  -13647,  -21040,    2271,   11555},
{  -2646,   15437,   -4589,   18638,   -4299,    -622},
{ -20064,    4169,   18115,   -1404,   13722,   -1825},
{ -16359,    9080,     744,   22021,     125,   10794},
{   9644,  -14607,  -18479,  -14714,   11174,  -20754},
{   -326,  -23762,    6144,    7909,     602,    1540},
{  -6650,    6634,  -12683,   21396,   20785,   -6839},
{   4252,  -21043,    5628,   18687,   23860,    8328},
{  17986,    5704,   -5245,  -18093,    -555,    3219},
{   6091,   14232,   -5117,  -17456,  -19452,  -11649},
{ -21586,   11302,   15434,   25590,    6777,  -26683},
{  21355,   -8244,    5877,   -3540,    6079,   -2567},
{   2603,   -2455,    5421,  -12286,  -19100,    5574},
{  -1721,  -26393,  -23664,   22904,    -349,    3787},
{   2189,   -1203,    5340,    3249,  -22617,     104},
{  -1664,  -11020,   -2857,  -20723,  -24049,   19900},
{  22873,   -7345,  -18481,  -14616,   -8400,  -12965},
{   3777,    3958,    8239,   20494,   -6991,   -1201},
{   -160,   -1613,    -793,   -8681,     573,     776},
{   4297,   -3786,   20373,    6082,   -5321,  -18400},
{  18745,    2463,   12546,   -7749,   -7734,   -2183},
{  11074,   -4720,   22119,    1825,  -24351,    4080},
{   1503,  -19178,   -1569,      13,    -313,     375},
{    318,    -575,    2544,     178,     102,      40},
{ -15996,  -26897,    5008,    3320,     686,    1159},
{  25755,   26886,     574,   -5930,   -3916,    1407},
{  -9148,   -7665,   -2875,   -8384,  -18663,   26400},
{  -7445,  -18040,  -18396,    8802,   -2252,  -21886},
{   7851,   11773,   27485,  -12847,   -1410,   19590},
{   2240,    5947,   11247,   15980,   -6499,   24280},
{  21673,  -18515,    9771,    6550,   -2730,     334},
{  -4149,    1576,  -11010,      89,  -24429,   -5710},
{   7720,    1478,   21412,  -25025,   -8385,       9},
{  -2448,   10218,  -12756,  -16079,    1161,  -21284},
{  -8757,  -14429,  -22918,  -14812,    2629,   13844},
{  -7252,    2843,   -9639,    2882,  -14625,   24497},
{   -674,   -6530,     414,  -23333,  -21343,     454},
{   2104,   -6312,   10887,   18087,   -1199,     175},
{   -493,    -562,   -2739,     118,   -1074,      93},
{ -10011,   -4075,  -28071,   22180,   15077,    -636},
{  -4637,  -16408,   -9003,  -20418,  -11608,  -20932},
{   4815,   15892,   24238,  -13634,   -3074,   -1059},
{  -6724,    4610,  -18772,  -15283,  -16685,   23988},
{  15349,    -674,   -3682,   21679,    4475,  -12088},
{   4756,    2593,    5354,    6001,   15063,   26490},
{ -23815,  -17251,    6944,     378,     694,     670},
{  23392,   -8839,  -14713,    7544,    -876,   11088},
{   3640,    3336,   22593,   -3495,   -2328,    -113},
{    284,    6914,    3097,   10171,    6638,  -18621},
{   2472,    5976,   11054,  -11936,    -603,    -663},
{  16175,   16441,   13164,   -4043,    4667,    7431},
{  19338,   15534,   -6533,    1681,   -4857,   17048},
{  17027,     532,  -19064,   -1441,   -5130,    1085},
{ -12617,  -17609,    2062,  -25332,   19009,  -16121},
{  10056,  -21000,  -13634,   -2949,   15367,   19934},
{   -648,   -1605,   10046,   -1592,   13296,   19808},
{  -1054,   10744,     538,   24938,    9630,   -9052},
{ -10099,    3042,  -25076,  -24052,   13971,     100},
{   6547,    6907,    7031,   10348,   23775,  -17886},
{ -22793,   -1984,   -1393,   -3330,    9267,   14317},
{ -14346,   -3967,    3042,   16254,  -17303,    9646},
{ -21393,   23628,   16773,     716,    2663,     114},
{ -19016,   -3038,    1574,    -245,    1463,    -793},
{  22410,   23441,  -14637,    -530,   17310,   13617},
{ -11582,    7935,  -13954,   23465,  -24628,   26550},
{  -1045,    3679,   -2218,   10572,   20999,   -3702},
{ -15513,     197,   16718,  -24603,    4945,       5},
{  10781,    4335,   26790,   -9059,  -16152,   -2840},
{  16075,  -24100,   -3933,   -6833,   12645,   -7029},
{   2096,  -25572,   -8370,    6814,      11,    1178},
{ -11848,    -583,   -8889,  -20543,  -10471,    -380},
{  -2487,   24777,  -21639,  -19341,    1660,    -732},
{   2313,   13679,    4085,   24549,   24691,  -21179},
{  -2366,    -504,   -4130,  -10570,   23668,    1961},
{  20379,   17809,   -9506,    3733,  -18954,   -6292},
{  -3856,   16802,    -929,  -20310,  -17739,    6797},
{  12431,    6078,  -11272,  -14450,    6913,   23476},
{   7636,   -1655,   23017,   10719,   -8292,     838},
{  -8559,   -1235,  -18096,    3897,   16093,    1490},
{  -3586,    8276,   15165,   -3791,  -21149,    1741},
{  -4497,   21739,    2366,    -278,   -4792,   15549},
{ -23122,  -13708,    7668,   16232,   24120,   15025},
{ -20043,   12821,  -20160,   16691,  -11655,  -16081},
{ -12601,   20239,    3496,   -2549,   -6745,  -11850},
{   4441,    7812,   20783,   17080,   11523,   -9643},
{  24766,    8494,  -23298,   -3262,   11101,   -7120},
{ -10107,   -7623,  -22152,  -18303,   26645,    9550},
{ -25549,     477,    7874,   -1538,    1123,    -168},
{    470,    9834,    -347,   23945,  -10381,   -9467},
{  -4096,   -9702,   -6856,  -21544,   20845,    7174},
{   5370,    9748,  -23765,   -1190,     512,   -1538},
{  -1006,  -10046,  -12649,   19234,   -1790,    -890},
{  15108,   23620,  -15646,   -2522,   -1203,   -1325},
{  -7406,   -2605,    1095,    -247,    -473,     177},
{   8089,       4,   12424,  -22284,   10405,   -7728},
{  22196,   10775,   -5043,     690,     534,    -212},
{  -3153,   -1418,  -16835,   18426,   15821,   22956},
{   5681,   -2229,    3196,   -3414,  -21817,  -14807},
{     19,     787,    1032,     170,   -8295,    -645},
{   -882,   -2319,  -27105,     432,   -4392,    1499},
{  -1354,  -11819,     -76,  -20380,  -10293,   11328},
{    211,   -4753,   -4675,   -6933,  -13538,   14479},
{   6043,    5260,    -459,    -462,     143,     -65},
{  -2572,    7256,   -3317,    9212,  -23184,   -9990},
{ -24882,   -9532,   18874,    6101,    2429,  -14482},
{   8314,    2277,   14192,    3512,   25881,   22000},
{    208,   20218,    -281,  -24778,     -63,   -1183},
{   1095,   -6034,    2706,  -21935,   -2655,     563},
{     23,   -5930,     243,   -8989,    5345,   20558},
{ -15466,   12699,    4160,   11087,   20621,  -10416},
{  20995,     -85,   -8468,     194,    1003,   -9515},
{ -19637,   -3335,  -14081,    3574,  -23381,    -667},
{  -2076,    3489,   -3192,  -19367,     539,   -1530},
{   7352,  -15213,   22596,   19369,    1043,   16627},
{  -1872,    -413,    1235,   -5276,   -3550,   21903},
{   7931,   -2008,   16968,   -6799,   29393,   -2475},
{ -13589,    8389,  -23636,  -22091,  -14178,  -14297},
{ -11575,  -20090,   16056,   -1848,   15721,    4500},
{   3849,  -16581,   20161,  -21155,    7778,   11864},
{  -6547,   -1273,  -18837,  -11218,   11636,    1044},
{   2528,   -6691,  -17917,  -11362,   -4894,   -1008},
{   1241,    4260,    2319,    6111,    3485,   20209},
{   3014,   -3048,    5316,   -4539,   20831,    8702},
{  -1790,  -14683,     278,   13956,  -10065,  -10547},
{ -22732,   -7957,   -1154,   13821,   -1484,   -1247},
{  -7317,    -615,   13094,   18927,    9897,    1452},
{   2552,   -2338,    3424,   -4630,   11124,  -19584},
{ -11125,  -20553,  -10855,  -10783,  -20767,    6833},
{    984,  -15095,    5775,   25125,    5377,  -19799},
{    517,   13272,   -7458,   -1711,   20612,   -6013},
{ -21417,   13251,  -20795,   13449,   17281,   13104},
{ -15811,  -16248,   23093,   -4037,   -8195,     871},
{    582,   12571,  -21129,  -14766,   -9187,    5685},
{   4318,   -1776,   11425,  -17763,   -9921,     577},
{   6013,   16830,   17655,  -25766,   -4400,   -3550},
{ -13744,  -16541,    3636,   -3330,  -21091,  -15886},
{   6565,  -11147,    8649,  -13114,   23345,  -13565},
{  -2542,   -9046,   -7558,   29240,    3701,    -383},
{ -10612,   24995,    1893,   -8210,   20920,  -16210},
{   5276,   16726,   10659,   19940,   -4799,  -19324},
{   -532,   -9300,   27856,    4965,    -241,     536},
{   -765,  -20706,   -3412,   18870,    2765,    1420},
{  -3059,    2708,  -19022,    -331,    3537,     116},
};
static const int16_t vq_hebap7[512][6] = {
{ -21173,   21893,   10390,   13646,   10718,   -9177},
{ -22519,   -8193,   18328,   -6629,   25518,  -10848},
{   6800,  -13758,  -13278,   22418,   14667,  -20938},
{   2347,   10516,    1125,   -3455,    5569,   27136},
{  -6617,   11851,  -24524,   22937,   20362,   -6019},
{ -21768,   10681,  -19615,  -15021,   -8478,   -2081},
{  -2745,    8684,   -4895,   27739,    7554,  -11961},
{  -1020,    2460,    -954,    4754,    -627,  -16368},
{ -19702,   23097,      75,  -13684,   -2644,    2108},
{   4049,   -2872,    5851,   -4459,   22150,   12560},
{ -21304,  -17129,    -730,    7419,  -11658,  -10523},
{  11332,    1792,   26666,   23518,  -19561,    -491},
{ -17827,  -16777,  -13606,  -14389,  -22029,   -2464},
{   1091,   -5967,   -7975,  -16977,  -20432,  -21931},
{  18388,   -1103,    1933,   13342,  -17463,   18114},
{  22646,   17345,   -9966,   17919,   18274,     698},
{   1484,   20297,   -5754,  -26515,    4941,  -22263},
{  -2603,    4587,   -5842,   18464,    8767,   -2568},
{  -2797,   -1602,   21713,    3099,  -25683,    3224},
{ -19027,    4693,   -5007,    6060,    1972,  -15095},
{  -2189,    9516,    -530,   20669,   -4662,   -8301},
{ -22325,   -8887,    2529,  -11352,    5476,     998},
{  22100,   -5052,    1651,   -2657,    4615,    2319},
{  20855,   -3078,   -3330,    4105,   13470,    3069},
{     85,   17289,   10264,  -14752,     214,      90},
{ -26365,  -18849,  -19352,   19244,  -10218,    9909},
{  -9739,   20497,   -6579,   -6983,    2891,    -738},
{  20575,  -15860,  -22913,    6870,      76,     327},
{   8744,  -12877,  -22945,   -2372,  -19424,   -9771},
{ -12886,   16183,   21084,    3821,     749,  -13792},
{ -15995,   18399,    2391,  -17661,   19484,   -6018},
{   1423,   11734,    4051,   19290,    6857,  -19681},
{  -5200,    9766,   18246,    2463,   18764,   -4852},
{   -597,   19498,    1323,   -9096,    -308,   -1104},
{  -3099,  -25731,  -15665,   25332,    4634,    2635},
{  19623,   -2384,   -7913,   11796,   -9333,  -14084},
{   2642,   26453,  -21091,  -10354,   -1693,   -1711},
{  22031,   21625,   11580,  -22915,   -4141,     129},
{  -6122,    3542,     915,    -261,     -17,    -383},
{   1696,    6704,   -1425,   20838,     857,   -4416},
{   1423,  -15280,   -8550,   -9667,    5210,    5687},
{  -4520,    -613,  -11683,    5618,    4230,     619},
{    937,   -4963,  -14102,  -17104,   -6906,   -5952},
{ -15068,    -481,   -7237,  -14894,   18876,   21673},
{ -25658,    2910,    1143,    -327,    -458,    -995},
{  -9656,    -819,  -24900,    2804,   20225,    1083},
{  -1111,   -3682,   -1788,  -19492,     966,     821},
{   7293,  -21759,   10790,   -7059,  -23293,   -1723},
{   -282,  -11093,     170,  -20950,  -28926,   12615},
{  17938,    3713,   -1563,     885,       5,     564},
{   6116,   22696,    2242,   -6951,    9975,   -6132},
{   4338,   26808,   -3705,    1976,   -1079,   -2570},
{   -661,   -7901,   -2668,  -15194,   17722,    4375},
{  -4174,  -11053,     717,  -22506,    1562,   12252},
{  -6405,   18334,    6103,    6983,    5956,   18195},
{   9851,    5370,   23604,   -6861,   -6569,     -62},
{  21964,   13359,    -683,    3785,    2168,     209},
{  -3569,   -1127,  -19724,   -1544,    1308,    -803},
{  -3083,   16049,  -13791,   -3077,    4294,   23713},
{  -9999,    9943,  -15872,   12934,  -23631,   21699},
{   9722,   22837,   12192,   15091,    5533,    4837},
{   2243,    2099,    1243,    4089,    4748,   12956},
{   4007,   -2468,    3353,   -3092,    8843,   17024},
{   4330,    6127,    5549,    9249,   11226,   28592},
{  -9586,   -8825,     236,    1009,     455,    -964},
{   6829,   19290,   -1018,     200,    1821,     578},
{   5196,     957,   10372,    3330,  -12800,    -127},
{  -3022,   -8193,  -14557,   22061,    5920,    1053},
{  10982,   25942,  -24546,  -23278,  -11905,   -6789},
{  22667,  -11010,    5736,    2567,   23705,  -10253},
{  -3343,   -4233,   -5458,   20667,  -10843,   -3605},
{  -4131,   -3612,    4575,    -829,    -350,    -847},
{  -3303,    3451,   -7398,  -11604,    3023,     455},
{   3200,   -9547,    3202,  -22893,   11184,  -26466},
{ -14093,   -4117,   15382,   14295,  -10915,  -20377},
{   3807,  -11016,   22052,   14370,  -15328,   -7733},
{  -6291,  -17719,   -1560,   12048,  -19805,    -443},
{  -6147,   -4234,    -160,    8363,   22638,   11911},
{  19197,    1175,    7422,   -9875,   -4136,    4704},
{    -72,   -7652,    -112,  -11955,   -3230,   27175},
{   3274,    5963,    7501,  -17019,     866,  -25452},
{    737,    1861,    1833,    2022,    2384,    4755},
{  -5217,    7512,    3323,    2715,    3065,   -1606},
{   4247,     565,    5629,    2497,   18019,   -4920},
{  -2833,  -17920,   -8062,   15738,   -1018,    2136},
{   3050,  -19483,   16930,   29835,  -10222,   15153},
{ -11346,     118,  -25796,  -13761,   15320,    -468},
{  -4824,    4960,   -4263,    1575,  -10593,   19561},
{  -8203,   -1409,    -763,   -1139,    -607,    1408},
{  -2203,  -11415,    2021,   -6388,   -2600,     711},
{   -413,   -2511,    -216,   -3519,  -28267,    1719},
{ -14446,   17050,   13917,   13499,  -25762,  -16121},
{  19228,    7341,  -12301,     682,   -3791,    -199},
{  -4193,   20746,  -15651,   11349,    5860,    -824},
{ -21490,   -3546,      -3,   -1705,   -3959,    9213},
{  15445,   -1876,    2012,  -19627,   16228,   -4845},
{  -2867,   -3733,   -7354,    -175,  -20119,   11174},
{  -3571,  -24587,   19700,    6654,     979,    -654},
{  21820,   -7430,   -6639,  -10767,   -8362,   15543},
{  14827,   17977,   -7204,   -3409,    1906,  -17288},
{   3525,   -3947,   -1415,   -2798,   17648,    2082},
{  -6580,  -15255,  -17913,    1337,   15338,   21158},
{   6210,    9698,   15155,  -24666,  -22507,   -3999},
{  -1740,    -593,    1095,   -7779,   25058,    5601},
{  21415,    -432,   -1658,   -6898,   -1438,  -14454},
{  -6943,     700,  -12139,    -745,  -24187,   22466},
{   6287,    3283,   11006,    3844,   19184,   14781},
{ -22502,   15274,    5443,   -2808,    -970,   -3343},
{   3257,   -3708,    4744,   -8301,   22814,  -10208},
{  24346,  -20970,   19846,     987,  -11958,   -6277},
{   3906,  -19701,   13060,   -1609,   18641,    7466},
{ -26409,  -22549,   16305,    2014,   10975,   18032},
{  -7039,    4655,  -14818,   18739,   15789,    1296},
{   9310,   -1681,   14667,   -3326,   26535,  -11853},
{   5728,    5917,   13400,   10020,   -2236,  -24704},
{   1741,   -6727,   12695,  -22009,    4080,    5450},
{  -2621,    9393,   21143,  -25938,   -3162,   -2529},
{  20672,   18894,  -13939,    6990,   -8260,   15811},
{ -23818,   11183,  -13639,   11868,   16045,    2630},
{  18361,  -10220,     829,     856,   -1010,     157},
{  14400,   -4678,    5153,  -13290,  -27434,  -11028},
{  21613,   11256,   17453,    7604,   13130,    -484},
{      7,    1236,     573,    4214,    5576,   -3081},
{    916,   -9092,    1285,   -8958,    1185,  -28699},
{  21587,   23695,   19116,   -2885,  -14282,   -8438},
{  23414,   -6161,   12978,    3061,   -9351,    2236},
{  -3070,   -7344,  -20140,    5788,     582,    -551},
{  -3993,     315,   -7773,    8224,  -28082,  -12465},
{  13766,  -15357,   19205,  -20624,   13043,  -19247},
{   3777,    -177,    8029,   -1001,   17812,    5162},
{  -7308,   -4327,  -18096,    -620,   -1350,   14932},
{  14756,   -1221,  -12819,  -14922,    -547,   27125},
{   2234,    1708,    2764,    5416,    7986,  -25163},
{   2873,    3636,    3992,    5344,   10142,   21259},
{   1158,    5379,     508,  -10514,     290,   -1615},
{   1114,   24789,   16575,  -25168,    -298,   -2832},
{  -1107,   -6144,   -1918,   -7791,   -2971,  -23276},
{   4016,   10793,   17317,   -4342,  -20982,   -3383},
{  -4494,    -207,   -9951,   -3575,    7947,    1154},
{  -7576,    8117,  -14047,   16982,  -26457,  -27540},
{ -15164,   16096,  -16844,   -8886,  -23720,   15906},
{  24922,    5680,   -1874,     420,     132,     117},
{   -506,  -19310,    -198,     412,    -311,     752},
{  -1906,    3981,   -7688,   16566,  -19291,  -14722},
{   -399,    -729,   -3807,   -4196,  -12395,    7639},
{   3368,    2330,    9092,   23686,  -10290,   -1705},
{  -3148,    2596,   -7986,   14602,   -4807,   16627},
{   8057,    1481,      49,   17205,   24869,    7474},
{ -19304,    -513,   11905,    2346,    5588,    3365},
{  -5063,  -21812,   11370,   10896,    4881,     261},
{   4794,   20577,    5109,   -6025,   -8049,   -1521},
{   8125,  -14756,   20639,  -14918,   23941,   -3650},
{  12451,    1381,    3613,    8687,  -24002,    4848},
{   6726,   10643,   10086,   25217,  -25159,   -1065},
{   6561,   13977,    2911,   21737,   16465,  -26050},
{  -1776,    2575,  -19606,  -16800,    3032,    6679},
{  15012,  -17910,   -8438,  -21554,  -27111,   11808},
{   3448,    -924,  -15913,   -1135,    5126,  -20613},
{   7720,    2226,   17463,    5434,   28942,   17552},
{   1246,   15614,  -11743,   24618,  -17539,    3272},
{   3215,   17950,    2783,    -722,  -22672,    5979},
{  -5678,   -3184,  -26087,   26034,    6583,    3302},
{  20310,   -3555,   -2715,    -444,   -1487,    1526},
{ -20640,  -21970,  -12207,  -25793,    8863,   -1036},
{  17888,     570,  -16102,    8329,   -2553,   15275},
{  -2677,    9950,   -1879,   16477,  -12762,  -29007},
{   -120,   -2221,     219,      97,     365,      35},
{   1270,    -718,    1480,   -2689,    1930,   -7527},
{   1896,    8750,    1906,   18235,  -12692,   -6174},
{  -3733,   13713,   -9882,  -15960,   -1376,   -7146},
{ -10600,    8496,   15967,   -8792,    7532,   20439},
{   3041,  -13457,    1032,  -26952,    5787,   24984},
{  -4590,   -8220,   -9322,   -6112,  -17243,   25745},
{ -17808,    6970,    3752,     626,    -114,    2178},
{   4449,   -4862,    7054,   -5404,    4738,   -2827},
{   4922,    -651,   18939,   -9866,     848,    1886},
{   -336,   -5410,    7234,   20444,   -9583,    -600},
{    781,  -19474,  -12648,    6634,    1414,     450},
{  -3399,  -16770,   11107,   13200,   -5498,   21663},
{  -3265,    4859,   -5961,    7530,  -10837,   28086},
{  10350,  -12901,   25699,   25640,    -639,     351},
{   1163,   18763,   -5466,  -15087,    -145,   -1377},
{ -14477,   27229,  -31383,  -32653,   21439,   -2894},
{  15420,   18823,   22128,   19398,   22583,   13587},
{ -10674,   10710,    5089,   -4756,     909,  -20760},
{ -12948,  -20660,    7410,    2722,    3427,   11585},
{  -1105,   18374,   19731,   -9650,   22442,   19634},
{   -296,   -6798,  -14677,   21603,   19796,   21399},
{ -19350,   -7501,   25446,   13144,    8588,  -25298},
{   3092,  -10618,   20896,    9249,   -3326,    1796},
{   -811,    1449,    3106,    4748,   12073,  -14262},
{ -20720,   14275,   -4332,  -25838,   -5781,  -21149},
{  -5132,   10554,  -14020,  -22150,    2840,    -554},
{  25533,   17648,   14886,  -21074,    2459,   25142},
{  -9370,   -1788,  -12862,   -5870,  -25811,  -11023},
{   6698,     819,   10313,     166,   27581,     523},
{    101,  -19388,    3413,    9638,      64,     806},
{  -2742,  -17931,   -2576,   22818,    8553,    1126},
{   2972,   15203,    1792,   25434,   -5728,  -17265},
{  -1419,    1604,    4398,   11452,    1731,   23787},
{  -5136,    4625,  -10653,   27981,    9897,   -2510},
{ -10528,  -28033,    2999,   -1530,    -832,    -830},
{ -11133,  -12511,   22206,   -7243,  -23578,  -21698},
{  16935,  -21892,    1861,   -9606,    9432,   19026},
{  10277,    9516,   26815,    2010,   -4943,   -9080},
{   5547,   -2210,   14270,  -15300,  -19316,    1822},
{  -4850,    -783,   -8959,   -3076,  -20056,   -3197},
{   8232,   -2794,  -17752,   13308,    3229,    -991},
{ -12237,   -6581,   10315,   -9552,    2260,  -20648},
{  -7000,    5529,   -7553,   -7490,  -10342,  -10266},
{   3641,   19479,   -5972,  -19097,  -18570,   12805},
{   1283,   -4164,    4198,  -28473,   -2498,    1866},
{  16047,   26826,  -13053,   -6316,     985,   -1597},
{   -403,   13680,    6457,   25070,   27124,  -20710},
{ -18070,   -1790,  -24986,    5953,    -954,   26600},
{ -24224,  -15383,   24788,    1953,   -1136,     187},
{  -2289,   12505,  -20738,    -904,   18324,   21258},
{   2658,   -6140,   16179,   22276,    -556,    2154},
{  -6087,   13950,  -25682,  -27713,    4049,   -4795},
{ -21452,   26473,   19435,   -9124,     895,     303},
{ -22200,  -26177,   -6026,   24729,  -22926,   -9030},
{ -14276,  -15982,   23732,  -22851,    9268,   -3841},
{  29482,   21923,   -6213,    1679,   -2059,   -1120},
{   -435,    9802,   -3891,   12359,   -4288,  -18971},
{  19768,     -86,    2467,    1990,   -1021,   -5354},
{  20986,   -8783,   -5329,  -23562,   -4730,    2673},
{  -5095,    5605,   -4629,   19150,   26037,  -12259},
{    972,    6858,    4551,   27949,   -4025,   -2272},
{   6075,   -3260,   -4989,    -373,   -1571,   -3730},
{  -7256,  -12992,   -8820,   -5109,   23054,    5054},
{    920,    2615,    7912,   -7353,   -4905,   20186},
{   -250,    5454,    3140,    6928,  -18723,   -2051},
{ -10299,   -4372,   19608,    4879,    -661,   -1885},
{  14816,   -8603,  -19815,    6135,  -21210,   14108},
{ -11945,   -2223,    5018,   11892,   22741,     406},
{ -13184,   -2613,  -13256,  -22433,  -12482,   -8380},
{  17066,   25267,   -2273,    5056,    -342,     145},
{   8401,  -17683,   19112,   10615,  -19453,   17083},
{  20821,   -5700,   12298,  -25598,   10391,    7692},
{   4550,   15779,   17338,  -19379,   -4768,    1206},
{  -7723,   10836,  -27164,  -11439,    6835,   -1776},
{   2542,    3199,    4442,   17513,   -3711,    -914},
{  20960,  -16774,   -5814,   11087,     -70,   22961},
{   3305,    2919,    6256,   -4800,  -20966,   -3230},
{   5924,  -16547,    2183,    2733,    3446,  -23306},
{  -6061,    -194,  -13852,  -10971,   19488,    1029},
{   4467,   -5964,  -19004,    1519,    -359,     855},
{  -1581,   -7607,   22070,  -11580,  -10032,   17102},
{ -12412,    2553,    4324,   22500,    5751,   12170},
{ -25127,   17996,   -6384,    1180,    1182,    9622},
{  23462,   -8471,   -4392,   -2669,    7638,  -16835},
{  -5511,   -2887,  -10757,  -20883,    7246,    1053},
{   2703,  -20602,   -7554,    7516,   -7740,    5868},
{  20670,   21901,     457,   14969,  -17657,  -11921},
{   3603,   -1595,   -2177,    -157,     -43,     605},
{   2513,    8954,   10527,   22559,  -16100,  -16041},
{   6002,    4951,    6795,   -4862,  -22400,   18849},
{   7590,   -1693,  -24688,   -3404,   14169,    1214},
{  -4398,   -6663,   -6870,  -10083,  -24596,    9253},
{  10468,   17751,   -7748,     147,   -6314,    4419},
{  16187,  -16557,   -4119,    4302,    7625,    5409},
{   3303,    2735,    7458,  -19902,   -2254,   -3702},
{  -2077,   21609,   14870,   12545,   -6081,   -1764},
{   4678,   11740,    2859,    6953,    1919,   -3871},
{   3522,  -21853,   -2469,  -10453,   18893,  -10742},
{   3759,  -10191,   -4866,   -2659,  -17831,   -1242},
{  14991,    9351,   11870,   -1573,   -4848,   22549},
{   9509,  -27152,   10734,   20851,  -26185,  -17878},
{  -7170,   -1392,  -19495,   12746,    8198,   -1988},
{   1883,   28158,    -846,   -7235,     249,     233},
{  -7200,     669,    -371,   -2948,   23234,   -5635},
{   3141,     288,    3223,   -1258,     -98,  -27607},
{  17373,  -23235,    5110,  -11199,   -2574,  -11487},
{  -4928,    1518,   -5456,     670,  -18278,    1951},
{  10334,  -19865,   -4649,     361,    -160,    -923},
{  18732,   14264,   -3155,   -7485,   -3328,    5959},
{  -3614,   21077,    7276,    3536,    8121,   -1528},
{  -8422,     500,  -19182,   18929,   26392,   -1039},
{  15639,   25668,    8375,    1903,    1945,  -11979},
{  -2716,    3389,   26850,   -4587,    1803,      22},
{   1177,    -655,    1233,   -2128,    7844,    1767},
{   -761,    8209,  -19290,   -4593,    1923,    -343},
{   -689,   -3530,   -3267,   -3804,   -2753,   18566},
{  -2110,    1962,   -1353,   16643,    2765,  -23102},
{   -433,    4905,     302,   13016,   15933,   -5905},
{   3203,    4126,   11181,   -5496,   -2529,   -1160},
{  -1091,   -6469,   -1415,    5682,    -268,     583},
{  -9405,  -19572,    6216,    1658,     993,     -75},
{  -1695,   -4504,   -2289,   -4088,   -6556,  -16577},
{   4760,    -892,  -10902,    6516,   24199,   -6011},
{   -253,    1000,      63,     -81,    -115,    -382},
{  -1333,   24224,    -698,   -4667,   -2801,  -19144},
{   -876,  -28866,  -21873,   12677,   -6344,    3235},
{  16847,   21145,  -26172,   -3183,    -396,     230},
{  18296,   -7790,  -12857,    -679,   -1473,       5},
{ -10488,   11429,   25805,   -1122,    1401,    -438},
{   3782,   -7429,   26720,   17567,   19257,   12542},
{   6332,    -746,   12789,    9316,  -22542,   -5354},
{   3418,  -22728,   26978,   18303,    1076,     956},
{ -27315,   -2988,     920,     235,    2233,      81},
{   6199,    5296,   16093,   14768,   -8429,   -1112},
{  -6432,   19244,    9921,   -3253,    1278,    -954},
{  24213,    2049,  -22931,    2585,   -2410,   -4216},
{   9286,   14282,  -19735,   -3985,   -2344,    1028},
{ -20128,   17993,   -9458,   23012,  -16983,    8625},
{  -6896,  -20730,    3762,   17415,   22341,   19024},
{    842,   24181,   25062,   -5839,     -78,     937},
{   -621,   19722,  -24204,   -1962,  -14854,     -56},
{  22766,   -5119,   17365,   23868,  -19480,   -6558},
{  -2158,   17490,  -21435,    3340,  -12819,  -20295},
{  -9621,   17325,     715,    2265,   -4123,    -492},
{   9156,   12947,   27303,  -21175,   -6072,   -9457},
{ -13164,  -23269,  -14006,   -4184,    6978,       2},
{    938,  -13381,    3520,  -24297,   22902,   19589},
{  -4911,  -19774,   19764,   -9310,  -12650,    3819},
{  -5462,   -4249,   -6987,   -6260,  -13943,  -25150},
{   9341,   10369,  -13862,   -6704,   22556,    -519},
{   6651,   18768,   -4855,   12570,   14730,  -10209},
{   -823,   18119,     398,   -1582,    -116,    -363},
{  -6935,  -12694,  -28392,    8552,    6961,    -239},
{  -2602,   -4704,   -1021,    2015,    5129,   23670},
{ -12559,   -8190,  -25028,   18544,   14179,    1663},
{   3813,   21036,   -9620,   -5051,   -1800,   -1087},
{ -22057,   16675,   14960,    9459,    2786,   16991},
{ -26040,  -19318,   -6414,    1104,    5798,  -18039},
{  -1737,   24825,   10417,  -11087,     896,   -5273},
{  -1855,   11661,   -2803,   24809,  -21435,  -19792},
{ -23473,  -16729,   -5782,    5643,    2636,    4940},
{  -1724,    4388,  -26673,  -13695,   10570,  -25895},
{  15358,  -19496,   26242,  -18493,    1736,    8054},
{   5684,   20890,    4091,  -19100,  -14588,  -10468},
{  17260,  -16291,   14859,  -17711,  -19174,   12435},
{ -27185,  -12573,    6743,    -562,     976,    -257},
{  12395,   -8618,  -22248,  -19843,   11013,    7762},
{   3799,   11853,  -27622,   -8473,    1089,   -1495},
{   4141,   -2182,  -26720,    -735,    -774,    1469},
{   3125,   13762,    4606,   29257,   18771,   -9958},
{ -17465,   -9445,  -17562,   -2530,   -6435,   -3726},
{  -1742,    4351,   -6841,  -19773,    9627,  -10654},
{   7251,    3525,   10835,    5601,   25198,  -23348},
{ -10300,  -17830,     631,   11640,    2044,  -20878},
{   -873,   -8502,   -1063,  -15674,  -10693,   14934},
{ -15957,   28137,    5268,     477,   -1053,    1158},
{  -1495,   -8814,   -5764,  -24965,   25988,    7907},
{  -1038,    -114,   -2308,   -1319,   -6480,    1472},
{   4895,  -17897,  -25850,    5301,    -188,    1581},
{   3200,   17225,    4346,   22101,  -18543,   22028},
{ -10250,     545,  -10932,    2276,  -28070,    8118},
{  15343,    2329,    9316,   20537,   14908,   21021},
{   6329,    6130,  -24508,     837,   -8637,   -5844},
{   7386,    -501,   10503,   20131,   11435,   -4755},
{  -2745,   24174,   -9274,   15273,   -8389,   -5835},
{   2992,   -2864,    6048,   -7473,   11687,  -19996},
{   -883,  -11954,   -9976,  -21829,   -4436,  -27178},
{   3458,   19626,    1280,    2597,   19849,    5255},
{  -5315,   19133,  -14518,   -8946,   13749,   -1352},
{  18642,   17655,   11001,    6817,  -18418,    6336},
{  -1697,    2244,   -4640,    3948,  -12890,   -5273},
{  20428,   10542,    4170,   -1012,   19439,   21691},
{  -2943,  -19735,   -4208,    1320,     909,   -8897},
{   9351,   -8066,   -2618,  -12933,   26582,    3507},
{   9705,  -22628,    8311,    8167,  -13293,    5608},
{   3222,    3749,   -1508,     165,     -52,    -196},
{    102,  -22744,   -8832,     903,  -11421,  -14662},
{   -120,    5998,   19765,   13401,    3628,    5197},
{   8528,    5827,   -1066,     774,     -39,    -166},
{   9411,   -9476,    9581,  -13004,   24456,   24900},
{  17878,    2235,  -21639,   20478,    4716,   -7190},
{  -2482,    9511,    1611,  -21943,   14230,   -1289},
{   9288,   -2291,   23215,   -3452,  -10842,      11},
{   9496,    3041,    5130,   -3890,  -21219,  -22589},
{  14262,   -9838,   20195,   14019,      91,  -17200},
{ -18591,     980,      17,     821,     120,    -574},
{  12285,  -19269,   13742,   16373,    -161,    6025},
{  -3364,    1530,   -4005,    2454,  -10872,  -23839},
{    105,    5085,    -260,    5790,    -588,   19170},
{   4121,    4169,   13439,   14644,   20899,    7434},
{   -175,   13101,   -3704,   23233,    3907,   10106},
{  -6101,   23467,    5204,   -1341,    1599,   13174},
{  -3217,   -3494,   15117,   -8387,  -11762,   -4750},
{   1146,    4675,  -19378,   14917,   -5091,     249},
{ -21506,   10136,  -16473,  -13305,   18382,   -8601},
{    628,    2447,    3344,    3130,   -5115,     119},
{  17900,  -22422,  -17633,   21967,  -16293,   -7676},
{  16863,   24214,    5612,   -3858,    -809,    3822},
{  -2291,   10091,   -2360,  -25109,   -1226,     312},
{   2957,   11256,   26745,  -13266,   -3455,   -1128},
{ -19762,   -2708,    4604,    6355,    1638,   25501},
{ -19593,   -7753,    3159,     -85,    -489,   -1855},
{    814,   12510,   19077,   -4681,   -2610,   -1474},
{ -23408,  -19027,    8137,   19878,    7912,    -282},
{    839,  -19652,   11927,   27278,   -3211,    2266},
{   4020,   -1110,    8226,   -1274,   20922,   25060},
{  26576,     325,   -8693,    -232,   -2218,    -699},
{ -11293,   -4200,    1805,   -6673,  -22940,   -1339},
{  -2005,  -15886,   -1047,  -27687,  -13235,   14370},
{ -22073,    1949,   13175,  -15656,   -1846,    8055},
{   3039,   12025,    7132,  -24632,     413,   -2347},
{ -24048,    -206,   12459,   -6654,    -417,  -10091},
{  18179,  -23688,  -20515,  -16396,    7230,     763},
{   5659,   -5085,   13878,  -23729,  -11077,  -19587},
{  11340,     501,   25040,    7616,  -19658,    1605},
{ -26650,    8878,   10544,     417,    1299,     261},
{  14460,   11369,   -3263,    9990,    8194,   18111},
{   1355,  -20838,   -9196,  -16060,   -8559,    -730},
{  -1918,  -20937,  -18293,   -2461,   -2651,    4316},
{  -2810,   24521,  -10996,  -25721,     308,   -1234},
{  -9075,  -17280,   -1833,  -29342,  -24213,  -16631},
{  -2843,   10165,   -5339,   -2888,   21858,  -21340},
{ -15832,   14849,  -23780,    5184,   10113,  -20639},
{ -19535,  -11361,    8413,    1486,  -23658,   -5759},
{  -7512,    1027,  -20794,   13732,   19892,  -21934},
{ -12132,   -7022,  -19175,   -8840,   22125,  -16490},
{   1937,    5210,   -6318,  -23788,   13141,   11082},
{   -205,    6036,    -380,    8658,    -233,   28020},
{  -5523,    7477,    7635,   23595,    9763,   -2590},
{  21658,  -28313,   -3086,    -300,   -1032,    1744},
{ -22352,   16646,     208,    6665,  -17400,   -3028},
{  18482,    9336,   -2737,  -19372,     407,   -4389},
{  -4913,  -17370,   18819,  -17654,   13416,   15232},
{   7749,    6368,   23135,  -18174,    7584,   -4248},
{  -1489,   -6523,     586,  -10157,   14964,   25568},
{   3844,   -6156,    4897,  -13045,  -22526,    5647},
{  -8491,   -2105,  -24774,     905,   -9326,    1456},
{  -3040,   -1476,    1166,   -4428,   11236,    9204},
{   3397,   -1451,   13598,  -15841,   24540,    5819},
{   8483,   -2993,   21547,  -16916,    7741,   24018},
{ -14932,  -23758,   -5332,   -6664,   -4497,   13267},
{  19379,   12916,   -2142,    -737,   21100,  -22101},
{   3393,   -4629,    5735,  -18913,   -6969,    2687},
{   1148,  -16147,  -21433,  -28095,    -630,  -14449},
{   7300,     672,   18530,  -17452,  -10149,     351},
{  11356,  -10974,   17212,    4624,     145,   17791},
{   -711,   -3479,   -2238,   15887,    2027,       0},
{ -28048,    1794,    -593,   -2758,  -21852,   11535},
{ -19683,    4937,   22004,   21523,   -3148,    1790},
{    813,    8231,    2633,   11981,   -3043,   22201},
{   8952,  -24760,    -690,   14873,   -2366,   -5372},
{   8406,   -5439,    -274,    -642,    -145,     778},
{  -6605,    7258,   20780,  -23507,  -18625,   22782},
{ -22896,  -25488,   10020,   -1614,    1508,   -1393},
{   7607,     407,  -24678,  -16385,   -1804,   -4699},
{ -10592,  -19139,   10462,   -3747,    8721,   -6919},
{  13010,    5292,   -6230,   -4884,  -20904,   -1797},
{  16891,  -13770,    -465,   19343,  -10741,  -12959},
{  25193,  -14799,   -5681,    -521,    -321,   -1211},
{   6917,   -3093,   20183,  -26903,  -12026,    1295},
{    305,    1992,   19457,    -985,      25,    -521},
{   6707,   -3698,    8365,   -8687,   21921,  -27166},
{   4668,    5997,    7117,   11696,   24401,  -10794},
{    744,   -9416,   19893,    1963,    7922,   -9824},
{   3430,   21282,   -1736,   10844,    8821,   27015},
{  -8813,    1521,  -24038,    1651,    7838,   -1208},
{   3911,  -11221,    3273,  -12541,    7168,   18402},
{  21642,    9117,  -11536,   -5256,    7077,    2382},
{    100,    3817,   -6713,    1244,    1518,    -321},
{   7946,  -18670,   10667,   -4866,     727,     776},
{ -15883,   -8150,   -2087,   22739,    1567,   -3482},
{   4380,   -2735,    8469,   -7025,  -11424,    1317},
{  26970,    4393,    7665,   17561,    -714,     650},
{ -16191,    -835,    8365,    1795,  -14314,   16297},
{   4504,  -10048,    7662,  -26690,  -17428,    2580},
{     48,   -3984,     564,   -5871,    2658,  -18658},
{  12579,  -26016,  -15642,    2672,   -1347,    -887},
{  -4950,    4208,   -6811,    2569,  -20621,   -8658},
{  -1836,  -14818,   -5571,  -23322,  -14800,   25867},
{   5434,  -28139,   -2357,   -2883,    -570,    2431},
{  13096,   -2771,   24994,  -12496,  -24723,   -1025},
{  -5676,   -4339,    1908,   18628,  -21323,   17366},
{  27660,  -27897,  -15409,    1436,   -7112,   -2241},
{   8019,    3847,   24568,    -469,    9674,   10683},
{   -903,  -10149,    1801,  -21260,    4795,   -8751},
{   1122,   -9582,    2625,   22791,     956,     882},
{   7876,   19075,   -9900,  -24266,    7496,    9277},
{    980,  -26764,   -5386,    5396,    1086,    1648},
{  28838,   -1270,    -447,       5,    -429,     -20},
{ -15283,    6132,   22812,    1252,   -9963,     511},
{    851,    7925,    -457,  -12210,    4261,    7579},
{  -4530,    8452,   -1246,   14501,  -24951,   -5760},
{ -17814,  -10727,    9887,  -23929,  -13432,    1878},
{ -15049,   10165,   16491,  -14603,  -11712,  -21156},
{  -3317,     840,   -5683,   22413,    1994,     586},
{  23158,   -5788,  -15043,  -10372,   -9271,  -13523},
{   -773,   -9509,   -3993,  -24264,    8463,    5804},
{  -8545,    -703,  -12440,   -3985,  -25122,  -28147},
{ -16659,   16001,    2746,    1611,    5097,   -1043},
{     41,   -7181,   19903,   31555,  -32237,   13927},
{  -5658,     845,  -12774,    5705,   16695,     -86},
{   5282,   14875,   27026,   21124,   15776,  -10477},
{  14712,   19648,  -11487,  -13361,  -20196,  -15229},
{   8597,   -9138,    -626,   10891,   -6015,    6346},
{  -1488,   -1272,   -1479,   -1303,   -3704,   -5485},
{  -3370,   17871,   -6604,   24930,   25886,   -3127},
{   8416,   27783,   -1385,    5350,   -4260,   19993},
{   5688,     362,   17246,    3809,   -3246,    1088},
{   -105,  -29607,    2747,   15223,    -167,    3722},
{   3502,   -3195,    8602,    7772,   -1566,    -915},
{   -491,    3257,   -2423,    5522,   20606,    -100},
{ -13948,  -11368,  -15375,  -21866,   -8520,   12221},
{   -616,    2424,   -2023,    4398,   -3805,    8108},
{  -7204,   21043,   21211,   -9395,  -19391,     896},
{  -5737,  -15160,  -21298,   17066,   -1006,    -366},
{   6261,    3240,  -11937,  -16213,  -15820,    6581},
{  -3155,   24796,    2733,   -1257,    -875,   -1597},
{ -20469,   11094,   24071,   -8987,   14136,    2220},
{ -14106,   11959,  -22495,    4135,   -1055,   -5420},
{    801,   -2655,      60,   -5324,    -790,    5937},
{  -7372,   -1764,  -22433,  -26060,   21707,    4178},
{  -5715,   -6648,  -14908,    1325,  -24044,    1493},
{  -6024,  -12488,   23930,    2950,    1601,    1173},
{  19067,   17630,   17929,  -10654,   10928,   -4958},
{   3231,   -3284,   27336,    4174,   -1683,     497},
};

const int16_t (* const ff_eac3_mantissa_vq[8])[6] = {
    NULL,
    vq_hebap1,
    vq_hebap2,
    vq_hebap3,
    vq_hebap4,
    vq_hebap5,
    vq_hebap6,
    vq_hebap7,
};

/**
 * Table E2.14 Frame Exponent Strategy Combinations
 */
const uint8_t ff_eac3_frm_expstr[32][6] = {
{    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE},
{    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,    EXP_D45},
{    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,    EXP_D25,  EXP_REUSE},
{    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,    EXP_D45,    EXP_D45},
{    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D25,  EXP_REUSE,  EXP_REUSE},
{    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D25,  EXP_REUSE,    EXP_D45},
{    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D45,    EXP_D25,  EXP_REUSE},
{    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D45,    EXP_D45,    EXP_D45},
{    EXP_D25,  EXP_REUSE,    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE},
{    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D45},
{    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE},
{    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D45},
{    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D25,  EXP_REUSE,  EXP_REUSE},
{    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D45},
{    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE},
{    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D45},
{    EXP_D45,    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE},
{    EXP_D45,    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE,    EXP_D45},
{    EXP_D45,    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D25,  EXP_REUSE},
{    EXP_D45,    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D45,    EXP_D45},
{    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE,  EXP_REUSE},
{    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE,    EXP_D45},
{    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D25,  EXP_REUSE},
{    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D45,    EXP_D45},
{    EXP_D45,    EXP_D45,    EXP_D15,  EXP_REUSE,  EXP_REUSE,  EXP_REUSE},
{    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE,  EXP_REUSE,    EXP_D45},
{    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D25,  EXP_REUSE},
{    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D45,    EXP_D45},
{    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE,  EXP_REUSE},
{    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE,    EXP_D45},
{    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D25,  EXP_REUSE},
{    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D45,    EXP_D45},
};

/**
 * Table E2.16 Default Coupling Banding Structure
 */
const uint8_t ff_eac3_default_cpl_band_struct[18] =
{ 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1 };

/**
 * Table of bin locations for rematrixing bands
 * reference: Section 7.5.2 Rematrixing : Frequency Band Definitions
 */
const uint8_t ff_ac3_rematrix_band_tab[5] = { 13, 25, 37, 61, 253 };
