

#ifndef __MYX_GRTPP_H__
#define __MYX_GRTPP_H__

#include "myx_grt_public_interface.h"

#include <string>
#include <vector>
#include <map>


namespace grt
{
  class Module;
  class Value;
  class Listener;

  class Error
  {
  public:
    virtual ~Error() {};
    virtual std::string descr() { return "error"; };
  };

  class GRTError : public Error
  {
    MYX_GRT_ERROR _error;
  public:
    GRTError(MYX_GRT_ERROR err) : _error(err) {};
    virtual std::string descr() { return myx_grt_error_string(_error); };
  };
  
  class InvalidGRTType : public Error
  {
  public:
    virtual std::string descr() { return "bad grt type"; };
  };
  
  class InvalidItem : public Error
  {
    std::string _descr;

  public:
    InvalidItem(unsigned int index, unsigned int count) { _descr= "index out of range"; };
    InvalidItem(const std::string &name) { _descr= "invalid item name"; };
    virtual std::string descr() { return _descr; };
  };

  class ReadOnlyItem : public Error
  {
  public:
    ReadOnlyItem(const std::string &name) { };
    virtual std::string descr() { return "item is read-only"; };
  };
  
  class TypeMismatch : public Error
  {
  public:
    TypeMismatch() {};
    TypeMismatch(MYX_GRT_VALUE_TYPE expected, MYX_GRT_VALUE_TYPE actual) {};
    virtual std::string descr() { return "type mismatch"; };
  };

  class ModuleError : public Error
  {
  public:
    virtual std::string descr() { return "module error"; };
  };
  
  class FunctionCallError : public ModuleError
  {
  public:
    FunctionCallError(const std::string &module, const std::string &function,
                      MYX_GRT_ERROR error) {};

    FunctionCallError(const std::string &module, const std::string &function,
                      const std::string &error, const std::string &detail) {};
  };
  
  class InvalidFunction : public ModuleError
  {
  public:
    InvalidFunction(const std::string &module, const std::string &function) {};
  };
  
  class InvalidFunctionReturn : public ModuleError
  {
  public:
    InvalidFunctionReturn(const std::string &module, const std::string &function, const Value &value) {};
  };
  


  //----------------------------------------------------------------------
  // Struct
  
  class Struct
  {
  protected:
    MYX_GRT *_grt;
    MYX_GRT_STRUCT *_struct;

  public:
    Struct(MYX_GRT *grt, MYX_GRT_STRUCT *gstruct)
      : _grt(grt), _struct(gstruct)
      {
      };

    inline bool is_valid() const { return _struct != 0; };
    inline bool is_a(Struct &other) const { return myx_grt_struct_is_or_inherits_from(_grt, myx_grt_struct_get_name(_struct), myx_grt_struct_get_name(other._struct)); };
    inline const std::string name() const { return myx_grt_struct_get_name(_struct); };
  };


  //----------------------------------------------------------------------
  // Value
  

  class Value
  {
  protected:    
    MYX_GRT_VALUE *_value;

  public:
    void __replace_value(MYX_GRT_VALUE *nvalue) { if (nvalue != _value) { if (_value) myx_grt_value_release(_value); _value= myx_grt_value_retain(nvalue); } };
    int refcount() { return myx_grt_value_get_current_reference_count(_value); };

    Value() : _value(0) { };

    Value(MYX_GRT_VALUE *value)
      : _value(value)
    {
      if (_value)
        myx_grt_value_retain(_value);
    };

    Value(const Value &value)
      : _value(value._value)
    {
      if (_value)
        myx_grt_value_retain(_value);
    };
    
    virtual ~Value()
    {
      if (_value)
        myx_grt_value_release(_value);
    };
    
    inline void release() { myx_grt_value_release(_value); };
    inline Value& retain() { myx_grt_value_retain(_value); return *this; };
    inline void clear() { if (_value) release(); _value= 0; };

    inline bool is_valid() const { return _value != 0; };

    inline MYX_GRT_VALUE *grt_value() const { return _value; };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_ANY_VALUE; };
    inline MYX_GRT_VALUE_TYPE type() const { return myx_grt_value_get_type(_value); };

    Value &operator= (const Value&other) { __replace_value(other._value); return *this; };
  };

  //----------------------------------------------------------------------
  // IntValue

  class IntValue : public Value
  {
  public:
    IntValue() : Value() {};

    IntValue(MYX_GRT_VALUE *ivalue)
      : Value()
    {
      if (myx_grt_value_get_type(ivalue) != MYX_INT_VALUE)
        throw InvalidGRTType();
      _value= myx_grt_value_retain(ivalue);
    };
    
    IntValue(const IntValue &ivalue)
      : Value(ivalue)
    {
    };

    static IntValue cast_from(const Value &ivalue)
    {
      if (ivalue.type() != MYX_INT_VALUE)
        throw InvalidGRTType();
      return IntValue(ivalue.grt_value());
    };

    IntValue(int value)
      : Value()
    {
      _value= myx_grt_value_from_int(value);
    };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_INT_VALUE; };

    static inline bool can_wrap(Value &value) { if (value.type() != MYX_INT_VALUE) return false; return true; };
    
    inline operator int () const { return myx_grt_value_as_int(_value); };

    IntValue &operator = (const IntValue &other);
    IntValue &operator = (int value);
  };

  
  //----------------------------------------------------------------------
  //DoubleValue

  class DoubleValue : public Value
  {
  public:
    DoubleValue() : Value() {};

    DoubleValue(MYX_GRT_VALUE *ivalue)
      : Value()
    {
      if (myx_grt_value_get_type(ivalue) != MYX_REAL_VALUE)
        throw InvalidGRTType();
      _value= myx_grt_value_retain(ivalue);
    };
    
    DoubleValue(const DoubleValue &ivalue)
      : Value(ivalue)
    {
    };

    static DoubleValue cast_from(const Value &ivalue)
    {
      if (ivalue.type() != MYX_REAL_VALUE)
        throw InvalidGRTType();
      return DoubleValue(ivalue.grt_value());
    };
    
    DoubleValue(double value)
      : Value()
    {
      _value= myx_grt_value_from_real(value);
    };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_REAL_VALUE; };

    static inline bool can_wrap(Value &value) { if (value.type() != MYX_REAL_VALUE) return false; return true; };

    inline operator double () { return myx_grt_value_as_real(_value); };

    DoubleValue &operator = (const DoubleValue &other);
    DoubleValue &operator = (double value);
  };


  //----------------------------------------------------------------------
  // StringValue

  class StringValue : public Value
  {
  public:
    StringValue() : Value() {};

    StringValue(MYX_GRT_VALUE *svalue)
      : Value()
    {
      if (myx_grt_value_get_type(svalue) != MYX_STRING_VALUE)
        throw InvalidGRTType();
      
      _value= myx_grt_value_retain(svalue);
    };

    StringValue(const StringValue &svalue)
      : Value(svalue)
    {
    };

    StringValue(const std::string &svalue)
      : Value()
    {
      _value= myx_grt_value_from_string(svalue.c_str());
    };

    static StringValue cast_from(const Value &svalue)
    {
      if (svalue.type() != MYX_STRING_VALUE)
        throw InvalidGRTType();
      return StringValue(svalue.grt_value());
    };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_STRING_VALUE; };

    static inline bool can_wrap(Value &value) { if (value.type() != MYX_STRING_VALUE) return false; return true; };

    inline operator std::string () { return myx_grt_value_as_string(_value); };

    StringValue &operator = (const StringValue &other);
    StringValue &operator = (const std::string &value);
  };

  
  //----------------------------------------------------------------------
  // Lists

  class BaseListValue : public Value
  {
  public:
    BaseListValue() : Value() {};
    BaseListValue(MYX_GRT_VALUE *lvalue)
      : Value(lvalue)
    {
      if (myx_grt_value_get_type(_value) != MYX_LIST_VALUE)
        throw InvalidGRTType();
    };

    inline void init(MYX_GRT_VALUE_TYPE type)
    {
      clear();
      _value= myx_grt_list_new(type, NULL);
    };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_LIST_VALUE; };

    static inline bool can_wrap(Value &value) { if (value.type() != MYX_LIST_VALUE) return false; return true; };

    inline MYX_GRT_VALUE_TYPE content_type() const { return myx_grt_list_content_get_type(_value); };
    inline std::string content_struct_name() const { const char *name= myx_grt_list_content_get_struct_name(_value); if (name) return name; return ""; };

    inline void remove(unsigned int index) { if (index >= count()) throw InvalidItem(index, count()); myx_grt_list_item_del(_value, index); };

    inline unsigned int count() const { return myx_grt_list_item_count(_value); };
    inline Value operator[](unsigned int index) const
    {
      MYX_GRT_VALUE *r= myx_grt_list_item_get(_value, index);
      if (!r) throw InvalidItem(index, count());
      return Value(r);
    };
  };

  //----------------------------------------------------------------------
  // ListValue

  
  //TODO: consider adding a readonly list class for use in bridged object wrappers
  
  template<class O> class ListValue : public BaseListValue
  {
  public:
    class Reference
    {
      ListValue<O> *_list;
      unsigned int _index;
      O *_tmp;

    public:
      Reference(ListValue<O> *list, unsigned int index)
        : _list(list), _index(index), _tmp(0) {};
      ~Reference() { delete _tmp; };

      void operator=(O &value)
      {
        _list->set(_index, value);
      };
      inline operator O ()
      {
        return _list->get(_index);
      };
      inline O operator * ()
      {
        return _list->get(_index);
      };
      inline O *operator->()
      {
        // hack
        if (!_tmp)
          _tmp= new O();
        _tmp->__replace_value(_list->get(_index).grt_value());
        return _tmp;
      };
    };

    ListValue()
      : BaseListValue()
    {
    };

    ListValue(MYX_GRT_VALUE *lvalue)
      : BaseListValue(lvalue)
    {
      if (myx_grt_list_content_get_type(_value) != MYX_OBJECT_VALUE)
        throw InvalidGRTType();
    };
    
    void init()
    {
      BaseListValue::init(MYX_OBJECT_VALUE);
    }

    static ListValue<O> cast_from(const Value &value)
    {
      if (value.type() != MYX_LIST_VALUE)
        throw InvalidGRTType();
      
      if (O::class_type() != MYX_ANY_VALUE &&
          myx_grt_list_content_get_type(value.grt_value()) !=
          O::class_type())
        throw InvalidGRTType();

      return value.grt_value();
    };

    static inline bool can_wrap(Value &value) 
    { 
      if (value.type() != MYX_LIST_VALUE) return false; 
      if (myx_grt_list_content_get_type(value.grt_value()) != MYX_OBJECT_VALUE)
        return false;
      Struct gstruct= O::static_get_struct();
      if (gstruct.is_valid() && !gstruct.is_a(Struct(myx_grt_obj_struct_get(value.grt_value()))))
        return false;
      return true;
    };

    void insert(const O &value, int index=-1)
    {
      myx_grt_list_item_insert(_value, index, value.grt_value());
    }

    inline O operator[](unsigned int index) const
    {
      return get(index);
    };

    inline Reference operator[](unsigned int index) 
    {
      if (index >= count())
        throw InvalidItem(index, count());
      return Reference(this, index);
    };

    inline O get(unsigned int index)
    {
      return O(myx_grt_list_item_get(_value, index));
    }
    inline void set(unsigned int index, const O &value)
    {
      myx_grt_list_item_set(_value, index, value.grt_value());
    }
  };

  //----------------------------------------------------------------------
  // ListValue<IntValue>

  template<> class ListValue<IntValue> : public BaseListValue
  {
  public:
    class Reference
    {
      ListValue<IntValue> *_list;
      unsigned int _index;

    public:
      Reference(ListValue<IntValue> *list, unsigned int index)
        : _list(list), _index(index) {};

      void operator=(IntValue &value)
      {
        _list->set(_index, value);
      };
      inline operator int ()
      {
        return _list->get(_index);
      };
    };

    ListValue()
      : BaseListValue()
    {
    };

    ListValue<IntValue>(MYX_GRT_VALUE *lvalue)
      : BaseListValue(lvalue)
    {
      if (myx_grt_list_content_get_type(_value) != MYX_INT_VALUE)
        throw InvalidGRTType();
    };
    
    void init()
    {
      BaseListValue::init(MYX_INT_VALUE);
    }

    static inline bool can_wrap(Value &value)
    {
      if (value.type() != MYX_LIST_VALUE) return false;
      if (myx_grt_list_content_get_type(value.grt_value()) != MYX_INT_VALUE) return false;
      return true; 
    };

    inline void insert(const IntValue &value, int index=-1) { myx_grt_list_item_insert(_value, index, value.grt_value()); };

    inline Reference operator[](unsigned int index) 
    {
      if (index >= count())
        throw InvalidItem(index, count());
      return Reference(this, index);
    };

    inline IntValue operator[](unsigned int index) const
    {
      return get(index);
    };
    inline IntValue get(unsigned int index) const
    {
      return IntValue(myx_grt_list_item_get(_value, index));
    };
    inline void set(unsigned int index, const IntValue &value)
    {
      myx_grt_list_item_set(_value, index, value.grt_value());
    };
  };


  //----------------------------------------------------------------------
  // ListValue<DoubleValue>

  template<> class ListValue<DoubleValue> : public BaseListValue
  {
  public:
    class Reference
    {
      ListValue<DoubleValue> *_list;
      unsigned int _index;

    public:
      Reference(ListValue<DoubleValue> *list, unsigned int index)
        : _list(list), _index(index) {};

      void operator=(DoubleValue &value)
      {
        _list->set(_index, value);
      };
      inline operator double ()
      {
        return _list->get(_index);
      };
    };

    ListValue()
      : BaseListValue()
    {
    };

    ListValue(MYX_GRT_VALUE *lvalue)
      : BaseListValue(lvalue)
    {
      if (myx_grt_list_content_get_type(_value) != MYX_REAL_VALUE)
        throw InvalidGRTType();
    };

    void init()
    {
      BaseListValue::init(MYX_REAL_VALUE);
    }
    
    static inline bool can_wrap(Value &value) { if (value.type() != MYX_LIST_VALUE) return false; if (myx_grt_list_content_get_type(value.grt_value()) != MYX_REAL_VALUE) return false; return true; };

    inline void insert(const DoubleValue &value, int index=-1) { myx_grt_list_item_insert(_value, index, value.grt_value()); };

    inline Reference operator[](unsigned int index) 
    {
      if (index >= count())
        throw InvalidItem(index, count());
      return Reference(this, index);
    };
    inline DoubleValue operator[](unsigned int index) const
    {
      return get(index);
    };
    inline DoubleValue get(unsigned int index) const
    {
      return DoubleValue(myx_grt_list_item_get(_value, index));
    };

    inline void set(unsigned int index, const DoubleValue &value)
    {
      myx_grt_list_item_set(_value, index, value.grt_value());
    };
  };


  //----------------------------------------------------------------------
  // ListValue<StringValue>

  template<> class ListValue<StringValue> : public BaseListValue
  {
  public:
    class Reference
    {
      ListValue<StringValue> *_list;
      unsigned int _index;

    public:
      Reference(ListValue<StringValue> *list, unsigned int index)
        : _list(list), _index(index) {};

      void operator=(StringValue &value)
      {
        _list->set(_index, value);
      };
      inline operator std::string ()
      {
        return _list->get(_index);
      };
    };

    ListValue()
      : BaseListValue()
    {
    }

    ListValue(MYX_GRT_VALUE *lvalue)
      : BaseListValue(lvalue)
    {
      if (myx_grt_list_content_get_type(_value) != MYX_STRING_VALUE)
        throw InvalidGRTType();
    };
    
    void init()
    {
      BaseListValue::init(MYX_STRING_VALUE);
    }

    static inline bool can_wrap(Value &value) { if (value.type() != MYX_LIST_VALUE) return false; if (myx_grt_list_content_get_type(value.grt_value()) != MYX_STRING_VALUE) return false; return true; };

    inline void insert(const StringValue &value, int index=-1) { myx_grt_list_item_insert(_value, index, value.grt_value()); };

    inline Reference operator[](unsigned int index) 
    {
      if (index >= count())
        throw InvalidItem(index, count());
      return Reference(this, index);
    };

    inline StringValue operator[](unsigned int index) const
    {
      return get(index);
    };

    inline StringValue get(unsigned int index) const
    {
      if (index >= count())
        throw InvalidItem(index, count());
      return myx_grt_list_item_get(_value, index);
    };

    inline void set(unsigned int index, const StringValue &value)
    {
      myx_grt_list_item_set(_value, index, value.grt_value());
    };
  };


  //----------------------------------------------------------------------
  // DictValue

  class DictValue : public Value
  {
  public:
    class Reference
    {
      DictValue *_dict;
      std::string _key;
      Value *_tmp;

    public:
      Reference(DictValue *dict, const std::string &key)
        : _dict(dict), _key(key), _tmp(0) {};
      ~Reference() { delete _tmp; };

      void operator=(Value &value)
      {
        _dict->set(_key, value);
      };
      inline operator Value ()
      {
        return _dict->get(_key);
      };
      inline Value operator*()
      {
        return _dict->get(_key);
      };
      inline Value *operator->()
      {
        // hack
        if (!_tmp)
          _tmp= new Value();
        _tmp->__replace_value(_dict->get(_key).grt_value());
        return _tmp;
      };
    };

    DictValue() : Value() {};
    DictValue(MYX_GRT_VALUE *dvalue)
      : Value(dvalue)
    {
      if (_value && myx_grt_value_get_type(_value) != MYX_DICT_VALUE)
        throw InvalidGRTType();
    };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_DICT_VALUE; };

    inline MYX_GRT_VALUE_TYPE content_type() { return myx_grt_dict_content_get_type(_value); };

    static DictValue cast_from(const Value &ivalue)
    {
      if (ivalue.type() != MYX_DICT_VALUE)
        throw InvalidGRTType();
      return DictValue(ivalue.grt_value());
    };

    inline bool has_key(const std::string &k)
    {
      if (myx_grt_dict_item_get_value(_value, k.c_str()))
        return true;
      return false;
    };

    inline Value operator[](const std::string &k) const
    {
      return get(k);
    };
    inline Reference operator[](const std::string &k)
    {
      if (!has_key(k))
        throw InvalidItem(k);
      return Reference(this, k);
    };
    
    Value get(const std::string &k) const
    {
      MYX_GRT_VALUE *r;
      r= myx_grt_dict_item_get_value(_value, k.c_str());
      if (!r) throw InvalidItem(k);
      return Value(r);
    }
    
    void set(const std::string &k, const Value &value)
    {
      myx_grt_dict_item_set_value(_value, k.c_str(), value.grt_value());
    }
  };

  //----------------------------------------------------------------------
  // ObjectValue

  class ObjectValue : public Value
  {
    class GRT *_grt;

  protected:
    ObjectValue(GRT *grt, const std::string &structname, const DictValue &args);
    
  public:
    ObjectValue() : Value() {};
    ObjectValue(MYX_GRT_VALUE *value);

    std::string id() { return myx_grt_obj_get_id(_value); };
    GRT *grt() { return _grt; };

    static MYX_GRT_VALUE_TYPE class_type() { return MYX_OBJECT_VALUE; };
    
    static ObjectValue cast_from(const Value &svalue)
    {
      if (svalue.type() != MYX_OBJECT_VALUE)
        throw InvalidGRTType();
      return ObjectValue(svalue.grt_value());
    };

    // must be overriden by each wrapped Object
    static std::string static_get_struct_name() { return ""; };

    inline Struct get_struct() { return Struct(myx_grt_obj_get_grt(_value), myx_grt_obj_struct_get(_value)); };
    inline std::string get_struct_name() { return myx_grt_obj_struct_get_name(_value); };

    bool isinstance(const std::string &struct_name) const;
    
    bool has_member(const std::string &name) const;
    
    Value get_member(const std::string &name) const;
    int get_int_member(const std::string &name) const;
    std::string get_string_member(const std::string &name) const;
    double get_double_member(const std::string &name) const;
    ObjectValue get_object_member(const std::string &name) const;

    void set_member(const std::string &name, const Value &value);
    void set_member(const std::string &name, int value);
    void set_member(const std::string &name, const std::string &value);
    void set_member(const std::string &name, double value);
    void set_member(const std::string &name, const ObjectValue &object);
    
    void add_to_member(const std::string &name, Value &value);
    void del_from_member(const std::string &name, unsigned int index);

    // get private data from bridge, if it's a c++ bridge
    class ObjectBridgeBase *get_bridge_private();
  };
  
  //----------------------------------------------------------------------
  
  template<class T> inline void assign(T &a, const Value &b)
  {
    a= T::cast_from(b);
  }

  //----------------------------------------------------------------------
  // GRT

  class GRT
  {
  protected:
    MYX_GRT *_grt;

    std::map<std::string, Module*> _modules;

    static void listener_callback(MYX_GRT_NOTIFICATION *notif, void *data);
  public:
    GRT();

    MYX_GRT *grt() { return _grt; };
    
    // module wrapper access
    Module *get_module(const std::string &name);
    bool register_module(Module *module);
   
    // structs
    void load_structs_from(const std::string &path);

    // values
    
    bool serialize(const Value &value, const std::string &path);
    Value unserialize(const std::string &path);

    inline Value root() { return Value(myx_grt_get_root(_grt)); };

    ObjectValue create_object(const std::string &struct_name, ...);

    inline Value get(const std::string &path) { return Value(myx_grt_value_item_get_by_path(_grt, root().grt_value(), path.c_str())); };
    inline void set(const std::string &path, const Value &value) { myx_grt_value_item_set_by_path(root().grt_value(), path.c_str(), value.grt_value()); };

    // notifications
    void enable_thread_notifications();
    void flush_notifications();

    void post_notification_now(const std::string &name, const Value &value, const Value &argdata);
    void post_notification_later(const std::string &name, const Value &value, const Value &argdata);

    void add_listener(Listener *listener, const std::string &name, const Value &value);
    void remove_listeners(Listener *listener);
    void remove_listeners(Listener *listener, const std::string &name, const Value &value);
    void remove_listeners(Listener *listener, const std::string &name);
    void remove_listeners(Listener *listener, const Value &value);
    void remove_thread_listeners();
  };


  //----------------------------------------------------------------------
  // Notifications

  class Listener {
    friend class GRT;

  protected:
    virtual void notify(const std::string &name, const Value &value, const Value &argdata)= 0;
  };

  //----------------------------------------------------------------------
  // Module wrappers

  // Wraps GRT modules and allow easy calling from C++
  class Module
  {
    GRT *_grt;
    MYX_GRT_MODULE *_module;

  protected:
    Module(GRT *grt, const std::string &name);

  public:
    inline std::string name() { return _module->name; };

    Value call_function(const std::string &function,
                        const ListValue<Value> &args);
  };


  //----------------------------------------------------------------------
  //----------------------------------------------------------------------
  // Object Bridge Support

  // Allows implementation of object bridges in C++ by subclassing

  // Implementations must use register_bridge() to register themselves
  // with the GRT environment and provide a static factory method
  // that will create and return a new instance of itself. It must
  // also override the virtual methods marked as such below.
  // 
  // One of initialize(), unserialize() or copy() will be called
  // to initialize the newly created instance, depending on how it
  // is created

  class ObjectBridgeBase
  {
    friend class ObjectValue;

  private:
    static int _initialize(MYX_GRT_VALUE *self, MYX_GRT_VALUE *argdict);
    static int _destroy(MYX_GRT_VALUE *self);
    static MYX_GRT_VALUE *_get_member(MYX_GRT_VALUE *self, const char *name);
    static MYX_GRT_ERROR _set_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value);
    static MYX_GRT_ERROR _add_to_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value);
    static MYX_GRT_ERROR _del_from_member(MYX_GRT_VALUE *self, const char *name, unsigned int index);
    static int _serialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr node);
    static int _unserialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr node);
    static int _copy(MYX_GRT_VALUE *self, MYX_GRT_VALUE *orig);

  public:
    typedef ObjectBridgeBase*(*FactoryMethod)(GRT *grt, MYX_GRT_VALUE *self,
                                              void *data);
  protected:
    MYX_GRT_VALUE *_self; // must not be retained!
    GRT *grt();

    // xml helpers
    void xset_int(xmlNodePtr node, const std::string &name, int value);
    void xset_string(xmlNodePtr node, const std::string &name, const std::string &value);
    void xset_double(xmlNodePtr node, const std::string &name, double value);
    xmlNodePtr xadd_child(xmlNodePtr node, const std::string &name, const std::string &value);

    ObjectBridgeBase(GRT *grt, MYX_GRT_VALUE *self);
    virtual ~ObjectBridgeBase();
    
    // The following methods must be overriden to specify the behaviour
    // of the bridge. All methods are mandatory, except for 
    // initialize() and destroy()
    virtual void initialize(const DictValue &args) {};
    virtual void destroy() {};
    virtual Value get_item(const std::string &name)= 0;
    virtual void set_item(const std::string &name, const Value &value)= 0;
    virtual void add_to_item(const std::string &name, const Value &value)= 0;
    virtual void del_from_item(const std::string &name, unsigned int index)= 0;
    virtual void serialize(xmlNodePtr node)= 0;
    virtual void unserialize(xmlNodePtr node)= 0;
    virtual void copy(ObjectBridgeBase *orig)= 0;

  public:
    static bool register_bridge(GRT *grt, const std::string &name,
                                FactoryMethod method, void *data);
  };


  //----------------------------------------------------------------------
  // C++ Module Support

  // Allows implementation of modules in C++ by subclassing
  
  class ModuleImplBase
  {
  public:
    
  };

  
  
  /*
   class SampleModuleImpl : public ModuleImplBase
   {


   public:
     
     IntValue copyToClipboard(const StringValue &text);
     IntValue createDir(const StringValue &path);


   public:

     int createDir(const std::string &path)
     {
       return 0;
     }
     
     static Value __createDir(ListValue &args, void *data)
     {
       SampleModuleImpl *self= (SampleModuleImpl*)data;
       return self->create_dir(args[0]);
     };
     
     static Value __copyToClipboard(ListValue &args, void *data)
     {
       SampleModuleImpl *self= (SampleModuleImpl*)data;
       return self->copy_to_clipboard(args[0]);
     };

     static void register_module(GRT *grt)
     {
       ModuleFunctions mf;
       
       mf.add("createDir:i:s path", &__create_dir);
       mf.add("copyToClipboard:i:s text", &__copy_to_clipboard);
       grt->register_module("SampleModuleImpl", mf);
     }
   };
   */

};



#endif
