/* Copyright (c) 2004, 2005 MySQL AB
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
  
   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
 */

/**
 * @file myx_grt.c 
 * @brief GRT environment functions
 * 
 * See also: <a href="../grt.html#GRT">GRT</a>
 */

#include <glib.h>

#include "myx_grt_private.h"
#include "myx_shared_util_functions.h"

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#include <objbase.h>
#elif defined(__APPLE__)
#include <CoreFoundation/CoreFoundation.h>
#else
// Linux specific
#include <uuid/uuid.h>
#endif



static void default_print(const char *msg, void *data)
{
  printf("%s", msg);
}


static void default_log(MYX_GRT *grt, int code, const char *msg, const char *detail)
{
  printf("%i: %s: %s\n", code, msg, detail?detail:"");
}

void myx_grt_init_threads()
{
  g_thread_init(NULL);
}

/**
 **************************************************************************** 
 * @brief Returns the version of the GRT environment
 *
 * @return a const char with the version information
 ****************************************************************************/
const char * myx_grt_version()
{
  return GRT_VERSION;
}


/**
 **************************************************************************** 
 * @brief Creates a new GRT environment.
 *
 *  Also:
 *  - sets a default print callback that's equivalent to printf() and
 * registers the built-in module loader type.
 *  - creates a server socket so that modules can connect to it for
 * communicating with the grt (passing progress info, for example)
 *   After creating the environment, you will want to call individual
 * initializers for each module loader you want to support.
 *
 * @return A newly created GRT.
 ****************************************************************************/
MYX_GRT *myx_grt_initialize(int options)
{
  MYX_GRT *grt= g_new0(MYX_GRT, 1);

  g_static_rec_mutex_init(&grt->global_mutex);
  g_static_rec_mutex_init(&grt->notification_queue_mutex);

  grt->options= options;

  grt->print= default_print;
  grt->print_data= NULL;
  
  grt->logfunc= default_log;
  
  grt->root= myx_grt_dict_new(NULL);

  grt->struct_icon_cache= g_hash_table_new_full(g_str_hash, g_str_equal,
                                                g_free, g_free);
  
  myx_grt_register_module_loader(grt, myx_builtin_init_loader(grt));

#ifdef remove_this_when_this_gets_used
  myx_grt_setup_messaging(grt);
#endif
  return grt;
}


/**
 ****************************************************************************
 * @brief Shutdown a GRT environment and frees all used resources
 *
 *   Will close and free all module loaders, modules and other resources
 *   allocated for it.
 * 
 * @param grt the GRT environment
 *
 * NOTE
 *   This is not freeing anything atm.
 ****************************************************************************/
void myx_grt_finalize(MYX_GRT *grt)
{
  //XXX TODO


}


void glib_message_handler(const gchar *log_domain, GLogLevelFlags log_level, 
                          const gchar *message, gpointer user_data)
{
  myx_grt_messages_stack_add_message((MYX_GRT *) user_data, message, NULL, 0);
}

/**
 ****************************************************************************
 * @brief Activates or deactivates the catching of glib messages
 * 
 *  Sets the function that is used when GRT needs to print messages.
 *
 * @param grt      the GRT environment
 * @param user_data a pointer to be passed to the callback function. 
 * @param process_output_func the callback function which should take 
 *        the text and the userdata pointer as arguments
 ****************************************************************************/
void myx_grt_catch_glib_messages(MYX_GRT *grt, int do_catch)
{
  if (do_catch)
    g_log_set_handler(NULL, G_LOG_LEVEL_MASK, glib_message_handler, grt);
  else
    g_log_default_handler(NULL, G_LOG_LEVEL_MASK, "", NULL);
}

/**
 ****************************************************************************
 * @brief Sets the function to be used to print values
 * 
 *  Sets the function that is used when GRT needs to print messages.
 *
 * @param grt      the GRT environment
 * @param user_data a pointer to be passed to the callback function. 
 * @param process_output_func the callback function which should take 
 *        the text and the userdata pointer as arguments
 ****************************************************************************/
void myx_grt_set_output_callback(MYX_GRT *grt, void *user_data, MYX_GRT_PRINT_CALLBACK process_output_func)
{
  GRT_ENTER(grt);

  grt->print= process_output_func;
  grt->print_data= user_data;
  
  GRT_LEAVE(grt);
}

/**
 ****************************************************************************
 * @brief Sets the function to be used to process messages
 * 
 *  Sets the function that is used when GRT needs to process messages.
 *
 * @param grt      the GRT environment
 * @param user_data a pointer to be passed to the callback function. 
 * @param process_output_func the callback function which should take 
 *        the text and the userdata pointer as arguments
 ****************************************************************************/
void myx_grt_set_message_callback(MYX_GRT *grt, void *user_data, 
                                  MYX_GRT_MESSAGE_CALLBACK process_messages_func)
{
  GRT_ENTER(grt);

  grt->process_messages= process_messages_func;
  grt->process_messages_data= user_data;
  
  GRT_LEAVE(grt);
}

/**
 ****************************************************************************
 * @brief Sets the function to be used to process messages
 * 
 *  Sets the function that is used when GRT needs to process messages.
 *
 * @param grt      the GRT environment
 * @param user_data a pointer to be passed to the callback function. 
 * @param process_output_func the callback function which should take 
 *        the text and the userdata pointer as arguments
 ****************************************************************************/
void myx_grt_set_input_callback(MYX_GRT *grt, void *user_data,
                                MYX_GRT_INPUT_CALLBACK process_input_func)
{
  GRT_ENTER(grt);

  grt->process_input= process_input_func;
  grt->process_input_data= user_data;
  
  GRT_LEAVE(grt);
}

void myx_grt_set_status_query_callback(MYX_GRT *grt, void *user_data,
                                 int (*process_status_query_func)(void *user_data))
{
  GRT_ENTER(grt);

  grt->process_status_query= process_status_query_func;
  grt->process_status_query_data= user_data;
  
  GRT_LEAVE(grt);
}


/**
 ****************************************************************************
 * @brief Sets callbacks for module logging
 * 
 *  This will set the callbacks that will be called when a module calls
 * a logging function, for errors or messages in general.
 *
 * @param grt      the GRT environment
 * @param log_func the function that will handle logging calls from modules
 ****************************************************************************/
void myx_grt_module_set_log_callback(MYX_GRT *grt, MYX_GRT_LOG_CALLBACK log_func)
{
  GRT_ENTER(grt);
  grt->logfunc= log_func;
  GRT_LEAVE(grt);
}


/**
 ****************************************************************************
 * @brief Registers a module loader with the GRT
 * 
 * Will register an already initialized module loader with the GRT.
 * You can only register only one module loader of each type.
 * After registration, you can scan for modules with myx_grt_scan_for_modules
 * or load them individually with myx_grt_module_init
 * 
 * @param grt    the GRT environment where the loader should be registered
 * @param loader an initialized loader object. 
 *
 * @return MYX_GRT_NO_ERROR, MYX_GRT_INTERNAL_ERROR
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_register_module_loader(MYX_GRT *grt, MYX_GRT_MODULE_LOADER *loader)
{  
  g_return_val_if_fail(grt != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(loader != NULL, MYX_GRT_INTERNAL_ERROR);

  GRT_ENTER(grt);

  grt->loaders_num++;
  grt->loaders= g_realloc(grt->loaders, sizeof(MYX_GRT_MODULE_LOADER*)*grt->loaders_num);

  grt->loaders[grt->loaders_num-1]= loader;

  GRT_RETURN(grt, MYX_GRT_NO_ERROR, MYX_GRT_ERROR);
}

/**
 ****************************************************************************
 * @brief Returns a copied list of the structs registered in the GRT
 * 
 * @param grt the GRT
 *
 * @return A copy of the registered structs
 ****************************************************************************/
MYX_GRT_STRUCTS * myx_grt_structs_get(MYX_GRT *grt)
{
  MYX_GRT_STRUCTS *structs= g_malloc(sizeof(MYX_GRT_STRUCTS));

  g_return_val_if_fail(grt != NULL, NULL);

  GRT_ENTER(grt);

  structs->structs_num= grt->structs_num;
  structs->structs= g_memdup(grt->structs, sizeof(MYX_GRT_STRUCT)*structs->structs_num);

  GRT_RETURN(grt, structs, MYX_GRT_STRUCTS*);
}

MYX_STRINGLIST * myx_grt_struct_packages(MYX_GRT *grt)
{
  unsigned int i;

  g_return_val_if_fail(grt != NULL, NULL);

  GRT_ENTER(grt);
  
  {
    MYX_STRINGLIST *packages= g_new0(MYX_STRINGLIST, 1);

    for (i= 0; i<grt->structs_num; i++)
    {
      MYX_GRT_STRUCT *str= grt->structs + i;
      char *str_pak= g_strdup(str->name);
      unsigned int j, l= (unsigned int)strlen(str_pak);

      for (j= l-1; j>0; j--)
      {
        if (str_pak[j] == '.')
          break;
      }
      str_pak[j]= 0;

      //See if the package name is already stored
      for (j= 0; j<packages->strings_num; j++)
      {
        if (strcmp3(str_pak, packages->strings[j]) == 0)
          break;
      }
      //If not, add the package name
      if (j == packages->strings_num)
      {
        packages->strings_num++;
        packages->strings= g_realloc(packages->strings, sizeof(char *)*packages->strings_num);
        packages->strings[packages->strings_num-1]= g_strdup(str_pak);
      }

      g_free(str_pak);
    }

    GRT_RETURN(grt, packages, MYX_STRINGLIST*);
  }
}

int myx_grt_package_count(MYX_GRT *grt)
{
  unsigned int c;
  MYX_STRINGLIST *packages;
  GRT_ENTER(grt);
  
  packages= myx_grt_struct_packages(grt);

  c= packages->strings_num;

  myx_free_stringlist(packages);

  GRT_RETURN(grt, c, int);
}

char * myx_grt_package_by_index(MYX_GRT *grt, unsigned int index)
{
  char *s;
  MYX_STRINGLIST *packages;
  
  GRT_ENTER(grt);

  packages= myx_grt_struct_packages(grt);

  s= g_strdup(packages->strings[index]);

  myx_free_stringlist(packages);

  GRT_RETURN(grt, s, char*);
}

/**
 ****************************************************************************
 * @brief Returns the number of structs registered in the GRT
 * 
 * @param grt the GRT
 *
 * @return The number of structs
 ****************************************************************************/
int myx_grt_struct_get_count(MYX_GRT *grt)
{
  g_return_val_if_fail(grt != NULL, -1);
  
  GRT_ENTER(grt);

  GRT_RETURN(grt, grt->structs_num, int);
}

/**
 ****************************************************************************
 * @brief Returns the number of structs registered in the GRT that have
 * the struct with the struct_name as their parent
 * 
 * @param grt the GRT
 * @param struct_name The name of the parent struct
 *
 * @return The number of structs
 ****************************************************************************/
int myx_grt_struct_get_child_count(MYX_GRT *grt, const char *struct_name)
{
  unsigned int i, c= 0;

  g_return_val_if_fail(grt != NULL, -1);
  
  GRT_ENTER(grt);

  for (i= 0; i<grt->structs_num; i++)
  {
    MYX_GRT_STRUCT *str= grt->structs + i;

    if (strcmp3(str->parent_struct_name, struct_name) == 0)
    {
      c++;
    }
  }

  GRT_RETURN(grt, c, int);
}

int myx_grt_package_struct_count(MYX_GRT *grt, const char *package_name)
{
  unsigned int i, c= 0, l= (unsigned int)strlen(package_name);

  g_return_val_if_fail(grt != NULL, 0);
  g_return_val_if_fail(package_name != NULL, 0);

  GRT_ENTER(grt);
  
  for (i= 0; i<grt->structs_num; i++)
  {
    MYX_GRT_STRUCT *str= grt->structs + i;   

    if (package_name[0])
    {
      char *begin_str= g_strdup_printf("%s.", package_name);

      if (str_beginswith(str->name, begin_str) &&
          (sub_str_count(".", str->name+l+1) == 0))
        c++;

      g_free(begin_str);
    }
    else
    {
      if (sub_str_count(".", str->name) == 0)
        c++;
    }
  }

  GRT_RETURN(grt, c, int);
}

/**
 ****************************************************************************
 * @brief Returns the GRT struct given by an index
 * 
 * @param grt the GRT
 * @param index the index of the requested struct
 * 
 * @return The struct with the given index
 ****************************************************************************/
MYX_GRT_STRUCT * myx_grt_struct_get_by_index(MYX_GRT *grt, unsigned int index)
{
  g_return_val_if_fail(grt != NULL, NULL);
  
  GRT_ENTER(grt);

  GRT_RETURN_VAL_IF_FAIL(grt, index < grt->structs_num, NULL);

  GRT_RETURN(grt, grt->structs + index, MYX_GRT_STRUCT*);
}

/**
 ****************************************************************************
 * @brief Returns the child GRT struct of the struct defined by struct_name 
 * with the given by an
 * 
 * @param grt the GRT
 * @param struct_name the name of the parent strcut
 * @param index the index of the requested struct
 * 
 * @return The struct with the given index or null if not found
 ****************************************************************************/
MYX_GRT_STRUCT * myx_grt_struct_get_child_by_index(MYX_GRT *grt, const char *struct_name, unsigned int index)
{
  unsigned int i, c= 0;

  g_return_val_if_fail(grt != NULL, NULL);
  
  GRT_ENTER(grt);

  for (i= 0; i<grt->structs_num; i++)
  {
    MYX_GRT_STRUCT *str= grt->structs + i;

    if (strcmp3(str->parent_struct_name, struct_name) == 0)
    {
      if (c == index)
        GRT_RETURN(grt, str, MYX_GRT_STRUCT*);

      c++;
    }
  }

  GRT_RETURN(grt, NULL, MYX_GRT_STRUCT*);
}

MYX_GRT_STRUCT * myx_grt_package_struct_by_index(MYX_GRT *grt, const char *package_name, unsigned int index)
{
  unsigned int i, c= 0, l= (unsigned int)strlen(package_name);

  g_return_val_if_fail(grt != NULL, NULL);

  GRT_ENTER(grt);
  
  for (i= 0; i<grt->structs_num; i++)
  {
    MYX_GRT_STRUCT *str= grt->structs + i;

    if (package_name[0])
    {
      char *begin_str= g_strdup_printf("%s.", package_name);

      if (str_beginswith(str->name, begin_str) &&
          (sub_str_count(".", str->name+l+1) == 0))
      {
        if (c == index)
        {
          g_free(begin_str);
          GRT_RETURN(grt, str, MYX_GRT_STRUCT*);
        }

        c++;
      }

      g_free(begin_str);
    }
    else
    {
      if (sub_str_count(".", str->name) == 0)
      {
        if (c == index)
          GRT_RETURN(grt, str, MYX_GRT_STRUCT*);
        
        c++;
      }
    }
  }

  GRT_RETURN(grt, NULL, MYX_GRT_STRUCT*);
}


/**
 ****************************************************************************
 * @brief Fetches the root object of the GRT
 * 
 * @param grt the GRT
 *
 * @return The root GRT value.
 ****************************************************************************/
MYX_GRT_VALUE *myx_grt_get_root(MYX_GRT *grt)
{
  g_return_val_if_fail(grt != NULL, NULL);

  GRT_ENTER(grt);
  
  GRT_RETURN(grt, grt->root, MYX_GRT_VALUE*);
}


/**
 ****************************************************************************
 * @brief "Loads" a value into the GRT and makes it the root object/value
 * 
 *   This will replace the root object of the GRT with the passed one.
 * The old object will be completely freed. 
 *
 * The GRT will take over ownership of the whole object passed. That means
 * you should not free it.
 *
 * @param grt the GRT environment
 * @param new_root a GRT object/value that will become the root object. It has to
 *    be a MYX_DICT_VALUE
 *
 * @return MYX_GRT_NO_ERROR if success
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_set_root(MYX_GRT *grt, MYX_GRT_VALUE *new_root)
{
  g_return_val_if_fail(grt != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(new_root != NULL, MYX_GRT_INTERNAL_ERROR);

  GRT_ENTER(grt);
  
  if (myx_grt_value_get_type(new_root) != MYX_DICT_VALUE)
    GRT_RETURN(grt, MYX_GRT_BAD_VALUE, MYX_GRT_ERROR);

  myx_grt_value_release(grt->root);

  grt->root= new_root;

  myx_grt_value_retain(new_root);
  
  GRT_RETURN(grt, MYX_GRT_NO_ERROR, MYX_GRT_ERROR);
}


/*static char *value_type_strings[]=
{
  "all",
    "int",
    "bigint",
    "real",
    "string",
    "list",
    "dict"
};*/


/**
 ****************************************************************************
 * @brief Prints a string using the print callback from GRT
 *
 *  Prints a formated message using the callback function set up in GRT.
 *
 * @param grt the GRT environment. A print callback must have been previously set.
 * @param fmt format string, accepts anything that printf() does.
 * @param ... arguments for the formatted message
 *****************************************************************************/
int myx_grt_printf(MYX_GRT *grt, const char *fmt, ...)
{
  char *tmp;
  
  va_list args;
  va_start(args, fmt);

  GRT_ENTER(grt);
  
  tmp= g_strdup_vprintf(fmt, args);
  va_end(args);
  
  MYX_PRINT(grt, tmp);
  g_free(tmp);
  
  GRT_RETURN(grt, 0, int);
}


/** 
 ****************************************************************************
 * @brief Converts messages that are stored in a GRT value to a GRT 
 * message list
 *
 * @param msgs_list  the GRT list value holding the messages
 *
 * @return Converted messages or NULL. The returned struct must be freed 
 * with myx_grt_messages_free()
 *****************************************************************************/
MYX_GRT_MSGS * myx_grt_messages_convert(MYX_GRT_VALUE *msgs_list)
{
  MYX_GRT_MSGS *msgs= NULL;

  if (msgs_list)
  {
    unsigned int i;
    msgs= g_new0(MYX_GRT_MSGS, 1);
    msgs->msgs_num= myx_grt_list_item_count(msgs_list);
    msgs->msgs= g_new0(MYX_GRT_MSG, msgs->msgs_num);

    for (i= 0; i < msgs->msgs_num; i++)
    {
      MYX_GRT_MSG *msg= msgs->msgs + i;
      MYX_GRT_VALUE *msg_value= myx_grt_list_item_get(msgs_list, i);
      MYX_GRT_VALUE *msg_detail_value= myx_grt_dict_item_get_value(msg_value, "details");

      msg->msg= g_strdup(myx_grt_dict_item_get_as_string(msg_value, "msg"));
      msg->msg_type= myx_grt_dict_item_get_as_int(msg_value, "msgType");
      msg->progress= myx_grt_dict_item_get_as_int(msg_value, "progress");

      if (msg_detail_value)
      {
        int count= myx_grt_list_item_count(msg_detail_value);

        if (count > 0)
        {
          int j;

          msg->msg_detail= g_new0(MYX_STRINGLIST, 1);
          msg->msg_detail->strings_num= count;
          msg->msg_detail->strings= g_malloc0(sizeof(char *) * count);

          for (j= 0; j < count; j++)
          {
            msg->msg_detail->strings[j]= g_strdup(myx_grt_list_item_get_as_string(msg_detail_value, j));
          }
        }
      }
    }
  }

  return msgs;
}


/** 
 ****************************************************************************
 * @brief 
 *
 * @param msgs messages to be freed
 *****************************************************************************/
void myx_grt_messages_free(MYX_GRT_MSGS *msgs)
{
  unsigned int i;
  
  if (msgs)
  {
    for (i= 0; i < msgs->msgs_num; i++)
    {
      g_free(msgs->msgs[i].msg);
      if (msgs->msgs[i].msg_detail)
        myx_free_stringlist(msgs->msgs[i].msg_detail);
    }
    g_free(msgs->msgs);
    g_free(msgs);
  }
}



/** 
 ****************************************************************************
 * @brief Add a message to the GRT message queue.
 *
 * @param grt
 * @param msg_type  
 * @param message  message string
 * @param details  optional string list for details of the message
 * @param copy_details if 1, the function will make a copy of the details 
 *              parameter
 * @param progress the current progress value (1 ... 100) or -1 
 *
 *****************************************************************************/
void myx_grt_messages_stack_add(MYX_GRT *grt, int msg_type, const char *message, MYX_STRINGLIST *details,
                     int copy_details, int progress)
{
  MYX_GRT_MSG *msg;
  
  GRT_ENTER(grt);
  
  if (!grt->msgs)
    grt->msgs= g_new0(MYX_GRT_MSGS, 1);

  grt->msgs->msgs_num++;
  grt->msgs->msgs= g_realloc(grt->msgs->msgs,
                             grt->msgs->msgs_num*sizeof(MYX_GRT_MSG));
  msg= grt->msgs->msgs+grt->msgs->msgs_num-1;

  msg->msg_type= msg_type;
  msg->msg= g_strdup(message);
  if (copy_details && details)
  {
    unsigned int i;
    msg->msg_detail= g_new0(MYX_STRINGLIST, 1);
    msg->msg_detail->strings= g_new0(char*, details->strings_num);
    for (i= 0; i < details->strings_num; i++)
      msg->msg_detail->strings[i]= g_strdup(details->strings[i]);
  }
  else
    msg->msg_detail= details;
  
  GRT_LEAVE(grt);
}

void myx_grt_messages_stack_add_message(MYX_GRT *grt, const char *message, 
                                        MYX_STRINGLIST *details, int copy_details, ...)
{
  char *msg;
  va_list args;

  va_start(args, copy_details);

  msg= g_strdup_vprintf(message, args);

  va_end(args);


  myx_grt_messages_stack_add(grt, 0, msg, details, copy_details, 0);

  myx_grt_messages_stack_flush(grt, 0);
}

void myx_grt_messages_stack_add_error(MYX_GRT *grt, const char *message, 
                                      MYX_STRINGLIST *details, int copy_details, ...)
{
  char *msg;
  va_list args;

  va_start(args, copy_details);

  msg= g_strdup_vprintf(message, args);

  va_end(args);

  myx_grt_messages_stack_add(grt, 1, msg, details, copy_details, 0);

  myx_grt_messages_stack_flush(grt, 0);
}

/** 
 ****************************************************************************
 * @brief Processes the messages stored on the GRT message stack.
 *
 * @param grt  
 * @param count  number of messages to fetch. 0 will return all messages.
 *
 * @return Fetched messages or NULL, if there's none. The returned struct
 * must be freed with myx_grt_free_msgs()
 *****************************************************************************/
void myx_grt_messages_stack_flush(MYX_GRT *grt, unsigned int count)
{
  //unsigned int i;
  MYX_GRT_MSGS *msgs;
  
  GRT_ENTER(grt);
  
  // return what was requested

  if (!grt->msgs)
    GRT_LEAVE(grt);
  
  if (count == 0)
  {
    msgs= grt->msgs;
    grt->msgs= NULL;
  }
  else
  {
    unsigned int i;

    if (count > grt->msgs->msgs_num)
      count= grt->msgs->msgs_num;

    msgs= g_new0(MYX_GRT_MSGS, 1);
    msgs->msgs= g_new0(MYX_GRT_MSG, count);
    for (i= 0; i < count; i++)
    {
      msgs->msgs[i]= grt->msgs->msgs[i];
    }
    memmove(grt->msgs->msgs, grt->msgs->msgs+count,
            (grt->msgs->msgs_num-count)*sizeof(MYX_GRT_MSG));
    grt->msgs->msgs_num-= count;
  }

  // call the callback function to process the messages
  if(grt->process_messages != NULL)
    grt->process_messages(msgs, grt->process_messages_data);

  myx_grt_messages_free(msgs);

  GRT_LEAVE(grt);
}


char * myx_grt_get_guid()
{
  /* GUIDs must be no more than 50 chars */
  
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  GUID guid;
  WCHAR guid_wstr[50];
  char guid_str[200];

  int len;

  CoCreateGuid(&guid);
  len= StringFromGUID2(&guid, (LPOLESTR) guid_wstr, 50);

  //Covert GUID from WideChar to utf8
  WideCharToMultiByte(CP_UTF8, 0, (LPCWSTR) guid_wstr, len,
    (LPSTR) guid_str, 200, NULL, NULL);

  return g_strdup(guid_str);
#elif defined(__APPLE__)
  CFUUIDRef uid;
  CFStringRef str;
  char *data;
  
  uid= CFUUIDCreate(NULL);
  str= CFUUIDCreateString(NULL, uid);
  
  data= g_malloc(sizeof(gchar)*(CFStringGetLength(str)+1));
  
  CFStringGetCString(str, data, CFStringGetLength(str)+1, kCFStringEncodingUTF8);
  
  CFRelease(uid);
  CFRelease(str);
  
  return data;
#else
  {
    uuid_t gid;
    char buffer[40];
    uuid_generate_time(gid);
    uuid_unparse(gid, buffer);
    return g_strdup(buffer);
  }
#endif
}



const char *myx_grt_error_string(MYX_GRT_ERROR error)
{
  switch (error)
  {
  case MYX_GRT_NO_ERROR:
    return "Success";
  case MYX_GRT_INTERNAL_ERROR:
    return "Internal error";
  case MYX_GRT_BAD_PATH:
    return "Invalid path";
  case MYX_GRT_CANT_OPEN_FILE:
    return "Cannot open file";
  case MYX_GRT_BAD_FUNCTION:
    return "Invalid function";
  case MYX_GRT_DUPLICATE_ENTRY:
    return "Insertion of duplicate name";
  case MYX_GRT_BAD_VALUE:
    return "Invalid value";
  case MYX_GRT_BAD_DATA:
    return "Invalid data";
  case MYX_GRT_VALIDATION_ERROR:
    return "Validation error";
  case MYX_GRT_FUNCTION_CALL_ERROR:
    return "Error calling function";
  case MYX_GRT_MODULE_INIT_ERROR:
    return "Error initializing module";
  case MYX_GRT_BAD_MODULE:
    return "Invalid module";
  case MYX_GRT_UNKNOWN_MODULE_TYPE:
    return "Unknown module type";
  case MYX_GRT_JAVA_NOT_FOUND:
    return "Java JRE not found";
  case MYX_GRT_JAVA_REGISTRY_CORRUPTED:
    return "Java registry corrupted";
  case MYX_GRT_JAVA_JRE_CANNOT_BE_LOADED:
    return "Java JRE cannot be loaded";

  case MYX_GRT_SHELL_UNSUPPORTED:
    return "Shell type is not supported";
  case MYX_GRT_SHELL_ALREADY_LOADED:
    return "Shell was already initialized";
  case MYX_GRT_SHELL_INIT_ERROR:
    return "Error initializing shell";

  default:
    return "Unknown error";
  }
}

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
int myx_free_lib_stringlist(MYX_STRINGLIST *stringlist)
{
  return myx_free_stringlist(stringlist);
}
#endif
