#include "myx_grt_private.h"


typedef struct {
  MYX_GRT *grt;
  GList *notifications;
} MYX_GRT_NOTIFICATION_QUEUE;



MYX_GRT_NOTIFICATION_NAME myx_grt_intern_notification_name(const char *name)
{
  static GHashTable *names= NULL;
  static GStaticMutex mutex;
  char *iname;
  
  g_static_mutex_lock(&mutex);
  
  if (!names)
    names= g_hash_table_new_full(g_str_hash, g_str_equal, NULL, g_free);
 
  iname= (char*)g_hash_table_lookup(names, name);
  if (!iname)
  {
    iname= g_strdup(name);
    g_hash_table_insert(names, iname, iname);
  }

  g_static_mutex_unlock(&mutex);

  return iname;
}


MYX_GRT_NOTIFICATION *myx_grt_notification_create(MYX_GRT_VALUE *object, const char *name, MYX_GRT_VALUE *data)
{
  MYX_GRT_NOTIFICATION *notif= g_new0(MYX_GRT_NOTIFICATION, 1);

  notif->object= myx_grt_value_retain(object);
  notif->name= myx_grt_intern_notification_name(name);
  notif->argdata= myx_grt_value_retain(data);
  notif->refcount= 1;

  return notif;
}


int myx_grt_notification_retain(MYX_GRT_NOTIFICATION *notif)
{
  g_return_val_if_fail(notif != NULL, -1);
  notif->refcount++;
  return 0;
}


int myx_grt_notification_release(MYX_GRT_NOTIFICATION *notif)
{
  g_return_val_if_fail(notif != NULL, -1);

  notif->refcount--;
  if (notif->refcount <= 0)
  {
    myx_grt_value_release(notif->object);
    myx_grt_value_release(notif->argdata);
    g_free(notif);
  }
  return 0;
}



/*
 * @brief Enables the notification listening for the current thread
 *
 * Deferred notifications will only be delivered to threads that call this function.
 * Non-deferred notifications are delivered normally in the thread that posts the notification.
 */
void myx_grt_notification_listening_enable(MYX_GRT *grt)
{
  g_static_rec_mutex_lock(&grt->notification_queue_mutex);
        
  if (!grt->notification_queues)
  {
    grt->notification_queues= g_hash_table_new(g_direct_hash, g_direct_equal);
  }

  if (!g_hash_table_lookup(grt->notification_queues, g_thread_self()))
  {
    MYX_GRT_NOTIFICATION_QUEUE *queue= g_new0(MYX_GRT_NOTIFICATION_QUEUE, 1);

    queue->grt= grt;

    g_hash_table_insert(grt->notification_queues, g_thread_self(), queue);
  }

  g_static_rec_mutex_unlock(&grt->notification_queue_mutex);
}



/*
 * @brief Posts a notification of the given name and object.
 *
 * @param object - object the notification is associated with
 * @param name - of the notification to be sent
 * @param later - if 0, the notification is sent immediately. if 1, it will be deferred and queued
 * 
 */
int myx_grt_notification_post_named(MYX_GRT *grt, MYX_GRT_VALUE *object, const char *name, MYX_GRT_VALUE *data, int later)
{
  MYX_GRT_NOTIFICATION *notif= myx_grt_notification_create(object, name, data);

  myx_grt_notification_post(grt, notif, later);

  myx_grt_notification_release(notif);

  return 0;
}


static int _listener_matches_notification(MYX_GRT_LISTENER *listener, MYX_GRT_NOTIFICATION *notification)
{
  // check if notif name matches
  if (listener->name && strcmp(notification->name, listener->name)!=0) return 0;
  // check if object matches
  if (listener->object && notification->object != listener->object) return 0;

  return 1;
}


static int _send_notification(MYX_GRT *grt, MYX_GRT_NOTIFICATION *notification)
{
  MYX_GRT_LISTENER *listener= grt->listeners;
  GThread *curthread= g_thread_self();

  while (listener)
  {
    if (listener->thread == curthread && _listener_matches_notification(listener, notification))
    {
      (*listener->callback)(notification, listener->userdata);
    }
    listener= listener->next;           
  }
  return 0;
}


static void queue_notification(gpointer key, gpointer value, gpointer data)
{
  MYX_GRT_NOTIFICATION_QUEUE *queue= (MYX_GRT_NOTIFICATION_QUEUE*)value;
  MYX_GRT_NOTIFICATION *notification= (MYX_GRT_NOTIFICATION*)data;
  GThread *curthread= g_thread_self();
  MYX_GRT_LISTENER *listener= queue->grt->listeners;
  int ok= 0;

  // check if there are listeners for this notification in this thread 
  // if not, don't add it
  while (listener)
  {
    if (listener->thread == curthread && _listener_matches_notification(listener, notification))
    {
      ok= 1;
      break;
    }
    listener= listener->next;           
  }

  if (ok)
  {
    queue->notifications= g_list_prepend(queue->notifications, notification);
    myx_grt_notification_retain(notification);
  }
}


/*
 * @brief Posts a notification to all listeners.
 *
 * The notification is posted to all listeners that match it. If later is 0, all listeners
 * in the *current thread only* will be called during execution of the function. If later is 1,
 * all listeners across all threads that have listeners and have enabled notification queues
 * will be called as soon as myx_grt_notifications_flush() is called in each thread (and the 
 * function returns immediately). Note that notifications will only be queued to threds that
 * have its queue enabled at posting time.
 *
 * @param notification - the notification to be posted
 * @param later - if 0, the notification is sent immediately. if 1, it will be deferred and queued
 *
 */
int myx_grt_notification_post(MYX_GRT *grt, MYX_GRT_NOTIFICATION *notification, int later)
{
  if (later)
  {
    g_return_val_if_fail(grt->notification_queues!=NULL, -1);

    g_static_rec_mutex_lock(&grt->notification_queue_mutex);

    g_hash_table_foreach(grt->notification_queues, queue_notification, notification);
  }
  else
  {
    g_static_rec_mutex_lock(&grt->notification_queue_mutex);

    _send_notification(grt, notification);
  }
  g_static_rec_mutex_unlock(&grt->notification_queue_mutex);
}


int myx_grt_notifications_flush(MYX_GRT *grt)
{
  MYX_GRT_NOTIFICATION_QUEUE *queue;
  
  g_static_rec_mutex_lock(&grt->notification_queue_mutex);

  queue= g_hash_table_lookup(grt->notification_queues, g_thread_self());
  if (!queue)
  {
    g_static_rec_mutex_unlock(&grt->notification_queue_mutex);
    g_warning("notification queue not enabled for current thread");
    return -1;
  }

  while (queue->notifications)
  {
    MYX_GRT_NOTIFICATION *notification= (MYX_GRT_NOTIFICATION*)queue->notifications->data;

    _send_notification(grt, notification);

    myx_grt_notification_release(notification);

    queue->notifications= g_list_remove(queue->notifications, notification);
  }
  g_static_rec_mutex_unlock(&grt->notification_queue_mutex);
  return 0;
}




/*
 * @brief Add a notification listener.
 *
 * Add a notification listener to the GRT environment. The current thread must have had 
 *
 * @param object - if not NULL, will restrict listening to notifications from the specific object
 * @param name - if not NULL, will restrict listening to notifications of the given name
 */
int myx_grt_listener_add(MYX_GRT *grt, MYX_GRT_VALUE *object, MYX_GRT_NOTIFICATION_NAME name, MYX_GRT_LISTENER_CALLBACK callback, void *data)
{
  MYX_GRT_LISTENER *listener;

#if 0
  if (g_list_find(grt->notification_enabled_threads, g_thread_self())!=NULL)
  {
    g_warning("adding listener to thread without a notification queue");
  }
#endif

  g_static_rec_mutex_lock(&grt->notification_queue_mutex);

  listener= g_new0(MYX_GRT_LISTENER, 1);
  listener->name= name;
  listener->object= object;
  listener->userdata= data;
  listener->callback= callback;
  listener->thread= g_thread_self();

  listener->next= grt->listeners;
  grt->listeners= listener;

  g_static_rec_mutex_unlock(&grt->notification_queue_mutex);

  return 0;
}


static void free_listener(MYX_GRT_LISTENER *listener)
{
  g_free(listener);
}


int myx_grt_listener_remove(MYX_GRT *grt, MYX_GRT_LISTENER_CALLBACK callback, void *data, MYX_GRT_VALUE *object, const char *name, MYX_GRT_LISTENER_MASK mask)
{
  MYX_GRT_LISTENER *node, *next, *prev;

  g_static_rec_mutex_lock(&grt->notification_queue_mutex);

  prev= 0;
  node= grt->listeners;
  while (node)
  {
    next= node->next;

    if ( ((mask & MYX_GRT_LMASK_CURRENT_THREAD) == 0 || node->thread == g_thread_self()) &&
         ((mask & MYX_GRT_LMASK_CALLBACK) == 0 || (callback == node->callback && data == node->userdata)) &&
         ((mask & MYX_GRT_LMASK_OBJECT) == 0 || object == node->object) &&
         ((mask & MYX_GRT_LMASK_NAME) == 0 || (name == node->name || (name && strcmp(name, node->name)==0))) )
    {
      free_listener(node);
      if (prev)
        prev->next= next;
      else
        grt->listeners= next;
    }
    else
      prev= node;
    node= next;
  }
  g_static_rec_mutex_unlock(&grt->notification_queue_mutex);
  return 0;
}


