/* Copyright (c) 2004, 2005 MySQL AB
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.
  
   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
 */


#include "myx_grt_private.h"

//----------------------------------------------------------------------------------------------------------------------

/*
 * @brief return the loader struct for the required type
 *
 * @param grt - the GRT environment
 * @param type - type of module loader to look up
 * @return NULL if error, the loader if it's found.
 */
MYX_GRT_MODULE_LOADER *myx_grt_get_loader_of_type(MYX_GRT *grt, MYX_GRT_MODULE_TYPE type)
{
  unsigned int i;
  for (i= 0; i < grt->loaders_num; i++)
  {
    if (grt->loaders[i]->loader_type == type)
      return grt->loaders[i];
  }
  return NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * @brief myx_grt_add_module -- adds a module to the list of modules of the GRT
 *
 * @param grt grt to add module
 * @param module to add. Must not be freed
 * @return MYX_GRT_NO_ERROR
 */
MYX_GRT_ERROR myx_grt_add_module(MYX_GRT *grt, MYX_GRT_MODULE *module)
{  
  if (grt->options & MYX_GRT_VERBOSE)
    myx_grt_printf(grt, "GRT registering module %s\n", module->name);

  grt->modules= g_realloc(grt->modules, sizeof(MYX_GRT_MODULE*)*(grt->modules_num+1));
  grt->modules[grt->modules_num++]= module;
  
  myx_grt_notify_listeners(grt, GRT_MODULE_ADD_NOTIFICATION, module);

  return MYX_GRT_NO_ERROR;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Searchs the internal module list and returns the first module with the given name (if there is one).
 *
 * @param grt The runtime to operate on.
 * @param name The name of the module to look for.
 * @return The module reference if one could be found. Otherwise NULL.
 */
MYX_GRT_MODULE* myx_grt_find_module(MYX_GRT *grt, char* name)
{
  MYX_GRT_MODULE* Result = NULL;
  unsigned int I;
  for (I = 0; I < grt->modules_num; ++I)
    if (strcmp(grt->modules[I]->name, name) == 0)
    {
      Result = grt->modules[I];
      break;
    };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Removes the given module from the list of modules in the given runtime.
 *
 * @param grt grt to add module
 * @param module to add. Must not be freed
 */
void myx_grt_remove_module(MYX_GRT *grt, MYX_GRT_MODULE *module)
{
  unsigned int I;
  for (I = 0; I < grt->modules_num; ++I)
    if (grt->modules[I] == module)
    {
      if (I < grt->modules_num - 1)
        memcpy(grt->modules[I], grt->modules[I + 1], sizeof(MYX_GRT_MODULE*) * (grt->modules_num - I - 1));
      grt->modules = g_realloc(grt->modules, sizeof(MYX_GRT_MODULE*) * (--grt->modules_num));
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------


  
/*
 * Function parameter list syntax:
 * 
 * param_list::= param[, param_list]
 * param::= type [ label]
 * label::= a name for the parameter
 * type::= i | r | s | l[<content_spec>] | d[<content_spec>] | o[@struct]
 * content_spec::= i | r | s | o@struct
 * struct::= name of a valid struct
 * 
 * Ex.:
 *  doSomething:s:i count,l<i> poslist,o<db.mysql.Table> table,d args
 */
int myx_grt_parse_function_spec(const char *spec, MYX_GRT_FUNCTION *func)
{  
  if (spec)
  {
    char **parts= g_strsplit(spec, ":", 0);
    char **args;
    int i, argc;
    
    if (g_strv_length(parts) != 3)
    {
      g_warning("Error parsing module function spec: %s", spec);
      g_strfreev(parts);
      return -1;
    }

    func->name= g_strdup(parts[0]);
    func->return_string= g_strdup(parts[1]);
    func->param_string= g_strdup(parts[2]);

    // parse return type
    if (*parts[1] == 'i' && strlen(parts[1])==1)
      func->return_type= MYX_INT_VALUE;
    else if (*parts[1] == 'r' && strlen(parts[1])==1)
      func->return_type= MYX_REAL_VALUE;
    else if (*parts[1] == 's' && strlen(parts[1])==1)
      func->return_type= MYX_STRING_VALUE;
    else if (*parts[1] == 'l' || *parts[1] == 'd')
    {
      if (*parts[1] == 'l')
        func->return_type= MYX_LIST_VALUE;
      else
        func->return_type= MYX_DICT_VALUE;
      if (parts[1][1]=='<')
      {
        if (parts[1][2]=='i' && parts[1][3]=='>')
          func->return_content_type= MYX_INT_VALUE;
        else if (parts[1][2]=='r' && parts[1][3]=='>')
          func->return_content_type= MYX_REAL_VALUE;
        else if (parts[1][2]=='s' && parts[1][3]=='>')
          func->return_content_type= MYX_STRING_VALUE;
        else if (parts[1][2]=='o')
        {
          func->return_content_type= MYX_OBJECT_VALUE;
          if (parts[1][3]=='@')
          {
            char *ptr;
            func->return_struct_name= g_strdup(parts[1]+4);
            ptr= strchr(func->return_struct_name, '>');
            if (ptr) *ptr= 0;
          }
        }
        else
        {
          g_warning("Error parsing module function spec: %s  '%s'", spec, parts[1]);
          g_strfreev(parts);
          return -1;
        }
      }
    }
    else if (*parts[1] == 'o')
    {
      func->return_type= MYX_OBJECT_VALUE;
      if (parts[1][1]== '@')
      {
        func->return_struct_name= g_strdup(parts[1]+2);
      }
    }
    else
    {
      g_warning("Error parsing module function spec: %s  '%s'", spec, parts[1]);
      g_strfreev(parts);
      return -1;
    }

    // parse arguments    
    args= g_strsplit(parts[2], ",", 0);
    g_strfreev(parts);

    argc= g_strv_length(args);
    func->params_num= argc;
    func->params= g_new0(MYX_GRT_FUNCTION_PARAM, argc);

    for (i= 0; i < argc; i++)
    {
      char *arg= args[i];
      MYX_GRT_FUNCTION_PARAM *fp= func->params+i;
      char *ptr;

      // the optional arg name
      ptr= strchr(arg, ' ');
      if (ptr)
      {
        fp->name= g_strdup(ptr+1);
        *ptr= 0;
      }

      if (*args[1] == 'i' && strlen(args[1])==1)
        fp->type= MYX_INT_VALUE;
      else if (*args[1] == 'r' && strlen(args[1])==1)
        fp->type= MYX_REAL_VALUE;
      else if (*args[1] == 's' && strlen(args[1])==1)
        fp->type= MYX_STRING_VALUE;
      else if (*args[1] == 'l' || *args[1] == 'd')
      {
        if (*args[1] == 'l')
          fp->type= MYX_LIST_VALUE;
        else
          fp->type= MYX_DICT_VALUE;
        if (args[1][1]=='<')
        {
          if (args[1][2]=='i' && args[1][3]=='>')
            fp->content_type= MYX_INT_VALUE;
          else if (args[1][2]=='r' && args[1][3]=='>')
            fp->content_type= MYX_REAL_VALUE;
          else if (args[1][2]=='s' && args[1][3]=='>')
            fp->content_type= MYX_STRING_VALUE;
          else if (args[1][2]=='o')
          {
            fp->content_type= MYX_OBJECT_VALUE;
            if (args[1][3]=='@')
            {
              char *ptr;
              fp->struct_name= g_strdup(args[1]+4);
              ptr= strchr(fp->struct_name, '>');
              if (ptr) *ptr= 0;
            }
          }
          else
          {
            g_warning("Error parsing module function spec: %s  '%s'", spec, args[1]);
            g_strfreev(args);
            return -1;
          }
        }
      }
      else if (*args[1] == 'o')
      {
        fp->type= MYX_OBJECT_VALUE;
        if (args[1][1]== '@')
        {
          fp->struct_name= g_strdup(args[1]+2);
        }
      }
      else
      {
        g_warning("Error parsing module function spec: %s  '%s'", spec, args[1]);
        g_strfreev(args);
        return -1;
      }
    }

    g_strfreev(args);
  }
  return 0;
}

