
#include "myx_grt_private.h"


//--------------------------------------------------------------------------------
// C++ Object Wrapper Generator
//--------------------------------------------------------------------------------

#define grt_lgpl "/*\n" \
" Generic Runtime Library (GRT)\n" \
" Copyright (C) 2005~2007 MySQL AB\n\n" \
" This library is free software; you can redistribute it and/or\n" \
" modify it under the terms of the GNU Lesser General Public\n" \
" License as published by the Free Software Foundation; either\n" \
" version 2.1 of the License, or (at your option) any later version.\n\n" \
" This library is distributed in the hope that it will be useful,\n" \
" but WITHOUT ANY WARRANTY; without even the implied warranty of\n" \
" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n" \
" Lesser General Public License for more details.\n\n" \
" You should have received a copy of the GNU Lesser General Public\n" \
" License along with this library; if not, write to the Free Software\n" \
" Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA\n" \
" */\n\n"

static const char *class_tmpl=
  "class %class_name% : public %base_class_name% {\n"
  "  typedef %base_class_name% super;\n"
  "protected:\n"
  "  %class_name%(GRT *grt, const std::string &structname, const grt::DictValue &args)\n"
  "    : super(grt, structname, args)\n"
  "  {\n"
  "  }\n"
  "public:\n"
  "  %class_name%() {};\n"
  "  %class_name%(MYX_GRT_VALUE *value) : %base_class_name%(value)\n"
  "  {\n"
  "    if (!myx_grt_obj_is_or_inherits_from(value, \"%struct_name%\"))\n"
  "      throw InvalidGRTType();\n"
  "  }\n"
  "  %class_name%(GRT *grt)\n"
  "    : super(grt, \"%struct_name%\", grt::DictValue())\n"
  "  {\n"
  "  }\n"
  "  static inline %class_name% cast_from(const Value &svalue) {\n"
  "    if (svalue.type() != MYX_OBJECT_VALUE)\n"
  "      throw InvalidGRTType();\n"
  "    return %class_name%(svalue.grt_value());\n"
  "  }\n"

  //cast_from()
  
  "\n"
  "%property_defs%"
  "};\n";


static const char *property_tmpl=
  "  inline %prop_type% %prop_name%() const { return %prop_type%::cast_from(get_member(\"%prop_name%\")); };\n"
  "  inline void %prop_name%(const %prop_type% &value) { set_member(\"%prop_name%\", value); };\n";

static const char *int_property_tmpl=
  "  inline int %prop_name%() const { return get_int_member(\"%prop_name%\"); };\n"
  "  inline void %prop_name%(const %prop_type% &value) { set_member(\"%prop_name%\", value); };\n"
  "  inline void %prop_name%(int value) { set_member(\"%prop_name%\", value); };\n";

static const char *double_property_tmpl=
  "  inline double %prop_name%() const { return get_double_member(\"%prop_name%\"); };\n"
  "  inline void %prop_name%(const %prop_type% &value) { set_member(\"%prop_name%\", value); };\n"
  "  inline void %prop_name%(double value) { set_member(\"%prop_name%\", value); };\n";

static const char *string_property_tmpl=
  "  inline std::string %prop_name%() const { return get_string_member(\"%prop_name%\"); };\n"
  "  inline void %prop_name%(const %prop_type% &value) { set_member(\"%prop_name%\", value); };\n"
  "  inline void %prop_name%(const std::string &value) { set_member(\"%prop_name%\", value); };\n";


static const char *obj_property_tmpl=
  "  inline %prop_type% %prop_name%() const;\n"
  "  inline void %prop_name%(const %prop_type% &value);\n";


static const char *list_property_tmpl=
  "  inline %prop_type% %prop_name%() const { return %prop_type%::cast_from(get_member(\"%prop_name%\")); };\n"
  "  inline void %prop_name%(const %prop_type% &value) { set_member(\"%prop_name%\", value); };\n"
  "  inline void %prop_name%_add(%prop_content_type% &value);\n"
  "  inline void %prop_name%_del(unsigned int index);\n";


static const char *obj_property_impl_tmpl=
  "inline %prop_type% %class_name%::%prop_name%() const { return %prop_type%::cast_from(get_member(\"%prop_name%\")); };\n"
  "inline void %class_name%::%prop_name%(const %prop_type% &value) { set_member(\"%prop_name%\", value); };\n";

static const char *list_property_impl_tmpl=
  "inline void %class_name%::%prop_name%_add(%prop_content_type% &value) { add_to_member(\"%prop_name%\", value); };\n"
  "inline void %class_name%::%prop_name%_del(unsigned int index) { del_from_member(\"%prop_name%\", index); };\n";




static MYX_GRT_ERROR export_cpp_struct_to_class(MYX_GRT_STRUCT *gstruct,
                                                FILE *f, GList **post)
{
  char **props;
  unsigned int i;
  char *s;
  char *tmp;
  char *class_name;

  if (gstruct->members_num > 0)
    props= g_new0(char*, gstruct->members_num+1);
  else
    props= NULL;
  
  class_name= str_g_subst(gstruct->name, ".", "_");  
  
  for (i= 0; i < gstruct->members_num; i++)
  {
    char *p;
    char *tmp;
    char *type= NULL;
    const char *templ= property_tmpl;
    char *ctype= NULL;
    
    switch (gstruct->members[i].value_type)
    {
    case MYX_INT_VALUE:
      type= g_strdup("grt::IntValue");
      templ= int_property_tmpl;
      break;
    case MYX_REAL_VALUE:
      type= g_strdup("grt::DoubleValue");
      templ= double_property_tmpl;
      break;
    case MYX_STRING_VALUE:
      type= g_strdup("grt::StringValue");
      templ= string_property_tmpl;
      break;
    case MYX_LIST_VALUE:
      switch (gstruct->members[i].content_type)
      {
      case MYX_INT_VALUE: 
        type= g_strdup("grt::ListValue<grt::IntValue>");
        ctype= g_strdup("grt::IntValue");
        break;
      case MYX_REAL_VALUE:
        type= g_strdup("grt::ListValue<grt::DoubleValue>");
        ctype= g_strdup("grt::DoubleValue");
        break;
      case MYX_STRING_VALUE:
        type= g_strdup("grt::ListValue<grt::StringValue>");
        ctype= g_strdup("grt::StringValue");
        break;
      case MYX_LIST_VALUE:
        type= g_strdup("grt::ListValue<grt::BaseListValue>");
        ctype= g_strdup("grt::BaseListValue");
        break;
      case MYX_DICT_VALUE:
        type= g_strdup("grt::ListValue<grt::DictValue>");
        ctype= g_strdup("grt::DictValue");
        break;
      case MYX_OBJECT_VALUE:
        tmp= str_g_subst(gstruct->members[i].content_struct_name, ".", "_");
        ctype= g_strdup_printf("%s ", tmp);
        type= g_strdup_printf("grt::ListValue<%s> ", tmp);
        g_free(tmp);
        break;
      }
      templ= list_property_tmpl;
      break;
    case MYX_DICT_VALUE:
      type= g_strdup("grt::DictValue");
      break;
    case MYX_OBJECT_VALUE:
      templ= obj_property_tmpl;
      tmp= str_g_subst(gstruct->members[i].object_struct_name, ".", "_");
      type= g_strdup_printf("%s", tmp);
      g_free(tmp);
      break;
    default:
      g_warning("invalid type in member '%s' of struct '%s'",
                gstruct->members[i].name, gstruct->name);
    }

    p= str_g_subst(templ, "%prop_name%", gstruct->members[i].name);
    p= str_g_replace(p, "%prop_type%", type);
    if (ctype)
    {
      char *imp;
      
      imp= str_g_subst(list_property_impl_tmpl, "%class_name%", class_name);
      imp= str_g_replace(imp, "%prop_content_type%", ctype);
      imp= str_g_replace(imp, "%prop_name%", gstruct->members[i].name);
      *post= g_list_prepend(*post, imp);

      p= str_g_replace(p, "%prop_content_type%", ctype);
      g_free(ctype);
    }
    else if (gstruct->members[i].value_type == MYX_OBJECT_VALUE)
    {
      char *imp;

      imp= str_g_subst(obj_property_impl_tmpl, "%class_name%", class_name);
      imp= str_g_replace(imp, "%prop_name%", gstruct->members[i].name);
      imp= str_g_replace(imp, "%prop_type%", type);
      *post= g_list_prepend(*post, imp);
    }    
    g_free(type);

    props[i]= p;
  }
  if (props)
    props[i]= NULL;
  
  s= str_g_subst(class_tmpl, "%struct_name%", gstruct->name);
  s= str_g_subst(s, "%class_name%", class_name);
  g_free(class_name);
  if (!gstruct->parent_struct_name || !*gstruct->parent_struct_name)
    s= str_g_replace(s, "%base_class_name%", "grt::ObjectValue");
  else
  {
    tmp= str_g_subst(gstruct->parent_struct_name, ".", "_");
    s= str_g_replace(s, "%base_class_name%", tmp);
    g_free(tmp);
  }

  if (props)
  {
    tmp= g_strjoinv("", props);
    s= str_g_replace(s, "%property_defs%", tmp);
    g_free(tmp);
    g_strfreev(props);
  }
  else
    s= str_g_replace(s, "%property_defs%", "");
  fprintf(f, "%s", s);
  g_free(s);
  return MYX_GRT_NO_ERROR;
}



MYX_GRT_ERROR myx_grt_struct_export_cpp_classes(MYX_GRT_STRUCTS *gstructs,
                                                const char *output_path)
{
  FILE *f= fopen(output_path, "w+");
  unsigned int i;
  GList *postponed= NULL;
  
  fprintf(f, "/* This file is automatically generated, do not edit */\n");
  fprintf(f, "%s\n", grt_lgpl);
  
  fprintf(f, "\n\n#include \"myx_grtpp.h\"\n\n");

  // forward declarations
  for (i= 0; i < gstructs->structs_num; i++)
  {
    char *tmp= str_g_subst(gstructs->structs[i].name, ".", "_");
    fprintf(f, "class %s;\n", tmp);
    g_free(tmp);
  }
  
  fprintf(f, "\n\n");
  
  // export classes
  for (i= 0; i < gstructs->structs_num; i++)
  {
    export_cpp_struct_to_class(gstructs->structs+i, f, &postponed);
    
    fprintf(f, "\n\n");
  }

  // export postponed stuff
  if (postponed)
  {
    GList *node= postponed;
    fprintf(f, "//\n");
    while (node)
    {
      fprintf(f, "%s", (char*)node->data);
      g_free(node->data);
      node= node->next;
    }
    g_list_free(postponed);
  }

  fclose(f);
  
  return MYX_GRT_NO_ERROR;
}
