

#include "myx_grt_private.h"
#include "myx_grt_public_interface.h"

typedef const char * (*dict_identification_func)(MYX_GRT_VALUE *dict);

static int myx_grt_generic_diff_make(MYX_GRT *grt, MYX_GRT_VALUE *list, 
                                      MYX_GRT_VALUE * path, MYX_GRT_VALUE *source, 
                                      MYX_GRT_VALUE *target, dict_identification_func dict_id_func);


static MYX_GRT_VALUE * myx_append_path(MYX_GRT_VALUE *path, int index, 
                                       const char *postfix, const char *action)
{
  char *path_new= NULL;
  const char *path_cchar= myx_grt_value_as_string(path);
  MYX_GRT_VALUE *retval= NULL;
  if(index >= 0)
  {
    char *idx= g_strdup_printf("%d", index);
    path_new= g_strconcat(action, path_cchar, "/", idx, NULL);
  }
  else
  {
    const char *slash= (postfix && *postfix) ? "/" : "";
    path_new= g_strconcat(action, path_cchar, slash, postfix, NULL);
  }
  retval= myx_grt_value_from_string(path_new);
  g_free(path_new);
  return retval;
}

/**
 ****************************************************************************
 * @brief Compares types of two dict objects
 *
 * @param grt pointer to the GRT environment
 * @param source the source value
 * @param target the target value
 *
 * @return true if the dicts hold the same type, false if not
 *****************************************************************************/
static int myx_grt_dict_compare_types(MYX_GRT *grt, MYX_GRT_VALUE *source, MYX_GRT_VALUE *target)
{
  const char *c1, *c2;
  g_return_val_if_fail(myx_grt_value_get_type(source) == MYX_DICT_VALUE, 0);
  g_return_val_if_fail(myx_grt_value_get_type(target) == MYX_DICT_VALUE, 0);
  g_return_val_if_fail(myx_grt_dict_content_get_type(source) == myx_grt_dict_content_get_type(target), 0);
  c1= myx_grt_dict_struct_get_name(source);
  c2= myx_grt_dict_struct_get_name(target);
  if(c1 && c2)
  {
    g_return_val_if_fail(strcmp(c1, c2) == 0, 0);
  }
  else if(c1 || c2)
  {
    return 0;
  }
  c1= myx_grt_dict_content_get_struct_name(source);
  c2= myx_grt_dict_content_get_struct_name(target);
  if(c1 && c2)
  {
    g_return_val_if_fail(strcmp(c1, c2) == 0, 0);
  }
  else if(c1 || c2)
  {
    return 0;
  }
  return 1;
}

/**
 ****************************************************************************
 * @brief Compares types of two list objects
 *
 * @param grt pointer to the GRT environment
 * @param source the source value
 * @param target the target value
 *
 * @return true if the lists hold the same type, false if not
 *****************************************************************************/
static int myx_grt_list_compare_types(MYX_GRT *grt, MYX_GRT_VALUE *source, MYX_GRT_VALUE *target)
{
  const char *c1, *c2;
  g_return_val_if_fail(myx_grt_value_get_type(source) == MYX_LIST_VALUE, 0);
  g_return_val_if_fail(myx_grt_value_get_type(target) == MYX_LIST_VALUE, 0);
  g_return_val_if_fail(myx_grt_list_content_get_type(source) == myx_grt_list_content_get_type(target), 0);
  c1= myx_grt_list_content_get_struct_name(source);
  c2= myx_grt_list_content_get_struct_name(target);
  if(c1 && c2)
  {
    g_return_val_if_fail(strcmp(c1 ,c2) == 0, 0);
  }
  else if(c1 || c2)
  {
    return 0;
  }
  return 1;
}

/**
 ****************************************************************************
 * @brief Compares types of two objects
 *
 * @param grt pointer to the GRT environment
 * @param source the source value
 * @param target the target value
 *
 * @return true if the values are of the same type, false if not
 *****************************************************************************/
static int myx_grt_value_compare_types(MYX_GRT *grt, MYX_GRT_VALUE *source, MYX_GRT_VALUE *target)
{
  MYX_GRT_VALUE_TYPE st= myx_grt_value_get_type(source);
  MYX_GRT_VALUE_TYPE tt= myx_grt_value_get_type(target);

  g_return_val_if_fail(st == tt, 0);
  if(!myx_grt_value_is_simple_type(source)) 
  {
    if(st == MYX_DICT_VALUE)
    {
      return myx_grt_dict_compare_types(grt, source, target);
    }
    else if(st == MYX_LIST_VALUE)
    {
      return myx_grt_list_compare_types(grt, source, target);
    }
    else
    {
      return 0;
    }
  }
  return 1;
}


static int myx_grt_simple_value_compare(MYX_GRT *grt, MYX_GRT_VALUE *source, MYX_GRT_VALUE *target)
{
  int int_s, int_t;
  double double_s, double_t;
  const char *cchar_s, *cchar_t;
  MYX_GRT_VALUE_TYPE st= myx_grt_value_get_type(source);
  MYX_GRT_VALUE_TYPE tt= myx_grt_value_get_type(target);
  g_return_val_if_fail(st == tt, -2);

  switch(st)
  {
  case MYX_INT_VALUE:
    int_s= myx_grt_value_as_int(source);
    int_t= myx_grt_value_as_int(target);
    return int_s < int_t ? -1 : int_s == int_t ? 0 : 1;
  case MYX_REAL_VALUE:
    double_s= myx_grt_value_as_real(source);
    double_t= myx_grt_value_as_real(target);
    return double_s < double_t ? -1 : double_s == double_t ? 0 : 1;
  case MYX_STRING_VALUE:
    cchar_s= myx_grt_value_as_string(source);
    cchar_t= myx_grt_value_as_string(target);
    return strcmp(cchar_s, cchar_t);
  default:
     break;
  }
  return -2;
}



// returns 0 if there were no changes
static int myx_grt_simple_value_diff_make(MYX_GRT *grt, MYX_GRT_VALUE *list, 
                                           MYX_GRT_VALUE * path, MYX_GRT_VALUE *source, 
                                           MYX_GRT_VALUE *target)
{
  const char *cchar_s, *cchar_t;
  double double_s, double_t;
  int int_s, int_t;
  int retval= 0;

  MYX_GRT_VALUE_TYPE st= myx_grt_value_get_type(source);
  MYX_GRT_VALUE_TYPE tt= myx_grt_value_get_type(target);
  g_return_val_if_fail(st == tt, 0);

  switch(st)
  {
  case MYX_INT_VALUE:
    int_s= myx_grt_value_as_int(source);
    int_t= myx_grt_value_as_int(target);
    if(int_s != int_t)
    {
      // change int value
      myx_grt_list_item_add(list, myx_append_path(path, -1, "", "/"));
      myx_grt_list_item_add(list, myx_grt_value_from_int(int_s));
      myx_grt_bridge_value_change_int(target, int_s);
      retval= 1;
    }
    break;
  case MYX_REAL_VALUE:
    double_s= myx_grt_value_as_real(source);
    double_t= myx_grt_value_as_real(target);
    if(double_s != double_t)
    {
      // change double value
      myx_grt_list_item_add(list, myx_append_path(path, -1, "", "/"));
      myx_grt_list_item_add(list, myx_grt_value_from_real(double_s));
      myx_grt_bridge_value_change_real(target, double_s);
      retval= 1;
    }
    break;
  case MYX_STRING_VALUE:
    cchar_s= myx_grt_value_as_string(source);
    cchar_t= myx_grt_value_as_string(target);
    if(strcmp(cchar_s, cchar_t) != 0)
    {
      // change const char* value
      myx_grt_list_item_add(list, myx_append_path(path, -1, "", "/"));
      myx_grt_list_item_add(list, myx_grt_value_from_string(cchar_s));
      myx_grt_bridge_value_change_string(target, cchar_s);
      retval= 1;
    }
    break;
  default:
    break;
  }

  return retval;
}

/*
static const char *get_dict_id(MYX_GRT_VALUE *dict)
{
  return myx_grt_dict_id_item_as_string(dict);
}
*/
static const char *get_dict_name(MYX_GRT_VALUE *dict)
{
  const char *old_name= myx_grt_dict_item_get_as_string(dict, "oldName");
  if(old_name)
    return old_name;
  return myx_grt_dict_item_get_as_string(dict, "name");
}

// returns 0 if there were no changes
static int myx_grt_list_diff_make(MYX_GRT *grt, MYX_GRT_VALUE *list, 
                                   MYX_GRT_VALUE * path, MYX_GRT_VALUE *source, 
                                   MYX_GRT_VALUE *target,
                                   dict_identification_func dict_id_func)
{
  int next, is_simple;
  const char *s_id, *t_id;
  unsigned int s_count= myx_grt_list_item_count(source);
  unsigned int t_count;
  MYX_GRT_VALUE *s_value, *t_value, *sub_path;
  unsigned int i, j;
  int retval= 0;

  // find possible / * +
restart1:
  for(i= 0; i < s_count; i++)
  {
    next= 0;
    s_value= myx_grt_list_item_get(source, i);
    t_count= myx_grt_list_item_count(target);
    if(myx_grt_value_is_simple_type(s_value))
    {
      for(j= 0; j < t_count; j++)
      {
        t_value= myx_grt_list_item_get(target, j);
        if(myx_grt_simple_value_compare(grt, s_value, t_value) == 0)
        {
          if(i != j)
          {
            // this value could be already matched earlier
            // this is detected by checking if source[j] is equal to target[j]
            if((j < i) && (myx_grt_simple_value_compare(grt, myx_grt_list_item_get(source, j), t_value) == 0))
              continue;
            
            // move s_value in diff
            sub_path= myx_append_path(path, i, "", "*");
            myx_grt_list_item_add(list, sub_path);
            myx_grt_value_release(sub_path);
            myx_grt_list_item_add(list, myx_grt_value_from_int(j));

            myx_grt_list_item_del(target, j);
            if(i > myx_grt_list_item_count(target))
              myx_grt_list_item_insert(target, myx_grt_list_item_count(target), s_value);
            else
              myx_grt_list_item_insert(target, i, s_value);

            retval= 1;
          }
          next= 1;
          break;
        }
      }
      if(next == 0) // s_value is not found in target
      {
        // add s_value to diff
        sub_path= myx_append_path(path, i, "", "+");
        myx_grt_list_item_add(list, sub_path);
        myx_grt_value_release(sub_path);
        s_value= myx_grt_value_dup(s_value, 1);
        myx_grt_list_item_add(list, s_value);
        if(i > myx_grt_list_item_count(target))
          myx_grt_list_item_insert(target, myx_grt_list_item_count(target), s_value);
        else
          myx_grt_list_item_insert(target, i, s_value);
        retval= 1;
        goto restart1;
      }
    }
    else  // s_value is a dict or a list
    {
      //s_id= myx_grt_dict_id_item_as_string(s_value);
      //s_id= myx_grt_dict_item_get_as_string(s_value, "name");

      s_id= dict_id_func(s_value);
      for(j= 0; j < t_count; j++)
      {
        t_value= myx_grt_list_item_get(target, j);
        //t_id= myx_grt_dict_id_item_as_string(t_value);
        //t_id= myx_grt_dict_item_get_as_string(t_value, "name");
        t_id= dict_id_func(t_value);
        if(strcmp(s_id, t_id) == 0)
        {
          sub_path= myx_append_path(path, i, "", "");
          retval= myx_grt_generic_diff_make(grt, list, sub_path, s_value, t_value, dict_id_func) || retval;
          if(i != j)
          {
            // move s_value in diff
            sub_path= myx_append_path(path, i, "", "*");
            myx_grt_list_item_add(list, sub_path);
            myx_grt_value_release(sub_path);
            myx_grt_list_item_add(list, myx_grt_value_from_int(j));

            myx_grt_list_item_del(target, j);
            if(i > myx_grt_list_item_count(target))
              myx_grt_list_item_insert(target, myx_grt_list_item_count(target), s_value);
            else
              myx_grt_list_item_insert(target, i, s_value);
            retval= 1;
          }
          next= 1;
          break;
        }
      }
      if(next == 0) // s_value is not found in target
      {
        // add s_value to diff
        sub_path= myx_append_path(path, i, "", "+");
        myx_grt_list_item_add(list, sub_path);
        myx_grt_value_release(sub_path);
        s_value= myx_grt_value_dup(s_value, 1);
        myx_grt_list_item_add(list, s_value);

        if(i > myx_grt_list_item_count(target))
          myx_grt_list_item_insert(target, myx_grt_list_item_count(target), s_value);
        else
          myx_grt_list_item_insert(target, i, s_value);
        retval= 1;
        goto restart1;
      }
    }
  } // for(i)

  s_count= myx_grt_list_item_count(source);
restart2:
  t_count= myx_grt_list_item_count(target);
  for(j= 0; j < t_count; j++)
  {
    next= 0;
    t_value= myx_grt_list_item_get(target, j);
    is_simple= myx_grt_value_is_simple_type(t_value);
    if(!is_simple)
    {
      //t_id= myx_grt_dict_id_item_as_string(t_value);
      t_id= dict_id_func(t_value);
    }
    for(i= 0; i < s_count; i++)
    {
      s_value= myx_grt_list_item_get(source, i);
      if(is_simple)
      {
        next= (myx_grt_simple_value_compare(grt, s_value, t_value) == 0);
      }
      else  // t_value is dict/list
      {
        //s_id= myx_grt_dict_id_item_as_string(s_value);
        s_id= dict_id_func(s_value);
        next= (strcmp(s_id, t_id) == 0);
      }
      if(next)
      {
        break;
      }
    }
    if(next == 0)
    {
      // del t_value from the diff
      sub_path= myx_append_path(path, j, "", "-");
      myx_grt_list_item_add(list, sub_path);
      myx_grt_value_release(sub_path);
      myx_grt_list_item_add(list, myx_grt_value_from_string(""));
      myx_grt_list_item_del(target, j);
      retval= 1;
      goto restart2;
    }
  }

  return retval;
}

// returns 0 if nothing was changed
static int myx_grt_dict_diff_make(MYX_GRT *grt, MYX_GRT_VALUE *list, 
                                   MYX_GRT_VALUE * path, MYX_GRT_VALUE *source, 
                                   MYX_GRT_VALUE *target, dict_identification_func dict_id_func)
{
  const char *s_key, *t_key;
  MYX_GRT_VALUE *s_value, *t_value, *sub_path;
  unsigned int i, j, next;
  unsigned int s_count;
  unsigned int t_count;
  MYX_GRT_VALUE_TYPE st;
  MYX_GRT_VALUE_TYPE tt;
  int retval= 0;  // the change flag

  // find possible / + -
restart1:
  s_count= myx_grt_dict_item_count(source);
  for(i= 0; i < s_count; i++)
  {
    next= 0;
    myx_grt_dict_item_by_index(source, i, &s_key, &s_value);
    t_count= myx_grt_dict_item_count(target);
    for(j= 0; j < t_count; j++)
    {
      myx_grt_dict_item_by_index(target, j, &t_key, &t_value);
      st= myx_grt_value_get_type(s_value);
      tt= myx_grt_value_get_type(t_value);

      if(strcmp(s_key, t_key) == 0)
      {
        if(st == tt) 
        {
          // diff s_value/t_value
          sub_path= myx_append_path(path, -1, s_key, "");
          retval= myx_grt_generic_diff_make(grt, list, sub_path, s_value, t_value, dict_id_func) || retval;
          myx_grt_value_release(sub_path);
        }
        else if(s_value != NULL)  // values have differenent types just exchange
        {
          sub_path= myx_append_path(path, -1, s_key, "/");
          myx_grt_list_item_add(list, sub_path);
          myx_grt_list_item_add(list, s_value);
          myx_grt_value_release(sub_path);
          retval= 1;
        }
        else 
        {
          myx_grt_list_item_add(list, myx_append_path(path, -1, t_key, "-"));
          myx_grt_list_item_add(list, myx_grt_value_from_string(""));
          myx_grt_dict_item_del(target, t_key);
          retval= 1;
        }
        next= 1;
        break;
      }
    }
    if(next == 0)
    {
      myx_grt_list_item_add(list, myx_append_path(path, -1, s_key, "+"));
      s_value= myx_grt_value_dup(s_value, 1);
      myx_grt_list_item_add(list, s_value);
      myx_grt_dict_item_set_value(target, g_strdup(s_key), s_value);
      retval= 1;
      goto restart1;
    }
  }

  // find possible -
  s_count= myx_grt_dict_item_count(source);
restart2:
  t_count= myx_grt_dict_item_count(target);
  for(j= 0; j < t_count; j++)
  {
    next= 0;
    myx_grt_dict_item_by_index(target, j, &t_key, &t_value);
    for(i= 0; i < s_count; i++)
    {
      myx_grt_dict_item_by_index(source, i, &s_key, &s_value);
      if(strcmp(s_key, t_key) == 0)
      {
        next= 1;
        break;
      }
    }
    if(next == 0)
    {
      // delete s_value from diff
      myx_grt_list_item_add(list, myx_append_path(path, -1, t_key, "-"));
      myx_grt_list_item_add(list, myx_grt_value_from_string(""));
      myx_grt_dict_item_del(target, t_key);
      retval= 1;
      goto restart2;
    }
  }

  return retval;
}

// returns 1 if diff found any changes
static int myx_grt_generic_diff_make(MYX_GRT *grt, MYX_GRT_VALUE *list, 
                                      MYX_GRT_VALUE * path, MYX_GRT_VALUE *source, 
                                      MYX_GRT_VALUE *target, dict_identification_func dict_id_func)
{
  MYX_GRT_VALUE_TYPE st= myx_grt_value_get_type(source);
  MYX_GRT_VALUE_TYPE tt= myx_grt_value_get_type(target);
  g_return_val_if_fail(st == tt, 0);

  switch(st)
  {
  case MYX_ANY_VALUE:
    break;
  case MYX_INT_VALUE:
  case MYX_REAL_VALUE:
  case MYX_STRING_VALUE:
    return myx_grt_simple_value_diff_make(grt, list, path, source, target);
  case MYX_LIST_VALUE:
    return myx_grt_list_diff_make(grt, list, path, source, target, dict_id_func);
  case MYX_DICT_VALUE:
    return myx_grt_dict_diff_make(grt, list, path, source, target, dict_id_func);
  }
  return 0;
}

MYX_GRT_VALUE * myx_grt_value_diff_make_with_params(MYX_GRT *grt, MYX_GRT_VALUE *source, 
                                                    MYX_GRT_VALUE *target, dict_identification_func dict_id_func)
{
  MYX_GRT_VALUE * retval= myx_grt_list_new(MYX_ANY_VALUE, NULL);
  MYX_GRT_VALUE * source_dup, *target_dup;
  MYX_GRT_VALUE * path;

  g_return_val_if_fail(retval != NULL, NULL);
  g_return_val_if_fail(myx_grt_value_compare_types(grt, source, target) == 1, NULL);

  path= myx_grt_value_from_string("");
  source_dup= myx_grt_value_dup(source, 1);
  target_dup= myx_grt_value_dup(target, 1);
  myx_grt_generic_diff_make(grt, retval, path, target_dup, source_dup, dict_id_func);

  return retval;
}

/**
 ****************************************************************************
 * @brief Creates a diff between two GRT values
 *
 *  Recursivly walks through the given value and its members and builds the 
 * diff as a GRT list, that is of the form 
 *
 * {path, change, path, change, path, change}.
 *
 * Example:
 * {"/name", MYX_STRING_VALUE, "./engine", MYX_STRING_VALUE, "-/columns/1", NULL, 
 *  "+/columns/2", MYX_DICT_VALUE}
 *
 * the encoding of the path is the following. If it starts with
 * / a value change
 * - a value remove
 * + a value added
 * * a value changed position in list
 *
 * for lists, the index that was removed or added is given
 *
 * @param grt pointer to the GRT environment
 * @param source the source value
 * @param target the target value
 *
 * @return the diff as a GRT value
 *****************************************************************************/
MYX_GRT_VALUE * myx_grt_value_diff_make(MYX_GRT *grt, MYX_GRT_VALUE *source, MYX_GRT_VALUE *target)
{
  return myx_grt_value_diff_make_with_params(grt, source, target, get_dict_name);
}

/**
 ****************************************************************************
 * @brief Applies a diff to a value
 *
 *  Applies a diff to the given value and also returns the updated value
 *
 * @param grt pointer to the GRT environment
 * @param value the value the diff will be applied to
 * @param diff the diff
 *
 * @return value updated (not duplicated)
 *****************************************************************************/
MYX_GRT_VALUE * myx_grt_value_diff_apply(MYX_GRT *grt, MYX_GRT_VALUE *value, MYX_GRT_VALUE *diff)
{
  unsigned int i, diff_count;
  
  diff_count= myx_grt_list_item_count(diff);
  for(i= 0; i < diff_count; i++)
  {
    unsigned move_from_index, move_to_index;
    char *parent_path;
    MYX_GRT_VALUE *parent_obj;
    const char *diff_path= myx_grt_value_as_string(myx_grt_list_item_get(diff, i++));
    MYX_GRT_VALUE *action_obj= myx_grt_list_item_get(diff, i);
    char diff_op= *diff_path++;
    MYX_GRT_VALUE_TYPE type;

    parent_path= myx_get_parent_path(diff_path);
    parent_path= (parent_path && parent_path[0]) ? parent_path : g_strdup("/");
    parent_obj= myx_grt_dict_item_get_by_path(grt, value, parent_path);
    //diff_path= diff_path+strlen(parent_path)-1;
    diff_path= diff_path+strlen(parent_path);
    if(diff_path[0] != '/') 
    {
      diff_path= g_strconcat("/", diff_path, NULL);
    }

    switch(diff_op)
    {
    case '+':
      type= myx_grt_value_get_type(parent_obj);
      if(type == MYX_LIST_VALUE)
      {
        if(diff_path[0] == '/')
        {
          ++diff_path;
        }
        myx_grt_list_item_insert(parent_obj, atoi(diff_path), action_obj);
      }
      else if(type == MYX_DICT_VALUE)
      {
        //myx_grt_dict_item_set_by_path(value, diff_path, action_obj);
        myx_grt_dict_item_set_by_path(parent_obj, diff_path, action_obj);
      }
      break;
    case '/':
      //myx_grt_dict_item_set_by_path(, diff_path, action_obj);
      myx_grt_dict_item_set_by_path(parent_obj, diff_path, action_obj);
      break;
    case '-':
      type= myx_grt_value_get_type(parent_obj);
      if(diff_path[0] == '/')
      {
        ++diff_path;
      }
      if(type == MYX_LIST_VALUE)
      {
        myx_grt_list_item_del(parent_obj, atoi(diff_path));
      }
      else if(type == MYX_DICT_VALUE)
      {
        myx_grt_dict_item_del(parent_obj, diff_path);
      }
      break;
    case '*':
      if(diff_path[0] == '/')
      {
        ++diff_path;
      }
      // assuming parent_obj is list (* applicable only to lists)
      move_to_index= atoi(diff_path);
      move_from_index= myx_grt_value_as_int(action_obj);
      action_obj= myx_grt_list_item_get(parent_obj, move_from_index);
      action_obj= myx_grt_value_retain(action_obj);
      myx_grt_list_item_del(parent_obj, move_from_index);
      if(myx_grt_list_item_count(parent_obj) < move_to_index) 
      {
        myx_grt_list_item_insert(parent_obj, myx_grt_list_item_count(parent_obj), action_obj);
      }
      else
      {
        myx_grt_list_item_insert(parent_obj, move_to_index, action_obj);
      }
      myx_grt_value_release(action_obj);
      break;
    }
    g_free(parent_path);
  }
  return value;
}
