

#include "myx_grt_private.h"


static int defobj_initialize(MYX_GRT_VALUE *self, MYX_GRT_VALUE *argdict);
static int defobj_destroy(MYX_GRT_VALUE *self);
static MYX_GRT_VALUE *defobj_get_member(MYX_GRT_VALUE *self, const char *name);
static MYX_GRT_ERROR defobj_set_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value);
static MYX_GRT_ERROR defobj_add_to_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value);
static MYX_GRT_ERROR defobj_del_from_member(MYX_GRT_VALUE *self, const char *name, unsigned int index);
static int defobj_serialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr node);
static int defobj_unserialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr parent);
static int defobj_copy(MYX_GRT_VALUE *self, MYX_GRT_VALUE *orig);



// these are the default callbacks used by GRT objects
// these can be replaced by custom implementations e.g. when
// exposing a C++ object to the GRT that will store its own attribute values
static MYX_GRT_OBJECT_BRIDGE default_object_bridge= {
    NULL, // name
    NULL, // owner
    defobj_initialize,
    defobj_destroy,
    defobj_get_member,
    defobj_set_member,
    defobj_add_to_member,
    defobj_del_from_member,
    defobj_serialize,
    defobj_unserialize,
    defobj_copy
};


typedef struct {
  // ordered by key
  const char **keys;
  MYX_GRT_VALUE **values;
} MYX_DEFOBJ_PRIVATE;



MYX_GRT_VALUE *myx_grt_obj_newv(MYX_GRT *grt, const char *struct_name, va_list args)
{
  MYX_GRT_VALUE *argdict;
  MYX_GRT_VALUE *res;
  const char *key;

  key= va_arg(args, const char*);
  if (key)
  {
    argdict= myx_grt_dict_new(NULL);
    while (key)
    {
      MYX_GRT_VALUE *value= va_arg(args, MYX_GRT_VALUE*);
      
      if (value)
        myx_grt_dict_item_set_value(argdict, key, value);
      key= va_arg(args, const char*);
    }
  }
  else
    argdict= NULL;
  
  res= myx_grt_obj_newd(grt, struct_name, argdict);

  if (argdict)
    myx_grt_value_release(argdict);

  return res;
}


MYX_GRT_VALUE *myx_grt_obj_new(MYX_GRT *grt, const char *struct_name, ...)
{
  va_list args;
  MYX_GRT_VALUE *value;

  va_start(args, struct_name);
  value= myx_grt_obj_newv(grt, struct_name, args);
  va_end(args);

  return value;
}


MYX_GRT_VALUE *myx_grt_obj_newd(MYX_GRT *grt, const char *struct_name, MYX_GRT_VALUE *argdict)
{
  MYX_GRT_STRUCT *gstruct;
  MYX_GRT_VALUE *value;
  MYX_GRT_OBJECT_BRIDGE *bridge;

  g_return_val_if_fail(grt!=NULL, NULL);
  g_return_val_if_fail(struct_name && *struct_name, NULL);

  gstruct= myx_grt_struct_get(grt, struct_name);
  g_return_val_if_fail(gstruct != NULL, NULL);

  value= g_new0(MYX_GRT_VALUE, 1);

  value->type= MYX_OBJECT_VALUE;
  value->value.o= g_new0(MYX_GRT_OBJECT, 1);
  value->refcount= 1;

  value->value.o->id= myx_grt_get_guid();
  value->value.o->grt= grt;
  value->value.o->gstruct= gstruct;
  
  bridge= myx_grt_struct_get_obj_bridge(grt, gstruct);

  if (!bridge)
    value->value.o->bridge= &default_object_bridge;
  else
    value->value.o->bridge= bridge;

  if ((*value->value.o->bridge->initialize)(value, argdict) < 0)
  {
    myx_grt_value_release(value);
    return NULL;
  }

  return value;
}


MYX_GRT_VALUE *myx_grt_obj_new_unserialized(MYX_GRT *grt, xmlNodePtr node)
{
  MYX_GRT_STRUCT *gstruct;
  xmlChar *prop;
  MYX_GRT_VALUE *value;
  MYX_GRT_OBJECT_BRIDGE *bridge;
  
  g_return_val_if_fail(grt!=NULL, NULL);
  g_return_val_if_fail(node!=NULL, NULL);

  prop= xmlGetProp(node, (xmlChar*)"type");
  if (!prop)
    return NULL;
  if (strcmp((char*)prop, "object")!=0)
  {
    xmlFree(prop);
    return NULL;
  }
  xmlFree(prop);

  prop= xmlGetProp(node, (xmlChar*)"struct-name");
  if (!prop)
    return NULL;

  gstruct= myx_grt_struct_get(grt, (char*)prop);
  if (!gstruct)
  {
    g_warning("error unserializing object: struct '%s' unknown", (char*)prop);
    xmlFree(prop);
    return NULL;
  }
  xmlFree(prop);

  value= g_new0(MYX_GRT_VALUE, 1);

  value->type= MYX_OBJECT_VALUE;
  value->value.o= g_new0(MYX_GRT_OBJECT, 1);
  value->refcount= 1;

  prop= xmlGetProp(node, (xmlChar*)"id");
  value->value.o->id= g_strdup((char*)prop);
  xmlFree(prop);
  value->value.o->grt= grt;
  value->value.o->gstruct= gstruct;
  
  bridge= myx_grt_struct_get_obj_bridge(grt, gstruct);

  if (!bridge)
    value->value.o->bridge= &default_object_bridge;
  else
    value->value.o->bridge= bridge;

  if ((*value->value.o->bridge->unserialize)(value, grt, node) < 0)
  {
    myx_grt_value_release(value);
    return NULL;
  }
  return value;
}



MYX_GRT_VALUE *myx_grt_obj_copy(MYX_GRT_VALUE *value)
{
  MYX_GRT_VALUE *copy;

  g_return_val_if_fail(value!=NULL, NULL);
  g_return_val_if_fail(value->type == MYX_OBJECT_VALUE, NULL);

  copy= g_new0(MYX_GRT_VALUE, 1);

  copy->type= MYX_OBJECT_VALUE;
  copy->value.o= g_new0(MYX_GRT_OBJECT, 1);
  copy->refcount= 1;

  copy->value.o->id= myx_grt_get_guid();
  copy->value.o->grt= value->value.o->grt;
  copy->value.o->gstruct= value->value.o->gstruct;
  copy->value.o->bridge= value->value.o->bridge;

  if ((*copy->value.o->bridge->copy)(copy, value) < 0)
  {
    myx_grt_value_release(copy);
    return NULL;
  }

  return copy;
}


xmlNodePtr myx_grt_obj_serialize(MYX_GRT *grt, MYX_GRT_VALUE *obj, xmlNodePtr parent)
{
  xmlNodePtr node;
  
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  node= xmlNewTextChild(parent, NULL, (xmlChar*)"value", NULL);
  xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"object");
  xmlNewProp(node, (xmlChar*)"struct-name", (xmlChar*)obj->value.o->gstruct->name);
  xmlNewProp(node, (xmlChar*)"id", (xmlChar*)obj->value.o->id);

  if ((*obj->value.o->bridge->serialize)(obj, grt, node) == 0)
    return node;
  
  xmlFreeNode(node);
  
  return NULL;
}


int myx_grt_free_object(MYX_GRT_VALUE *obj)
{
  g_return_val_if_fail(obj != NULL, -1);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, -1);

  (*obj->value.o->bridge->destroy)(obj);

  g_free(obj->value.o->id);
  g_free(obj->value.o);

  return 0;
}


MYX_GRT * myx_grt_obj_get_grt(MYX_GRT_VALUE *obj)
{
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);
  
  return obj->value.o->grt;
}


const char * myx_grt_obj_struct_get_name(MYX_GRT_VALUE *obj)
{
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  return obj->value.o->gstruct->name;
}


MYX_GRT_STRUCT * myx_grt_obj_struct_get(MYX_GRT_VALUE *obj)
{
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  return obj->value.o->gstruct;
}


int myx_grt_obj_inherits_from(MYX_GRT_VALUE *obj, const char *struct_name)
{
  MYX_GRT_OBJECT *object;
  MYX_GRT *grt;
  const char *parent_struct_name;
  
  g_return_val_if_fail(obj != NULL, 0);
  g_return_val_if_fail(struct_name != NULL, 0);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, 0);

  object= obj->value.o;
  grt= object->grt;
  parent_struct_name= object->gstruct->parent_struct_name;
  
  while (parent_struct_name)
  {
    MYX_GRT_STRUCT *pstruct;

    if (strcmp(parent_struct_name, struct_name)==0)
      return 1;
    
    pstruct= myx_grt_struct_get(grt, parent_struct_name);
    if (!pstruct)
      break;

    parent_struct_name= pstruct->parent_struct_name;
  }

  return 0;
}


int myx_grt_obj_is_or_inherits_from(MYX_GRT_VALUE *obj, const char *struct_name)
{
  g_return_val_if_fail(obj != NULL, 0);
  g_return_val_if_fail(struct_name != NULL, 0);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, 0);

  if (strcmp(obj->value.o->gstruct->name, struct_name)==0)
    return 1;

  return myx_grt_obj_inherits_from(obj, struct_name);
}


const char *myx_grt_obj_get_id(MYX_GRT_VALUE *obj)
{
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  return obj->value.o->id;
}


int myx_grt_obj_is_same(MYX_GRT_VALUE *obj, MYX_GRT_VALUE *other)
{
  if (obj == NULL && other == NULL)
    return 1;
  if (!obj || !other)
    return 0;

  if (obj->type != MYX_OBJECT_VALUE || other->type != MYX_OBJECT_VALUE)
    return -1;
  
  return strcmp(myx_grt_obj_get_id(obj), myx_grt_obj_get_id(other))==0;
}


int myx_grt_obj_has_member(MYX_GRT_VALUE *obj, const char *member)
{
  g_return_val_if_fail(obj != NULL, -1);
  g_return_val_if_fail(member != NULL, -1);
  g_return_val_if_fail(obj->type != MYX_OBJECT_VALUE, -1);

  return myx_grt_struct_get_member_by_name(obj->value.o->grt, myx_grt_obj_struct_get(obj), member, 1) != NULL;
}


MYX_GRT_VALUE *myx_grt_obj_get_value(MYX_GRT_VALUE *obj, const char *member)
{
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(member != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  return (*obj->value.o->bridge->get_member)(obj, member);
}
  

MYX_GRT_ERROR myx_grt_obj_set_value(MYX_GRT_VALUE *obj, const char *member, MYX_GRT_VALUE *value)
{
  MYX_GRT_STRUCT *gstruct;
  MYX_GRT_STRUCT_MEMBER *smember;
  
  g_return_val_if_fail(obj != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(member != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, MYX_GRT_INTERNAL_ERROR);

  gstruct= obj->value.o->gstruct;
  smember= myx_grt_struct_get_member_by_name(obj->value.o->grt, gstruct, member, 1);
  if (!smember)
    return MYX_GRT_BAD_ITEM;

  if (smember->value_type != myx_grt_value_get_type(value))
    return MYX_GRT_VALIDATION_ERROR;
  
  if (smember->value_type == MYX_OBJECT_VALUE)
  {
    if (!myx_grt_obj_is_or_inherits_from(value, smember->object_struct_name))
      return MYX_GRT_VALIDATION_ERROR;
  }
  
  return (*obj->value.o->bridge->set_member)(obj, member, value);
}


MYX_GRT_ERROR myx_grt_obj_add_list_value(MYX_GRT_VALUE *obj, const char *member, MYX_GRT_VALUE *value)
{
  MYX_GRT_STRUCT *gstruct;
  MYX_GRT_STRUCT_MEMBER *smember;
  
  g_return_val_if_fail(obj != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(member != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, MYX_GRT_INTERNAL_ERROR);

  gstruct= obj->value.o->gstruct;
  smember= myx_grt_struct_get_member_by_name(obj->value.o->grt, gstruct, member, 1);
  if (!smember)
    return MYX_GRT_BAD_ITEM;

  if (smember->value_type != MYX_LIST_VALUE)
    return MYX_GRT_VALIDATION_ERROR;

  if (smember->content_type != myx_grt_value_get_type(value))
    return MYX_GRT_VALIDATION_ERROR;
  
  if (smember->content_type == MYX_OBJECT_VALUE)
  {
    if (!myx_grt_obj_is_or_inherits_from(value, smember->content_struct_name))
      return MYX_GRT_VALIDATION_ERROR;
  }
  
  return (*obj->value.o->bridge->add_to_member)(obj, member, value);
}


MYX_GRT_ERROR myx_grt_obj_del_list_value(MYX_GRT_VALUE *obj, const char *member, unsigned int index)
{
  MYX_GRT_STRUCT *gstruct;
  MYX_GRT_STRUCT_MEMBER *smember;
  
  g_return_val_if_fail(obj != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(member != NULL, MYX_GRT_INTERNAL_ERROR);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, MYX_GRT_INTERNAL_ERROR);

  gstruct= obj->value.o->gstruct;
  smember= myx_grt_struct_get_member_by_name(obj->value.o->grt, gstruct, member, 1);
  if (!smember)
    return MYX_GRT_BAD_ITEM;

  if (smember->value_type != MYX_LIST_VALUE)
    return MYX_GRT_VALIDATION_ERROR;

  return (*obj->value.o->bridge->del_from_member)(obj, member, index);
}



unsigned int myx_grt_obj_item_count(MYX_GRT_VALUE *obj)
{
  g_return_val_if_fail(obj != NULL, 0);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, 0);

  return myx_grt_struct_get_member_count_total(obj->value.o->grt, obj->value.o->gstruct);
}


const char * myx_grt_obj_item_name_by_index(MYX_GRT_VALUE *obj, unsigned int index)
{
  MYX_GRT_STRUCT_MEMBER *member;
  
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  member= myx_grt_struct_get_member_by_index_total(obj->value.o->grt, obj->value.o->gstruct, index);
  if (member)
    return member->name;
  return NULL;
}


MYX_GRT_VALUE * myx_grt_obj_item_value_by_index(MYX_GRT_VALUE *obj, unsigned int index)
{
  const char *name;
  
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  name= myx_grt_obj_item_name_by_index(obj, index);
  if (name)
    return myx_grt_obj_get_value(obj, name);
  return NULL;
}


MYX_GRT_STRUCT_MEMBER *myx_grt_obj_item_struct_member_by_index(MYX_GRT_VALUE *obj, unsigned int index)
{
  MYX_GRT_STRUCT_MEMBER *member;
  
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, NULL);

  return myx_grt_struct_get_member_by_index_total(obj->value.o->grt, obj->value.o->gstruct, index);
}




int myx_grt_obj_item_by_index(MYX_GRT_VALUE *obj, unsigned int index,
                              const char **retname, MYX_GRT_VALUE **retvalue)
{
  MYX_GRT_STRUCT_MEMBER *member;
  
  g_return_val_if_fail(obj != NULL, -1);
  g_return_val_if_fail(obj->type == MYX_OBJECT_VALUE, -1);

  member= myx_grt_struct_get_member_by_index_total(obj->value.o->grt, obj->value.o->gstruct, index);
  if (member)
  {
    *retname= member->name;
    *retvalue= myx_grt_obj_get_value(obj, member);
    return 0;
  }
  return -1;
}


//--------------------------------------------------------------------------------
// Default Object Implementation
//--------------------------------------------------------------------------------

static int sstrcmp(const void *a, const void *b)
{
  return strcmp(*(char**)a, *(char**)b);
}


static int defobj_initialize(MYX_GRT_VALUE *self, MYX_GRT_VALUE *argdict)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_GRT *grt= obj->grt;
  MYX_GRT_STRUCT *gstruct= obj->gstruct;
  unsigned int i;
  unsigned int count= myx_grt_struct_get_member_count_total(grt, gstruct);
  MYX_DEFOBJ_PRIVATE *priv= g_new0(MYX_DEFOBJ_PRIVATE, 1);

  
  priv->keys= g_new0(const char*, count);
  priv->values= g_new0(MYX_GRT_VALUE*, count);
  obj->bridge_data= priv;
  
  // init with empty values
  for (i= 0; i < count; i++)
  {
    MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index_total(grt, gstruct, i);
    const char *member_name= myx_grt_struct_get_member_name(member);

    priv->keys[i]= member_name;
  }
  // sort
  qsort(priv->keys, count, sizeof(char*), sstrcmp);

  // create child values
  for (i= 0; i < count; i++)
  {
    MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index_total(grt, gstruct, i);
    MYX_GRT_VALUE_TYPE member_type= myx_grt_struct_member_get_type(member);
    const char *member_name= myx_grt_struct_get_member_name(member);
    const char *member_default= myx_grt_struct_get_member_default(member);

    MYX_GRT_VALUE *ivalue= NULL;
    
    if (argdict)
    {
      ivalue= myx_grt_dict_item_get_value(argdict, member_name);
      if (ivalue)
      {
        myx_grt_obj_set_value(self, member_name, ivalue);
        continue;
      }
    }

    switch (member_type)
    {
    case MYX_STRING_VALUE:
      {
        MYX_GRT_VALUE *child_value;

        if (!member_default || !*member_default || strcasecmp(member_default, "null")==0)
          child_value= myx_grt_value_from_string("");
        else
          child_value= myx_grt_value_from_string(member_default);
        myx_grt_obj_set_value(self, member_name, child_value);

        if (child_value)
          myx_grt_value_release(child_value);
      }
      break;
    case MYX_INT_VALUE:
      {
        MYX_GRT_VALUE *child_value;
        
        if (!member_default || !*member_default || strcasecmp(member_default, "null")==0)      
          child_value= myx_grt_value_from_int(0);
        else
          child_value= myx_grt_value_from_int(atoi(member_default));

        myx_grt_obj_set_value(self, member_name, child_value);

        if (child_value)
          myx_grt_value_release(child_value);
      }
      break;
    case MYX_REAL_VALUE:
      {
        MYX_GRT_VALUE *child_value;
        
        if (!member_default || !*member_default || strcasecmp(member_default, "null")==0)
          child_value= myx_grt_value_from_real(0.0);
        else
          child_value= myx_grt_value_from_real(atof(member_default));

        myx_grt_obj_set_value(self, member_name, child_value);

        if (child_value)
          myx_grt_value_release(child_value);
      }
      break;
    case MYX_LIST_VALUE:
      {
        MYX_GRT_VALUE_TYPE content_type= myx_grt_struct_member_get_content_type(member);
        const char *content_struct= myx_grt_struct_member_get_content_struct_name(member);
        MYX_GRT_VALUE *child_value= myx_grt_list_new(content_type, content_struct);

        myx_grt_obj_set_value(self, member_name, child_value);

        myx_grt_value_release(child_value);
      }
      break;
    case MYX_DICT_VALUE:
      {
        MYX_GRT_VALUE_TYPE content_type= myx_grt_struct_member_get_content_type(member);
        const char *content_struct= myx_grt_struct_member_get_content_struct_name(member);
        MYX_GRT_VALUE *child_value= NULL;
        
        if (content_type != MYX_ANY_VALUE)
          child_value= myx_grt_dict_new_typed(content_type, content_struct);

        myx_grt_obj_set_value(self, member_name, child_value);
        
        if (child_value)
          myx_grt_value_release(child_value);
      }
      break;
    case MYX_OBJECT_VALUE:
      if (!member_default || !*member_default || strcasecmp(member_default, "null")==0)
      {
        myx_grt_obj_set_value(self, member_name, NULL);
      }
      else
      {
        const char *struct_name= myx_grt_struct_member_get_obj_struct_name(member);
        MYX_GRT_VALUE *child_value= myx_grt_obj_new(grt, struct_name);

        myx_grt_obj_set_value(self, member_name, child_value);

        myx_grt_value_release(child_value);
      }
      break;
    }
  }

  return 0;
}




static MYX_GRT_VALUE *defobj_get_member(MYX_GRT_VALUE *self, const char *name)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_DEFOBJ_PRIVATE *priv= (MYX_DEFOBJ_PRIVATE*)obj->bridge_data;
  int member_count= myx_grt_struct_get_member_count_total(obj->grt, obj->gstruct);
  int i;
  const char **ptr= bsearch(&name, priv->keys,
                     member_count,
                     sizeof(char*), sstrcmp);

  if (!ptr)
    return NULL;

  i= ptr - priv->keys;

  g_return_val_if_fail(i >= 0 && i < member_count, NULL);

  return priv->values[i];
}


static MYX_GRT_ERROR defobj_set_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_DEFOBJ_PRIVATE *priv= (MYX_DEFOBJ_PRIVATE*)obj->bridge_data;
  int member_count= myx_grt_struct_get_member_count_total(obj->grt, obj->gstruct);
  int i;
  const char **ptr= bsearch(&name, priv->keys,
                     member_count,
                     sizeof(char*), sstrcmp);

  if (!ptr)
    return MYX_GRT_BAD_ITEM;

  i= ptr - priv->keys;

  g_return_val_if_fail(i >= 0 && i < member_count, -1);


  if (!priv->values[i])
  {
    // the key was not in the obj already
    if (value)
      priv->values[i]= myx_grt_value_retain(value);
    else
      priv->values[i]= NULL;
  }
  else
  {
    // Replace the value.
    MYX_GRT_VALUE* currentValue = priv->values[i];
    if (currentValue != value)
    {
      priv->values[i]= myx_grt_value_retain(value);
      myx_grt_value_release(currentValue);
    }
  }
  return MYX_GRT_NO_ERROR;
}


static MYX_GRT_ERROR defobj_add_to_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_DEFOBJ_PRIVATE *priv= (MYX_DEFOBJ_PRIVATE*)obj->bridge_data;
  int member_count= myx_grt_struct_get_member_count_total(obj->grt, obj->gstruct);
  MYX_GRT_STRUCT_MEMBER *member;
  int i;
  const char **ptr= bsearch(&name, priv->keys,
                     member_count,
                     sizeof(char*), sstrcmp);

  if (!ptr)
    return MYX_GRT_BAD_ITEM;

  i= ptr - priv->keys;

  g_return_val_if_fail(i >= 0 && i < member_count, -1);

  member= myx_grt_struct_get_member_by_index(obj->gstruct, i);

  if (!priv->values[i])
  {
    // not initialized yet
    priv->values[i]= myx_grt_list_new(member->content_type,
                                      member->content_struct_name);
  }

  myx_grt_list_item_add(priv->values[i], value);

  return MYX_GRT_NO_ERROR;
}


static MYX_GRT_ERROR defobj_del_from_member(MYX_GRT_VALUE *self, const char *name, unsigned int index)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_DEFOBJ_PRIVATE *priv= (MYX_DEFOBJ_PRIVATE*)obj->bridge_data;
  int member_count= myx_grt_struct_get_member_count_total(obj->grt, obj->gstruct);
  MYX_GRT_STRUCT_MEMBER *member;
  int i;
  const char **ptr= bsearch(&name, priv->keys,
                     member_count,
                     sizeof(char*), sstrcmp);

  if (!ptr)
    return MYX_GRT_BAD_ITEM;

  i= ptr - priv->keys;

  g_return_val_if_fail(i >= 0 && i < member_count, -1);

  member= myx_grt_struct_get_member_by_index(obj->gstruct, i);

  if (!priv->values[i])
    return MYX_GRT_BAD_ITEM;

  if (index >= myx_grt_list_item_count(priv->values[i]))
    return MYX_GRT_BAD_ITEM;

  myx_grt_list_item_del(priv->values[i], index);    

  return MYX_GRT_NO_ERROR;
}


static int defobj_serialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr node)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_DEFOBJ_PRIVATE *priv= (MYX_DEFOBJ_PRIVATE*)obj->bridge_data;
  MYX_GRT_STRUCT *gstruct= obj->gstruct;
  int i;
  //TODO: if xmlname is specified, it should use it as the element name
  // and use its prefix as member prefixes (member names become
  // sub-element names)

  for (i= 0; i < gstruct->members_num; i++)
  {
    xmlNodePtr child;
    
    if (priv->values[i])
    {
      child= myx_grt_serialize_to_xml(grt, node, priv->values[i]);
      xmlNewProp(child, (xmlChar*)"key", (xmlChar*)priv->keys[i]);
    }
  }
  return 0;
}


static int defobj_unserialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr parent)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_GRT_STRUCT *gstruct= myx_grt_obj_struct_get(self);
  unsigned int i;
  unsigned int count= myx_grt_struct_get_member_count_total(grt, gstruct);
  MYX_DEFOBJ_PRIVATE *priv= g_new0(MYX_DEFOBJ_PRIVATE, 1);

  priv->keys= g_new0(const char*, count);
  priv->values= g_new0(MYX_GRT_VALUE*, count);
  obj->bridge_data= priv;
  
  // init with empty values
  for (i= 0; i < count; i++)
  {
    MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index_total(grt, gstruct, i);
    const char *member_name= myx_grt_struct_get_member_name(member);

    priv->keys[i]= member_name;
  }
  // sort
  qsort(priv->keys, count, sizeof(char*), sstrcmp);

  // load values
  {
    xmlNodePtr child;

    child= parent->children;
    while (child)
    {
      MYX_GRT_VALUE *sub_value;

      if (child->type == XML_ELEMENT_NODE)
      {
        xmlChar *key= xmlGetProp(child, (xmlChar*)"key");

        if (key)
        {
          sub_value= myx_grt_unserialize_from_xml(grt, child);
          if (sub_value)
          {
            for (i= 0; i < count; i++)
            {
              if (strcmp((char*)key, priv->keys[i])==0)
              {
                if (priv->values[i])
                {
                  // duplicate value, but we can ignore the 1st instance
                  myx_grt_value_release(priv->values[i]);
                }
                priv->values[i]= sub_value;
              break;
              }
            }
            if (i == count)// there were unknown items in the xml
              myx_grt_value_release(sub_value);
          }
          xmlFree(key);
        }
      }
      child= child->next;
    }
  }
}


static int defobj_copy(MYX_GRT_VALUE *self, MYX_GRT_VALUE *orig)
{
  MYX_GRT_STRUCT *gstruct= myx_grt_obj_struct_get(self);
  unsigned int i;
  MYX_DEFOBJ_PRIVATE *priv= g_new0(MYX_DEFOBJ_PRIVATE, 1);
  MYX_DEFOBJ_PRIVATE *opriv= (MYX_DEFOBJ_PRIVATE*)orig->value.o->bridge_data;
  unsigned int count= myx_grt_struct_get_member_count_total(self->value.o->grt, self->value.o->gstruct);

  priv->keys= g_new0(const char*, count);
  priv->values= g_new0(MYX_GRT_VALUE*, count);
  self->value.o->bridge_data= priv;
  
  for (i= 0; i < count; i++)
  {
    priv->keys[i]= opriv->keys[i];
    priv->values[i]= myx_grt_value_retain(opriv->values[i]);

    //TODO XXX
    // perhaps there should be a flag in the struct xml telling
    // whether a member should be duplicated or just re-referenced
  }
  return 0;
}


static int defobj_destroy(MYX_GRT_VALUE *self)
{
  MYX_GRT_OBJECT *obj= self->value.o;
  MYX_DEFOBJ_PRIVATE *priv= (MYX_DEFOBJ_PRIVATE*)obj->bridge_data;
  int i;
  int member_count= myx_grt_struct_get_member_count_total(obj->grt, obj->gstruct);

  for (i= 0; i < member_count; i++)
  {
    if (priv->values[i])
      myx_grt_value_release(priv->values[i]);
  }
  g_free(priv->values);
  g_free(priv->keys);
  g_free(priv);

  return 0;
}

  

//--------------------------------------------------------------------------------
// Bridge Management
//--------------------------------------------------------------------------------


MYX_GRT_ERROR myx_grt_obj_bridge_register(MYX_GRT *grt, MYX_GRT_OBJECT_BRIDGE *bridge)
{
  if (myx_grt_obj_bridge_get(grt, bridge->name))
    return MYX_GRT_DUPLICATE_ENTRY;

  grt->obj_bridges_num++;
  grt->obj_bridges= g_realloc(grt->obj_bridges, grt->obj_bridges_num*sizeof(MYX_GRT_OBJECT_BRIDGE));

  grt->obj_bridges[grt->obj_bridges_num-1]= *bridge;

  return MYX_GRT_NO_ERROR;
}


MYX_GRT_OBJECT_BRIDGE *myx_grt_obj_bridge_get(MYX_GRT *grt, const char *name)
{
  unsigned int i;
  
  g_return_val_if_fail(grt!=NULL, NULL);
  g_return_val_if_fail(name!=NULL, NULL);

  for (i= 0; i < grt->obj_bridges_num; i++)
  {
    if (strcmp(grt->obj_bridges[i].name, name)==0)
      return grt->obj_bridges+i;
  }
  return NULL;
}


const char *myx_grt_obj_bridge_get_name(MYX_GRT_OBJECT_BRIDGE *bridge)
{
  g_return_val_if_fail(bridge!=NULL, NULL);

  return bridge->name;
}


unsigned int myx_grt_obj_bridge_count(MYX_GRT *grt)
{
  g_return_val_if_fail(grt!=NULL, 0);

  return grt->obj_bridges_num;
}


MYX_GRT_OBJECT_BRIDGE *myx_grt_obj_bridge_get_by_index(MYX_GRT *grt, unsigned int index)
{
  g_return_val_if_fail(grt!=NULL, NULL);
  g_return_val_if_fail(index < grt->obj_bridges_num, NULL);

  return grt->obj_bridges+index;
}

