
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include <libxml/parser.h>

#include "myx_grt_private.h"
#include "myx_xml_util_functions.h"

#define ser_new_node(node, name, value)\
  xmlNewTextChild(node, NULL, (xmlChar*)name, (xmlChar*)value)

xmlNodePtr myx_ser_new_int_node(xmlNodePtr node, const char *name, int value)
{
  char buffer[32];

#ifdef _WINDOWS
  _snprintf(buffer, sizeof(buffer), "%i", value);
#else
  snprintf(buffer, sizeof(buffer), "%i", value);
#endif

  return ser_new_node(node, name, buffer);
}

#define ser_set_prop(node, name, value)\
  xmlNewProp(node, (xmlChar*)name, (xmlChar*)value)

static void ser_set_int_prop(xmlNodePtr node, const char *name, int value)
{
  char buffer[32];

#ifdef _WINDOWS
  _snprintf(buffer, sizeof(buffer), "%i", value);
#else
  snprintf(buffer, sizeof(buffer), "%i", value);
#endif

  ser_set_prop(node, name, buffer);
}
  







/**
 ****************************************************************************
 * @brief Loads a previously stored GRT value from a file
 * 
 *   Loads a GRT value from a GRT XML file. Use myx_grt_store_to_file to
 * create files in that format.
 *   Read myx_grt_store_to_file for more details.
 *
 * @param filename the name of the file to load. The file must be in the GRT XML format.
 *     Usually something stored by myx_grt_store_to_file
 *
 * @return NULL on error and the value if the file could be correctly loaded.
 ****************************************************************************/
MYX_GRT_VALUE *myx_grt_retrieve_from_file(MYX_GRT *grt, const char *filename)
{
  xmlDocPtr doc;
  xmlNodePtr root;
  MYX_GRT_VALUE *value;
  
  g_return_val_if_fail(grt->unserializer_cache==NULL, NULL);

  if (!(doc= myx_xmlParseFile(filename)))
    return NULL;

  root= xmlDocGetRootElement(doc);
  if (root)
  {
    root= root->children;
    while (root && xmlStrcmp(root->name, (xmlChar*)"value")!=0) root= root->next;
    if (root)
    {
      grt->unserializer_cache= g_hash_table_new_full(g_str_hash, g_str_equal,
                                                     xmlFree, 
                                                     myx_grt_value_release);
      value= myx_grt_unserialize_from_xml(grt, root);
      g_hash_table_destroy(grt->unserializer_cache);
      grt->unserializer_cache= NULL;
    }
  }

  xmlFreeDoc(doc);

  return value;
}


/**
 ****************************************************************************
 * @brief Stores a GRT value to a file
 * 
 *   This will serialize the value to XML and store it in a file that can
 * later be retrieved with myx_grt_retrieve_from_file.
 * NOTE: This function is not reentrant.
 *
 * @param  value the GRT value to store
 * @param  filename name of file to store data
 *
 * @return  MYX_GRT_NO_ERROR if there were no errors.
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_store_to_file(MYX_GRT *grt, MYX_GRT_VALUE *value, const char *filename)
{
  xmlDocPtr doc;
  int res;
  
  g_return_val_if_fail(grt->serializer_cache==NULL, MYX_GRT_INTERNAL_ERROR);
  
  doc= xmlNewDoc((xmlChar*)"1.0");
  doc->children= xmlNewDocRawNode(doc, NULL, (xmlChar*)"data", NULL);
  
  grt->serializer_cache= g_hash_table_new(g_direct_hash, g_direct_equal);
  
  myx_grt_serialize_to_xml(grt, doc->children, value);

  g_hash_table_destroy(grt->serializer_cache);
  grt->serializer_cache= NULL;
  
  res= myx_xmlSaveFile(filename, doc);

  xmlFreeDoc(doc);

  return res == -1 ? MYX_GRT_CANT_OPEN_FILE : MYX_GRT_NO_ERROR;
}




static int checkIfSeen(MYX_GRT *grt, MYX_GRT_VALUE *value)
{
  if (g_hash_table_lookup(grt->serializer_cache, value))
    return 1;

  // value is not yet in XML tree. put it in and record the ID
  g_hash_table_insert(grt->serializer_cache, (gpointer)value, "dummy");
  return 0;
}


/*
 *****************************************************************************
 * Encodes a GRT value (and its sub-values) to a XML tree.
 * 
 * The implementation will store simple values directly and complex values
 * (lists, dicts and objects) directly at the first reference and as
 * links on further ones. 
 * 
 *****************************************************************************
 */
xmlNodePtr myx_grt_serialize_to_xml(MYX_GRT *grt, xmlNodePtr parent, MYX_GRT_VALUE *value)
{
  MYX_GRT_STRUCT *gstruct;
  unsigned int i;
  char buffer[100];
  const char *id;
  xmlNodePtr node= NULL;

  switch (value->type)
  {
  case MYX_INT_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%i", value->value.i);
    node= ser_new_node(parent, "value", buffer);

    ser_set_prop(node, "type", "int");
    break;
  case MYX_REAL_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%f", value->value.r);
    node= ser_new_node(parent, "value", buffer);

    ser_set_prop(node, "type", "real");
    break;
  case MYX_STRING_VALUE:
    node= ser_new_node(parent, "value", value->value.s);

    ser_set_prop(node, "type", "string");
    break;
  case MYX_LIST_VALUE:
    if (checkIfSeen(grt, value))
    {
      g_snprintf(buffer, sizeof(buffer), "%p", value);
      return ser_new_node(parent, "link", buffer);
    }
    
    node= ser_new_node(parent, "value", NULL);

    g_snprintf(buffer, sizeof(buffer), "%p", value);
    ser_set_prop(node, "_ptr_", buffer);
    ser_set_prop(node, "type", "list");
    ser_set_prop(node, "content-type", myx_get_value_type_as_string(value->value.l->content_type));

    if (value->value.l->content_struct_name)
      ser_set_prop(node, "content-struct-name", value->value.l->content_struct_name);

    for (i= 0; i < value->value.l->items_num; i++)
    {
      g_return_val_if_fail((value->value.l->items[i]->type == value->value.l->content_type) ||
         (value->value.l->content_type == MYX_ANY_VALUE), NULL);
      myx_grt_serialize_to_xml(grt, node, value->value.l->items[i]);
    }
    break;
  case MYX_DICT_VALUE:
    if (checkIfSeen(grt, value))
    {
      g_snprintf(buffer, sizeof(buffer), "%p", value);
      return ser_new_node(parent, "link", buffer);
    }

    gstruct= NULL;
    node= ser_new_node(parent, "value", NULL);

    g_snprintf(buffer, sizeof(buffer), "%p", value);
    ser_set_prop(node, "_ptr_", buffer);
    ser_set_prop(node, "type", "dict");

    if (value->value.d->content_type != MYX_ANY_VALUE)
      ser_set_prop(node, "content-type", myx_get_value_type_as_string(value->value.d->content_type));

    if ((value->value.d->content_struct_name) && (value->value.d->content_struct_name[0]))
      ser_set_prop(node, "content-struct-name", value->value.d->content_struct_name);

    for (i= 0; i < value->value.d->items_num; i++)
    {
      xmlNodePtr child;

      if (value->value.d->items[i].value)
      {
        child= myx_grt_serialize_to_xml(grt, node, value->value.d->items[i].value);
        xmlNewProp(child, (xmlChar*)"key", (xmlChar*)value->value.d->items[i].key);
      }
    }
    break;
  case MYX_OBJECT_VALUE:
    if (checkIfSeen(grt, value))
    {
      return ser_new_node(parent, "link", value->value.o->id);
    }
    node= myx_grt_obj_serialize(grt, value, parent);
    break;
  case MYX_ANY_VALUE:
    break;
  }
  return node;
}


MYX_GRT_VALUE *myx_grt_unserialize_from_xml(MYX_GRT *grt, xmlNodePtr node)
{
  xmlChar *str;
  xmlChar *node_type;
  MYX_GRT_VALUE *value= NULL;
  MYX_GRT_VALUE_TYPE vtype;
  MYX_GRT_ERROR error;
  
  if (strcmp((char*)node->name, "link")==0)
  {
    // this is a link instead of a value, look up for the original value and
    // return it
    str= xmlNodeGetContent(node);
    value= g_hash_table_lookup(grt->unserializer_cache, (char*)str);
    xmlFree(str);

    // retain the value to keep consistent with non-links
    return myx_grt_value_retain(value);
  }
  else if (strcmp((char*)node->name, "value")!=0)
    return NULL;

  node_type= xmlGetProp(node, (xmlChar*)"type");  
  if (!node_type)
  {
    g_warning("Node '%s' in xml doesn't have a type property", node->name);
    return NULL;
  }

  vtype= myx_get_value_type_from_string((char*)node_type, &error);
  if (error != MYX_GRT_NO_ERROR)
  {
    return NULL;
  }
  
  if (vtype == MYX_DICT_VALUE)
  {
    if (xmlHasProp(node, (xmlChar*)"struct-name"))
    {
      g_warning("dict value has deprecated struct-name property. replacing dict with object");
      vtype= MYX_OBJECT_VALUE;
    }
  }
  
  switch (vtype)
  {
  case MYX_INT_VALUE:
    str= xmlNodeGetContent(node);
    value= myx_grt_value_from_int(strtol((char*)str, NULL, 0));
    xmlFree(str);
    break;
  case MYX_REAL_VALUE:
    str= xmlNodeGetContent(node);
    value= myx_grt_value_from_real(strtod((char*)str, NULL));
    xmlFree(str);
    break;
  case MYX_STRING_VALUE:
    str= xmlNodeGetContent(node);
    value= myx_grt_value_from_string((char*)str);
    xmlFree(str);
  break;
  case MYX_DICT_VALUE:
    {
      //xmlChar *global_object_path= xmlGetProp(node, (xmlChar*)"global-object-path");
      xmlNodePtr child;
      xmlChar *prop;
    /*
    if (global_object_path)
    {
      xmlFree(link_id);
      return unserialize_from_xml_global_object(grt, node);
    }
    xmlFree(global_object_path);
     */
      
      MYX_GRT_VALUE_TYPE content_type= MYX_ANY_VALUE;
      MYX_GRT_ERROR error;
      
      prop= xmlGetProp(node, (xmlChar*)"struct-name");
      if (prop)
      {
        g_warning("dict value has deprecated struct-name property");
        xmlFree(prop);
      }
      
      prop= xmlGetProp(node, (xmlChar*)"content-type");

    if (prop)
    {
      content_type= myx_get_value_type_from_string((char*)prop, &error);
      if (error == MYX_GRT_NO_ERROR)
      {
        char *content_struct_name= (char*)xmlGetProp(node, (xmlChar*)"content-struct-name");

#if 0
        if (content_type != MYX_OBJECT_VALUE && content_struct_name != NULL)
        {
          // content-struct is only valid for object type
          g_warning("content-struct-name property ignored for dict of non object content-type");
          
          value= myx_grt_dict_new_typed(content_type, NULL);
        }
        else
          value= myx_grt_dict_new_typed(content_type, content_struct_name);
#else 
        if (content_type != MYX_OBJECT_VALUE && content_struct_name != NULL)
        {
          // content-struct is only valid for object type
          g_warning("content-struct-name property for dict is deprecated. replacing dict with object");

          content_type= MYX_OBJECT_VALUE;
        }
        value= myx_grt_dict_new_typed(content_type, content_struct_name);
#endif
        xmlFree(content_struct_name);
      }
      xmlFree(prop);
      if (error != MYX_GRT_NO_ERROR)
        return NULL;
    }
    else
      value= myx_grt_dict_new();

    prop= xmlGetProp(node, (xmlChar*)"_ptr_");
    g_hash_table_insert(grt->unserializer_cache, prop, myx_grt_value_retain(value));

    child= node->children;
    while (child)
    {
      MYX_GRT_VALUE *sub_value;
      
      if (child->type == XML_ELEMENT_NODE)
      {
        xmlChar *key= xmlGetProp(child, (xmlChar*)"key");
        if (key)
        {
          sub_value= myx_grt_unserialize_from_xml(grt, child);
          if (sub_value)
          {
            myx_grt_dict_item_set_value(value, (char*)key, sub_value);
            myx_grt_value_release(sub_value);
          }
          else
          {
            // error!
            myx_grt_value_release(value);
            value= NULL;
            break;
          }
          xmlFree(key);
        }
      }
      child= child->next;
    }
    }
    break;
  case MYX_LIST_VALUE:
    {
      xmlChar *ctype= xmlGetProp(node, (xmlChar*)"content-type");
      xmlChar *cstruct_name= xmlGetProp(node, (xmlChar*)"content-struct-name");
      xmlNodePtr child;
      MYX_GRT_ERROR error;
      MYX_GRT_VALUE_TYPE content_type= myx_get_value_type_from_string((char*)ctype, &error);

      xmlFree(ctype);
      
      if (error == MYX_GRT_NO_ERROR)
      {
        xmlChar *prop;

        if (content_type != MYX_OBJECT_VALUE && cstruct_name != NULL)
        {
          if (content_type != MYX_DICT_VALUE)
          {
            g_warning("content-struct-name attribute ignored for list of non object content-type");
            xmlFree(cstruct_name);
            cstruct_name= NULL;
          }
          else
          {
            content_type= MYX_OBJECT_VALUE;
            g_warning("content-struct-name is deprecated for dicts. replacing dict with object");
          }
        }
          
        // this does the work of myx_grt_list_new()
        value= g_new0(MYX_GRT_VALUE, 1);
        value->type= MYX_LIST_VALUE;
        value->value.l= g_new0(MYX_GRT_LIST, 1);
        value->value.l->content_type= content_type;
        value->refcount= 1;
        value->value.l->content_struct_name= g_strdup((char*)cstruct_name);
        
        prop= xmlGetProp(node, (xmlChar*)"_ptr_");
        g_hash_table_insert(grt->unserializer_cache, prop, myx_grt_value_retain(value));
        
        child= node->children;
        while (child)
        {
          MYX_GRT_VALUE *sub_value;
          
          if (child->type == XML_ELEMENT_NODE)
          {
            sub_value= myx_grt_unserialize_from_xml(grt, child);
            if (sub_value)
            {
              myx_grt_list_item_add(value, sub_value);
              myx_grt_value_release(sub_value);
            }
            else
            {
              //error!
              myx_grt_value_release(value);
              value= NULL;
              break;
            }
          }
          
          child= child->next;
        }
        if (cstruct_name)
          xmlFree(cstruct_name);
      }
    }
    break;
  case MYX_OBJECT_VALUE:
    value= myx_grt_obj_new_unserialized(grt, node);
    if (value)
    {
      g_hash_table_insert(grt->unserializer_cache, 
                          xmlGetProp(node, (xmlChar*)"id"),
                          myx_grt_value_retain(value));
    }
    break;
  }

  return value;
}


#if 0
static xmlNodePtr serialize_to_xml_global_object(const char *objectPath, xmlNodePtr parent, MYX_GRT_VALUE *value,
                                   GHashTable *saved_ids)
{
  //unsigned int i;
  char buffer[100];
  //const char *id;
  xmlNodePtr node;

  switch (value->type)
  {
  case MYX_INT_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%i", value->value.i);
    node= xmlNewTextChild(parent, NULL, (xmlChar*)"value", (xmlChar*)buffer);
    
    xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"int");
    xmlNewProp(node, (xmlChar*)"globalObjectPath", (xmlChar*)objectPath);
    break;
  case MYX_REAL_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%f", value->value.r);
    node= xmlNewTextChild(parent, NULL, (xmlChar*)"value", (xmlChar*)buffer);

    xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"real");

    xmlNewProp(node, (xmlChar*)"global-object-path", (xmlChar*)objectPath);
    break;
  case MYX_STRING_VALUE:
    node= xmlNewTextChild(parent, NULL, (xmlChar*)"value", (xmlChar*)value->value.s);

    xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"string");
    xmlNewProp(node, (xmlChar*)"global-object-path", (xmlChar*)objectPath);
    break;
  case MYX_LIST_VALUE:
    node= xmlNewTextChild(parent, NULL, (xmlChar*)"value", NULL);
    xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"list");
    xmlNewProp(node, (xmlChar*)"content-type", (xmlChar*)myx_get_value_type_as_string(value->value.l->content_type));

    if (value->value.l->content_struct_name)
      xmlNewProp(node, (xmlChar*)"content-struct-name", (xmlChar*)value->value.l->content_struct_name);

    /*for (i= 0; i < value->value.l->items_num; i++)
    {
      g_return_val_if_fail((value->value.l->items[i]->type == value->value.l->content_type) ||
         (value->value.l->content_type == MYX_ANY_VALUE), NULL);
      serialize_to_xml(node, value->value.l->items[i], saved_ids);
    }*/
    xmlNewProp(node, (xmlChar*)"global-object-path", (xmlChar*)objectPath);
    break;
  case MYX_DICT_VALUE:
    node= xmlNewTextChild(parent, NULL, (xmlChar*)"value", NULL);
    xmlNewProp(node, (xmlChar*)"type", (xmlChar*)"dict");

/*XXX    if ((value->value.d->struct_name) && (value->value.d->struct_name[0]))
      xmlNewProp(node, (xmlChar*)"struct-name", (xmlChar*)value->value.d->struct_name);
*/
    if (value->value.d->content_type != MYX_ANY_VALUE)
      xmlNewProp(node, (xmlChar*)"content-type", (xmlChar*)myx_get_value_type_as_string(value->value.d->content_type));

    if ((value->value.d->content_struct_name) && (value->value.d->content_struct_name[0]))
      xmlNewProp(node, (xmlChar*)"content-struct-name", (xmlChar*)value->value.d->content_struct_name);

    /*if ((id= myx_grt_dict_id_item_as_string(value)))
    {
      if (g_hash_table_lookup(saved_ids, id))
      {
        // the object is already in the XML tree, just leave a link to it
        xmlNewProp(node, "link", id);

        return node;
      }
      else
        // object is not yet in XML tree. put it in and record the ID
        g_hash_table_insert(saved_ids, (gpointer)id, "X");      
    }

    for (i= 0; i < value->value.d->items_num; i++)
    {
      xmlNodePtr child;
      
      child= serialize_to_xml(node, value->value.d->items[i].value, saved_ids);
      xmlNewProp(child, "key", value->value.d->items[i].key);
    }*/
    xmlNewProp(node, (xmlChar*)"global-object-path", (xmlChar*)objectPath);
    break;
  case MYX_ANY_VALUE:
    break;
  }
  return node;
}

static MYX_GRT_VALUE *unserialize_from_xml_global_object(MYX_GRT *grt, xmlNodePtr node, GHashTable *objects_by_id)
{
  char *node_type= (char*)xmlGetProp(node, (xmlChar*)"type");
  MYX_GRT_VALUE *value= NULL;

  if (!node_type)
  {
    g_warning("Node '%s' in xml doesn't have a type property", node->name);
    return NULL;
  }

  GRT_ENTER(grt);
  
  // check if we have a global object
  if ((strcmp(node_type, "dict")==0) || (strcmp(node_type, "list")==0))
  {
    xmlChar *global_object_path= xmlGetProp(node, (xmlChar*)"global-object-path");

    if (global_object_path)
    {
      value= myx_grt_dict_item_get_by_path(grt, myx_grt_get_root(grt), (char*)global_object_path);
      xmlFree(global_object_path);
      GRT_RETURN(grt, value, MYX_GRT_VALUE*);
    }
  }

  //GRT_RETURN(grt, unserialize_from_xml(grt, node, objects_by_id), MYX_GRT_VALUE*);
  GRT_RETURN(grt, myx_grt_unserialize_from_xml(grt, node, objects_by_id), MYX_GRT_VALUE*);
}
#endif
/**
 ****************************************************************************
 * @brief Convert a GRT value into a XML string
 *
 * Produces a XML representation of the GRT value.
 * NOTE: not reentrant with other XML serialization functions
 * 
 * @param value a GRT value
 *
 * @return String containing the value as a XML or NULL if there's an error.
 * 
 * @see myx_grt_value_from_xml
 *****************************************************************************/
char *myx_grt_value_to_xml(MYX_GRT *grt, MYX_GRT_VALUE *value)
{
  xmlDocPtr doc;
  xmlNodePtr root;
  xmlChar *buffer= NULL;
  int size;
  
  g_return_val_if_fail(grt->serializer_cache==NULL, NULL);
  
  if (value)
  {
    doc= xmlNewDoc((xmlChar*)"1.0");
    doc->children= root= xmlNewDocRawNode(doc, NULL, (xmlChar*)"data", NULL);
    
    grt->serializer_cache= g_hash_table_new(g_str_hash, g_str_equal);
    
    myx_grt_serialize_to_xml(grt, root, value);
    
    g_hash_table_destroy(grt->serializer_cache);
    grt->serializer_cache= NULL;

    xmlDocDumpFormatMemory(doc, &buffer, &size, 1);

    xmlFreeDoc(doc);

    return (char*)buffer;
  }
  else
    return NULL;
}


/**
 ****************************************************************************
 * @brief Parse a XML representation of a GRT value.
 *
 * Parses a XML string and rebuilds the GRT value that corresponds to it.
 *
 * @param str the string in XML format containing a serialized GRT value
 * @param size length of the string
 *
 * @return The value corresponding to the passed in string or NULL if there's
 *   an error.
 * 
 * @see myx_grt_value_to_xml
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_xml(MYX_GRT *grt, const char *str, size_t size)
{
  xmlDocPtr doc= xmlParseMemory(str, (int)size);
  xmlNodePtr root;
  MYX_GRT_VALUE *value;

  if (!doc)
  {
    xmlErrorPtr error= xmlGetLastError();

    if (error) 
    {
      myx_grt_messages_stack_add_error(grt, "Could not parse XML data. Line %d, %s", NULL, 0, 
        error->line, error->message);
    }
    else
    {
      g_warning("Could not parse XML data");
    }
    return NULL;
  }

  root= xmlDocGetRootElement(doc);

  //check if we have a <data> rootnode
  if (root && (xmlStrcmp(root->name, (xmlChar*)"data")==0))
    root= root->children;

  //skip all nodes that are no <value> nodes
  while (root && xmlStrcmp(root->name, (xmlChar*)"value")!=0) 
    root= root->next;

  if (root)
  {
    GHashTable *objects_by_id= g_hash_table_new(g_str_hash, g_str_equal);
    value= myx_grt_unserialize_from_xml(grt, root);
    g_hash_table_destroy(objects_by_id);
  }
  else
    value= NULL;

  xmlFreeDoc(doc);

  return value;
}


#if 0
/**
 ****************************************************************************
 * @brief Parse a XML representation of a GRT value that might be a global object.
 *
 * Parses a XML string and rebuilds the GRT value that corresponds to it.
 *
 * @param grt the GRT environment
 * @param str the string in XML format containing a serialized GRT value
 * @param size length of the string
 *
 * @return The value corresponding to the passed in string or NULL if there's
 *   an error.
 * 
 * @see myx_grt_value_to_xml
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_xml_global_object(MYX_GRT *grt, const char *str, size_t size)
{
  xmlDocPtr doc= xmlParseMemory(str, (int)size);
  xmlNodePtr root;
  MYX_GRT_VALUE *value;

  if (!doc)
  {
    g_warning("Could not parse XML data");
    return NULL;
  }

  GRT_ENTER(grt);
  
  root= xmlDocGetRootElement(doc);

  //check if we have a <data> rootnode
  if (root && (xmlStrcmp(root->name, (xmlChar*)"data")==0))
    root= root->children;

  //skip all nodes that are no <value> nodes
  while (root && xmlStrcmp(root->name, (xmlChar*)"value")!=0) 
    root= root->next;

  if (root)
  {
    GHashTable *objects_by_id= g_hash_table_new(g_str_hash, g_str_equal);
    value= unserialize_from_xml_global_object(grt, root, objects_by_id);
    g_hash_table_destroy(objects_by_id);
  }
  else
    value= NULL;

  xmlFreeDoc(doc);

  GRT_RETURN(grt, value, MYX_GRT_VALUE*);
}

/**
 ****************************************************************************
 * @brief Convert a GRT value into a XML global object string
 *
 * Produces a XML representation of the GRT value as a global object
 * 
 * @param value a GRT value
 *
 * @return String containing the value as a XML or NULL if there's an error.
 * 
 * @see myx_grt_value_to_xml
 *****************************************************************************/
char *myx_grt_value_to_xml_global_object(const char *objectPath, MYX_GRT_VALUE *value)
{
  xmlDocPtr doc;
  xmlNodePtr root;
  xmlChar *buffer= NULL;
  GHashTable *saved_ids;
  int size;
  
  if (value)
  {
    doc= xmlNewDoc((xmlChar*)"1.0");
    doc->children= root= xmlNewDocRawNode(doc, NULL, (xmlChar*)"data", NULL);
    
    saved_ids= g_hash_table_new(g_str_hash, g_str_equal);
    
    myx_grt_serialize_to_xml_global_object(objectPath, root, value, saved_ids);
    
    g_hash_table_destroy(saved_ids);

    xmlDocDumpFormatMemory(doc, &buffer, &size, 1);

    xmlFreeDoc(doc);

    return (char*)buffer;
  }
  else
    return NULL;
}
#endif 
