
#include "myx_grtpp.h"
#include <stdarg.h>

namespace grt {
  
//----------------------------------------------------------------------
// GRT
#if 0
}
#endif

GRT::GRT()
{
  _grt= myx_grt_initialize(0);
}


Module *GRT::get_module(const std::string &name)
{
  if (_modules.find(name) != _modules.end())
    return _modules[name];
  return 0;
}


bool GRT::register_module(Module *module)
{
  if (_modules.find(module->name()) == _modules.end())
  {
    _modules[module->name()]= module;
    return true;
  }
  return false;
}

void GRT::load_structs_from(const std::string &path)
{
  MYX_GRT_ERROR error= myx_grt_struct_load_and_register(_grt, path.c_str());

  if (error != MYX_GRT_NO_ERROR)
    throw GRTError(error);
}

ObjectValue GRT::create_object(const std::string &struct_name, ...)
{
  MYX_GRT_VALUE *obj;
  
  va_list args;
  va_start(args, struct_name);
  obj= myx_grt_obj_newv(_grt, struct_name.c_str(), args);
  va_end(args);

  return ObjectValue(obj);
}


bool GRT::serialize(const Value &value, const std::string &path)
{
  MYX_GRT_ERROR err= myx_grt_store_to_file(_grt, value.grt_value(), path.c_str());
  
  return err == MYX_GRT_NO_ERROR;
}


Value GRT::unserialize(const std::string &path)
{  
  return myx_grt_retrieve_from_file(_grt, path.c_str());
}


void GRT::enable_thread_notifications()
{
  myx_grt_notification_listening_enable(_grt);
}

void GRT::flush_notifications()
{
  myx_grt_notifications_flush(_grt);
}

void GRT::post_notification_now(const std::string &name, const Value &value, const Value &argdata)
{
  myx_grt_notification_post_named(_grt, value.grt_value(), name.c_str(), argdata.grt_value(), 0);
}

void GRT::post_notification_later(const std::string &name, const Value &value, const Value &argdata)
{
  myx_grt_notification_post_named(_grt, value.grt_value(), name.c_str(), argdata.grt_value(), 1);
}


void GRT::listener_callback(MYX_GRT_NOTIFICATION *notif, void *data)
{
  Listener *listener= static_cast<Listener*>(data);

  listener->notify(notif->name, notif->object, notif->argdata);
}


void GRT::add_listener(Listener *listener, const std::string &name, const Value &value)
{
  myx_grt_listener_add(_grt, value.grt_value(), myx_grt_intern_notification_name(name.c_str()), &listener_callback, listener);
}

void GRT::remove_listeners(Listener *listener)
{
  myx_grt_listener_remove(_grt, &listener_callback, listener, NULL, NULL, MYX_GRT_LMASK_CALLBACK);
}

void GRT::remove_listeners(Listener *listener, const std::string &name, const Value &value)
{
  myx_grt_listener_remove(_grt, &listener_callback, listener, value.grt_value(), name.c_str(), (MYX_GRT_LISTENER_MASK)(MYX_GRT_LMASK_CALLBACK|MYX_GRT_LMASK_NAME|MYX_GRT_LMASK_OBJECT));
}

void GRT::remove_listeners(Listener *listener, const std::string &name)
{
  myx_grt_listener_remove(_grt, &listener_callback, listener, NULL, name.c_str(), (MYX_GRT_LISTENER_MASK)(MYX_GRT_LMASK_CALLBACK|MYX_GRT_LMASK_NAME));
}

void GRT::remove_listeners(Listener *listener, const Value &value)
{
  myx_grt_listener_remove(_grt, &listener_callback, listener, value.grt_value(), NULL, (MYX_GRT_LISTENER_MASK)(MYX_GRT_LMASK_CALLBACK|MYX_GRT_LMASK_OBJECT));
}

void GRT::remove_thread_listeners()
{
  myx_grt_listener_remove(_grt, NULL, NULL, NULL, NULL, MYX_GRT_LMASK_CURRENT_THREAD);
}


//----------------------------------------------------------------------
// Values

/*
Value &Value::operator = (const Value &other)
{
  if (_value)
    myx_grt_value_release(_value);
  _value= myx_grt_value_retain(other._value);
  return *this;
}
*/
  
  
IntValue &IntValue::operator = (const IntValue &other)
{
  if (_value != other._value)
  {
    if (_value)
      myx_grt_value_release(_value);
    _value= myx_grt_value_retain(other._value);
  }
  return *this;
}


IntValue &IntValue::operator = (int value)
{
  if (_value)
    myx_grt_value_release(_value);
  _value= myx_grt_value_from_int(value);
  return *this;
}

//----------------------------------------------------------------------


DoubleValue &DoubleValue::operator = (const DoubleValue &other)
{
  if (_value != other._value)
  {
    if (_value)
      myx_grt_value_release(_value);
    _value= myx_grt_value_retain(other._value);
  }
  return *this;
}


DoubleValue &DoubleValue::operator = (double value)
{
  if (_value)
    myx_grt_value_release(_value);
  _value= myx_grt_value_from_real(value);
  return *this;
}

//----------------------------------------------------------------------


StringValue &StringValue::operator = (const StringValue &other)
{
  if (_value != other._value)
  {
    if (_value)
      myx_grt_value_release(_value);
    _value= myx_grt_value_retain(other._value);
  }
  return *this;
}


StringValue &StringValue::operator = (const std::string &value)
{
  if (_value)
    myx_grt_value_release(_value);
  _value= myx_grt_value_from_string(value.c_str());
  return *this;
}

//----------------------------------------------------------------------

/*
template<class O> void ListValue<O>::insert(const O &value, int index)
{
  if (content_type() != value.type())
    throw TypeMismatch(content_type(), value.type());
  
  std::string content_struct= content_struct_name();
  
  if (value.type() == MYX_OBJECT_VALUE && !content_struct.empty() &&
      !ObjectValue::cast_from(value).isinstance(content_struct))
    throw TypeMismatch();

  myx_grt_list_item_insert(_value, index, value.grt_value());
};
 */


//----------------------------------------------------------------------
// ObjectValue

ObjectValue::ObjectValue(MYX_GRT_VALUE *value)
    : Value(value)
{
  if (myx_grt_value_get_type(_value) != MYX_OBJECT_VALUE)
    throw InvalidGRTType();
}


ObjectValue::ObjectValue(GRT *grt, const std::string &structname, const DictValue &args)
    : Value()
{
  _value= myx_grt_obj_newd(grt->grt(), structname.c_str(), args.grt_value());
}


bool ObjectValue::isinstance(const std::string &struct_name) const
{
  return myx_grt_obj_is_or_inherits_from(_value, struct_name.c_str());
}
  

bool ObjectValue::has_member(const std::string &name) const
{
  return myx_grt_obj_has_member(_value, name.c_str());
}
  

Value ObjectValue::get_member(const std::string &name) const
{
  MYX_GRT_VALUE *r= myx_grt_obj_get_value(_value, name.c_str());
  if (!r)
    throw InvalidItem(name);
  return r;
}


int ObjectValue::get_int_member(const std::string &name) const
{
  MYX_GRT_VALUE *r= myx_grt_obj_get_value(_value, name.c_str());
  if (!r)
    throw InvalidItem(name);
  return myx_grt_value_as_int(r);
}


std::string ObjectValue::get_string_member(const std::string &name) const
{
  MYX_GRT_VALUE *r= myx_grt_obj_get_value(_value, name.c_str());
  if (!r)
    throw InvalidItem(name);
  return myx_grt_value_as_string(r);
}


double ObjectValue::get_double_member(const std::string &name) const
{
  MYX_GRT_VALUE *r= myx_grt_obj_get_value(_value, name.c_str());
  if (!r)
    throw InvalidItem(name);
  return myx_grt_value_as_real(r);
}

  
void ObjectValue::add_to_member(const std::string &name, Value &value)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_add_list_value(_value, name.c_str(), value.grt_value());
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err == MYX_GRT_VALIDATION_ERROR)
    throw TypeMismatch();
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}


void ObjectValue::del_from_member(const std::string &name, unsigned int index)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_del_list_value(_value, name.c_str(), index);
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}

void ObjectValue::set_member(const std::string &name, const Value &value)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_set_value(_value, name.c_str(), value.grt_value());
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err == MYX_GRT_VALIDATION_ERROR)
    throw TypeMismatch();
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}

void ObjectValue::set_member(const std::string &name, int value)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_set_value(_value, name.c_str(), IntValue(value).grt_value());
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err == MYX_GRT_VALIDATION_ERROR)
    throw TypeMismatch();
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}

void ObjectValue::set_member(const std::string &name, const std::string &value)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_set_value(_value, name.c_str(), StringValue(value).grt_value());
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err == MYX_GRT_VALIDATION_ERROR)
    throw TypeMismatch();
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}

void ObjectValue::set_member(const std::string &name, double value)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_set_value(_value, name.c_str(), DoubleValue(value).grt_value());
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err == MYX_GRT_VALIDATION_ERROR)
    throw TypeMismatch();
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}

void ObjectValue::set_member(const std::string &name, const ObjectValue &object)
{
  MYX_GRT_ERROR err;
  err= myx_grt_obj_set_value(_value, name.c_str(), object.grt_value());
  if (err == MYX_GRT_BAD_ITEM)
    throw InvalidItem(name);
  else if (err == MYX_GRT_VALIDATION_ERROR)
    throw TypeMismatch();
  else if (err != MYX_GRT_NO_ERROR)
    throw GRTError(err);
}


ObjectBridgeBase *ObjectValue::get_bridge_private()
{
  MYX_GRT_OBJECT_BRIDGE *bridge= _value->value.o->bridge;

  // check if this is really a c++ bridge
  if (bridge->initialize == &ObjectBridgeBase::_initialize)
  {
    return static_cast<ObjectBridgeBase*>(_value->value.o->bridge_data);
  }
  return 0;
}


//----------------------------------------------------------------------
// Module

Module::Module(GRT *grt, const std::string &name)
    : _grt(grt)
{
  _module= myx_grt_module_get(grt->grt(), name.c_str());
}
    

Value Module::call_function(const std::string &function,
                            const ListValue<Value> &args)
{
  MYX_GRT_FUNCTION *func;
  MYX_GRT_VALUE *ret;
  MYX_GRT_ERROR error;

  func= myx_grt_module_function_get(_module, function.c_str());
  if (!func)
    throw InvalidFunction(_module->name, function);

  ret= myx_grt_function_call(_grt->grt(), func, args.grt_value(), &error);
  
  if (error != MYX_GRT_NO_ERROR)
  {
    if (ret && myx_grt_value_get_type(ret) == MYX_DICT_VALUE)
    {
      const char *err= myx_grt_dict_item_get_as_string(ret, "error");
      const char *det= myx_grt_dict_item_get_as_string(ret, "detail");
      
      throw FunctionCallError(_module->name, function, err, det);
    }
    else
      throw FunctionCallError(_module->name, function, error);
  }
  else
  {
    if (ret && myx_grt_value_get_type(ret) == MYX_DICT_VALUE)
    {
      return myx_grt_dict_item_get_value(ret, "value");
    }
    else
    {
      g_warning("Function %s.%s returned invalid typed result",
                _module->name, function.c_str());

      throw InvalidFunctionReturn(_module->name, function, ret);
    }
  }
}



//----------------------------------------------------------------------
// Object Bridge

// "global" bridge data, shared by all instances of the same bridge class
struct ObjectBridgeBaseData {
  GRT *grt;
  ObjectBridgeBase::FactoryMethod factory;
  void *data;
};


void ObjectBridgeBase::xset_int(xmlNodePtr node, const std::string &name, int value)
{
  char buffer[50];
  sprintf(buffer, "%i", value);
  xmlNewProp(node, (xmlChar*)name.c_str(), (xmlChar*)buffer);
}


void ObjectBridgeBase::xset_string(xmlNodePtr node, const std::string &name, const std::string &value)
{
  xmlNewProp(node, (xmlChar*)name.c_str(), (xmlChar*)value.c_str());
}


void ObjectBridgeBase::xset_double(xmlNodePtr node, const std::string &name, double value)
{
  char buffer[100];
  sprintf(buffer, "%f", value);
  xmlNewProp(node, (xmlChar*)name.c_str(), (xmlChar*)buffer);
}


xmlNodePtr ObjectBridgeBase::xadd_child(xmlNodePtr node, const std::string &name, const std::string &value)
{
  if (value.empty())
    return xmlNewTextChild(node, NULL, (xmlChar*)name.c_str(), NULL);

  return xmlNewTextChild(node, NULL, (xmlChar*)name.c_str(), (xmlChar*)value.c_str());
}


GRT *ObjectBridgeBase::grt()
{
  ObjectBridgeBaseData *bdata= (ObjectBridgeBaseData*)_self->value.o->bridge->data;
  
  return bdata->grt;
}


//XXX TODO catch exceptions

int ObjectBridgeBase::_initialize(MYX_GRT_VALUE *self, MYX_GRT_VALUE *argdict)
{
  // instantiate and initialize
  ObjectBridgeBase *base;
  ObjectBridgeBaseData *bdata= (ObjectBridgeBaseData*)self->value.o->bridge->data;

  base= (*bdata->factory)(bdata->grt, self, bdata->data);
  self->value.o->bridge_data= base;

  base->initialize(DictValue(argdict));

  return 0;
}


int ObjectBridgeBase::_destroy(MYX_GRT_VALUE *self)
{
  ObjectBridgeBase *base= (ObjectBridgeBase*)self->value.o->bridge_data;

  base->destroy();
  
  delete base;
  
  return 0;
}


MYX_GRT_VALUE *ObjectBridgeBase::_get_member(MYX_GRT_VALUE *self, const char *name)
{
  ObjectBridgeBase *base= (ObjectBridgeBase*)self->value.o->bridge_data;

  Value ret;

  ret= base->get_item(name);

  // here we count how many reference counts the returned value has.
  // if it is 1, it means that the only ref held is by "ret"
  // that, in turn means that the subclass that implemented get_item()
  // is not holding a reference to it.
  // so as soon as we return from here, the return GRT value will be
  // released and destroyed, which we don't want.
  // since we don't have a global autorelease pool in the GRT, we'll have to
  // let the subclass handle that.
  if (ret.is_valid() && ret.refcount() <= 1)
  {
    g_error("get_member() for bridge %s has returned a value that's not retained! (%i)",
            self->value.o->bridge->name, ret.refcount());
  }

  return ret.grt_value();
}


MYX_GRT_ERROR ObjectBridgeBase::_set_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value)
{
  ObjectBridgeBase *base= (ObjectBridgeBase*)self->value.o->bridge_data;

  base->set_item(name, Value(value));
  
  return MYX_GRT_NO_ERROR;
}


MYX_GRT_ERROR ObjectBridgeBase::_add_to_member(MYX_GRT_VALUE *self, const char *name, MYX_GRT_VALUE *value)
{
  ObjectBridgeBase *base= (ObjectBridgeBase*)self->value.o->bridge_data;

  base->add_to_item(name, Value(value));
  return MYX_GRT_NO_ERROR;
}


MYX_GRT_ERROR ObjectBridgeBase::_del_from_member(MYX_GRT_VALUE *self, const char *name, unsigned int index)
{
  ObjectBridgeBase *base= (ObjectBridgeBase*)self->value.o->bridge_data;

  base->del_from_item(name, index);
  return MYX_GRT_NO_ERROR;
}


int ObjectBridgeBase::_serialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr node)
{
  ObjectBridgeBase *base= (ObjectBridgeBase*)self->value.o->bridge_data;

  base->serialize(node);
  return 0;
}


int ObjectBridgeBase::_unserialize(MYX_GRT_VALUE *self, MYX_GRT *grt, xmlNodePtr node)
{
  // instantiate and unserialize
  ObjectBridgeBase *base;
  ObjectBridgeBaseData *bdata= (ObjectBridgeBaseData*)self->value.o->bridge->data;

  base= (*bdata->factory)(bdata->grt, self, bdata->data);
  self->value.o->bridge_data= base;

  base->unserialize(node);
  return 0;
}


int ObjectBridgeBase::_copy(MYX_GRT_VALUE *self, MYX_GRT_VALUE *orig)
{
  // instantiate and copy
  ObjectBridgeBase *obase= (ObjectBridgeBase*)orig->value.o->bridge_data;;
  ObjectBridgeBase *base;
  ObjectBridgeBaseData *bdata= (ObjectBridgeBaseData*)self->value.o->bridge->data;

  base= (*bdata->factory)(bdata->grt, self, bdata->data);
  self->value.o->bridge_data= base;

  base->copy(obase);
  return 0;
}


bool ObjectBridgeBase::register_bridge(GRT *grt, const std::string &name,
                                       FactoryMethod method, void *data)
{
  MYX_GRT_OBJECT_BRIDGE *bridge;
  ObjectBridgeBaseData *bridge_data= new ObjectBridgeBaseData;

  bridge_data->grt= grt;
  bridge_data->factory= method;
  bridge_data->data= data;

  bridge= g_new0(MYX_GRT_OBJECT_BRIDGE, 1);

  bridge->name= g_strdup(name.c_str());
  bridge->data= bridge_data;
  bridge->initialize= &_initialize;
  bridge->destroy= &_destroy;
  bridge->get_member= &_get_member;
  bridge->set_member= &_set_member;
  bridge->add_to_member= &_add_to_member;
  bridge->del_from_member= &_del_from_member;
  bridge->serialize= &_serialize;
  bridge->unserialize= &_unserialize;
  bridge->copy= &_copy;

  if (myx_grt_obj_bridge_register(grt->grt(), bridge) != MYX_GRT_NO_ERROR)
  {
    delete bridge_data;
    g_free(bridge->name);
    g_free(bridge);

    return false;
  }

  return true;
}



ObjectBridgeBase::ObjectBridgeBase(GRT *grt, MYX_GRT_VALUE *self)
  : _self(self) // must not retain, or it would be a reference cycle
{
}

ObjectBridgeBase::~ObjectBridgeBase()
{
}

};


