#include "testgrt.h"
#include "test_objects.h"


BEGIN_TEST_DATA_CLASS(grt_object_value)
public:
  GRT *grt;
TEST_DATA_CONSTRUCTOR(grt_object_value) 
{
  grt= new GRT();
};

END_TEST_DATA_CLASS


TEST_MODULE(grt_object_value, "GRT: object values");


/*
 * - load sample structs xml
 * - create objects
 * - check getting/setting values
 * - check duplication
 * - check if all members exist
 * - check adding invalid member
 * 
 * - for bridged object
 * - register bridge
 * - allocate object
 * - getting/setting values
 * - getting/setting bad values
 * - duplication
 * - destroy
 * 
 */


TEST_FUNCTION(1)
{
  // load structs
  grt->load_structs_from("../generic-runtime-3/testing/unit-tests/structs.xml");

  ensure_equals("load structs", myx_grt_struct_get_count(grt->grt()), 5);
}


TEST_FUNCTION(5)
{
  ObjectValue obj(grt->create_object("test.Book", NULL));

  obj.set_member("title", "Harry Potter");
  ensure_equals("get_member", obj.get_string_member("title"), "Harry Potter");

  obj.set_member("price", 123.45);
  ensure_equals("get_dbl_member", obj.get_double_member("price"), 123.45);

  
  ObjectValue author(grt->create_object("test.Author",
                                        "name", StringValue("Some One").grt_value(),
                                        NULL));
  
  ensure_equals("constructor", author.get_string_member("name"), "Some One");

  bool flag= false;
  try {
    obj.add_to_member("authors", author);
  } catch (...) {
    flag= true;
  };
  ensure("add value to list", flag==false);

  try {
    obj.add_to_member("authors", obj);
  } catch (...) {
    flag= true;
  };
  ensure("add bad value to list", flag==true);
}


TEST_FUNCTION(6)
{
  ObjectValue obj(grt->create_object("test.Book", NULL));
  bool flag= false;
  try {
    obj.set_member("invalid", "XXX");
  } catch (InvalidItem exc) {
    flag= true;
  };
  ensure("set invalid member", flag==true);

  flag= false;
  try {
    obj.get_int_member("invalid");
  } catch (InvalidItem exc) {
    flag= true;
  };
  ensure("get invalid member", flag==true);

  flag= false;
  try { obj.set_member("title", 1234); } catch (TypeMismatch exc) { flag= true; };
  ensure("set bad type1", flag==true);
  flag= false;
  try { obj.set_member("title", 1234.123); } catch (TypeMismatch exc) { flag= true; };
  ensure("set bad type2", flag==true);
  flag= false;
  try { obj.set_member("price", "hello"); } catch (TypeMismatch exc) { flag= true; };
  ensure("set bad type3", flag==true);
  flag= false;
  try { obj.set_member("authors", "joe"); } catch (TypeMismatch exc) { flag= true; };
  ensure("set bad type4", flag==true);
  flag= false;
  try { obj.set_member("authors", 1234); } catch (TypeMismatch exc) { flag= true; };
  ensure("set bad type5", flag==true);
  flag= false;
  try { obj.set_member("pages", 1234.456); } catch (TypeMismatch exc) { flag= true; };
  ensure("set bad type6", flag==true);
  
}



// generated object tests

TEST_FUNCTION(10)
{
  test_Book book(grt);
  
  book.title("Harry Potter");
  book.title(book.title()+ " XXV");
  book.price(500.23);
  ensure("bad title", !(book.title() == "Harry Potter"));

  ensure("title", book.title() == "Harry Potter XXV");

  test_Author author(grt);
  
  book.authors_add(author);
  ensure("author add", book.authors().count()==1);

  book.authors().get(0).name("J.K.Bowling");
  ensure("indirect author name", author.name()=="J.K.Bowling");

  book.authors()[0]->name("J.K.Bowling");
  ensure("indirect author name", author.name()=="J.K.Bowling");

  book.authors_del(0);

  bool flag= false;
  try {
    book.authors_del(0);
  } catch (InvalidItem &exc) {
    flag= true;
  };
  ensure("bad author del", flag == true);

}




// bridged object test

class TestBridge : public ObjectBridgeBase {
public:
  grt::IntValue x;
  grt::IntValue y;
  grt::StringValue myname;
  grt::ListValue<test_Book> books;
  bool *flag;

protected:
  virtual void initialize(const DictValue &args)
  {
    x= grt::IntValue(0);
    y= grt::IntValue(0);
    myname= grt::StringValue("hello");
    books.init();
  }
  virtual void destroy()
  {
    *flag= true;
  }

  virtual Value get_item(const std::string &name)
  {
    if (name == "x")
      return x;
    if (name == "y")
      return y;
    if (name == "name")
      return myname;
    if (name == "books")
      return books;
    return Value();
  }
  
  virtual void set_item(const std::string &name, const Value &value)
  {
    if (name == "x")
      assign(x, value);
    if (name == "y")
      assign(y, value);
    if (name == "name")
      assign(myname, value);
    if (name == "books")
      throw ReadOnlyItem(name);
  }
  
  virtual void add_to_item(const std::string &name, const Value &value)
  {
    if (name == "books")
    {
      test_Book book(test_Book::cast_from(value));
      books.insert(book);
    }
  }

  virtual void del_from_item(const std::string &name, unsigned int index)
  {
    if (name == "books")
    {
      books.remove(index);
    }
  }

  virtual void serialize(xmlNodePtr node)
  {
  }
  
  virtual void unserialize(xmlNodePtr node)
  {
  }
  
  virtual void copy(ObjectBridgeBase *orig)
  {
  }

  
public:
  TestBridge(GRT *grt, MYX_GRT_VALUE *self) : ObjectBridgeBase(grt, self) {};

  static ObjectBridgeBase*  create(GRT *grt, MYX_GRT_VALUE *self, void *data)
  {
    return new TestBridge(grt, self);
  }
};


TEST_FUNCTION(20)
{
  bool ret;
  
  ret= TestBridge::register_bridge(grt, "cppbridge", &TestBridge::create, NULL);
  ensure_equals("bridge registration", ret, true);
}


TEST_FUNCTION(21)
{
  bool bridge_destroyed= false;
  
  {
    test_Bridged bridged(grt);
    test_Book book(grt);
  
    book.title("Harry Potter");
    book.title(book.title()+ " XXV");
    book.price(500.23);

    TestBridge *bridge_data;
    
    ensure("get_value", bridged.name() == "hello");
    
    bridge_data= (TestBridge*)bridged.get_bridge_private();
    
    bridge_data->flag= &bridge_destroyed;
    
    ensure("get private", bridge_data!=0);
    ensure("check private", (std::string)bridge_data->myname == bridged.name());
    
    bridged.name("xyz");
    ensure("check change", (std::string)bridge_data->myname == "xyz");
    
    bridged.x(1234);
    ensure_equals("change x", bridged.x(), 1234);
    
    ensure_equals("list count", (int)bridged.books().count(), 0);
    
    bridged.books_add(book);
    
    ensure_equals("list count after add", (int)bridged.books().count(), 1);

    ensure_equals("book title", bridged.books().get(0).title(),
                  "Harry Potter XXV");
  
    bridged.books_del(0);
    
    ensure_equals("list count after del", (int)bridged.books().count(), 0);

    bool flag= false;
    try {
      bridged.books_del(0);
    } catch (InvalidItem &exc) {
      flag= true;
    };
    ensure("del invalid item", flag==true);
  }
  // leaving the context should destroy the objects

  ensure_equals("bridge destroy callback", bridge_destroyed, true);
}


END_TESTS
