/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include <glib.h>
#include <glib/gstdio.h>
#include "string_utilities.h"
#include "wb_sql_editor_snippets.h"
#include "sqlide/wb_sql_editor_form.h"
#include "mforms/utilities.h"
#include "mforms/filechooser.h"
#include "file_functions.h"
#include "util_functions.h"

static DbSqlEditorSnippets *singleton= 0;

void DbSqlEditorSnippets::setup(wb::WBContextSQLIDE *sqlide, const std::string &path)
{
  if (singleton != 0)
    return;
  
  singleton = new DbSqlEditorSnippets(sqlide, path);
}


DbSqlEditorSnippets *DbSqlEditorSnippets::get_instance()
{
  return singleton;
}


bec::ToolbarItemList DbSqlEditorSnippets::get_toolbar_items()
{
  bec::ToolbarItemList items;
  bec::ToolbarItem item;
  bec::IconManager *im = bec::IconManager::get_instance();

  item.icon = im->get_icon_id("snippet_add.png");
  item.caption = "Add Snippet";
  item.name = "add_snippet";
  item.command = "add_snippet";
  item.tooltip = "Add SQL Code in Editor to Snippet List";
  item.type = bec::ToolbarAction;
  item.enabled = true;
  items.push_back(item);

  item.icon = im->get_icon_id("snippet_del.png");
  item.caption = "Delete Snippet";
  item.name = "del_snippet";
  item.command = "del_snippet";
  item.tooltip = "Delete Selected Snippet from List";
  item.type = bec::ToolbarAction;
  item.enabled = true;
  items.push_back(item);

  item.type = bec::ToolbarSeparator;
  items.push_back(item);

  item.icon = im->get_icon_id("snippet_use.png");
  item.caption = "Use Snippet";
  item.name = "replace_text";
  item.command = "replace_text";
  item.tooltip = "Replace active SQL editor contents with the selected snippet";
  item.type = bec::ToolbarAction;
  item.enabled = true;
  items.push_back(item);

  item.icon = im->get_icon_id("snippet_insert.png");
  item.caption = "Insert Snippet";
  item.name = "insert_text";
  item.command = "insert_text";
  item.tooltip = "Insert selected snippet to cursor position in the active SQL editor";
  item.type = bec::ToolbarAction;
  item.enabled = true;
  items.push_back(item);

  item.type = bec::ToolbarSeparator;
  items.push_back(item);

  return items;
}


bool DbSqlEditorSnippets::activate_toolbar_item(const bec::NodeId &selected, const std::string &name)
{
  if (name == "add_snippet")
  {
    Db_sql_editor *db_sql_editor = _sqlide->get_active_sql_editor();
    if (db_sql_editor)
    {
      int start, end;
      std::string text = db_sql_editor->sql_editor()->sql();
      if (db_sql_editor->sql_editor()->selected_range(start, end))
        text = text.substr(start, end - start);
      if (db_sql_editor->save_snippet(text))
        _sqlide->get_grt_manager()->replace_status_text("SQL saved to snippets list.");
    }
  }
  else if (name == "del_snippet" && selected.is_valid() && selected[0] >= 0 && selected[0] < (int) _entries.size())
  {
    delete_node(selected);
    Db_sql_editor *editor = _sqlide->get_active_sql_editor();
    if (editor)
      editor->do_partial_ui_refresh(Db_sql_editor::RefreshSnippets);
  }
  else if ((name == "replace_text" || "insert_text") && selected.is_valid() && selected[0] >= 0 && selected[0] < (int) _entries.size())
  {
    std::string script;
    
    script.append(_entries[selected[0]].second);
    script.append("\n");

    Db_sql_editor *editor = _sqlide->get_active_sql_editor();
    if (editor)
    {
      bool do_ui_refresh= true;
      if (name == "replace_text")
      {
        editor->sql_editor()->is_refresh_enabled(true);
        editor->sql_editor()->sql(script);
      }
      else if (name == "insert_text")
      {
        editor->sql_editor()->is_refresh_enabled(true);
        editor->sql_editor()->set_selected_text(script);
//#if defined(_WIN32)
        //do_ui_refresh= false;
//#endif
      }
      if (do_ui_refresh)
        editor->do_partial_ui_refresh(Db_sql_editor::RefreshEditor);
    }
    else
      return false;
  }
  else
    return false;

  return true;
}


std::vector<std::string> DbSqlEditorSnippets::get_category_list()
{
  std::vector<std::string> categories;

  GDir *dir = g_dir_open(_path.c_str(), 0, NULL);
  if (dir)
  {
    const gchar *name;
    while ((name = g_dir_read_name(dir)))
    {
      if (g_str_has_suffix(name, ".txt"))
        categories.push_back(std::string(name, strlen(name)-4));
    }
    g_dir_close(dir);
  }

  std::vector<std::string>::iterator iter;
  // move up User Snippets to 1st entry
  if ((iter = std::find(categories.begin(), categories.end(), "User Snippets")) != categories.end())
    categories.erase(iter);
  categories.insert(categories.begin(), "User Snippets");
  /*
  if (std::find(categories.begin(), categories.end(), "SQL DML Statements") == categories.end())
    categories.push_back("SQL DML Statements");
  if (std::find(categories.begin(), categories.end(), "SQL DDL Statements") == categories.end())
    categories.push_back("SQL DDL Statements");
*/
  //categories.push_back("Import From File...");

  return categories;
}

void DbSqlEditorSnippets::select_category(const std::string &category)
{
  /*if (category == "Import From File...")
  {
    mforms::FileChooser chooser(mforms::OpenFile);
    chooser.set_title("Import SQL Snippets");

    if (chooser.run_modal())
    {
      std::string path = chooser.get_path();


      
      Db_sql_editor *editor = _sqlide->get_active_sql_editor();
      if (editor)
        editor->do_partial_ui_refresh(Db_sql_editor::RefreshSnippetCollections);
    }
  }
  else*/
  {
    _selected_category = category;
    load();
    Db_sql_editor *editor = _sqlide->get_active_sql_editor();
    if (editor)
      editor->do_partial_ui_refresh(Db_sql_editor::RefreshSnippets);
  }
}

void DbSqlEditorSnippets::load()
{
  _entries.clear();

  FILE *f = base_fopen(base::strfmt("%s/%s.txt", _path.c_str(), _selected_category.c_str()).c_str(), "r");
  if (f)
  {
    char line[4096];

    while (fgets(line, sizeof(line), f))
    {
      char *ptr = strchr(line, '\n');
      if (ptr)
        *ptr= 0;
      std::string name = line;
      std::string script = "";
      bool cont = false;

      while (fgets(line, sizeof(line)-1, f) && line[0] == ' ')
      {
        script.append(line+(cont ? 0 : 1));
        cont = strchr(line, '\n') != 0;
      }

      // Remove the last line break, we added that, not the user.
      if (script.size() > 0)
        script.erase(script.size() - 1, 1);
      _entries.push_back(std::make_pair(name, script));
    }
    
    fclose(f);
  }
}


void DbSqlEditorSnippets::save()
{
  FILE *f = base_fopen(base::strfmt("%s/%s.txt", _path.c_str(), _selected_category.c_str()).c_str(), "w+");
  if (f)
  {
    for (std::vector<std::pair<std::string, std::string> >::const_iterator i = _entries.begin();
         i != _entries.end(); ++i)
    {
      std::vector<std::string> lines = bec::split_string(i->second, "\n");
      
      fprintf(f, "%s\n", i->first.c_str());
      for (std::vector<std::string>::const_iterator l = lines.begin(); l != lines.end(); ++l)
        fprintf(f, " %s\n", l->c_str());
      fprintf(f, "\n");
    }
    fclose(f);
  }  
}


DbSqlEditorSnippets::DbSqlEditorSnippets(wb::WBContextSQLIDE *sqlide, const std::string &path)
: _sqlide(sqlide), _path(path)
{
  // check if the old snippets file exist and move it to the new location
  if (g_file_test(std::string(_path+"/../sql_snippets.txt").c_str(), G_FILE_TEST_EXISTS))
  {
    g_mkdir_with_parents(_path.c_str(), 0700);
    g_rename(std::string(_path+"/../sql_snippets.txt").c_str(), std::string(_path+"/User Snippets.txt").c_str());
  }
  else
    g_mkdir_with_parents(_path.c_str(), 0700);
  
  // copy the standard files
  std::string datadir = _sqlide->get_grt_manager()->get_data_file_path("snippets");
  {
    GDir *dir = g_dir_open(datadir.c_str(), 0, NULL);
    if (dir)
    {
      const gchar *name;
      while ((name = g_dir_read_name(dir)))
      {
        std::string dest = std::string(_path).append("/").append(name);
        if (!g_file_test(dest.c_str(), G_FILE_TEST_EXISTS))
        {
          std::string source = std::string(datadir).append("/").append(name);
          copy_file(source.c_str(), dest.c_str());
        }
      }
      g_dir_close(dir);
    }
  }

  load();
}


void DbSqlEditorSnippets::add_snippet(const std::string &name, const std::string &snippet)
{
  _entries.push_back(std::make_pair(name, snippet));
  save();
}



int DbSqlEditorSnippets::count()
{
  return (int)_entries.size();
}


bool DbSqlEditorSnippets::get_field(const bec::NodeId &node, int column, std::string &value)
{
  if (node.is_valid() && node[0] >= 0 && node[0] < (int)_entries.size())
  {
    switch ((Column)column)
    {
      case Letter:
        if (node[0] <= 'Z'-'A')
        {
          value = "A";
          value[0] += node[0];
        }
        else
          value = "";
        break;
      case Description:
        {
          std::string::size_type p;
          value = _entries[node[0]].first;
          
          p = _entries[node[0]].second.find('\n');
          if (p != std::string::npos)
            value.append("\n").append(_entries[node[0]].second.substr(0, p));
          else
            value.append("\n").append(_entries[node[0]].second);
          break;
        }
      case Script:
        value = _entries[node[0]].second;
        break;
    }
    return true;
  }
  return false;
}

bool DbSqlEditorSnippets::set_field(const bec::NodeId &node, int column, const std::string &value)
{
  if (node.is_valid() && node[0] >= 0 && node[0] < (int)_entries.size() && column == 1)
  {
    _entries[node[0]].first = value;
    return true;
  }
  return false;
}

bool DbSqlEditorSnippets::activate_node(const bec::NodeId &node)
{
  if (node.is_valid())
  {
    std::string script = _entries[node[0]].second;
    
    Db_sql_editor *editor = _sqlide->get_active_sql_editor();
    if (editor)
    {
      editor->sql_editor()->is_refresh_enabled(true);
      editor->sql_editor()->sql(script);
      editor->do_partial_ui_refresh(Db_sql_editor::RefreshEditor);
      return true;
    }
  }
  return false;
}


bec::MenuItemList DbSqlEditorSnippets::get_popup_items_for_nodes(const std::vector<bec::NodeId> &nodes)
{
  bec::MenuItemList items;
  
  bec::MenuItem item;
  
  item.caption = "Replace Text in SQL Area";
  item.enabled = nodes.size() > 0;
  item.name = "replace_text";
  item.type = bec::MenuAction;
  items.push_back(item);

  item.caption = "Insert Text into SQL Area";
  item.enabled = nodes.size() > 0;
  item.name = "insert_text";
  item.type = bec::MenuAction;
  items.push_back(item);
  
  item.caption = "Copy to Clipboard";
  item.enabled = nodes.size() > 0;
  item.name = "copy_to_clipboard";
  item.type = bec::MenuAction;
  items.push_back(item);
  
  item.caption = "";
  item.enabled = true;
  item.name = "";
  item.type = bec::MenuSeparator;
  items.push_back(item);
  
  item.caption = "Delete";
  item.enabled = nodes.size() > 0;
  item.name = "delete";
  item.type = bec::MenuAction;
  items.push_back(item);
  
  return items;
}


bool DbSqlEditorSnippets::activate_popup_item_for_nodes(const std::string &name, const std::vector<bec::NodeId> &nodes)
{  
  if (name == "delete")
  {
    for (std::vector<bec::NodeId>::const_reverse_iterator n = nodes.rbegin(); n != nodes.rend(); ++n)
      delete_node(*n);
    Db_sql_editor *editor = _sqlide->get_active_sql_editor();
    if (editor)
      editor->do_partial_ui_refresh(Db_sql_editor::RefreshSnippets);
  }
  else if (name == "replace_text" || name == "copy_to_clipboard" || name == "insert_text")
  {
    std::string script;
    
    for (std::vector<bec::NodeId>::const_iterator n = nodes.begin(); n != nodes.end(); ++n)
    {
      script.append(_entries[(*n)[0]].second);
      script.append("\n");
    }

    if (name == "copy_to_clipboard")
    {
      mforms::Utilities::set_clipboard_text(script);
    }
    else
    {
      Db_sql_editor *editor = _sqlide->get_active_sql_editor();
      if (editor)
      {
        bool do_ui_refresh= true;
        if (name == "replace_text")
        {
          editor->sql_editor()->is_refresh_enabled(true);
          editor->sql_editor()->sql(script);
        }
        else if (name == "insert_text")
        {
          editor->sql_editor()->is_refresh_enabled(true);
          editor->sql_editor()->set_selected_text(script);
//#if defined(_WIN32)
//          do_ui_refresh= false;
//#endif
        }
        if (do_ui_refresh)
          editor->do_partial_ui_refresh(Db_sql_editor::RefreshEditor);
      }
      else
        return false;
    }
  }
  else
    return false;
  return true;
}


bool DbSqlEditorSnippets::can_delete_node(const bec::NodeId &node)
{
  return node.is_valid() && node[0] < (int)_entries.size();
}


bool DbSqlEditorSnippets::delete_node(const bec::NodeId &node)
{
  if (node.is_valid() && node[0] >= 0 && node[0] < (int)_entries.size())
  {
    _entries.erase(_entries.begin() + node[0]);
    save();
    return true;
  }
  return false;
}

