/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "grt_code_editor.h"
#include "string_utilities.h"
#include "grt_shell_window.h"

using namespace mforms;


GRTCodeEditor::GRTCodeEditor(GRTShellWindow *owner, bool module, const std::string &language)
  : Splitter(false), _top(false), _owner(owner), _toolbar(true), _output(VerticalScrollBar),
#ifdef _WIN32
   _text()
#else
   _text(VerticalScrollBar)
#endif
{
  _dirty= false;
  _editing_module = module;
  _language= language;

#ifdef _WIN32
  _toolbar.set_size(-1, 24);
  _toolbar.set_back_color("#BCC7D8");
  set_back_color("#283752");
#endif

  _toolbar.set_padding(2);
  _toolbar.set_spacing(4);
  _top.add(&_toolbar, false, true);
  _top.add(&_text, true, true);

#ifdef _WIN32
  if (_language == "lua")
    _text.set_language(LanguageLua);
  else
    _text.set_language(LanguagePython);
#else
  _text.set_monospaced(true);
#endif

  _text.set_font(_owner->grtm()->get_app_option_string("workbench.general.Editor:Font"));
  _output.set_read_only(true);
  
  add(&_top);
  if (!module)
    add(&_output);
  
  _text.signal_changed().connect(sigc::mem_fun(this, &GRTCodeEditor::text_changed));
  
  add_tool_button("tiny_save.png", sigc::hide_return(sigc::bind(sigc::mem_fun(this, &GRTCodeEditor::save), false)),
                  "Save file");
  add_tool_button("tiny_saveas.png", sigc::hide_return(sigc::bind(sigc::mem_fun(this, &GRTCodeEditor::save), true)),
                  "Save file with a new name");
  
  if (_editing_module)
  {
    //   add_tool_button("tiny_refresh.png", sigc::mem_fun(this, &GRTCodeEditor::execute),
    //                   "Refresh the module");
  }
  else
  {
    add_tool_button("tiny_execute.png", sigc::mem_fun(this, &GRTCodeEditor::execute),
                    "Execute Script");
    
    add_tool_button("clear_output.png", sigc::bind(sigc::mem_fun(_output, &mforms::TextBox::set_value), ""), 
                    "Clear script output");
  }  
  
#ifndef _WIN32
  // TODO: remove as soon as all platforms support closable tabs.
  add_tool_button("Discard.png", sigc::mem_fun(this, &GRTCodeEditor::close),
                  "Close this script tab", false);
#endif
}


void GRTCodeEditor::set_path(const std::string &path)
{
  _filename= path;
  _owner->set_editor_title(this, get_title());
}


void GRTCodeEditor::set_text(const std::string &text)
{
  _text.set_value(text);
}


std::string GRTCodeEditor::get_title()
{
  if (_filename.empty())
    return "Unnamed Script";
  return g_basename(_filename.c_str());
}

bool GRTCodeEditor::save(bool choose_file)
{
  if (choose_file || _filename.empty())
  {
    FileChooser chooser(SaveFile);
    
    chooser.set_title("Save File");
    if (chooser.run_modal())
      _filename = chooser.get_path();
    else
      return false;
  }
  
  std::string data = _text.get_string_value();
  GError *error = 0;
  if (!g_file_set_contents(_filename.c_str(), data.c_str(), data.size(), &error))
  {
    Utilities::show_error("Error Saving File",
                          base::strfmt("Could not save to %s:\n%s", 
                                       _filename.c_str(), error->message),
                          "OK", "", "");
    g_error_free(error);
    return false;
  }
  
  add_output(base::strfmt("Script saved as %s\n", _filename.c_str()));
  
  _owner->set_editor_title(this, get_title());
  _owner->refresh_files();
  
  _dirty= false;
  return true;
}


void GRTCodeEditor::execute()
{
  std::string script = _text.get_string_value();
  
  _owner->grtm()->push_output_callback(sigc::mem_fun(this, &GRTCodeEditor::add_output));
  
  bool ret = _owner->execute_script(script, _language);

  _owner->grtm()->pop_output_callback();
  
  if (ret)
    add_output("\nScript finished.\n");
  else
    add_output("\nError executing script.\n");
}


void GRTCodeEditor::add_output(const std::string &text)
{
  _output.append_text(text, true);
}


void GRTCodeEditor::add_tool_button(const std::string &image,
                                    const sigc::slot<void> &action,
                                    const std::string &tooltip,
                                    bool left)

{
  App *app= App::get();
  Button *b = manage(new Button(ToolButton));
  b->set_icon(app->get_resource_path(image));
  b->set_tooltip(tooltip);
  b->signal_clicked().connect(action);
  if (left)
    _toolbar.add(b, false, true);
  else
    _toolbar.add_end(b, false, true);
}


void GRTCodeEditor::text_changed()
{
  _dirty= true;
}

//--------------------------------------------------------------------------------------------------

bool GRTCodeEditor::can_close()
{
  if (_dirty)
  {
    int r;
    r= mforms::Utilities::show_message("Close Editor", 
      base::strfmt("%s has unsaved changes, would you like to save them?", get_title().c_str()),
      _("Save"), _("Cancel"), _("Don't Save"));

    if (r == mforms::ResultOk)
    {
      if (!save(false))
        return false;
    }
    else if (r == mforms::ResultCancel)
      return false;
  }

  return true;
}

//--------------------------------------------------------------------------------------------------

void GRTCodeEditor::close()
{
  if (can_close())
    _owner->close_editor(this);
}

//--------------------------------------------------------------------------------------------------
