/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"
#include "grt_shell_window.h"
#include "file_functions.h"
#include "string_utilities.h"
#include <errno.h>

#include "grt/common.h"
#include "grt_code_editor.h"
#include "grt_plugin_wizard.h"

using namespace MySQL::Geometry;

//--------------------------------------------------------------------------------------------------

GRTShellWindow::GRTShellWindow(wb::WBContext* context)
: mforms::Form(0, (mforms::FormFlag)(mforms::FormResizable | mforms::FormMinimizable | mforms::FormHideOnClose)), 
  _context(context),
  _hsplitter(true), _side_tab(mforms::TabViewPalette), _main_tab(mforms::TabViewDocumentClosable),
  _header_panel(mforms::FilledHeaderPanel), _files_tree((mforms::TreeOptions)0),
  _global_box1(false), _global_box2(false), _global_splitter(false), _global_combo(mforms::SelectorCombobox),
  _global_tree((mforms::TreeOptions)0), _global_list(true), 
  _classes_box(false), _classes_splitter(false), _classes_tree(false), _classes_text(mforms::VerticalScrollBar),
  _modules_splitter(false), _modules_tree(false), _modules_text(mforms::VerticalScrollBar),
  _shell_box(false), _shell_text(mforms::VerticalScrollBar), _shell_hbox(true),
  _snippet_box(false), _snippet_vbox(false), _snippet_bbox(true), _snippet_list(mforms::TreeShowHeader),
#ifdef _WIN32 // Can be removed once we have code editors for all platforms.
  _snippet_text()
#else
  _snippet_text(mforms::BothScrollBars)
#endif
{
  set_title(("Workbench Scripting Shell"));
  set_content(&_hsplitter);
  signal_closed().connect(sigc::mem_fun(this, &GRTShellWindow::shell_closed));
  
#ifdef _WIN32
  set_padding(Padding(6, 6, 6, 6));
  set_back_color("#283752");
    _hsplitter.set_back_color("#283752");
  _header_panel.add(&_side_tab);
  _hsplitter.add(&_header_panel);
#else
  _hsplitter.add(&_side_tab);
#endif

  _side_tab.signal_tab_changed().connect(sigc::mem_fun(this, &GRTShellWindow::side_tab_changed));

  _hsplitter.add(&_main_tab);

  // Side bar consists of 4 pages: files, globals tree, classes tree and modules tree.
  _first_show= true;

  // Files
  mforms::Box *files_box= mforms::manage(new mforms::Box(false));
  
  {
    mforms::Box *bar = mforms::manage(new mforms::Box(true));
    files_box->add(bar, false, true);
    
#ifdef _WIN32
    bar->set_size(-1, 24);
    bar->set_back_color("#BCC7D8");
#endif

    bar->set_padding(2);
    bar->set_spacing(4);
    
    App *app= App::get();
    Button *b = manage(new Button(ToolButton));
    b->set_icon(app->get_resource_path("tiny_new_doc.png"));
    b->set_tooltip("Create a new file from a template");
    b->signal_clicked().connect(sigc::mem_fun(this, &GRTShellWindow::add_new_script));
    bar->add(b, false, true);

    b = manage(new Button(ToolButton));
    b->set_icon(app->get_resource_path("tiny_load.png"));
    b->set_tooltip("Open a script file");
    b->signal_clicked().connect(sigc::mem_fun(this, &GRTShellWindow::open_script_file));
    bar->add(b, false, true);

    b = manage(new Button(ToolButton));
    b->set_icon(app->get_resource_path("Discard.png"));
    b->set_tooltip("Delete the selected file");
    b->signal_clicked().connect(sigc::mem_fun(this, &GRTShellWindow::delete_selected_file));
    bar->add_end(b, false, true);    
  }

  _files_tree.add_column(StringColumnType, "", 400, false);
  _files_tree.end_columns();

  _files_tree.signal_row_activated().connect(sigc::mem_fun(this, &GRTShellWindow::file_list_activated));
  
  files_box->add(&_files_tree, true, true);
#ifdef _WIN32
  files_box->set_back_color("#FFFFFF");
  ((View*) files_box)->set_padding(Padding(0, 0, 0, 2));
#endif
  _side_tab.add_page(files_box, "Files");
      
  // 1) Globals tree
  _global_tree_be= 0;
#ifdef _WIN32
  _global_splitter.set_back_color("#FFFFFF");
  _global_splitter.set_padding(Padding(0, 0, 0, 2));
#endif
  _side_tab.add_page(&_global_splitter, "Globals");
  _global_splitter.add(&_global_box1, 0);
  _global_splitter.add(&_global_box2, 0);
#ifndef _WIN32
  _global_box1.set_spacing(4);
  _global_box2.set_spacing(4);
#endif
  _global_box1.add(&_global_combo, false, false);
  _global_box1.add(&_global_tree, true, true);

  _global_box2.add(&_global_entry, false, true);
  _global_entry.set_read_only(true);
  _global_entry.set_back_color("#FFFFFF");
  _global_box2.add(&_global_list, true, true);
  _global_list.add_column(mforms::IconStringGRTColumnType, 0, "Name");
  _global_list.add_column(mforms::StringGRTColumnType, 1, "Value");
  _global_list.set_column_width(0, 100);
  _global_list.set_column_width(1, 100);
  _global_tree.add_column(mforms::IconStringGRTColumnType, 0, "GRT Globals");
  _global_tree.set_column_width(0, 220);
  _global_tree.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::global_selected));
  _global_combo.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::change_global_path));
  _inspector= 0;

  // 2) Classes tree
#ifdef _WIN32
  _classes_splitter.set_back_color("#FFFFFF");
  _classes_splitter.set_padding(Padding(0, 0, 0, 2));
#endif
  _side_tab.add_page(&_classes_splitter, "Classes");
  _classes_splitter.add(&_classes_box, 0);
  _classes_box.set_spacing(4);
  _classes_box.add(&_classes_sorting, false, true);
  _classes_box.add(&_classes_tree, true, true);

  _classes_splitter.add(&_classes_text, 0);
  _classes_text.set_read_only(true);
  _classes_text.set_back_color("#FFFFFF");
  _classes_tree.add_column(mforms::IconStringGRTColumnType, bec::StructsTreeBE::Name, "Name");
  _classes_tree.add_column(mforms::StringGRTColumnType, bec::StructsTreeBE::Type, "Type");
  _classes_tree.set_column_width(0, 150);
  _classes_tree.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::class_selected));
  _classes_sorting.add_item("Group by Name");
  _classes_sorting.add_item("Group by Hierarchy");
  _classes_sorting.add_item("Group by Package");
  _classes_sorting.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::change_class_sorting));
  
  // 3) Modules tree
#ifdef _WIN32
  _modules_splitter.set_back_color("#FFFFFF");
  _modules_splitter.set_padding(Padding(0, 0, 0, 2));
#endif
  _side_tab.add_page(&_modules_splitter, "Modules");
  _modules_splitter.add(&_modules_tree, 0);
  _modules_splitter.add(&_modules_text, 0);
  _modules_text.set_read_only(true);
  _modules_text.set_back_color("#FFFFFF");
  _modules_tree.add_column(mforms::IconStringGRTColumnType, 0, "GRT Modules");
  _modules_tree.set_column_width(0, 220);
  _modules_tree.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::module_selected));
  
  // setup shell
  _shell_box.add(&_shell_text, true, true);
  _shell_text.set_monospaced(true);
  _shell_text.set_read_only(true);
  _shell_text.set_front_color("#E0E0E0");
  _shell_text.set_back_color("#000000");
  _shell_text.set_padding(2);
  _shell_box.add(&_shell_hbox, false, true);
  _shell_hbox.add(&_shell_prompt, false, true);
  _shell_hbox.add(&_shell_entry, true, true);
  _main_tab.add_page(&_shell_box, "Shell");
  
  _shell_entry.signal_action().connect(sigc::mem_fun(this, &GRTShellWindow::shell_action));

  
  // snippets tab
  _snippet_vbox.set_spacing(8);
  _snippet_vbox.set_padding(8);
  _snippet_vbox.add(&_snippet_box, true, true);
  _snippet_vbox.add(&_snippet_bbox, false, true);

  //_snippet_box.set_spacing(8);
  _snippet_box.add(&_snippet_list);
  _snippet_box.add(&_snippet_text);

  _snippet_bbox.set_spacing(12);
  _snippet_bbox.add(&_snippet_add, false, true);
  _snippet_bbox.add(&_snippet_del, false, true);
  _snippet_bbox.add_end(&_snippet_copy, false, true);

  _snippet_menu.add_item("Execute Snippet", "execute");
  _snippet_menu.add_item("Send to Script Editor", "new_with_snippet");
  _snippet_menu.add_separator();
  _snippet_menu.add_item("Copy to Clipboard", "copy_clipboard");
  _snippet_menu.add_separator();
  _snippet_menu.add_item("Delete Snippet", "delete");

  _snippet_menu.set_handler(sigc::mem_fun(this, &GRTShellWindow::snippet_menu_activate));

  _snippet_list.set_context_menu(&_snippet_menu);

  _snippet_del.set_text("Delete");
  _snippet_add.set_text("Add");
  _snippet_copy.set_text("Copy to Clipboard");
  _snippet_add.signal_clicked().connect(sigc::mem_fun(this, &GRTShellWindow::add_snippet));
  _snippet_del.signal_clicked().connect(sigc::mem_fun(this, &GRTShellWindow::del_snippet));
  _snippet_copy.signal_clicked().connect(sigc::mem_fun(this, &GRTShellWindow::copy_snippet));

  _snippet_list.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::snippet_selected));
#ifdef _WIN32
  _snippet_text.set_language(LanguagePython);
#endif

  _snippet_text.set_font(grtm()->get_app_option_string("workbench.general.Editor:Font"));
  _snippet_text.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::snippet_changed));

#ifndef _WIN32
  _snippet_text.set_monospaced(true);
#endif
  
  _snippet_list.add_column(mforms::StringColumnType, "Snippet", 500, false);
  _snippet_list.end_columns();
  _main_tab.add_page(&_snippet_vbox, "Snippets");

  _main_tab.signal_tab_closing().connect(sigc::mem_fun(this, &GRTShellWindow::on_tab_closing));
  _main_tab.signal_tab_closed().connect(sigc::mem_fun(this, &GRTShellWindow::on_tab_closed));

  _snippet_box.set_position(400);

  // Minimum size for the entire window.
  set_size(800, 600);
  _hsplitter.set_position(250);
  _global_splitter.set_position(400);
  _modules_splitter.set_position(400);
  _classes_splitter.set_position(400);
  
  grtm()->get_shell()->set_ready_handler(sigc::mem_fun(this, &GRTShellWindow::handle_prompt));
  grtm()->get_shell()->set_output_handler(sigc::mem_fun(this, &GRTShellWindow::handle_output));

  snippet_selected();
  side_tab_changed();
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::shell_action(mforms::TextEntryAction action)
{ 
  switch (action)
  {
    case mforms::EntryActivate:
    {
      std::string command= _shell_entry.get_string_value();
      _shell_entry.set_value("");
      //  _completion->add_completion_text(command);
      command += '\n';
      grtm()->get_shell()->write(grtm()->get_grt()->get_shell()->get_prompt()+" "+command);
      grtm()->get_shell()->process_line_async(command);
      break;
    }
    case mforms::EntryKeyUp:
    {
      std::string line;
      if (grtm()->get_shell()->previous_history_line(_shell_entry.get_string_value(), line))
        _shell_entry.set_value(line);
      break;
    }
    case mforms::EntryCKeyUp:
      break;
    case mforms::EntryKeyDown:
    {
      std::string line;
      if (grtm()->get_shell()->next_history_line(line))
        _shell_entry.set_value(line);
      break;
    }      
    case mforms::EntryCKeyDown:
      break;
  }
}

void GRTShellWindow::show(bool flag)
{
  if (flag)
    refresh_all();
  mforms::Form::show(flag);

  load_state();
}


void GRTShellWindow::refresh_all()
{
  if (_first_show)
  {
    _first_show = false;
    set_globals_tree_path(_global_combo.get_string_value());
    _classes_tree.set_model(grtm()->get_shared_structs_tree());
    _modules_tree.set_model(grtm()->get_shared_modules_tree());
  }
  
  refresh_files();

  int idx= 0;
  std::string root = _global_tree_be->get_path_for_node(bec::NodeId(), true);
  std::vector<std::string> l = grtm()->get_shell()->get_grt_tree_bookmarks();
  _global_combo.clear();
  for (std::vector<std::string>::const_iterator i = l.begin(); i != l.end(); ++i, ++idx)
  {
    _global_combo.add_item(*i);
    if (root == *i)
      _global_combo.set_selected(idx);
  }
  
  //refresh values
  _global_tree.refresh(bec::NodeId());
  grtm()->get_shared_structs_tree()->refresh();
  grtm()->get_shared_modules_tree()->refresh();
  
///  _global_tree.expand_row(Gtk::TreePath("0"), false);
  _global_list.refresh(bec::NodeId());
  
  //refresh _struct
  _classes_tree.refresh(bec::NodeId());
  
  // refresh modules
  _modules_tree.refresh(bec::NodeId());

  if (grtm()->get_grt()->shell_type() == "lua")
  {
    _script_extension = ".lua";
    _comment_prefix = "-- ";
  }
  else if (grtm()->get_grt()->shell_type() == "python")
  {
    _script_extension = ".py";
    _comment_prefix = "# ";
  }
  
  refresh_snippets();
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::side_tab_changed()
{
  static std::string side_bar_titles[] = {
    _("File Browser"),
    _("Globals Tree"),
    _("Classes List"),
    _("Modules List")
  };

  _header_panel.set_title(side_bar_titles[_side_tab.get_active_tab()]);
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::handle_output(const std::string &text)
{
  _shell_text.append_text(text, true);
}


void GRTShellWindow::handle_error(const std::string &text, const std::string &detail)
{
  _shell_text.append_text(text);
  
  _shell_text.append_text(detail);
  
}


void GRTShellWindow::handle_prompt(const std::string &text)
{
  _shell_prompt.set_text(text);
}



void GRTShellWindow::set_globals_tree_path(const std::string &path)
{
  try
  {
    if (path.empty() || path[0] != '/')
      _global_tree_be = grtm()->get_shared_value_tree("/");
    else
      _global_tree_be = grtm()->get_shared_value_tree(path);
    _global_tree.set_model(_global_tree_be);
    _global_tree.refresh(bec::NodeId());
    _global_list.refresh(bec::NodeId());
  }
  catch (std::exception &exc)
  {
    g_message("error: %s", exc.what());
  }
}

void GRTShellWindow::change_global_path()
{
  set_globals_tree_path(_global_combo.get_string_value());
}

void GRTShellWindow::change_class_sorting()
{
  grtm()->get_shared_structs_tree()->set_display_mode((bec::StructsTreeBE::DisplayMode)_classes_sorting.get_selected_index());
  _classes_tree.refresh(bec::NodeId());
}


void GRTShellWindow::global_selected()
{
  if (_inspector)
  {
    _global_list.set_model(0);
    delete _inspector;
    _inspector= 0;
  }
  
  bec::NodeId selected;
  
  if (_global_tree.get_selected_node(selected))
  {
    grt::ValueRef value(_global_tree_be->get_node_value(selected));
    
    if (value.is_valid())
    {
      _inspector= grtm()->get_new_value_inspector(value, false);
      _global_list.set_model(_inspector);
    }
    
    _global_entry.set_value(_global_tree_be->get_path_for_node(selected, true));
  }
  else
    _global_entry.set_value("");
}

void GRTShellWindow::class_selected()
{
  bec::NodeId selected;
  
  if (_classes_tree.get_selected_node(selected))
  {
    std::string text;
   
    grtm()->get_shared_structs_tree()->get_field(selected, bec::StructsTreeBE::Caption, text);
    text.append("\n");
    text.append(grtm()->get_shared_structs_tree()->get_field_description(selected, 0));
   
    _classes_text.set_value(text);
  }
  else
    _classes_text.set_value("");
}

void GRTShellWindow::module_selected()
{
  bec::NodeId selected;
  
  if (_modules_tree.get_selected_node(selected))
  {
    std::string text(grtm()->get_shared_modules_tree()->get_field_description(selected, 0));
    
    _modules_text.set_value(text);
  }
  else
    _modules_text.set_value("");
}


void GRTShellWindow::save_snippets()
{
  std::string path = bec::make_path(grtm()->get_user_datadir(), "shell_snippets"+_script_extension);
  FILE *f = base_fopen(path.c_str(), "w+");
  if (!f)
  {
    _shell_text.append_text(base::strfmt("Cannot save snippets to %s: %s", 
                                         path.c_str(), g_strerror(errno)));
    return;
  }
  
  int c = _snippet_list.count();
  for (int i = _global_snippet_count; i < c; i++)
  {
    std::string snippet = _snippet_list.get_row_tag(i);
    std::string::size_type p = 0, l = snippet.size();

    while (p < l)
    {
      std::string::size_type eol = snippet.find('\n', p);
      if (eol == std::string::npos)
        eol = l;
      else
        eol++;
      fwrite(" ", 1, 1, f);
      fwrite(snippet.data()+p, 1, eol-p, f);
      p = eol;
    }
    fwrite("\n", 1, 1, f);
  }
  
  fclose(f);
}

void GRTShellWindow::load_snippets_from(const std::string &path)
{
  FILE *f = base_fopen(path.c_str(), "r");
  if (f)
  {
    char line[4096];
    
    while (fgets(line, sizeof(line), f))
    {
      std::string script = line+1;
      char *ptr = strchr(line, '\n');
      if (ptr)
        *ptr= 0;
      std::string name = line+1;
      
      while (fgets(line, sizeof(line)-1, f) && line[0] == ' ')
      {
        script.append(line+1);
      }
      
      // Remove the last line break, we added that, not the user.
      if (script.size() > 0)
        script.erase(script.size() - 1, 1);
      
      int row = _snippet_list.add_row();
      _snippet_list.set(row, 0, name);
      _snippet_list.set_row_tag(row, script);
    }
    fclose(f);
  }
}


void GRTShellWindow::refresh_snippets()
{
  _snippet_list.clear_rows();
  
  load_snippets_from(grtm()->get_data_file_path("shell_snippets"+_script_extension));
  _global_snippet_count = _snippet_list.count();
  load_snippets_from(bec::make_path(grtm()->get_user_datadir(), "shell_snippets"+_script_extension));
  
  snippet_selected();
}


void GRTShellWindow::open_script_file()
{
  mforms::FileChooser chooser(mforms::OpenFile);
  chooser.set_title(_("Open GRT Script"));
  if (chooser.run_modal())
  {
    open_file_in_editor(chooser.get_path(), true);
  }
}


bool GRTShellWindow::execute_script(const std::string &script, const std::string &language)
{
  bool result = grtm()->get_shell()->run_script(script, language);
  save_state();

  return result;
}


void GRTShellWindow::add_snippet()
{
  std::string snippet = _comment_prefix + " new snippet\n";
  
  int row = _snippet_list.add_row();
  _snippet_list.set_row_tag(row, snippet);
  _snippet_list.set_selected(row);

  snippet_selected(); // force snippet to be displayed
  snippet_changed(); // force setting of the snippet name

  save_state();
}


void GRTShellWindow::del_snippet()
{
  int sel = _snippet_list.get_selected();
  if (sel >= 0)
  {
    _snippet_list.delete_row(sel);
    snippet_selected();
    save_snippets();
  }
}


void GRTShellWindow::copy_snippet()
{
  int sel = _snippet_list.get_selected();
  if (sel >= 0)
    mforms::Utilities::set_clipboard_text(_snippet_list.get_row_tag(sel));
}


void GRTShellWindow::scriptize_snippet()
{
  int sel = _snippet_list.get_selected();
  if (sel >= 0)
  {
    std::string snippet = _snippet_list.get_row_tag(sel);
    std::string language = "python";
    
    // kind of a hack.. pick language depending on comment type used
    if (g_str_has_prefix(snippet.c_str(), "#"))
      language= "python";
    else if (g_str_has_prefix(snippet.c_str(), "--"))
      language= "lua";
    
    GRTCodeEditor *editor= add_editor(true, language);
    editor->set_text(snippet);
  }
}


void GRTShellWindow::run_snippet()
{
  int sel = _snippet_list.get_selected();
  if (sel >= 0)
  {
    std::string script = _snippet_list.get_row_tag(sel);

    // auto-select the tab where output goes
    _main_tab.set_active_tab(0);

    handle_output("Running snippet...\n");
    // redirect snippet output to the shell
    grtm()->push_output_callback(sigc::mem_fun(this, &GRTShellWindow::handle_output));
    
    try
    {
      std::string language = "python";
      
      // kind of a hack.. pick language depending on comment type used
      if (g_str_has_prefix(script.c_str(), "#"))
        language= "python";
      else if (g_str_has_prefix(script.c_str(), "--"))
        language= "lua";

      bool ret = execute_script(script, language);
      grtm()->pop_output_callback();
      if (!ret)
        handle_output("Snippet execution finished with an error\n");
    }
    catch (const std::exception &exc)
    {
      grtm()->pop_output_callback();
      
      handle_output("Exception caught while executing snippet:\n");
      handle_output(std::string(exc.what()).append("\n"));
    }
  }

  save_state();
}


void GRTShellWindow::snippet_selected()
{
  bool read_only = false;
  _snippet_text.set_read_only(false); // Necessary to be able to change the text.
  int sel = _snippet_list.get_selected();
  if (sel < 0)
  {
    _snippet_del.set_enabled(false);
    _snippet_copy.set_enabled(false);
    _snippet_text.set_value("");
    read_only = true;
    
    for (int i= 0; i < 6; i++)
      _snippet_menu.set_item_enabled(i, false);    
  }
  else
  {
    if (sel < _global_snippet_count)
    {
      read_only = true;
      _snippet_del.set_enabled(false);
      
      for (int i= 0; i < 6; i++)
      {
        if (i != 5)
          _snippet_menu.set_item_enabled(i, true);
        else
          _snippet_menu.set_item_enabled(i, false); // 5 is delete
      }
    }
    else
    {
      _snippet_del.set_enabled(true);
      
      for (int i= 0; i < 6; i++)
        _snippet_menu.set_item_enabled(i, true);
    }
    _snippet_text.set_value(_snippet_list.get_row_tag(sel));
    _snippet_copy.set_enabled(true);
  }

  _snippet_text.set_read_only(read_only);
}


void GRTShellWindow::snippet_changed()
{
  std::string snippet = _snippet_text.get_string_value();
  int sel = _snippet_list.get_selected();
  
  if (sel >= 0)
  {
    _snippet_list.set_row_tag(sel, snippet);
    
    std::string::size_type p = snippet.find('\n');
    if (p != std::string::npos)
      snippet = snippet.substr(0, p);
    _snippet_list.set(sel, 0, snippet);
    
    save_snippets();
  }
}


void GRTShellWindow::snippet_menu_activate(const std::string &action)
{
  if (action == "execute")
    run_snippet();
  else if (action == "new_with_snippet")
    scriptize_snippet();
  else if (action == "copy_clipboard")
    copy_snippet();
  else if (action == "delete")
    del_snippet();
}


GRTCodeEditor *GRTShellWindow::add_editor(bool is_script, const std::string &language)
{
  GRTCodeEditor *editor = manage(new GRTCodeEditor(this, !is_script, language));
  
  _editors.push_back(editor);
  
  int page = _main_tab.add_page(editor, editor->get_title());
  _main_tab.set_active_tab(page);
  
  editor->set_position(_main_tab.get_height() - (is_script ? 200 : 0));

  save_state();

  return editor;
}


void GRTShellWindow::close_editor(GRTCodeEditor *editor)
{
  for (std::vector<GRTCodeEditor*>::iterator iter= _editors.begin(); iter != _editors.end(); ++iter)
  {
    if ((*iter) == editor)
    {
      _editors.erase(iter);
      break;
    }
  }

  _main_tab.remove_page(editor); 
  delete editor;

  save_state();
}


void GRTShellWindow::open_file_in_editor(const std::string &path, bool is_script)
{
  if (get_editor_for(path, true) != NULL)
    return;

  char *data= NULL;
  gsize length= 0;
  GError *error= NULL;
  
  if (!g_file_get_contents(path.c_str(), &data, &length, &error))
  {
    Utilities::show_error("Open File",
                          base::strfmt("Could not open %s: %s", path.c_str(),
                                       error->message), 
                          "OK");
    g_error_free(error);
    return;
  }

  std::string language = "python"; // Python module or plugin.
  if (g_str_has_suffix(path.c_str(), ".py"))
    language = "python"; // Python script
  else
    if (g_str_has_suffix(path.c_str(), ".lua"))
      language = "lua"; // Lua script
  GRTCodeEditor *editor= add_editor(is_script, language);
  editor->set_path(path);
  editor->set_text(std::string(data, length));
  g_free(data);
}


void GRTShellWindow::add_new_script()
{
  NewPluginDialog wizard(this, grtm()->get_data_file_path("script_templates"));
  std::string path;
  std::string code;
  bool is_script;
  std::string language;
  
  if (wizard.run(path, code, is_script, language))
  {
    GRTCodeEditor *editor= add_editor(is_script, language);
    if (!path.empty() && g_basename(path.c_str()) == path)
      path= bec::make_path(grtm()->get_user_script_path(), path);
    editor->set_path(path);
    editor->set_text(code);
  }

  save_state();
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::set_editor_title(GRTCodeEditor *editor, const std::string &title)
{
  int index= _main_tab.get_page_index(editor);
  if (index >= 0)
    _main_tab.set_tab_title(index, editor->get_title());
}


//--------------------------------------------------------------------------------------------------

/**
 * Called from the UI context when WB is about to quit. Check if we have pending changes.
 * Return true if we are clear, false otherwise.
 */
bool GRTShellWindow::request_quit()
{
  for (std::vector<GRTCodeEditor*>::iterator editor = _editors.begin(); editor != _editors.end(); editor++)
    if (!(*editor)->can_close())
      return false;

  return true;
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::add_files_from_dir(const std::string &dirname, bool is_script)
{
  GDir *dir= g_dir_open(dirname.c_str(), 0, NULL);
  if (!dir)
    return;
  
  while (const gchar *name= g_dir_read_name(dir))
  {
    if (g_str_has_suffix(name, ".py") || g_str_has_suffix(name, ".lua"))
    {
      int row= _files_tree.add_row();
      _files_tree.set(row, 0, std::string("      ")+name);
      if (is_script)
        _files_tree.set_row_tag(row, std::string("s").append(dirname).append(G_DIR_SEPARATOR_S).append(name));
      else
        _files_tree.set_row_tag(row, std::string("m").append(dirname).append(G_DIR_SEPARATOR_S).append(name));
    }
  }

  g_dir_close(dir);
}


void GRTShellWindow::refresh_files()
{
  int row;
  
  _files_tree.clear_rows();
  
  row= _files_tree.add_row();
  _files_tree.set(row, 0, "User Scripts");
  add_files_from_dir(grtm()->get_user_script_path(), true);

  row= _files_tree.add_row();
  _files_tree.set(row, 0, "User Modules");
  add_files_from_dir(grtm()->get_user_module_path(), false);

  row= _files_tree.add_row();
  _files_tree.set(row, 0, "User Libraries");
  add_files_from_dir(grtm()->get_user_library_path(), true);  
}


void GRTShellWindow::file_list_activated(int row, int column)
{
  std::string path= _files_tree.get_row_tag(row);
  if (!path.empty())
  {
    open_file_in_editor(path.substr(1), path[0] == 's');
  }
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::delete_selected_file()
{
  std::string path= _files_tree.get_row_tag(_files_tree.get_selected());
  if (!path.empty())
  {
    std::string fn = path.substr(1);
    if (mforms::Utilities::show_message(_("Delete File"),
                                        base::strfmt(_("Really delete '%s' from disk? This operation cannot be undone."),
                                                     fn.c_str()),
                                        _("Delete"), _("Cancel")) == mforms::ResultOk)
    {
      g_remove(fn.c_str());
      refresh_files();
    }
  }
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::load_state()
{
  // Restore form's size and position.
  int x = _context->read_state("left", "scripting-shell", 100);
  int y = _context->read_state("top", "scripting-shell", 100);
  int width = _context->read_state("width", "scripting-shell", 800);
  int height = _context->read_state("height", "scripting-shell", 600);

  set_size(width, height);
  // TODO: This crashed WB in linux, cause parent view is not set
  set_position(x, y);

  // Restore divider positions.
  _hsplitter.set_position(_context->read_state("main-splitter", "scripting-shell", 250));
  _global_splitter.set_position(_context->read_state("global-splitter", "scripting-shell", 400));
  _modules_splitter.set_position(_context->read_state("modules-splitter", "scripting-shell", 400));
  _classes_splitter.set_position(_context->read_state("classes-splitter", "scripting-shell", 400));

  _shell_text.set_font(grtm()->get_app_option_string("workbench.scripting.ScriptingShell:Font"));
  _snippet_text.set_font(grtm()->get_app_option_string("workbench.scripting.ScriptingEditor:Font"));
  for (std::vector<GRTCodeEditor*>::iterator editor = _editors.begin(); editor != _editors.end(); editor++)
    (*editor)->set_font(grtm()->get_app_option_string("workbench.scripting.ScriptingEditor:Font"));
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::save_state()
{
  // Store form's size and position.
  _context->save_state("left", "scripting-shell", get_x());
  _context->save_state("top", "scripting-shell", get_y());
  _context->save_state("width", "scripting-shell", get_width());
  _context->save_state("height", "scripting-shell", get_height());

  // Store all divider positions.
  _context->save_state("main-splitter", "scripting-shell", _hsplitter.get_position());
  _context->save_state("global-splitter", "scripting-shell", _global_splitter.get_position());
  _context->save_state("modules-splitter", "scripting-shell", _modules_splitter.get_position());
  _context->save_state("classes-splitter", "scripting-shell", _classes_splitter.get_position());
}

//--------------------------------------------------------------------------------------------------

/**
 *  Triggered when the shell window was closed by the user. We can use this event to store our state.
 */
void GRTShellWindow::shell_closed()
{
  save_state();
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered when a tab is about to close. Don't allow shell and snippets to close and check if
 * editors are dirty.
 */
bool GRTShellWindow::on_tab_closing(int index)
{
  if (index == 0 || index == 1)
    return false;

  GRTCodeEditor* editor = _editors[index - 2];
  return editor->can_close();
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::on_tab_closed(int index)
{
  GRTCodeEditor* editor = _editors[index - 2];
  close_editor(editor);
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns the editor which is currently editing the given file.
 */
GRTCodeEditor* GRTShellWindow::get_editor_for(const std::string& path, bool select_tab)
{
  for (std::vector<GRTCodeEditor*>::iterator editor = _editors.begin(); editor != _editors.end(); editor++)
    if ((*editor)->get_path() == path)
    {
      if (select_tab)
        _main_tab.set_active_tab(editor - _editors.begin() + 2);
      return *editor;
    }

  return NULL;
}

//--------------------------------------------------------------------------------------------------
