#include "query_sidebar.h"
#include "sqlide/utils_fe.h"
#include "linux_utilities/gtk_helpers.h"
#include <gtkmm/combobox.h>
#include <gtkmm/textview.h>


QuerySidebar::QuerySidebar(Db_sql_editor::Ref editor, MGGladeXML *xml)
  : _ui(xml), _editor(editor), _refreshing_active_schema_selector(false)
{
  // active schema selector
  {
    Gtk::ComboBox *combo;
    _ui->get("schema_combo", &combo);
#if GTK_VERSION_GT(2,10)
    combo->set_tooltip_text("Default schema for queries to be executed against");
#endif
    combo->signal_changed().connect(sigc::mem_fun(this, &QuerySidebar::schema_changed));
    setup_combo_for_string_list(combo);
  }

  // schemata tree
  {
    _ui->get("schema_tree", &_schema_tree_view);
  
    _editor->get_schema_tree()->pre_refresh_signal.connect(sigc::mem_fun(this, &QuerySidebar::pre_refresh_live_tree));
    
    _schema_tree_model= TreeModelWrapper::create(_editor->get_schema_tree(), _schema_tree_view, "schematree");
    _schema_tree_model->model().append_string_column(0, "Object", RO, WITH_ICON);
    _schema_tree_model->set_delay_expanding_nodes(true);

    _schema_tree_view->signal_row_activated().connect(sigc::mem_fun(this, &QuerySidebar::activate_row));
    _schema_tree_model->children_count_enabled(false);
    _schema_tree_view->set_model(_schema_tree_model);
    _schema_tree_model->children_count_enabled(true);

    _schema_tree_view->get_selection()->signal_changed().connect(sigc::mem_fun(this, &QuerySidebar::on_schema_tree_view_selection_changed));
  
    _schema_tree_model->set_fe_menu_handler(sigc::mem_fun(this, &QuerySidebar::handle_context_menu_item));

    _editor->get_schema_tree()->refresh_node_ui.connect(sigc::mem_fun(this, &QuerySidebar::refresh_node_ui));

    _ui->get("info_text", &_info_text);
  }

  // sql snippets
  {
    Gtk::Container *placeholder;
    _ui->get("sql_snippets_placeholder", &placeholder);

    wb::WBContextUI *wbui= editor->wbsql()->get_wbui();
    _sql_snippets_view= Gtk::manage(new SqlSnippetsView(wbui, _ui));
    placeholder->add(*_sql_snippets_view);
    _sql_snippets_view->show_all();
  }
}


QuerySidebar::~QuerySidebar()
{
}


void QuerySidebar::handle_context_menu_item(const std::string &name, const std::vector<bec::NodeId> &nodes)
{
  if (name == "refresh")
    refresh(false);
}


void QuerySidebar::schema_changed()
{
  if (_refreshing_active_schema_selector)
    return;

  Gtk::ComboBox *combo;
  _ui->get("schema_combo", &combo);

  std::string schema= get_selected_combo_item(combo);
  _editor->active_schema(schema);
}


void QuerySidebar::pre_refresh_live_tree()
{
  _schema_tree_view->collapse_all();
  _schema_tree_model->set_be_model(0);
  _schema_tree_view->unset_model();
}


void QuerySidebar::refresh(bool is_active_schema_set_needed)
{
  refresh_active_schema_selector(true);

  _schema_tree_model->set_be_model(_editor->get_schema_tree());
  _schema_tree_view->set_model(_schema_tree_model);

  if (is_active_schema_set_needed)
  {
    int active_schema_index = _editor->active_schema_index();
    if (-1 != active_schema_index)
    {
      Gtk::TreeModel::Path path;
      path.append_index(active_schema_index);
      _schema_tree_view->expand_row(path, false);
    }
  }
  
  on_schema_tree_view_selection_changed();
}


void QuerySidebar::refresh_active_schema_selector(bool reload_list)
{
  _refreshing_active_schema_selector= true;

  Gtk::ComboBox *combo;
  _ui->get("schema_combo", &combo);

  if (reload_list)
  {
    std::list<std::string> schemata;
    _editor->get_schemata(schemata);
    combo->set_model(model_from_string_list(schemata));
  }

  combo->set_active(_editor->active_schema_index());

  _refreshing_active_schema_selector= false;
}


void QuerySidebar::refresh_schema_tree()
{
  refresh(false);
}


void QuerySidebar::activate_row(const Gtk::TreeModel::Path &path, Gtk::TreeViewColumn* column)
{
  bec::NodeId node= _schema_tree_model->node_for_iter(_schema_tree_model->get_iter(path));

  _editor->get_schema_tree()->activate_node(node);
}


void QuerySidebar::refresh_node_ui(const bec::NodeId &node)
{
  Gtk::TreeModel::Path node_path= node2path(node);
  bool expanded= _schema_tree_view->row_expanded(node_path);
  if (expanded)
  {
    _schema_tree_view->collapse_row(node_path);
    _schema_tree_view->expand_row(node_path, false);
  }
}

void QuerySidebar::on_schema_tree_view_selection_changed()
{
  std::string obj_info_text= "";
  Glib::ListHandle<Gtk::TreeModel::Path, Gtk::TreePath_Traits> selected_rows= _schema_tree_view->get_selection()->get_selected_rows();
  if (!selected_rows.empty())
  {
    Gtk::TreeModel::Path node_path= *selected_rows.begin();
    bec::NodeId node= bec::NodeId(node_path.to_string());
    obj_info_text= _schema_tree_model->get_be_model()->get_field_description(node, 0);
    // notify treeview that the node has child nodes now
    _schema_tree_model->row_has_child_toggled(node_path, _schema_tree_model->get_iter(node_path));
  }
  else
    obj_info_text= _editor->connection_info();
  _info_text->get_buffer()->set_text(obj_info_text);
}
