using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Reflection;

using MySQL.Grt;
using MySQL.Workbench;
using MySQL.Utilities;
using MySQL.Forms;
using System.Threading;
using MySQL.Utilities.SysUtils;

namespace MySQL.GUI.Workbench
{
	static class Program
  {
    #region Static Variables and Enums

    // The types of application metadata information
    public enum ApplicationMetaInfo { Company, Copyright, Version, Revision, Configuration };

		// The Workbench Context
		private static WbContext wbContext = null;

		// The GRT Manager
		private static GrtManager grtManager = null;

    // Timer to guarantee idle handler is called periodically when nothing happens
    private static System.Windows.Forms.Timer timer = null;

    // Flag to collect garbage
    private static bool gcRequested = false;
    public static void CollectGarbageOnIdle()
    {
      gcRequested = true;
    }

    // Used to synchronize with for messages triggered by background threads.
    private static MainForm mainForm = null;

    #endregion
    
    /// <summary>
		/// The main entry point for the application.
		/// </summary>
    
    [STAThread]
	  static void Main(string[] Args)
    {
      // Give the main thread a proper name, so we can later check for it when needed.
      Thread.CurrentThread.Name = "mainthread";

      // Get the base directory of the executable and change the working dir to it
      // TODO: the working dir should be untouched and the app should work
      // from any basedir
      System.Reflection.Assembly asm = System.Reflection.Assembly.GetEntryAssembly();
      string baseDir = System.IO.Path.GetDirectoryName(asm.Location);
      string workdir = System.IO.Directory.GetCurrentDirectory();
      System.IO.Directory.SetCurrentDirectory(baseDir);

      // Some people don't have c:\windows\system32 in PATH, so we need to set it here
      // for WBA to find the needed commands
      String systemdir = Environment.GetFolderPath(Environment.SpecialFolder.System);
      String syspath = Environment.GetEnvironmentVariable("PATH");
      String []paths= syspath.Split(new char[]{';'});
      syspath = "";
      // strip all python related dirs from PATH to avoid conflicts with other Python installations
      foreach (String path in paths)
      {
        if (!path.Contains("Python") && !path.Contains("python"))
          syspath = syspath+";"+path;
      }
      Environment.SetEnvironmentVariable("PATH", systemdir + syspath);

      // Initialize forms stuff.
      MySQL.Forms.Manager formsManager = new MySQL.Forms.Manager();

      #region Command line argument parsing and handling

      // Connect application to console to have proper output there if requested.
      Win32.RedirectConsole();

      // Command line parsing
      Arguments CommandLine = new Arguments();

      CommandLine.AddOptionName("-verbose", true);
      CommandLine.AddOptionName("-v", true);
      CommandLine.AddOptionName("-help", true);
      CommandLine.AddOptionName("-h", true);
      CommandLine.AddOptionName("-open", false);
      CommandLine.AddOptionName("-nologo", true);
      CommandLine.AddOptionName("-version", true);
      CommandLine.AddOptionName("-grtversion", true);
      CommandLine.AddOptionName("-swrendering", false);
      CommandLine.AddOptionName("-log", true);
      CommandLine.AddOptionName("-query", false);
      CommandLine.AddOptionName("-model", false);
      CommandLine.AddOptionName("-admin", false);
      CommandLine.AddOptionName("-script", false);
      CommandLine.AddOptionName("-run", false);
      CommandLine.AddOptionName("-run-python", false);
      CommandLine.AddOptionName("-run-script", false);
      CommandLine.AddOptionName("-quit-when-done", true);

      CommandLine.Parse(Args);

      // Command line handling
      if (ProcessCommandLineArguments(CommandLine))
      {
        Win32.ReleaseConsole();
        return;
      }

      #endregion

      Application.EnableVisualStyles();
      Application.SetCompatibleTextRenderingDefault(false);

      #region Runtime path check

      // Currently WB has trouble running from a path containing non-ASCII characters.
      // Actually, our third party libraries have (namely lua, python, ctemplate, yassl etc.), 
      // as they don't consider Unicode file names (encoded as UTF-8) which leads to file-not-found
      // errors. Refuse to work in such a path for now.
      foreach (Char c in baseDir)
        if (c > 0x7f)
        {
          MessageBox.Show("MySQL Workbench cannot be executed from a path that contains non-ASCII characters.\n"+
            "This problem is imposed by used third-party libraries.\n" +
            "Please run this application from the default installation path or at least a path which is all ASCII characters.",
            "MySQL Workbench Execution Problem", MessageBoxButtons.OK, MessageBoxIcon.Error);
          return;
        }

      #endregion

      #region Release check (outdated beta or rc version)

      // check the date of the executable and suggest to install a new version if this is a beta or rc
      if (GetApplicationMetaInfo(ApplicationMetaInfo.Configuration).ToUpper().IndexOf("BETA") >= 0 ||
        GetApplicationMetaInfo(ApplicationMetaInfo.Configuration).ToUpper().IndexOf("RC") >= 0)
      {
        DateTime fileDate = System.IO.File.GetCreationTime(Application.ExecutablePath);

        if (DateTime.Now.Subtract(fileDate).TotalDays > 45)
        {
          if (MessageBox.Show("This version of MySQL Workbench is older than 45 days and most probably outdated. "
            + Environment.NewLine
            + "It is recommended to upgrade to a newer version if available. "
            + Environment.NewLine
            + "Press [OK] to check for a new version and exit the application. "
            + "Press [Cancel] to continue using this version.",
            "MySQL Worbench Version Outdated", MessageBoxButtons.OKCancel,
            MessageBoxIcon.Warning, MessageBoxDefaultButton.Button1) == DialogResult.OK)
          {
            CheckForNewVersion();
            return;
          }
        }
      }

      #endregion

      #region Variables and Splashscreen

      // Show splash screen if not turned off
      if (CommandLine["nologo"] == null)
        SplashScreen.Show(typeof(WorkbenchSplashScreen));

      #endregion

      #region Initialize GRT

      // Try to instantiate the Workbench context and the GRT Manager and catch exceptions
		  try
		  {
			  // Create Workbench Context
			  wbContext = new WbContext((CommandLine["verbose"] != null || CommandLine["v"] != null));

			  if (wbContext != null)
			  {
				  // Create the GRT Manager instance
				  grtManager = wbContext.get_grt_manager();
			  }
		  }
		  catch (Exception ex)
		  {
        MainForm.HandleException(ex);
      }

      #endregion

      #region Initialize Callbacks and Mainform

      // If the Workbench Context and GRT Manager were successfully created, 
		  // initialize the application
		  if (wbContext != null && grtManager != null)
		  {
			  mainForm = new MainForm(wbContext);

			  // Initialize the Workbench context
			  WbFrontendCallbacks wbFrontendCallback = new WbFrontendCallbacks(
          formsManager,
          //new WbFrontendCallbacks.IntStrStrStrStrStrDelegate(ShowConfirmDialog),
          new WbFrontendCallbacks.StrStrStrStrDelegate(mainForm.ShowFileDialog),
				  new WbFrontendCallbacks.VoidStrDelegate(mainForm.ShowStatusText),
				  new WbFrontendCallbacks.BoolStrStrFloatDelegate(mainForm.ShowProgress),
          new WbFrontendCallbacks.BoolStrIntStrPtrDelegate(mainForm.RequestInput),
				  new WbFrontendCallbacks.CanvasViewStringStringDelegate(mainForm.CreateNewDiagram),
				  new WbFrontendCallbacks.VoidCanvasViewDelegate(mainForm.DestroyView),
				  new WbFrontendCallbacks.VoidCanvasViewDelegate(mainForm.SwitchedView),
				  new WbFrontendCallbacks.VoidCanvasViewDelegate(mainForm.ToolChanged),
          new WbFrontendCallbacks.VoidStrUIFormDelegate(mainForm.CreateMainFormView),
          new WbFrontendCallbacks.IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate(mainForm.OpenPlugin),
          new WbFrontendCallbacks.VoidIntPtrDelegate(mainForm.ShowPlugin),
          new WbFrontendCallbacks.VoidIntPtrDelegate(mainForm.HidePlugin),
				  new WbFrontendCallbacks.VoidRefreshTypeStringIntPtrDelegate(mainForm.RefreshGUI),
          new WbFrontendCallbacks.VoidBoolDelegate(mainForm.LockGUI),
          new WbFrontendCallbacks.VoidMenuItemListIntIntDelegate(mainForm.PopupCanvasMenu),
          new WbFrontendCallbacks.VoidStrDelegate(mainForm.PerformCommand),
          new WbFrontendCallbacks.VoidDelegate(mainForm.QuitApplication),
          new WbFrontendCallbacks.StringAppCommandAppViewStringDelegate(mainForm.ApplicationCommand));


        /// mainForm.PrintOutputText
        /// mainForm.ShowOutputForm

        string openAtStartupType = "";
        string openAtStartup = null;

        if (CommandLine["open"] != null)
        {
          openAtStartup = CommandLine["open"];
          openAtStartupType = "";
        }
        else if (CommandLine["model"] != null)
        {
          openAtStartup = CommandLine["model"];
          openAtStartupType = "model";
        }
        else if (CommandLine["query"] != null)
        {
          openAtStartup = CommandLine["query"];
          openAtStartupType = "query";
        }
        else if (CommandLine["admin"] != null)
        {
          openAtStartup = CommandLine["admin"];
          openAtStartupType = "admin";
        }
        else if (CommandLine["script"] != null)
        {
          openAtStartup = CommandLine["script"];
          openAtStartupType = "script";
        }

        if (openAtStartup == null)
        {
          openAtStartup = "";
          if (CommandLine.Leftovers.Count > 0)
            openAtStartup = CommandLine.Leftovers[CommandLine.Leftovers.Count-1];
        }
        if (openAtStartup != null && openAtStartup != "" && (openAtStartupType == "" || openAtStartupType == "model" || openAtStartupType == "script"))
        {
          if (!System.IO.Path.IsPathRooted(openAtStartup))
            openAtStartup = System.IO.Path.Combine(workdir, openAtStartup);
        }

        String runScript = "";
        String runScriptLanguage = "";
        if (CommandLine["run"] != null)
          runScript = CommandLine["run"];
        else if (CommandLine["run-python"] != null)
        {
          runScript = CommandLine["run-python"];
          runScriptLanguage = "python";
        }
        else if (CommandLine["run-lua"] != null)
        {
          runScript = CommandLine["run-lua"];
          runScriptLanguage = "lua";
        }

        WbOptions wbOptions = new WbOptions(baseDir, // basedir
          baseDir, // plugin
          "",       // struct
				  baseDir + "/modules", // modules
				  baseDir, // library
				  System.IO.Path.Combine(System.IO.Path.Combine(
						  Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
						  "MySQL"), "Workbench"),
          openAtStartup,
          openAtStartupType,
          runScript, runScriptLanguage,
          (CommandLine["swrendering"] != null),
          (CommandLine["openglrendering"] != null),
          (CommandLine["log"] != null),
          (CommandLine["quit-when-done"] != null || CommandLine["quitwhendone"] != null));

			  wbContext.init(wbFrontendCallback, wbOptions);

			  // Set the Application.Idle event handler
			  Application.Idle += new EventHandler(OnApplicationIdle);

        // Don't call the idle handler too often.
        timer = new System.Windows.Forms.Timer();
        timer.Interval = 100;
        timer.Tick += new EventHandler(timer_Tick);
        timer.Start();

			  // Trigger GRT idle tasks
			  grtManager.perform_idle_tasks();
        
        // Setup Menus
        mainForm.SetupUI();
        mainForm.Show();

			  // Tell the backend our main UI is ready. This will also load a model if it was given via command line
        // and opens the overview form for it.
        wbContext.finished_loading(wbOptions);

        // Start the Application
        try
        {
          Application.Run(new MySQLApplicationContext(mainForm));
        }
        catch (Exception e)
        {
          MainForm.HandleException(e);
        }
  			
        // shutdown wb context
        if (wbContext != null)
        {
          while (wbContext.is_busy())
            wbContext.flush_idle_tasks();

          wbContext.finalize();
        }
      }
      
      #endregion

      Win32.ReleaseConsole();
    }
     
    //----------------------------------------------------------------------------------------------

    #region Application Idle Handler

    static void timer_Tick(object sender, EventArgs e)
    {
      System.Windows.Forms.Timer timer = sender as System.Windows.Forms.Timer;

      // flush_idle_tasks needs to be reentrant so that functions
      // that crosses threads have their own idle tasks executed
      if (wbContext != null)
      {
        try
        {
          wbContext.flush_idle_tasks();
        }
        catch (Exception exception)
        {
          // temporarily stop the timer so that we don't get into a loop from showing the msgbox
          timer.Stop();
          MainForm.HandleException(exception);
          timer.Start();
        }
      }
    }

    private static void OnApplicationIdle(object sender, EventArgs e)
		{
      if (gcRequested)
      {
        gcRequested = false;
        GC.Collect();
      }
    }

    #endregion

    #region Message Callbacks

    private static int ShowStringInputDialog(String Title, String Prompt,
      ref String InputString)
    {
      DialogResult result;
      result = StringInputForm.ShowModal(Title, "", Prompt, ref InputString);
      if (result == DialogResult.OK)
        return 1;
      else
        return 0;
    }

    #endregion
    
    #region Various Static Functions

    /// <summary>
    /// Retrieves application metadata and returns it as a string
    /// </summary>
    /// <param name="kind">The type of metadata information to return</param>
    /// <returns></returns>
    static public string GetApplicationMetaInfo(ApplicationMetaInfo kind)
    {
      object[] attributes;
      Version version;

      Assembly assembly = Assembly.GetExecutingAssembly();

      string value = "";

      switch (kind)
      {
        case ApplicationMetaInfo.Company:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyCompanyAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyCompanyAttribute).Company;
          break;

        case ApplicationMetaInfo.Copyright:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyCopyrightAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyCopyrightAttribute).Copyright;
          break;

        case ApplicationMetaInfo.Configuration:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyConfigurationAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyConfigurationAttribute).Configuration;
          break;

        case ApplicationMetaInfo.Version:
          version = new Version(Application.ProductVersion);
          value = string.Format("{0}.{1}.{2}",
            version.Major, version.Minor, version.Build);
          break;

        case ApplicationMetaInfo.Revision:
          version = new Version(Application.ProductVersion);
          value = string.Format("{0}", version.MinorRevision);
          break;
      }

      return value;
    }

    /// <summary>
    /// Deals with the given command line arguments
    /// </summary>
    /// <param name="CommandLine">The command line arguments passed to the application</param>
    /// <returns></returns>
    static bool ProcessCommandLineArguments(Arguments CommandLine)
    {
      if (CommandLine["help"] != null || CommandLine["h"] != null)
      {
        Console.WriteLine("MySQL Workbench {0} {1}. " 
          + "(C) {2} by {3}. All rights reserved." + Environment.NewLine,
          GetApplicationMetaInfo(ApplicationMetaInfo.Version), 
          GetApplicationMetaInfo(ApplicationMetaInfo.Configuration),
          GetApplicationMetaInfo(ApplicationMetaInfo.Copyright),
          GetApplicationMetaInfo(ApplicationMetaInfo.Company));
        Console.WriteLine("Usage: MySQLWorkbench [options] [model file]" + Environment.NewLine);

        Console.WriteLine("Options" + Environment.NewLine
          + "  -help (-h) ...... Print this output" + Environment.NewLine
          + "  -open filename .. Open the given filename at startup" + Environment.NewLine
          + "  -nologo ......... Do not display the splash screen" + Environment.NewLine
          + "  -verbose (-v) ... Print verbose output in the GRT Shell" + Environment.NewLine
          + "  -version ........ Print the version information" + Environment.NewLine
          + "  -grtversion ..... Print the GRT version information" + Environment.NewLine
          + "  -swrendering .... Force the canvas to use software rendering instead of OpenGL" + Environment.NewLine
          + "  -log ............ Instruction to save messages (other debug info) to file" + Environment.NewLine);

        /*Console.WriteLine("GRT Options" + Environment.NewLine
          + "{0}",
          Grt.option_description);*/

        return true;
      }
      else if (CommandLine["version"] != null)
      {
        Console.WriteLine("MySQL Workbench {0}.{1} {2}",
          GetApplicationMetaInfo(ApplicationMetaInfo.Version), 
          GetApplicationMetaInfo(ApplicationMetaInfo.Revision),
          GetApplicationMetaInfo(ApplicationMetaInfo.Configuration));
        return true;
      }
      else if (CommandLine["grtversion"] != null)
      {
        Console.WriteLine("GRT Environment {0}", GRT.version());
        return true;
      }

      return false;
    }

    static public bool CheckForNewVersion()
    {
      try
      {
				System.Diagnostics.Process.Start("http://wb.mysql.com/workbench/version-check.php?config="
          + GetApplicationMetaInfo(ApplicationMetaInfo.Configuration).Replace(' ', '_')
          + "&version="
          + GetApplicationMetaInfo(ApplicationMetaInfo.Version)
          + "&revision="
          + GetApplicationMetaInfo(ApplicationMetaInfo.Revision));
      }
      catch (Exception e)
      {
        MessageBox.Show(e.Message.ToString(), "Error Opening Browser", 
          MessageBoxButtons.OK, MessageBoxIcon.Error, MessageBoxDefaultButton.Button1);
      }

      return true;
    }

    #endregion

  }
}