/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/**
 * Implementation of the textbox (multi line edit) wrapper for the mforms library.
 */

#include "stdafx.h"
#include "wf_textbox.h"

using namespace MySQL::Forms;

using namespace System::Windows::Forms;
using namespace System::Text;

//--------------------------------------------------------------------------------------------------

bool TextBoxImpl::create(::mforms::TextBox *self, ::mforms::ScrollBars scroll_bars)
{
  const ScrollBars mapper[]= 
    {ScrollBars::None, ScrollBars::Horizontal, ScrollBars::Vertical, ScrollBars::Both};

  TextBoxImpl ^text= gcnew TextBoxImpl(self);

  if (text != nullptr)
  {
    TextBox ^textbox= ViewImpl::create<TextBox>(self, text);
    textbox->Multiline= true;
    textbox->AcceptsReturn= true;
    textbox->ScrollBars= mapper[scroll_bars];
    textbox->TextChanged += gcnew EventHandler(change);
    textbox->Size= Size(100, textbox->PreferredSize.Height); // DefaultSize is not accessible here. 
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::change(Object^ sender, EventArgs^ args)
{
  TextBox^ textbox= (TextBox^) sender;

  if (textbox->Tag != nullptr)
  {
    ::mforms::TextBox* box= ViewImpl::get_backend_control<::mforms::TextBox>(textbox);
    if (box != NULL)
      box->callback();
  }
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::set_bordered(::mforms::TextBox *self, bool bordered)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
    textbox->get_control<TextBox>()->BorderStyle= bordered ? BorderStyle::FixedSingle : BorderStyle::None;
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::set_text(::mforms::TextBox *self, const std::string &text)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
  {
    // Convert LF only line breaks into Windows line breaks.
    StringBuilder^ builder= gcnew StringBuilder(CppStringToNative(text));
    builder->Replace("\n", Environment::NewLine);

    textbox->get_control<Control>()->Text= builder->ToString();
  }
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::append_text(::mforms::TextBox *self, const std::string &text, bool scroll_to_end)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
  {
    // Convert LF only line breaks into Windows line breaks.
    StringBuilder^ builder= gcnew StringBuilder(CppStringToNative(text));
    builder->Replace("\n", Environment::NewLine);

    TextBox^ native_box= textbox->get_control<TextBox>();
    native_box->Text += builder->ToString();
    if (scroll_to_end && native_box->Text->Length > 0)
    {
      native_box->Select(native_box->Text->Length - 1, 0);
      native_box->ScrollToCaret();
    }
  }
}

//--------------------------------------------------------------------------------------------------

std::string TextBoxImpl::get_text(::mforms::TextBox *self)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
  {
    // Convert Windows line breaks to LF.
    StringBuilder^ builder= gcnew StringBuilder(textbox->get_control<Control>()->Text);
    builder->Replace(Environment::NewLine, "\n");
    return NativeToCppString(builder->ToString());
  }
  return "";
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::set_read_only(::mforms::TextBox *self, bool flag)
{
   TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
    textbox->get_control<TextBox>()->ReadOnly= flag;
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::set_padding(::mforms::TextBox *self, int pad)
{
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::clear(::mforms::TextBox *self)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
    textbox->get_control<TextBox>()->Clear();
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::set_monospaced(::mforms::TextBox *self, bool flag)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
  {
    if (flag)
      textbox->get_control<TextBox>()->Font = gcnew System::Drawing::Font("Bitstream Vera Sans Mono", textbox->get_control<TextBox>()->Font->Size-2);
    else
      textbox->get_control<TextBox>()->ResetFont();
  }
}

//--------------------------------------------------------------------------------------------------

void TextBoxImpl::get_selected_range(::mforms::TextBox *self, int &start, int &end)
{
  TextBoxImpl^ textbox= (TextBoxImpl^)ObjectImpl::FromUnmanaged(self);

  if (textbox != nullptr)
  {
    start = textbox->get_control<TextBox>()->SelectionStart;
    end = start + textbox->get_control<TextBox>()->SelectionLength;
  }
}

//--------------------------------------------------------------------------------------------------

TextBoxImpl::TextBoxImpl(::mforms::TextBox *text)
  : ViewImpl(text)
{
}

//--------------------------------------------------------------------------------------------------

