/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"
#include "wf_webbrowser.h"

using namespace MySQL::Forms;

//----------------- WebBrowserImpl -----------------------------------------------------------------

bool WebBrowserImpl::create(::mforms::WebBrowser *self)
{
  WebBrowserImpl^ browser= gcnew WebBrowserImpl(self);
  if (browser != nullptr)
  {
    WebBrowser^ control= ViewImpl::create<WebBrowser>(self, browser);
    control->ScriptErrorsSuppressed = true; // No script error popup please.

    control->DocumentCompleted += gcnew WebBrowserDocumentCompletedEventHandler(browser, &DocumentLoaded);
    control->NewWindow += gcnew CancelEventHandler(browser, &OnNewWindow);
    control->StatusTextChanged += gcnew EventHandler(browser, &StatusTextChanged);

    return true;
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

void WebBrowserImpl::set_html(::mforms::WebBrowser *self, const std::string& code)
{
  WebBrowserImpl^ browser= (WebBrowserImpl^)ObjectImpl::FromUnmanaged(self);

  if (browser != nullptr)
    browser->set_html(code);
}

//-------------------------------------------------------------------------------------------------

void WebBrowserImpl::navigate(::mforms::WebBrowser *self, const std::string& url)
{
  WebBrowserImpl^ browser= (WebBrowserImpl^)ObjectImpl::FromUnmanaged(self);

  if (browser != nullptr)
    browser->navigate(url);
}

//-------------------------------------------------------------------------------------------------

std::string WebBrowserImpl::get_document_title(::mforms::WebBrowser *self)
{
  WebBrowserImpl^ browser= (WebBrowserImpl^)ObjectImpl::FromUnmanaged(self);

  if (browser != nullptr)
    return browser->get_document_title();

  return "";
}

//-------------------------------------------------------------------------------------------------

void WebBrowserImpl::DocumentLoaded(Object^ sender, WebBrowserDocumentCompletedEventArgs^ e)
{
  WebBrowser^ control= (WebBrowser^) sender;

  if (control->Tag != nullptr)
  {
    mforms::WebBrowser* browser= get_backend_control<mforms::WebBrowser>(control);
    if (browser != 0)
      browser->document_loaded(NativeToCppString(e->Url->AbsoluteUri));
  }
}

//-------------------------------------------------------------------------------------------------

void WebBrowserImpl::OnNewWindow(Object^ sender, CancelEventArgs^ e)
{
  WebBrowser^ control= dynamic_cast<WebBrowser^>(sender);

  HtmlElement^ element= control->Document->ActiveElement;
  String^ url= element->GetAttribute("href");

  // We don't want to let IE open an external IE window if a new window is requested.
  // Instead we cancel the action here and open the default browser.
  mforms::Utilities::open_url(NativeToCppString(url));
  e->Cancel= true;
}

//-------------------------------------------------------------------------------------------------

void WebBrowserImpl::StatusTextChanged(Object^ sender, EventArgs^ e)
{
  WebBrowser^ control= dynamic_cast<WebBrowser^>(sender);

  mforms::App::get()->set_status_text(NativeToCppString(control->StatusText));
}

//----------------- WebBrowserImpl actual implementation -------------------------------------------

void WebBrowserImpl::set_html(const std::string& code)
{
  WebBrowser^ browser= get_control<WebBrowser>();
  browser->DocumentText = CppStringToNative(code);
}

//-------------------------------------------------------------------------------------------------

void WebBrowserImpl::navigate(const std::string& url)
{
  WebBrowser^ browser= get_control<WebBrowser>();
  browser->Navigate(CppStringToNative(url));
}

//-------------------------------------------------------------------------------------------------

std::string WebBrowserImpl::get_document_title()
{
  WebBrowser^ browser= get_control<WebBrowser>();
  return NativeToCppString(browser->DocumentTitle);
}

//-------------------------------------------------------------------------------------------------
