/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "mysql_invalid_sql_parser.h"
#include "mysql_sql_parser_fe.h"
#include "mysql_sql_parser_utils.h"
#include "grtsqlparser/module_utils.h"
#include "string_utilities.h"
#include <sstream>


using namespace grt;
using namespace base;


Mysql_invalid_sql_parser::Null_state_keeper::~Null_state_keeper()
{
  _sql_parser->_next_group_routine_seqno= 0;
  _sql_parser->_next_trigger_seqno= 0;
  _sql_parser->_stub_num= 0;
  _sql_parser->_stub_name= std::string();
  _sql_parser->_active_obj_list= grt::ListRef<db_DatabaseDdlObject>();
  _sql_parser->_active_obj_list2= grt::ListRef<db_DatabaseDdlObject>();
  _sql_parser->_active_obj= db_DatabaseDdlObjectRef();
  _sql_parser->_active_grand_obj= db_DatabaseObjectRef();
  _sql_parser->_create_stub_object.disconnect();
  _sql_parser->_remove_stub_object.disconnect();
}
#define NULL_STATE_KEEPER Null_state_keeper _nsk(this);


Mysql_invalid_sql_parser::Mysql_invalid_sql_parser(grt::GRT *grt)
:
Sql_parser_base(grt),
Mysql_sql_parser(grt),
Invalid_sql_parser(grt)
{
  NULL_STATE_KEEPER
}


int Mysql_invalid_sql_parser::parse_inserts(db_TableRef table, const std::string &sql)
{
  NULL_STATE_KEEPER

  return pr_processed; //!
}


int Mysql_invalid_sql_parser::parse_triggers(db_TableRef table, const std::string &sql)
{
  NULL_STATE_KEEPER

  _active_grand_obj= table;
  _active_obj_list= ListRef<db_DatabaseDdlObject>::cast_from(table->triggers());
  _stub_name= "SYNTAX_ERROR_";
  _process_specific_create_statement= sigc::mem_fun(this, &Mysql_invalid_sql_parser::process_create_trigger_statement);
  _create_stub_object= sigc::mem_fun(this, &Mysql_invalid_sql_parser::create_stub_trigger);
  _shape_trigger= sigc::mem_fun(this, &Mysql_invalid_sql_parser::shape_trigger);

  _triggers_owner_table= db_mysql_TableRef::cast_from(table);

  return parse_invalid_sql_script(sql);
}


int Mysql_invalid_sql_parser::parse_routines(db_RoutineGroupRef routine_group, const std::string &sql)
{
  NULL_STATE_KEEPER

  _active_grand_obj= routine_group;
  _active_obj_list= ListRef<db_DatabaseDdlObject>::cast_from(
    db_mysql_SchemaRef::cast_from(_active_grand_obj->owner())->routines());
  _active_obj_list2= ListRef<db_DatabaseDdlObject>::cast_from(routine_group->routines());
  _stub_name= (*routine_group->name()).append("_SYNTAX_ERROR_");
  _process_specific_create_statement= sigc::mem_fun(this, &Mysql_invalid_sql_parser::process_create_routine_statement);
  _create_stub_object= sigc::mem_fun(this, &Mysql_invalid_sql_parser::create_stub_group_routine);
  _remove_stub_object= sigc::mem_fun(this, &Mysql_invalid_sql_parser::remove_stub_group_routine);
  _shape_routine= sigc::mem_fun(this, &Mysql_invalid_sql_parser::shape_group_routine);
  _case_sensitive_identifiers= false; //! identifiers within a mysql routine body are always case-insensitive?

  return parse_invalid_sql_script(sql);
}


int Mysql_invalid_sql_parser::parse_routine(db_RoutineRef routine, const std::string &sql)
{
  NULL_STATE_KEEPER

  _active_obj= routine;
  _active_grand_obj= _active_obj;
  _active_obj_list= ListRef<db_DatabaseDdlObject>::cast_from(
    db_mysql_SchemaRef::cast_from(_active_obj->owner())->routines());
  _stub_name= "SYNTAX_ERROR_";
  _process_specific_create_statement= sigc::mem_fun(this, &Mysql_invalid_sql_parser::process_create_routine_statement);
  _create_stub_object= sigc::mem_fun(this, &Mysql_invalid_sql_parser::create_stub_routine);
  _case_sensitive_identifiers= false; //! identifiers within a mysql routine body are always case-insensitive?

  return parse_invalid_sql_script(sql);
}


int Mysql_invalid_sql_parser::parse_view(db_ViewRef view, const std::string &sql)
{
  NULL_STATE_KEEPER

  _active_obj= view;
  _active_grand_obj= _active_obj;
  _active_obj_list= ListRef<db_DatabaseDdlObject>::cast_from(
    db_mysql_SchemaRef::cast_from(_active_obj->owner())->views());
  _stub_name= "SYNTAX_ERROR_";
  _process_specific_create_statement= sigc::mem_fun(this, &Mysql_invalid_sql_parser::process_create_view_statement);
  _create_stub_object= sigc::mem_fun(this, &Mysql_invalid_sql_parser::create_stub_view);

  _sql_script_preamble= "DELIMITER " + _non_std_sql_delimiter + EOL;
  std::string sql_= _sql_script_preamble + sql;

  return parse_invalid_sql_script(sql_);
}


int Mysql_invalid_sql_parser::parse_invalid_sql_script(const std::string &sql)
{
  set_options(DictRef());

  if (!_active_obj_list2.is_valid())
    _active_obj_list2= _active_obj_list;

  _active_schema= db_mysql_SchemaRef::cast_from(_active_grand_obj->owner());
  _catalog= db_mysql_CatalogRef(_grt);
  _catalog->schemata().insert(_active_schema);

  // take simple datatypes & other major attributes from given catalog
  // simple datatypes may be needed for parsing routine parameters & return value types
  {
    db_mysql_CatalogRef given_catalog= db_mysql_CatalogRef::cast_from(_active_schema->owner());
    _catalog->version(given_catalog->version());
    _catalog->defaultCharacterSetName(given_catalog->defaultCharacterSetName());
    _catalog->defaultCollationName(given_catalog->defaultCollationName());
    replace_contents(_catalog->simpleDatatypes(), given_catalog->simpleDatatypes());
  }

  _created_objects= grt::ListRef<GrtObject>(_grt);
  _reuse_existing_objects= true;
  _stick_to_active_schema= true;
  _set_old_names= false;
  _messages_enabled= false;
  _strip_sql= false;

  build_datatype_cache();

  _process_sql_statement= sigc::mem_fun(this, &Mysql_invalid_sql_parser::process_sql_statement);

  Mysql_sql_parser_fe sql_parser_fe(_grtm->get_grt());
  sql_parser_fe.ignore_dml= false;

  int res= Mysql_sql_parser_base::parse_sql_script(sql_parser_fe, sql);

  // remove from initial list objects that were not created during parsing
  // for parsing single objects (_active_obj.is_valid()) this is not appropriate
  if (_active_obj_list2.is_valid() && !_active_obj.is_valid())
  {
    for (size_t n= _active_obj_list2.count(); n > 0; --n)
    {
      db_DatabaseDdlObjectRef obj= _active_obj_list2.get(n-1);
      if (!find_named_object_in_list(_created_objects, obj->name(), _case_sensitive_identifiers).is_valid())
      {
        _active_obj_list.remove_value(obj);
        
        /*
          objects of some types are registered in several lists
          (e.g. routine in db_Schema::routines & db_RoutineGroup::routines)
          this call allows to make additional cleanup for such objects
        */
        _remove_stub_object(obj);
      }
    }
  }

  return res;
}


int Mysql_invalid_sql_parser::process_sql_statement(const SqlAstNode *tree)
{
  int err= Mysql_sql_parser::process_sql_statement(tree);
  if (0 != err)
  {
    ++_stub_num;

    // try to reuse existing stub
    db_DatabaseDdlObjectRef obj= find_named_object_in_list(_active_obj_list, stub_obj_name(), _case_sensitive_identifiers);

    if (obj.is_valid())
      setup_stub_obj(obj, false);
    else
    {
      _create_stub_object(obj);
      if (!_active_obj.is_valid())
        _active_obj_list.insert(obj);
    }

    _created_objects.insert(obj);
  }
  return err;
}


void Mysql_invalid_sql_parser::create_stub_view(db_DatabaseDdlObjectRef &obj)
{
  obj= db_mysql_ViewRef::cast_from(_active_obj);
  obj->sqlDefinition(strip_sql_statement(sql_statement(), _strip_sql));
}


void Mysql_invalid_sql_parser::create_stub_routine(db_DatabaseDdlObjectRef &obj)
{
  obj= db_mysql_RoutineRef::cast_from(_active_obj);
  obj->sqlDefinition(strip_sql_statement(sql_statement(), _strip_sql));
}


void Mysql_invalid_sql_parser::create_stub_group_routine(db_DatabaseDdlObjectRef &obj)
{
  db_mysql_RoutineRef routine(_grt);
  routine->owner(_active_schema);
  setup_stub_obj(routine, true);
  routine->routineType("<stub>");

  _active_obj_list2.insert(routine);

  obj= routine;
}


void Mysql_invalid_sql_parser::remove_stub_group_routine(db_DatabaseDdlObjectRef &obj)
{
  _active_obj_list2.remove_value(obj);
}


void Mysql_invalid_sql_parser::shape_group_routine(db_mysql_RoutineRef &obj)
{
  if (!find_named_object_in_list(_active_obj_list2, obj->name(), _case_sensitive_identifiers).is_valid())
    _active_obj_list2.insert(obj);
  obj->sequenceNumber(_next_group_routine_seqno++);
}


void Mysql_invalid_sql_parser::create_stub_trigger(db_DatabaseDdlObjectRef &obj)
{
  db_mysql_TriggerRef trigger(_grt);
  trigger->owner(_active_grand_obj);
  setup_stub_obj(trigger, true);

  obj= trigger;
}


void Mysql_invalid_sql_parser::shape_trigger(db_mysql_TriggerRef &obj)
{
  obj->sequenceNumber(_next_trigger_seqno++);
}


void Mysql_invalid_sql_parser::setup_stub_obj(db_DatabaseDdlObjectRef obj, bool set_name)
{
  if (set_name)
    obj->name(stub_obj_name());
  obj->sqlDefinition(strip_sql_statement(sql_statement(), _strip_sql));

  if (db_mysql_TriggerRef::can_wrap(obj))
    db_mysql_TriggerRef::cast_from(obj)->sequenceNumber(_next_trigger_seqno++);
  else if (db_mysql_RoutineRef::can_wrap(obj) && db_RoutineGroupRef::can_wrap(_active_grand_obj))
    db_mysql_RoutineRef::cast_from(obj)->sequenceNumber(_next_group_routine_seqno++);
}


std::string Mysql_invalid_sql_parser::stub_obj_name()
{
  std::ostringstream oss;
  oss << _stub_name << _stub_num;
  return oss.str();
}
