/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include "workbench/wb_context_ui.h"
#include "sqlide/wb_context_sqlide.h"
#include "sqlide/wb_sql_editor_form.h"
#include "sqlide/wb_sqlide_module.h"
#include "sqlide/wb_sql_editor_snippets.h"
#include "recordset_export_form.h"
#include "grt/common.h"
#include "string_utilities.h"

#include "objimpl/db.query/db_query_Editor.h"
#include "objimpl/db.query/db_query_Resultset.h"
#include "objimpl/db.query/db_query_EditableResultset.h"
#include "objimpl/db.query/db_query_QueryBuffer.h"
#include "grts/structs.db.query.h"

using namespace wb;
using namespace bec;
using namespace base;


//------------------------------------------------------------------------------------------------


class MYSQLWBBACKEND_PUBLIC_FUNC db_query_EditorConcreteImplData : public db_query_Editor::ImplData, public sigc::trackable
{
  void sql_editor_list_changed(Sql_editor::Ref editor, bool added)
  {
    if (added)
    {
      editor->grtobj()->owner(_self);
      _self->queryBuffers().insert(editor->grtobj());
    }
    else
      _self->queryBuffers().remove_value(editor->grtobj());
  }

  void recordset_list_changed(Recordset::Ref rset, bool added)
  {
    if (added)
    {
      if (!rset->is_readonly())
        _self->resultsets().insert(grtwrap_editablerecordset(_self, rset));
      else
        _self->resultsets().insert(grtwrap_recordset(_self, rset));
    }
    else
    {
      for (size_t c= _self->resultsets().count(), i= 0; i < c; i++)
      {
        db_query_ResultsetRef e(_self->resultsets()[i]);
        
        if (e->get_data()->recordset == rset)
        {
          _self->resultsets().remove(i);
          break;
        }
      }
    }
  }
  
  
  virtual void refresh_editor(Sql_editor::Ref editor)
  {
    int oindex = _editor->active_sql_editor_index();
    
    for (size_t c= _self->queryBuffers().count(), i= 0; i < c; i++)
    {
      db_query_QueryBufferRef e(_self->queryBuffers()[i]);
      if (e == editor->grtobj())
      {
        _editor->active_sql_editor_index(i);
        _editor->do_partial_ui_refresh(Db_sql_editor::RefreshEditor);      
        break;
      }
    }
    _editor->active_sql_editor_index(oindex);
  }
  
public:
  db_query_EditorConcreteImplData(boost::shared_ptr<Db_sql_editor> editor,
                                  const db_query_EditorRef &self)
  : _editor(editor), _self(dynamic_cast<db_query_Editor*>(self.valueptr()))
  {
    for (size_t i= 0; i < (size_t) editor->sql_editor_count(); i++)
    {
      db_query_QueryBufferRef qb(editor->sql_editor(i)->grtobj());
      qb->owner(self);
      _self->queryBuffers().insert(qb);
    }
    
    editor->recordset_list_changed.connect(sigc::mem_fun(this, &db_query_EditorConcreteImplData::recordset_list_changed));
    editor->sql_editor_list_changed.connect(sigc::mem_fun(this, &db_query_EditorConcreteImplData::sql_editor_list_changed));
  }
  
  boost::shared_ptr<Db_sql_editor> editor_object() const { return _editor; }
  
  virtual db_mgmt_ConnectionRef connection() const
  {
    return _editor->connection_descriptor();
  }
  
  virtual db_query_QueryBufferRef addQueryBuffer()
  {
    _editor->new_sql_script_file();

    return _editor->sql_editor()->grtobj();
  }
  
  virtual grt::IntegerRef addToOutput(const std::string &text, long bringToFront)
  {
    _editor->output_text_slot(text, bringToFront != 0);
    
    return grt::IntegerRef(0);
  }
  
  virtual grt::ListRef<db_query_Resultset> executeScript(const std::string &sql)
  {
    _editor->grt_manager()->replace_status_text("Executing query...");

    std::vector<Recordset::Ref> rsets(_editor->exec_sql_returning_results(sql, true));
    
    grt::ListRef<db_query_Resultset> result(_self->get_grt());
    for (std::vector<Recordset::Ref>::const_iterator iter= rsets.begin(); iter != rsets.end(); ++iter)
      result.insert(grtwrap_recordset(_self, *iter));

    _editor->grt_manager()->replace_status_text("Query finished.");

    return result;
  }
  
  virtual grt::IntegerRef executeScriptAndOutputToGrid(const std::string &sql)
  {
    _editor->exec_sql_retaining_editor_contents(sql, true);

    return grt::IntegerRef(0);
  }
  
  virtual db_query_EditableResultsetRef createTableEditResultset(const std::string &schema, const std::string &table, const std::string &where, bool showGrid)
  {
    std::string query;
    
    query= base::strfmt("EDIT `%s`.`%s`", 
                       base::escape_backticks(schema).c_str(),
                       base::escape_backticks(table).c_str());
    //XXX WHERE statement not handled for now, we have to make sure it doesn't contain anything dangerous
    
    if (showGrid)
    {
      executeScriptAndOutputToGrid(query);
    }
    else
    {
      std::vector<Recordset::Ref> rsets(_editor->exec_sql_returning_results(query, true));
      
      if (rsets.size() == 1 && !rsets[0]->is_readonly())
        return grtwrap_editablerecordset(_self, rsets[0]);
    }
    return db_query_EditableResultsetRef();
  }
  
  virtual void activeSchema(const std::string &schema)
  {
    _editor->active_schema(schema);
  }
  
  virtual std::string activeSchema()
  {
    return _editor->active_schema();
  }
  
  virtual db_query_QueryBufferRef activeQueryBuffer()
  {
    Sql_editor::Ref editor(_editor->sql_editor());
    if (editor)
      return editor->grtobj();
    return db_query_QueryBufferRef();
  }
  
  virtual db_query_ResultsetRef activeResultset()
  {
    Recordset::Ref rset(_editor->active_recordset());
    
    if (rset)
    {
      for (size_t c= _self->resultsets().count(), i= 0; i < c; i++)
      {
        db_query_ResultsetRef e(_self->resultsets()[i]);

        if (e->get_data()->recordset == rset)
          return e;
      }
    }
    return db_query_ResultsetRef();
  }
    
protected:  
  db_query_Editor *_self;
  boost::shared_ptr<Db_sql_editor> _editor;
};


//------------------------------------------------------------------------------------------------

static void call_commit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->commit();
}

static bool validate_commit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query() && !db_sql_editor->auto_commit());
}


static void call_rollback(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->rollback();  
}

static bool validate_rollback(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query() && !db_sql_editor->auto_commit());
}


static void call_autocommit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->auto_commit(!db_sql_editor->auto_commit());
}

static wb::CommandItemValidationState validate_autocommit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return wb::ItemDisabled;
  // Note: meaning has been switched here. Auto commit on means the button is NOT checked.
  return db_sql_editor->auto_commit() ? wb::ItemEnabled : wb::ItemEnabledAndChecked;
}


static void call_export(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
  {
    RecordsetExportForm exporter(db_sql_editor);
    exporter.run();
  }
}

static bool validate_export(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && sqlide->get_active_sql_editor()->recordset_count());
}



static void call_cancel(wb::WBContextSQLIDE *sqlide)
{
  sqlide->get_active_sql_editor()->cancel_query();
}

static bool validate_cancel(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && db_sql_editor->is_running_query());
}


static void call_new_file(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *editor= sqlide->get_active_sql_editor();
  if (editor)
    editor->new_sql_script_file();
}

static bool validate_new_file(wb::WBContextSQLIDE *sqlide)
{
  return (NULL != sqlide->get_active_sql_editor());
}


static void call_open_file(wb::WBContextSQLIDE *sqlide)
{
  std::string path= sqlide->get_wbui()->get_wb()->show_file_dialog("open", _("Open SQL Script"), "SQL Files (*.sql)|*.sql");
  if (!path.empty())
    sqlide->open_document(path);
}

static bool validate_open_file(wb::WBContextSQLIDE *sqlide)
{
  return (NULL != sqlide->get_active_sql_editor());
}

static void call_save_file(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *editor = sqlide->get_active_sql_editor();
  if (editor)
  {
    int i = editor->active_sql_editor_index();
    if (i >= 0)
    {
      editor->save_sql_script_file(editor->sql_editor_path(i), i);
    }
  }
}

static void call_save_file_as(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *editor = sqlide->get_active_sql_editor();
  if (editor)
  {
    int i = editor->active_sql_editor_index();
    if (i >= 0)
    {
      editor->save_sql_script_file("", i);
    }
  }
}

static bool validate_save_file(wb::WBContextSQLIDE *sqlide)
{
  return (NULL != sqlide->get_active_sql_editor());
}

static void call_save_snippet(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
  {
    int start, end;
    std::string text = db_sql_editor->sql_editor()->sql();
    if (db_sql_editor->sql_editor()->selected_range(start, end))
      text = text.substr(start, end - start);
    if (db_sql_editor->save_snippet(text))
      sqlide->get_grt_manager()->replace_status_text("SQL saved to snippets list.");
  }
}


static bool validate_save_snippet(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  // don't check contents as the toolbar is not updated when these change atm
  return ((NULL != db_sql_editor)/* && !db_sql_editor->sql_editor()->sql().empty()*/);
}


static void call_continue_on_error(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->continue_on_error(!db_sql_editor->continue_on_error());
}

static wb::CommandItemValidationState validate_continue_on_error(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    return wb::ItemDisabled;
  return db_sql_editor->continue_on_error() ? wb::ItemEnabled : wb::ItemEnabledAndChecked;
}


static void call_recordsets_are_pinned_by_default(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->recordsets_are_pinned_by_default(!db_sql_editor->recordsets_are_pinned_by_default());
}

static wb::CommandItemValidationState validate_recordsets_are_pinned_by_default(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    return wb::ItemDisabled;
  return db_sql_editor->recordsets_are_pinned_by_default() ? wb::ItemEnabledAndChecked : wb::ItemEnabled;
}


static void call_explain_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->explain_sql(false);
}

static bool validate_explain_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query());
}


static void call_explain_current_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->explain_current_statement();
}

static bool validate_explain_current_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query());
}


static void call_reconnect(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor && !db_sql_editor->is_running_query())
    db_sql_editor->connect();
}

static bool validate_reconnect(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query());
}


static void call_refresh_physical_overview(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->refresh_physical_overview(false);
}

static bool validate_refresh_physical_overview(wb::WBContextSQLIDE *sqlide)
{
  return (NULL != sqlide->get_active_sql_editor());
}


static void call_new_tab(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
  {
    db_mgmt_ConnectionRef conn(db_sql_editor->connection_descriptor());
    sqlide->get_wbui()->get_wb()->add_new_query_window(conn);
  }
}

static bool validate_new_tab(wb::WBContextSQLIDE *sqlide)
{
  return (NULL != sqlide->get_active_sql_editor());
}


#ifdef _WIN32
static void call_exec_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->action_list().trigger_action("exec_sql");
}

static bool validate_exec_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query());
}


static void call_exec_current_sql_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->action_list().trigger_action("exec_current_sql_statement");
}

static bool validate_exec_current_sql_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  return (db_sql_editor && !db_sql_editor->is_running_query());
}


static void call_save_edits(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    db_sql_editor->action_list().trigger_action("save_active_recordset_edits");
}

static bool validate_save_edits(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    return false;
  Recordset::Ref rs= db_sql_editor->active_recordset();
  return (rs && rs->has_pending_changes());
}


static void call_discard_edits(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    db_sql_editor->action_list().trigger_action("discard_active_recordset_edits");
}

static bool validate_discard_edits(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor= sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    return false;
  Recordset::Ref rs= db_sql_editor->active_recordset();
  return (rs && rs->has_pending_changes());
}

#endif // _WIN32

//--------------------------------------------------------------------------------------------------

WBContextSQLIDE::WBContextSQLIDE(WBContextUI *wbui)
: _wbui(wbui)
{
}

//--------------------------------------------------------------------------------------------------

WBContextSQLIDE::~WBContextSQLIDE()
{
}

//--------------------------------------------------------------------------------------------------

bec::GRTManager *WBContextSQLIDE::get_grt_manager()
{
  return _wbui->get_wb()->get_grt_manager();
}


void WBContextSQLIDE::init()
{
  DbSqlEditorSnippets::setup(this, bec::make_path(get_grt_manager()->get_user_datadir(), "snippets"));
  
  // register the builtin module
  /*WbSqlIdeImpl *module= */_wbui->get_wb()->get_grt()->get_native_module<WbSqlIdeImpl>();
  _wbui->signal_main_form_change().connect(sigc::mem_fun(this, &WBContextSQLIDE::active_form_changed));
  
  // setup some builtin commands handled by ourselves for the SQL IDE
  _wbui->get_command_ui()->add_builtin_command("query.commit", sigc::bind(sigc::ptr_fun(call_commit), this), 
                                              sigc::bind(sigc::ptr_fun(validate_commit), this));
  _wbui->get_command_ui()->add_builtin_command("query.rollback", sigc::bind(sigc::ptr_fun(call_rollback), this), 
                                              sigc::bind(sigc::ptr_fun(validate_rollback), this));
  _wbui->get_command_ui()->add_builtin_toggle_command("query.autocommit",
                                               sigc::bind(sigc::ptr_fun(call_autocommit), this),
                                               sigc::bind(sigc::ptr_fun(validate_autocommit), this));

  _wbui->get_command_ui()->add_builtin_command("query.savesnippet", sigc::bind(sigc::ptr_fun(call_save_snippet), this), 
                                               sigc::bind(sigc::ptr_fun(validate_save_snippet), this));

  _wbui->get_command_ui()->add_builtin_command("query.newtab", sigc::bind(sigc::ptr_fun(call_new_tab), this), 
                                               sigc::bind(sigc::ptr_fun(validate_new_tab), this));

  _wbui->get_command_ui()->add_builtin_command("query.newFile", sigc::bind(sigc::ptr_fun(call_new_file), this), 
                                               sigc::bind(sigc::ptr_fun(validate_new_file), this));
  _wbui->get_command_ui()->add_builtin_command("query.openFile", sigc::bind(sigc::ptr_fun(call_open_file), this), 
                                               sigc::bind(sigc::ptr_fun(validate_open_file), this));
  _wbui->get_command_ui()->add_builtin_command("query.saveFile", sigc::bind(sigc::ptr_fun(call_save_file), this), 
                                               sigc::bind(sigc::ptr_fun(validate_save_file), this));
  _wbui->get_command_ui()->add_builtin_command("query.saveFileAs", sigc::bind(sigc::ptr_fun(call_save_file_as), this), 
                                               sigc::bind(sigc::ptr_fun(validate_save_file), this));

  _wbui->get_command_ui()->add_builtin_command("query.export", sigc::bind(sigc::ptr_fun(call_export), this), 
                                              sigc::bind(sigc::ptr_fun(validate_export), this));
  
  _wbui->get_command_ui()->add_builtin_command("query.cancel", sigc::bind(sigc::ptr_fun(call_cancel), this), 
                                               sigc::bind(sigc::ptr_fun(validate_cancel), this));

  _wbui->get_command_ui()->add_builtin_command("query.reconnect", sigc::bind(sigc::ptr_fun(call_reconnect), this), 
                                               sigc::bind(sigc::ptr_fun(validate_reconnect), this));

  _wbui->get_command_ui()->add_builtin_command("query.refresh_physical_overview", sigc::bind(sigc::ptr_fun(call_refresh_physical_overview), this), 
                                               sigc::bind(sigc::ptr_fun(validate_refresh_physical_overview), this));

  _wbui->get_command_ui()->add_builtin_toggle_command("query.stopOnError",
                                               sigc::bind(sigc::ptr_fun(call_continue_on_error), this),
                                               sigc::bind(sigc::ptr_fun(validate_continue_on_error), this));
  _wbui->get_command_ui()->add_builtin_toggle_command("query.recordsets_are_pinned_by_default",
                                               sigc::bind(sigc::ptr_fun(call_recordsets_are_pinned_by_default), this), 
                                               sigc::bind(sigc::ptr_fun(validate_recordsets_are_pinned_by_default), this));

  _wbui->get_command_ui()->add_builtin_command("query.explain",
                                               sigc::bind(sigc::ptr_fun(call_explain_sql), this),
                                               sigc::bind(sigc::ptr_fun(validate_explain_sql), this));
  _wbui->get_command_ui()->add_builtin_command("query.explain_current_statement",
                                               sigc::bind(sigc::ptr_fun(call_explain_current_statement), this),
                                               sigc::bind(sigc::ptr_fun(validate_explain_current_statement), this));

  // other commands declared in frontend code
#ifdef _WIN32
  _wbui->get_command_ui()->add_builtin_command("query.execute",
    sigc::bind(sigc::ptr_fun(call_exec_sql), this), sigc::bind(sigc::ptr_fun(validate_exec_sql), this));
  _wbui->get_command_ui()->add_builtin_command("query.execute_current_statement",
    sigc::bind(sigc::ptr_fun(call_exec_current_sql_statement), this), sigc::bind(sigc::ptr_fun(validate_exec_current_sql_statement), this));
  _wbui->get_command_ui()->add_builtin_command("query.save_edits",
    sigc::bind(sigc::ptr_fun(call_save_edits), this), sigc::bind(sigc::ptr_fun(validate_save_edits), this));
  _wbui->get_command_ui()->add_builtin_command("query.discard_edits",
    sigc::bind(sigc::ptr_fun(call_discard_edits), this), sigc::bind(sigc::ptr_fun(validate_discard_edits), this));
#endif // _WIN32
}


grt::ListRef<app_ToolbarItem> WBContextSQLIDE::get_toolbar_items(const std::string &name)
{
  WBContext *wb= _wbui->get_wb();
 
  if (name == "main")
    return app_ToolbarRef::cast_from(wb->get_grt()->unserialize(make_path(wb->get_datadir(), "data/dbquery_toolbar.xml")))->items();
  
  return grt::ListRef<app_ToolbarItem>();
}


Db_sql_editor::Ref WBContextSQLIDE::create_connected_editor(const db_mgmt_ConnectionRef &conn)
{
  Db_sql_editor::Ref editor(Db_sql_editor::create(this, conn));
  {
    // create entry for grt tree
    db_query_EditorRef object(_wbui->get_wb()->get_grt());
    object->owner(_wbui->get_wb()->get_root());
    object->name(conn->name());
    
    object->set_data(new db_query_EditorConcreteImplData(editor, object));

    _wbui->get_wb()->get_root()->sqlEditors().insert(object);
  }
    
  _open_editors.push_back(editor);

  return editor;
}


Db_sql_editor* WBContextSQLIDE::get_active_sql_editor()
{
  bec::UIForm *form= _wbui->get_active_main_form();
  if (form)
    return dynamic_cast<Db_sql_editor*>(form);
  return 0;
}


void WBContextSQLIDE::active_form_changed(bec::UIForm *form)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (editor)
  {
    // no need in following refresh any more, since sidebar is embedded into db sql editor form

    // tell frontend to update the sidebar
    //editor->do_partial_ui_refresh(Db_sql_editor::RefreshSidebar);
  }
}


bool WBContextSQLIDE::activate_live_object(GrtObjectRef object)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (!editor)
    return false;
  return editor->activate_live_object(object);
}


bool WBContextSQLIDE::create_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (!editor)
    return false;
  return editor->create_live_object(object_type, owner_name, obj_name);
}


bool WBContextSQLIDE::drop_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (!editor)
    return false;
  return editor->drop_live_object(object_type, owner_name, obj_name);
}

//--------------------------------------------------------------------------------------------------

void WBContextSQLIDE::open_document(const std::string &path)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (editor)
    editor->open_sql_script_file(path, false);
  else
    mforms::Utilities::show_error(_("Open SQL Script"),
                                  _("Please select a connected SQL Editor tab to open a script file."),
                                  _("OK"));
}

static bool compare(Db_sql_editor::Ptr ptr, Db_sql_editor *editor)
{
  return ptr.lock().get() == editor;
}


void WBContextSQLIDE::editor_will_close(Db_sql_editor* editor)
{
  std::list<Db_sql_editor::Ptr>::iterator iter = std::find_if(_open_editors.begin(), _open_editors.end(), 
                                                              sigc::bind(sigc::ptr_fun(compare), editor));
  if (iter != _open_editors.end())
  {
    // delete entry from grt tree
    grt::ListRef<db_query_Editor> editors(_wbui->get_wb()->get_root()->sqlEditors());
    
    for (size_t c= editors.count(), i= 0; i < c; i++)
    {
      if (dynamic_cast<db_query_EditorConcreteImplData*>(editors[i]->get_data())->editor_object().get() == editor)
      {
        editors.remove(i);
        break;
      }
    }

    _open_editors.erase(iter);
  }
}


bool WBContextSQLIDE::request_quit()
{  
  for (std::list<Db_sql_editor::Ptr>::iterator ed = _open_editors.begin(); ed != _open_editors.end(); ++ed)
  {
    if (!ed->expired() && !ed->lock()->can_close())
      return false;
  }
  return true;
}


void WBContextSQLIDE::update_plugin_arguments_pool(bec::ArgumentPool &args)
{
  Db_sql_editor *editor_ptr= get_active_sql_editor();
  if (editor_ptr)
  {
    db_query_EditorRef editor(get_grt_editor_object(editor_ptr));
    db_query_ResultsetRef rset(editor->activeResultset());

    args.add_entries_for_object("activeSQLEditor", editor);
    args.add_entries_for_object("activeQueryBuffer", editor->activeQueryBuffer());
    args.add_entries_for_object("", editor->activeQueryBuffer());

    if (rset.is_valid())
      args.add_entries_for_object("activeResultset", rset, "db.query.Resultset");
  }
}


db_query_EditorRef WBContextSQLIDE::get_grt_editor_object(Db_sql_editor *editor)
{
  if (editor)
  {
    grt::ListRef<db_query_Editor> list(_wbui->get_wb()->get_root()->sqlEditors());
    for (grt::ListRef<db_query_Editor>::const_iterator ed= list.begin(); ed != list.end(); ++ed)
    {
      if (dynamic_cast<db_query_EditorConcreteImplData*>((*ed)->get_data())->editor_object().get() == editor)
        return *ed;
    }
  }
  return db_query_EditorRef();
}


