/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "recordset_text_storage.h"
#include "recordset_be.h"
#include "string_utilities.h"

#include <sqlite/query.hpp>
#include <boost/foreach.hpp>
#include <fstream>
#include <memory>

#define WIN32 // required by ctemplate to compile on win

#include <ctemplate/template.h>

using namespace bec;
using namespace grt;
using namespace base;

using ctemplate::Template;
using ctemplate::TemplateDictionary;

typedef std::map<std::string, std::string> Templates;
Templates _templates; // data format name -> template name

#define APPEND(literal)  out->Emit("" literal "", sizeof(literal)-1)

// string escaper for CSV tokens, encloses fields with " if needed, depending on the separator
class CSVTokenQuote : public ctemplate::TemplateModifier {
  void Modify(const char* in, size_t inlen,
              const ctemplate::PerExpandData* per_expand_data,
              ctemplate::ExpandEmitter* out, const std::string& arg) const 
  {
    int ch;
    if (arg == "=comma")
      ch = ',';
    else if (arg == "=tab")
      ch = '\t';
    else if (arg == "=semicolon")
      ch = ';';
    else
      ch = ';';
    // check if quotation needed
    if (memchr(in, ch, inlen) || memchr(in, ' ', inlen) || memchr(in, '"', inlen)
        || memchr(in, '\t', inlen) || memchr(in, '\r', inlen) || memchr(in, '\n', inlen))
    {
      out->Emit(std::string("\""));
      for (size_t i = 0; i < inlen; ++i) {
        if (in[i] == '"') 
          APPEND("\"\"");
        else 
          out->Emit(in[i]);
      } 
      out->Emit(std::string("\""));
    }
    else
      out->Emit(std::string(in, inlen));
  }
};
CSVTokenQuote csv_quote;


Recordset_text_storage::Recordset_text_storage(GRTManager *grtm)
:
Recordset_data_storage(grtm)
{
  static bool registered_csvquote = false;
  if (!registered_csvquote)
  {
    registered_csvquote = true;
    ctemplate::AddModifier("x-csv_quote=", &csv_quote);
    //XXX ctemplate::AddModifier("x-sql_quote", &sql_quote);
  }
  
  {
    class Template_names_initializer
    {
    public:
      Template_names_initializer()
      {
        _templates["CSV"]= "CSV.tpl";
        _templates["CSV_semicolon"]= "CSV_semicolon.tpl";
        _templates["tab"]= "tab.tpl";
        _templates["HTML"]= "HTML.tpl";
        _templates["XML"]= "XML.tpl";
        _templates["SQL_inserts"]= "SQL_inserts.tpl";
      }
    };
    static Template_names_initializer template_names_initializer;
  }
}


Recordset_text_storage::~Recordset_text_storage()
{
}


ColumnId Recordset_text_storage::aux_column_count()
{
  throw std::runtime_error("Recordset_text_storage::aux_column_count is not implemented");
}


void Recordset_text_storage::do_apply_changes(const Recordset *recordset, sqlite::connection *data_swap_db)
{
  throw std::runtime_error("Recordset_text_storage::apply_changes is not implemented");
}


void Recordset_text_storage::do_serialize(const Recordset *recordset, sqlite::connection *data_swap_db)
{
  std::string template_name= this->template_name(_data_format);
  if (template_name.empty())
    throw std::runtime_error(strfmt("Unknown data format: %s", _data_format.c_str()));

  std::string tpl_path= _grtm->get_data_file_path("modules/data/sqlide/" + template_name);
  Template *tpl= Template::GetTemplate(tpl_path, ctemplate::DO_NOT_STRIP);
  if (!tpl)
  {
    Template::ClearCache();
    throw std::runtime_error(strfmt("Failed to open template file: `%s`", tpl_path.c_str()));
  }
  
  {
    if (!g_file_set_contents(_file_path.c_str(), "", 1, NULL))
      throw std::runtime_error(strfmt("Failed to open output file: `%s`", _file_path.c_str()));
  }

  tpl->ReloadIfChanged();

  std::auto_ptr<TemplateDictionary> dict(new TemplateDictionary("/"));
  BOOST_FOREACH (const Parameters::value_type &param, _parameters)
    dict->SetValue(param.first, param.second);

  const Recordset::Column_names *column_names= recordset->column_names();
  ColumnId visible_col_count= recordset->get_column_count();
  const sqlide::VarToStr *var_to_str= recordset->var2str_convertor();

  // misc subst variables
  dict->SetValue("IDENT", "\t");

  // headers
  for (ColumnId col= 0; col < visible_col_count; ++col)
  {
    TemplateDictionary* col_dict = dict->AddSectionDictionary("COLUMN");
    col_dict->SetValueWithoutCopy("COLUMN_NAME", (*column_names)[col]);
  }
  // data
  {
    const size_t partition_count= recordset->data_swap_db_partition_count();
    std::list<boost::shared_ptr<sqlite::query> > data_queries(partition_count);
    Recordset::prepare_partition_queries(data_swap_db, "select * from `data%s`", data_queries);
    std::vector<boost::shared_ptr<sqlite::result> > data_results(data_queries.size());
    if (Recordset::emit_partition_queries(data_swap_db, data_queries, data_results))
    {
      bool next_row_exists= true;
      sqlite::Variant v;
      do
      {
        TemplateDictionary* row_dict = dict->AddSectionDictionary("ROW");
        for (size_t partition= 0; partition < partition_count; ++partition)
        {
          boost::shared_ptr<sqlite::result> &data_rs= data_results[partition];
          for (ColumnId col_begin= partition * Recordset::DATA_SWAP_DB_TABLE_MAX_COL_COUNT, col= col_begin,
            col_end= std::min<ColumnId>(visible_col_count, (partition + 1) * Recordset::DATA_SWAP_DB_TABLE_MAX_COL_COUNT); col < col_end; ++col)
          {
            ColumnId partition_column= col - col_begin;
            data_rs->get_variant(partition_column, v);
            TemplateDictionary* field_dict = row_dict->AddSectionDictionary("FIELD");
            field_dict->SetValueWithoutCopy("FIELD_NAME", (*column_names)[col]);
            std::string field_value= boost::apply_visitor(*var_to_str, v);
            field_dict->SetValue("FIELD_VALUE", field_value);
            //XXX bool is_string= !is_var_null(v);
            //field_dict->SetIntValue("IS_STR", is_string);
          }
        }
        BOOST_FOREACH (boost::shared_ptr<sqlite::result> &data_rs, data_results)
          next_row_exists= data_rs->next_row();
      }
      while (next_row_exists);
    }
  }

  // expand tempalte & flush result
  {
    std::string result_text;
    tpl->Expand(&result_text, dict.get());
    GError *error = 0;

    // use g_file_set because it can handle utf8 filenames
    if (!g_file_set_contents(_file_path.c_str(), result_text.data(), result_text.size(), &error))
    {
      std::string message = error->message;
      g_free(error);
      throw std::runtime_error(strfmt("Failed to write to output file `%s`: %s", _file_path.c_str(), message.c_str()));
    }
  }
}


void Recordset_text_storage::do_unserialize(Recordset *recordset, sqlite::connection *data_swap_db)
{
  throw std::runtime_error("Recordset_text_storage::unserialize is not implemented");
}


void Recordset_text_storage::do_fetch_blob_value(Recordset *recordset, sqlite::connection *data_swap_db, RowId rowid, ColumnId column, sqlite::Variant &blob_value)
{
}


std::string Recordset_text_storage::template_name(const std::string &format) const
{
  Templates::const_iterator i= _templates.find(format);
  return (_templates.end() != i) ? i->second : std::string();
}


std::string Recordset_text_storage::parameter_value(const std::string &name) const
{
  Parameters::const_iterator i= _parameters.find(name);
  return (_parameters.end() != i) ? i->second : std::string();
}
