/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _STRING_UTILITIES_H_
#define _STRING_UTILITIES_H_

#include "common.h"
#include <string>
#include <vector>

#include "glib.h"

#define _(s) s // TODO: replace with localization code.

namespace base
{
  #define SPACES " \t\r\n"

  // Trimming, cleanup etc.
  BASELIBRARY_PUBLIC_FUNC std::string trim_right(const std::string& s, const std::string& t = SPACES);
  BASELIBRARY_PUBLIC_FUNC std::string trim_left(const std::string& s, const std::string& t = SPACES);
  BASELIBRARY_PUBLIC_FUNC std::string trim(const std::string& s, const std::string& t = SPACES);
  BASELIBRARY_PUBLIC_FUNC void tolower(std::string& s);
  BASELIBRARY_PUBLIC_FUNC std::string tolower(const std::string& s);
  BASELIBRARY_PUBLIC_FUNC void toupper(std::string& s);
  BASELIBRARY_PUBLIC_FUNC std::string toupper(const std::string& s);

  // Parsing/Formatting.
  BASELIBRARY_PUBLIC_FUNC std::string get_identifier(const std::string& id, std::string::const_iterator& start);
  BASELIBRARY_PUBLIC_FUNC std::vector<std::string> split_qualified_identifier(const std::string& id);
  BASELIBRARY_PUBLIC_FUNC std::string strfmt(const char *fmt, ...) G_GNUC_PRINTF(1, 2);
  BASELIBRARY_PUBLIC_FUNC std::string pop_path_front(std::string &path);
  BASELIBRARY_PUBLIC_FUNC std::string pop_path_back(std::string &path);
  BASELIBRARY_PUBLIC_FUNC std::string strip_text(const std::string &text, bool left= true, bool right= true);

  BASELIBRARY_PUBLIC_FUNC std::string normalize_path_extension(std::string filename, std::string extension);

  BASELIBRARY_PUBLIC_FUNC std::string escape_sql_string(const std::string &string); // "strings" or 'strings'
  BASELIBRARY_PUBLIC_FUNC std::string escape_backticks(const std::string &string);  // `identifier`

  // Searching, splitting etc.
  BASELIBRARY_PUBLIC_FUNC bool starts_with(const std::string& s, const std::string& part);
  BASELIBRARY_PUBLIC_FUNC void replace(std::string& value, const std::string& search, const std::string& replacement);

  // These functions will perform newline conversion depending on OS. in case of error an exception is thrown.
  BASELIBRARY_PUBLIC_FUNC void set_text_file_contents(const std::string &filename, const std::string &data);
  BASELIBRARY_PUBLIC_FUNC std::string get_text_file_contents(const std::string &filename);

#ifdef __APPLE__
#undef check
#endif

  class BASELIBRARY_PUBLIC_FUNC EolHelpers
  {
  public:
    enum Eol_format { eol_lf, eol_cr, eol_crlf };
    static Eol_format detect(const std::string &text); // detects eol format based on the first eol occurence, line endings consistency is implied
    static int count_lines(const std::string &text); // counts lines in the text, even if line endings are inconsistent
    static bool check(const std::string &text); // checks whether line endings are consistent (same throughout the text)
    static void conv(const std::string &src_text, Eol_format src_eol_format, std::string &dest_text, Eol_format dest_eol_format); // converts between 2 known eol formats, line endings consistency is implied
    static void fix(const std::string &src_text, std::string &dest_text, Eol_format eol_format); // aligns eol format to the specified

    static bool is_eol(const char *sym_ptr) // reliably determines whether the given string position contains last symbol of eol sequence, line ending inconsistency is allowed
    {
      switch (*sym_ptr)
      {
      case '\n': return true;
      case '\r': return ('\n' != *(sym_ptr+1));
      default: return false;
      }
    }

    static Eol_format default_eol_format() // platform default eol format
    {
#if defined(_WIN32)
      return eol_crlf;
#elif defined(__APPLE__)
      return eol_cr;
#else
      return eol_lf;
#endif
    }

    static const std::string & eol(Eol_format eol= default_eol_format()) // returns eol sequence by eol format
    {
      static std::string eol_crlf_seq= "\r\n";
      static std::string eol_cr_seq= "\r";
      static std::string eol_lf_seq= "\n";
      switch (eol)
      {
      case eol_crlf: return eol_crlf_seq;
      case eol_cr: return eol_cr_seq;
      default: return eol_lf_seq;
      }
    }
  };

} // namespace base

#endif // _STRING_UTILITIES_H_
