/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"
#include <windows.h>

#include "mdc_canvas_view_windows.h"
#include "mdc_canvas_view_image.h"
#include "mdc_rectangle.h"

#include <cairo-win32.h>

#ifndef GL_BGRA
#define GL_BGRA GL_BGRA_EXT
#endif

using namespace mdc;

//--------------------------------------------------------------------------------------------------

/**
 * Determines a proper pixel format needed for creating an OpenGL rendering context.
 */
void FindPixelFormatForDeviceContext(HDC context)
{
  PIXELFORMATDESCRIPTOR pfd = { 
    sizeof(PIXELFORMATDESCRIPTOR),    // size of this pfd 
    1,                                // version number 
    PFD_DRAW_TO_WINDOW |              // support window 
    PFD_SUPPORT_OPENGL |              // support OpenGL 
    // PFD_SUPPORT_GDI |                 // support GDI
    PFD_DOUBLEBUFFER,                 // double buffered 
    PFD_TYPE_RGBA,                    // RGBA type 
    32,                               // color with alpha
    0, 0, 0, 0, 0, 0,                 // no explicit color bits assignment
    0, 0,                             // no explicit alpha bits assignment
    0,                                // no accumulation buffer 
    0, 0, 0, 0,                       // accum bits ignored 
    0,                                // no z-buffer (depth testing disabled for 2D drawing)
    0,                                // no stencil buffer 
    0,                                // no auxiliary buffer 
    PFD_MAIN_PLANE,                   // main layer 
    0,                                // reserved 
    0, 0, 0                           // layer masks ignored 
  }; 
  int  iPixelFormat; 
   
  // Get the device context's best available pixel format match.
  iPixelFormat= ChoosePixelFormat(context, &pfd); 
   
  // Make that match the device context's current pixel format .
  SetPixelFormat(context, iPixelFormat, &pfd); 
}

//--------------------------------------------------------------------------------------------------

WindowsGLCanvasView::WindowsGLCanvasView(HWND window, int width, int height)
  : OpenGLCanvasView(width, height), _glrc(0), _window(window)
{
  // A surface used to get a cairo context outside of a paint cycle.
  _offline_surface= cairo_win32_surface_create_with_dib(CAIRO_FORMAT_RGB24, 1, 1);
}

//--------------------------------------------------------------------------------------------------

WindowsGLCanvasView::~WindowsGLCanvasView()
{
  wglDeleteContext(_glrc);
  cairo_surface_destroy(_offline_surface);
}

//--------------------------------------------------------------------------------------------------

bool WindowsGLCanvasView::initialize()
{
  // Find a proper pixel format.
  _hdc= GetDC(_window);
  FindPixelFormatForDeviceContext(_hdc);

  _glrc= wglCreateContext(_hdc);
  if (!_glrc)
    throw canvas_error("Could not initialize WGL context.");

  // The make_current() call will set a proper surface.
  // We need an active context now, as the base initializer will set a few default values.
  _cairo= new CairoCtx();
  make_current();

  bool result= OpenGLCanvasView::initialize();
  remove_current();
  ReleaseDC(_window, _hdc);

  return result;
}

//--------------------------------------------------------------------------------------------------

void WindowsGLCanvasView::make_current()
{
  _hdc= GetDC(_window);
  wglMakeCurrent(_hdc, _glrc);

  _crsurface= cairo_win32_surface_create(_hdc);
  _cairo->update_cairo_backend(_crsurface);
  cairo_set_tolerance(_cairo->get_cr(), 0.1);
}

//--------------------------------------------------------------------------------------------------

void WindowsGLCanvasView::remove_current()
{
  cairo_surface_destroy(_crsurface);
  _crsurface= NULL;
  _cairo->update_cairo_backend(_offline_surface);

  wglMakeCurrent(0, 0);
  ReleaseDC(_window, _hdc);
}

//--------------------------------------------------------------------------------------------------

void WindowsGLCanvasView::swap_buffers()
{
  SwapBuffers(_hdc);
}

//--------------------------------------------------------------------------------------------------

void WindowsGLCanvasView::update_view_size(int width, int height)
{
  if (_view_width != width || _view_height != height)
  {
    _view_width= width;
    _view_height= height;

    update_offsets();
    queue_repaint();
    _viewport_changed_signal.emit();
  }
}

//----------------- WindowsCanvasView --------------------------------------------------------------

WindowsCanvasView::WindowsCanvasView(int width, int height)
  : CanvasView(width, height)
{
  _hdc= 0;
  _crsurface= 0;

  // A surface used to get a cairo context outside of a paint cycle (usually for font measurement).
  _offline_surface= cairo_win32_surface_create_with_dib(CAIRO_FORMAT_RGB24, 1, 1);
  _cairo= new CairoCtx(_offline_surface);
}

//--------------------------------------------------------------------------------------------------

WindowsCanvasView::~WindowsCanvasView()
{
  delete _cairo;

  if (_offline_surface)
    cairo_surface_destroy(_offline_surface);

  if (_crsurface)
    cairo_surface_destroy(_crsurface);

}

//--------------------------------------------------------------------------------------------------

bool WindowsCanvasView::initialize()
{
  return CanvasView::initialize();
}

//--------------------------------------------------------------------------------------------------

void WindowsCanvasView::update_view_size(int width, int height)
{
  if (_view_width != width || _view_height != height)
  {
    _view_width= width;
    _view_height= height;

    update_offsets();
    queue_repaint();

    _viewport_changed_signal.emit();
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * For drawing we need the current GDI device context (which might change between calls).
 * As the base class does not allow to pass it in the repaint() function an additional one is needed
 * to set the context for the next paint cycle.
 */
void WindowsCanvasView::set_target_context(HDC hdc)
{
  _hdc= hdc;
}

//--------------------------------------------------------------------------------------------------

void WindowsCanvasView::begin_repaint(int x, int y, int w, int h)
{
  _crsurface= cairo_win32_surface_create(_hdc);
  _cairo->update_cairo_backend(_crsurface);
  cairo_set_tolerance(_cairo->get_cr(), 0.1);
}

//--------------------------------------------------------------------------------------------------

void WindowsCanvasView::end_repaint()
{
  _hdc= 0;

  cairo_surface_destroy(_crsurface);
  _crsurface = NULL;

  _cairo->update_cairo_backend(_offline_surface);
}

//--------------------------------------------------------------------------------------------------

std::string mdc::detect_opengl_version()
{
  std::string version;

  // Create a temporary window which can be used to determine the OpenGL support.
  // Using the desktop window seems to crash on certain boards (mostly Intel on-board graphics
  // currently).
  WNDCLASSEX window_class;
  ATOM atom;

  window_class.cbSize = sizeof(window_class);
  window_class.style = 0;
  window_class.lpfnWndProc = DefWindowProc;
  window_class.cbClsExtra = 0;
  window_class.cbWndExtra = 0;
  window_class.hInstance = GetModuleHandle(NULL);
  window_class.hIcon = NULL;
  window_class.hCursor = LoadCursor (NULL, IDC_ARROW);
  window_class.hbrBackground = NULL;
  window_class.lpszMenuName = NULL;
  window_class.lpszClassName = L"temp-gl-window-class";
  window_class.hIconSm = NULL;

  atom = RegisterClassEx(&window_class);
  if (atom == 0)
    return "";

  HWND temp_window = CreateWindowEx(CS_OWNDC, (LPCTSTR) (DWORD) atom, L"temp-gl-window", WS_OVERLAPPEDWINDOW,
    CW_USEDEFAULT, CW_USEDEFAULT, 100, 100, GetDesktopWindow(), NULL, GetModuleHandle(NULL), NULL);

  HDC context= GetDC(temp_window);
  FindPixelFormatForDeviceContext(context);
  HGLRC ogl_context= wglCreateContext(context);
  wglMakeCurrent(context, ogl_context);
  const GLubyte* temp= glGetString(GL_VERSION);
  if (temp != NULL)
    version= (const char*) temp;

  wglMakeCurrent(0, 0);
  wglDeleteContext(ogl_context);
  ReleaseDC(temp_window, context);
  DestroyWindow(temp_window);

  return version;
}

//--------------------------------------------------------------------------------------------------

