/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA 02110-1301  USA
 */
#ifndef _MFORMS_TEXTBOX_H_
#define _MFORMS_TEXTBOX_H_

#include <mforms/view.h>

#include <sigc++/sigc++.h>

namespace mforms {
  enum ScrollBars {
    NoScrollBar= 0,      //!< Don't show any scroll bar.
    HorizontalScrollBar, //!< Show only the horizontal scroll bar.
    VerticalScrollBar,   //!< Show only the vertical scroll bar.
    BothScrollBars       //!< Show both scroll bars.
  };

  class TextBox;

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#ifndef SWIG
  struct TextBoxImplPtrs
  {
    bool (__stdcall *create)(TextBox *self, ScrollBars scroll_bars);
    void (__stdcall *set_bordered)(TextBox *self, bool flag);
    void (__stdcall *set_read_only)(TextBox *self, bool flag);
    void (__stdcall *set_text)(TextBox *self, const std::string &text);
    std::string (__stdcall *get_text)(TextBox *self);
    void (__stdcall *append_text)(TextBox *self, const std::string &text, bool scroll_to_end);
    void (__stdcall *set_padding)(TextBox *self, int pad);
    void (__stdcall *set_monospaced)(TextBox *self, bool flag);
    void (__stdcall *get_selected_range)(TextBox *self, int &start, int &end);
    void (__stdcall *clear)(TextBox *self);
  };
#endif
#endif

  /** Multiline text editing control */
  class MFORMS_EXPORT TextBox : public View
  {
#ifdef SWIG
// workaround for problem with unicode strings and default args in swig
%rename(append_text_and_scroll) append_text(const std::string &text, bool scroll_to_end);
%rename(append_text) append_text(const std::string &text);

%rename(append_text_with_encoding_and_scroll) append_text_with_encoding(const std::string &text, const std::string &encoding, bool scroll_to_end);
%rename(append_text_with_encoding) append_text_with_encoding(const std::string &text, const std::string &encoding);
#endif
  public:
    TextBox(ScrollBars scroll_bars);

    /** Sets whether to draw a border around the text box */
    void set_bordered(bool flag);
    /** Sets whether text should be displayed in a monospaced font */
    void set_monospaced(bool flag);
    
    /** Sets whether the content of the text box is editable by the user */
    void set_read_only(bool flag);
    /** Sets the text contents of the control */
    void set_value(const std::string &text);
    /** Gets the text contents of the control */
    virtual std::string get_string_value();
    
    /** Gets range of selected text */
    void get_selected_range(int &start, int &end);
    
    /** Appends text to the end of the text box. 
     
     If scroll_to_end is true, it will also scroll to make the end of the text box visible. */
    void append_text(const std::string &text, bool scroll_to_end= false);
    
    /** Appends text to the end of the text box, in the given character set encoding.
     
     If scroll_to_end is true, it will also scroll to make the end of the text box visible. */
    void append_text_with_encoding(const std::string &text, const std::string &encoding, bool scroll_to_end= false);
   
    /** Sets padding */
    void set_padding(int pad);
    
    /** Clear contents */
    void clear();

#ifndef SWIG
    /** Signal emitted when contents are edited */
    sigc::signal<void> signal_changed() { return _signal_changed; }
#ifndef DOXYGEN_SHOULD_SKIP_THIS
  public:
    void callback();
#endif
#endif
  protected:
    TextBoxImplPtrs *_textbox_impl;
    bool _updating;
    
    sigc::signal<void> _signal_changed;
  };
};

#endif
