/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */


/**
 * Utilities is a support class to trigger message boxes and the like in the front end.
 */

#include "stdafx.h"

#include <mforms/utilities.h>
#include <mforms/mforms.h>
#include "string_utilities.h"

#include <math.h>

using namespace mforms;

GThread *_mforms_main_thread=0;

//--------------------------------------------------------------------------------------------------

int Utilities::show_message(const std::string &title, const std::string &text,
                            const std::string &ok, const std::string &cancel,
                            const std::string &other)
{
  return ControlFactory::get_instance()->_utilities_impl.show_message(title, text, ok, cancel, other);
}

//--------------------------------------------------------------------------------------------------

int Utilities::show_error(const std::string &title, const std::string &text,
                          const std::string &ok, const std::string &cancel,
                          const std::string &other)
{
  return ControlFactory::get_instance()->_utilities_impl.show_error(title, text, ok, cancel, other);
}

//--------------------------------------------------------------------------------------------------

int Utilities::show_warning(const std::string &title, const std::string &text,
                            const std::string &ok, const std::string &cancel,
                            const std::string &other)
{
  return ControlFactory::get_instance()->_utilities_impl.show_warning(title, text, ok, cancel, other);
}

//--------------------------------------------------------------------------------------------------

void Utilities::show_wait_message(const std::string &title, const std::string &text)
{
  ControlFactory::get_instance()->_utilities_impl.show_wait_message(title, text);
}

//--------------------------------------------------------------------------------------------------

bool Utilities::hide_wait_message()
{
  return ControlFactory::get_instance()->_utilities_impl.hide_wait_message();
}

//--------------------------------------------------------------------------------------------------

void Utilities::set_clipboard_text(const std::string &text)
{
  ControlFactory::get_instance()->_utilities_impl.set_clipboard_text(text);
}

//--------------------------------------------------------------------------------------------------

std::string Utilities::get_clipboard_text()
{
  return ControlFactory::get_instance()->_utilities_impl.get_clipboard_text();
}

//--------------------------------------------------------------------------------------------------

std::string Utilities::get_special_folder(FolderType type)
{
  return ControlFactory::get_instance()->_utilities_impl.get_special_folder(type);
}

//--------------------------------------------------------------------------------------------------

void Utilities::open_url(const std::string &url)
{
  return ControlFactory::get_instance()->_utilities_impl.open_url(url);
}

//--------------------------------------------------------------------------------------------------

void Utilities::add_timeout(float interval, const sigc::slot<bool> &callback)
{
  ControlFactory::get_instance()->_utilities_impl.add_timeout(interval, callback);
}

//--------------------------------------------------------------------------------------------------

void Utilities::add_end_ok_cancel_buttons(mforms::Box *box, mforms::Button *ok, mforms::Button *cancel)
{
#ifdef __APPLE__
  box->add_end(ok, false, true);
  box->add_end(cancel, false, true);
#else
  box->add_end(cancel, false, true);
  box->add_end(ok, false, true);
#endif
}

//--------------------------------------------------------------------------------------------------

bool Utilities::request_input(const std::string &title, const std::string &description,
                              const std::string &default_value, std::string &ret_value)
{
  mforms::Form input_form(NULL, (FormFlag) (FormDialogFrame | FormStayOnTop));
  mforms::Table content;
  mforms::ImageBox icon;
  mforms::Label description_label("");
  mforms::TextEntry edit;
  mforms::Box button_box(true);
  mforms::Button ok_button;
  mforms::Button cancel_button;

  input_form.set_title(title.empty() ? _("Enter a value") : title);

  content.set_padding(12);
  content.set_row_count(2);
  content.set_row_spacing(10);
  content.set_column_count(3);
  content.set_column_spacing(4);

  icon.set_image("message_edit.png");
  content.add(&icon, 0, 1, 0, 2, HFillFlag | VFillFlag);

  description_label.set_text(description);
  description_label.set_style(BoldStyle);

  edit.set_size(150, -1);
  edit.set_value(default_value);
  content.add(&description_label, 1, 2, 0, 1, HFillFlag | VFillFlag);
  content.add(&edit, 2, 3, 0, 1, HFillFlag | VFillFlag);

  button_box.set_spacing(10);
  ok_button.set_text(_("OK"));
  ok_button.set_size(75, -1);
  cancel_button.set_text(_("Cancel"));
  cancel_button.set_size(75, -1);
  Utilities::add_end_ok_cancel_buttons(&button_box, &ok_button, &cancel_button);
  content.add(&button_box, 1, 3, 1, 2, HFillFlag);

  input_form.set_content(&content);
  input_form.center();
  bool result= input_form.run_modal(&ok_button, &cancel_button);
  if (result)
    ret_value = edit.get_string_value();

  return result;  
}

//--------------------------------------------------------------------------------------------------

void Utilities::store_password(const std::string &service, const std::string &account, const std::string &password)
{
  ControlFactory::get_instance()->_utilities_impl.store_password(service, account, password);
}

//--------------------------------------------------------------------------------------------------

bool Utilities::find_password(const std::string &service, const std::string &account, std::string &password)
{
  return ControlFactory::get_instance()->_utilities_impl.find_password(service, account, password);
}

//--------------------------------------------------------------------------------------------------

void Utilities::forget_password(const std::string &service, const std::string &account)
{
  ControlFactory::get_instance()->_utilities_impl.forget_password(service, account);
}

//--------------------------------------------------------------------------------------------------

static bool _ask_for_password(const std::string &title, const std::string &service, std::string &username /*in/out*/,
                             bool prompt_storage, std::string &ret_password /*out*/, bool &ret_store /*out*/)
{
  mforms::Form password_form(NULL, (FormFlag) (FormDialogFrame | FormStayOnTop));
  mforms::Table content;
  mforms::ImageBox icon;
  mforms::Label title_label("");
  mforms::Label service_description_label("");
  mforms::Label service_label("");
  mforms::Label user_description_label("");
  mforms::Label pw_description_label("");
  mforms::TextEntry password_edit(PasswordEntry);
  mforms::CheckBox save_password_box;
  mforms::Box button_box(true);
  mforms::Button ok_button;
  mforms::Button cancel_button;
  
  password_form.set_title(title.empty() ? _("MySQL Workbench Authentication") : title);
  
  content.set_padding(12);
  content.set_row_count(6);
  content.set_row_spacing(prompt_storage ? 8 : 7);
  content.set_column_count(3);
  content.set_column_spacing(4);
  
  icon.set_image("message_wb_lock.png");
  content.add(&icon, 0, 1, 0, 6, HFillFlag | VFillFlag);
  
  title_label.set_text(_("Please enter password for the following service:"));
  title_label.set_wrap_text(true);
  title_label.set_style(BigBoldStyle);
  title_label.set_size(250, -1);
  content.add(&title_label, 1, 3, 0, 1, HFillFlag | HExpandFlag | VFillFlag);
  
  service_description_label.set_text(_("Service:"));
  service_description_label.set_text_align(MiddleRight);
  service_description_label.set_style(BoldStyle);
  service_label.set_text(service);
  content.add(&service_description_label, 1, 2, 1, 2, HFillFlag | VFillFlag);
  content.add(&service_label, 2, 3, 1, 2, HFillFlag | VFillFlag);
  
  user_description_label.set_text(_("User:"));
  user_description_label.set_text_align(MiddleRight);
  user_description_label.set_style(BoldStyle);

  // Create an edit box for the user name if the given one is not set, otherwise just display the name.
  mforms::TextEntry* user_edit = NULL;
  if (username.empty())
  {
    user_edit = mforms::manage(new mforms::TextEntry());
    user_edit->set_value(_("<user name>"));
    content.add(&user_description_label, 1, 2, 2, 3, HFillFlag | VFillFlag);
    content.add(user_edit, 2, 3, 2, 3, HFillFlag | VFillFlag);
  }
  else
  {
    mforms::Label* user_label = mforms::manage(new mforms::Label(username));
    content.add(&user_description_label, 1, 2, 2, 3, HFillFlag | VFillFlag);
    content.add(user_label, 2, 3, 2, 3, HFillFlag | VFillFlag);
  }
  
  pw_description_label.set_text(_("Password:"));
  pw_description_label.set_text_align(MiddleRight);
  pw_description_label.set_style(BoldStyle);
  content.add(&pw_description_label, 1, 2, 3, 4, HFillFlag | VFillFlag);
  content.add(&password_edit, 2, 3, 3, 4, HFillFlag | HExpandFlag);
  
  if (prompt_storage)
  {
#ifdef _WIN32
    save_password_box.set_text(_("Save password in vault"));
#else
    save_password_box.set_text(_("Save password in keychain"));
#endif
    content.add(&save_password_box, 2, 3, 4, 5, HExpandFlag | HFillFlag);
  }

  button_box.set_spacing(10);
  ok_button.set_text(_("OK"));
  ok_button.set_size(75, -1);
  cancel_button.set_text(_("Cancel"));
  cancel_button.set_size(75, -1);
  Utilities::add_end_ok_cancel_buttons(&button_box, &ok_button, &cancel_button);
  if (prompt_storage)
    content.add(&button_box, 1, 3, 5, 6, HFillFlag);
  else
    content.add(&button_box, 1, 3, 4, 5, HFillFlag);
  
  password_form.set_content(&content);
  password_form.center();
  bool result= password_form.run_modal(&ok_button, &cancel_button);
  if (result)
  {
    ret_password= password_edit.get_string_value();
    ret_store= save_password_box.get_active();

    if (user_edit != NULL)
      username = user_edit->get_string_value();
  }
  
  return result;  
}

//--------------------------------------------------------------------------------------------------

bool Utilities::ask_for_password(const std::string &title, const std::string &service,
                                 std::string &username /*in/out*/, std::string &ret_password /*out*/)
{
  bool dummy= false;
  return _ask_for_password(title, service, username, false, ret_password, dummy);
}

//--------------------------------------------------------------------------------------------------

/**
 * Shows a dialog to request the password for the given service and user name.
 *
 * @param title Optional title describing the reason for the password request (eg. "Connect to MySQL Server")
 * @param service A description for what the password is required (e.g. "MySQL Server 5.1 on Thunder").
 * @param username [in/out] The name of the user for which to request the password.
 * @param password [out] Contains the password on return.
 *
 * @return True if the user pressed OK, otherwise false.
 */
bool Utilities::ask_for_password_check_store(const std::string &title, const std::string &service, 
                                             std::string &username, std::string &password, bool &store)
{
  return _ask_for_password(title, service, username, true, password, store);
}

//--------------------------------------------------------------------------------------------------

/**
 * Shows a dialog to request the password for the given service and user name.
 * If requested by the user the password will be saved using either system facilities like OS X Keychain or
 * Gnome Keyring, or an encrypted file.
 *
 * @param title Optional title describing the reason for the password request (eg. "Connect to MySQL Server")
 * @param service A description for what the password is required (e.g. "MySQL Server 5.1 on Thunder").
 * @param username The name of the user for which to request the password. If empty on enter then the user name can
 *        also be edited.
 * @param reset_password Delete stored password and ask for a new one.
 * @param password [out] Contains the password on return.
 *
 * @return True if the user pressed OK, otherwise false.
 */
bool Utilities::credentials_for_service(const std::string &title, const std::string &service, std::string &username /*in/out*/,
                                         bool reset_password, std::string &password /*out*/)
{
  if (!reset_password && find_password(service, username, password))
    return true;

  if (reset_password)
    forget_password(service, username);  

  bool should_store_password_out = false;
  if (ask_for_password_check_store(title, service, username, password, should_store_password_out))
  {
    if (should_store_password_out)
    {
      try
      {
        store_password(service, username, password);
      }
      catch (std::exception &exc)
      {
        show_warning(title.empty() ? _("Error Storing Password") : title, 
                     std::string("There was an error storing the password:\n")+exc.what(), "OK");
      }
    }
    return true;
  }
  return false;
}


//--------------------------------------------------------------------------------------------------

#ifdef _WIN32

static int modal_loops = 0;

void Utilities::enter_modal_loop()
{
  modal_loops++;
}


void Utilities::leave_modal_loop()
{
  modal_loops--;
}


bool Utilities::in_modal_loop()
{
  return modal_loops > 0;
}

#endif

//--------------------------------------------------------------------------------------------------

/**
 * Helper function to simplify icon loading. Returns NULL if the icon could not be found or
 * something wrong happened while loading.
 */
cairo_surface_t* Utilities::load_icon(const std::string& name)
{
  if (name.empty())
    return NULL;

  std::string icon_path= App::get()->get_resource_path(name);
  cairo_surface_t* result= cairo_image_surface_create_from_png(icon_path.c_str());
  if (result && cairo_surface_status(result) != CAIRO_STATUS_SUCCESS)
  {
    cairo_surface_destroy(result);
    result= NULL;
  }

  return result;
}

//--------------------------------------------------------------------------------------------------

/**
 * Shortens the string so that it fits into the given width. If there is already enough room then
 * the input is simply returned. Otherwise letters are removed (via binary search) and ellipses
 * are added so that the entire result fits into that width.
 */
std::string Utilities::shorten_string(cairo_t* cr, const std::string& text, double width)
{
  int ellipsis_width= 0;
  int length;
  int l, h, n, w;
  cairo_text_extents_t extents;
  
  // If the text fits already, return the input.
  cairo_text_extents(cr, text.c_str(), &extents);
  if (extents.width <= width)
    return text;
  
  length= text.size();
  if (length == 0 || width <= 0)
    return "";
  else
  {
    cairo_text_extents(cr, "...", &extents);
    ellipsis_width= (int) ceil(extents.width);
  }
  
  const gchar* head= text.c_str();
  if (width <= ellipsis_width)
    return "";
  else
  {
    // Do a binary search for the optimal string length which fits into the given width.
    l= 0;
    h= length - 1;
    while (l < h)
    {
      n= (l + h) / 2;

      // Skip to the nth position, which needs the following loop as we don't have direct
      // access to a char in an utf-8 buffer (one of the limitations of that transformation format).
      const gchar* tail= head;
      for (int i= 0; i < n; i++)
        tail= g_utf8_next_char(tail);
      gchar* part= g_strndup(head, tail - head);
      cairo_text_extents(cr, part, &extents);
      g_free(part);
      w= (int) ceil(extents.width) + ellipsis_width;
      if (w <= width)
        l= n + 1;
      else
        h= n;
    }
    return text.substr(0, l - 1) + "...";
  }
  
  return "";
}

//--------------------------------------------------------------------------------------------------

bool Utilities::in_main_thread()
{
  return g_thread_self() == _mforms_main_thread;
}

